﻿// $Id: DisparityEstimatorBase.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Reflection;
using Stereo.GpGpuLib;
using VideoLib.Parameters;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// The public base class for disparity estimators.
	/// </summary>
	public abstract class DisparityEstimatorBase : IDisposable
	{
		internal readonly List<StereoNodeFactory>[] factories;
		internal readonly RegisteredCollection<IStereoNode> activeNodes;
		internal readonly List<TechniqueDescription>[] techniqueDescriptions;

		public delegate void FrameDebugHandler(byte[] imageL, byte[] imageR, int imWidth, int imHeight, float[] costs, int width, int height, int depth);
		public event FrameDebugHandler NextFrameDebug;

		public DisparityEstimatorBase()
		{
			// Create the set of lists about the factories
			factories = new List<StereoNodeFactory>[TechniqueDescription.NTechniqueTypes];
			for (int i = 0; i < TechniqueDescription.NTechniqueTypes; ++i)
				factories[i] = new List<StereoNodeFactory>();

			// Find factories capable of creating StereoNodes
			FindFactories();

			// Create the space for technique descriptions
			techniqueDescriptions = new List<TechniqueDescription>[TechniqueDescription.NTechniqueTypes];
			for (int n = 0; n < techniqueDescriptions.Length; ++n)
			{
				List<TechniqueDescription> techniqueList = techniqueDescriptions[n] = new List<TechniqueDescription>();
				TechniqueType techniqueType = (TechniqueType)n;

				// Add 'none' as a viable option to all technique lists
				//techniqueList.Add(new TechniqueDescription("None", "", null, techniqueType));

				// Add a technique description for each factory to the list
				List<StereoNodeFactory> factoryList = factories[n];
				for (int i = 0; i < factoryList.Count; ++i)
					techniqueList.Add(new TechniqueDescription(factoryList[i].Name, factoryList[i].GetType().Name, factoryList[i], techniqueType));
			}

			// Create space for a list of currently active nodes
			activeNodes = new RegisteredCollection<IStereoNode>();

			// Perform an initial check that the factories loaded are valid
			ValidateFactories();
		}

		#region IDisposable (and associated) Members
		/// <summary>
		/// When overridden in derived classes: releases unmanaged resources associated with the object
		/// </summary>
		public abstract void Dispose();

		/// <summary>
		/// Releases unmanaged resources associated with an object, and reassigns the reference passed to null
		/// </summary>
		/// <typeparam name="T">The type of IStereoNode to destroy</typeparam>
		/// <param name="node">The node to destroy</param>
		protected void DisposeObject<T>(ref T node) where T : IStereoNode
		{
			activeNodes.Remove(node);
			if (node is IDisposable)
				(node as IDisposable).Dispose();
			node = default(T);
		}
		#endregion

		// Gets a list of all factories defined in the current assembly (VideoLib.Stereo.GpGpu) - using reflection
		private void FindFactories()
		{
			Assembly currentAssembly = Assembly.GetExecutingAssembly();
			Type[] assemblyTypes = currentAssembly.GetTypes();

			// Find all factory types in this assembly
			foreach (Type t in assemblyTypes)
				if (!t.IsAbstract && t.IsSubclassOf(typeof(StereoNodeFactory)))
				{
					// Travel up the inheritance hierarchy until the required generic base type is found
					Type baseT = t.BaseType;
					while (!baseT.IsGenericType || baseT.GetGenericTypeDefinition() != typeof(StereoNodeFactory<>))
					{
						baseT = baseT.BaseType;
						if (baseT == typeof(StereoNodeFactory))
							throw new NotSupportedException(
								"Stereo node factories should derive from StereoNodeFactory<T>, and not from StereoNodeFactory directly");
					}

					// Get the type parameter of StereoNodeFactory<>
					Type genericTypeParam = baseT.GetGenericArguments()[0];

					// Add the factory to the relevant list
					for (int i = 0; i < TechniqueDescription.NTechniqueTypes; ++i)
						if (genericTypeParam == TechniqueDescription.TechniqueNodeTypes[i])
							factories[i].Add((StereoNodeFactory)t.GetConstructor(Type.EmptyTypes).Invoke(null));
				}

			// Sort each list of factories in alphabetical order (so that the technique descriptions will also be alphabetical)
			for (int i = 0; i < factories.Length; ++i)
				factories[i].Sort(new Comparison<StereoNodeFactory>((x, y) => x.SortKey.CompareTo(y.SortKey)));
		}

		// Ensures all factories are compatible with the current settings (i.e. mainly the cost node)
		protected void ValidateFactories()
		{
			InputImageType inputType;
			CostSpaceType costType;
			DepthMapType depthType;

			GetCurrentConstraints(out inputType, out costType, out depthType);

			for (int n = 0; n < factories.Length; ++n)
			{
				TechniqueType type = (TechniqueType)n;
				factories[n].ForEach(x => x.CheckValid(inputType, costType, depthType));

				// Set technique validity for each description (note the -1 offset, as "none" is an option)
				List<StereoNodeFactory> factoryList = factories[n];
				List<TechniqueDescription> techniqueList = techniqueDescriptions[n];
				for (int i = 0; i < techniqueList.Count; ++i)
					techniqueList[i].IsValid = factoryList[i].IsValid;
			}
		}

		// Allows derived classes to provide frame debug information to a once-only event handler
		protected void OnNextFrameDebug(byte[] imageL, byte[] imageR, int imWidth, int imHeight, CostSpace costSpace, int width, int height, int depth)
		{
			if (NextFrameDebug != null)
			{
				float[] costs;
				costSpace.CopyDataOut(out costs);
				NextFrameDebug(imageL, imageR, imWidth, imHeight, costs, width, height, depth);
				NextFrameDebug = null;
			}
		}

		/// <summary>
		/// Gets a list of currently active 'parameter target' objects (stereo nodes)
		/// </summary>
		public RegisteredCollection<IStereoNode> ActiveParamObjects
		{
			get { return activeNodes; }
		}

		/// <summary>
		/// Enumerates all of the factories that are 'valid', and instantiate nodes implementing the given technique type
		/// </summary>
		/// <param name="type">The type of technique to list descriptions of</param>
		/// <returns>A list of 'valid' factories that produce techniques of type '<paramref>type</paramref>' </returns>
		public List<TechniqueDescription> GetTechniques(TechniqueType type)
		{
			return techniqueDescriptions[(int)type];
		}

		// Gets the current data type constraints on techniques (which determines which techniques are valid)
		protected abstract void GetCurrentConstraints(out InputImageType inputType, out CostSpaceType costType, out DepthMapType depthType);

		#region Abstract public interface

		/// <summary>
		/// Instantiates and applies the technique given by the technique description
		/// </summary>
		/// <param name="description">A description of the technique to use (returned from <see cref="GetTechniques"/>)</param>
		public abstract void SetTechnique(TechniqueDescription description);

		/// <summary>
		/// Fills an image with a greyscale depth-disparity map computed from the two input images
		/// </summary>
		/// <param name="inImageL">The left view (laid out without padding, width-first), internally formatted as an array of uints</param>
		/// <param name="inImageR">The right view (laid out without padding, width-first), internally formatted as an array of uints</param>
		/// <param name="width">The width of both of the input images</param>
		/// <param name="height">The height of both of the input images</param>
		/// <param name="outImage">
		/// The estimated depth map (should be created beforehand, with the same dimensions as the two input images), which is internally structured
		/// as a single-precision floating point array.
		/// </param>
		public abstract void ProcessFrame(byte[] inImageL, byte[] inImageR, int width, int height, byte[] outImage);

		#endregion

		#region Static factory methods
		/// <summary>
		/// Creates an instance of DisparityEstimatorBase, which implements 'standard' stereo image processing 
		/// (without occlusion checking)
		/// </summary>
		/// <returns>An instance of <see cref="DisparityEstimator"/></returns>
		public static DisparityEstimatorBase CreateStandard()
		{
			return new DisparityEstimator();
		}
		#endregion
	}
}
