/*
    Title:      ExternalSolverV2.java
    Authors:    Emina Torlek, MIT
                Jasmin Blanchette, TU Muenchen
    License:    See COPYRIGHT for details.
*/
/*
 * This file was adapted from ExternalSolver.java distributed with Kodkod
 * (2008-11-25 version) by Jasmin Christian Blanchette. The original
 * copyright notice is reproduced below. The changes to the original file
 * are Copyright 2009 Technische Universitaet Muenchen, like the rest of the
 * Kodkodi software.
 */    
/*    
 * Kodkod -- Copyright (c) 2005-2007, Emina Torlak
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package isabelle.kodkodi;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.nio.file.Files;
import java.nio.charset.Charset;
import java.util.BitSet;
import kodkod.engine.satlab.SATAbortedException;
import kodkod.engine.satlab.SATFactory;
import kodkod.engine.satlab.SATSolver;    

/**
 * An implementation of a wrapper for an external SAT solver,    
 * executed in a separate process.
 * @author Emina Torlak
 */
final class ExternalSolverV2 implements SATSolver {
    private final Charset UTF8 = Charset.forName("UTF-8");
    private final StringBuilder buffer;
    private final String executable;
    private final boolean outTemp;
    private final String[] options;
    private final BitSet solution;
    private volatile Boolean sat;
    private volatile int vars, clauses;
    private String satTag, solutionTag, unsatTag;    
    /**
     * Constructs an ExternalSolverV2 that will execute the specified binary
     * with the given options on a tempory input file: it will all clauses
     * added to this solver via the {@link #addClause(int[])} method.
     * If outTemp is false, the solver is assumed to write its output
     * to standard out; otherwise, it is assumed to write its output to a
     * temporary file given as second argument.
     *
     * satTag, solutionTag, and unsatTag specify the file format.
     */
    ExternalSolverV2(String executable, boolean outTemp,
                     String satTag, String solutionTag, String unsatTag,
                     String... options)
    {
        this.buffer = new StringBuilder(65536);
        this.sat = null;
        this.solution = new BitSet();
        this.vars = 0;
        this.clauses = 0;
        this.executable = executable;
        this.options = options;
        this.outTemp = outTemp;
        this.satTag = satTag;
        this.solutionTag = solutionTag;
        this.unsatTag = unsatTag;
    }

    /**
     * {@inheritDoc}
     * @see kodkod.engine.satlab.SATSolver#addClause(int[])
     */
    public boolean addClause(int[] lits) {
        if (lits.length>0) {
            clauses++;
            for(int lit: lits) {
                buffer.append(lit);
                buffer.append(" ");
            }
            buffer.append("0\n");
            return true;
        }
        return false;
    }    
    /**
     * @see kodkod.engine.satlab.SATSolver#addVariables(int)
     */
    public void addVariables(int numVars) {
        if (numVars < 0)
            throw new IllegalArgumentException("vars < 0: " + numVars);
        vars += numVars;
    }    
    /**
     * @see kodkod.engine.satlab.SATSolver#free()
     */
    public synchronized void free() { }
    /**
     * @see kodkod.engine.satlab.SATSolver#numberOfClauses()
     */
    public int numberOfClauses() {
        return clauses;
    }    
    /**
     * @see kodkod.engine.satlab.SATSolver#numberOfVariables()
     */
    public int numberOfVariables() {
        return vars;
    }    
    /**
     * @effects |lit| <= this.vars && lit != 0 => this.solution'.set(|lit|, lit>0)
     * @throws RuntimeException - lit=0 || |lit|>this.vars
     */
    private final void updateSolution(int lit) {
        int abs = StrictMath.abs(lit);
        if (abs<=vars && abs>0)
            solution.set(abs-1, lit>0);
        else
            throw new RuntimeException("invalid variable value: |" + lit + "| !in [1.."+vars+"]");
    }    
    /** Helper class that drains the stderr. */
    private final class Drainer implements Runnable {
        /** The stream to drain. */
        private final InputStream input;
        /** Constructor that constructs a new Drainer. */
        public Drainer(InputStream input) {
            this.input=input;
        }
        /** The run method that keeps reading from the InputStream until end-of-file. */
        public void run() {
            try {
                byte[] buffer=new byte[8192];
                while(true) {
                    int n=input.read(buffer);
                    if (n<0) break;
                }
            } catch (IOException ex) {
            }
            try { input.close(); } catch(IOException ex) { }
        }
    }    
    /**
     * @see kodkod.engine.satlab.SATSolver#solve()
     */
    public boolean solve() throws SATAbortedException {
        if (sat == null) {
            Path inpath = null;
            Path outpath = null;
            Process proc = null;
            try {
                inpath = Files.createTempFile("tmp", "cnf");
                outpath = outTemp ? Files.createTempFile("tmp", "out") : null;
                BufferedWriter cnf = Files.newBufferedWriter(inpath, UTF8);
                cnf.write("p cnf " + vars + " " + clauses + "\n");
                cnf.write(buffer.toString());
                cnf.write("\n");
                cnf.close();
                buffer.setLength(0);

                int command_prefix = outTemp ? 3 : 2;
                final String[] command = new String[command_prefix + options.length];
                command[0] = executable;
                command[1] = inpath.toString();
                if (outTemp) { command[2] = outpath.toString(); }
                System.arraycopy(options, 0, command, command_prefix, options.length);

                proc = Runtime.getRuntime().exec(command);
                new Thread(new Drainer(proc.getErrorStream())).start();
                final BufferedReader out;
                if (outTemp) {
                    new Thread(new Drainer(proc.getInputStream())).start();
                    try { proc.waitFor(); }
                    catch (InterruptedException e) {
                        throw new RuntimeException("could not wait for " + executable, e);
                    }
                    out = new BufferedReader(Files.newBufferedReader(outpath, UTF8));
                } else {
                    out = new BufferedReader(new InputStreamReader(proc.getInputStream(), UTF8));
                }

                String line;
                while((line = out.readLine()) != null) {
                    int j = 0;
                    if (line.indexOf(unsatTag) != -1) {
                        sat = Boolean.FALSE;
                        break;
                    } else if (line.indexOf(satTag) != -1) {
                        sat = Boolean.TRUE;
                        continue;
                    } else if (solutionTag.length() == 0) {
                        if (sat == Boolean.TRUE) {
                            final String[] tokens = line.split("\\s");
                            int last = tokens.length - 1;
                            try {
                                for (int i = 0; i <= last; i++) {
                                    int n = Integer.parseInt(tokens[i]);
                                    if (n != 0)
                                        updateSolution(n);
                                }
                            } catch (NumberFormatException e) {
                                // done (e.g., ran into the word "Random" in
                                // zChaff's output)
                                break;
                            }
                            break;
                        }
                    } else if ((j = line.indexOf(solutionTag)) != -1) {
                        line = line.substring(j + solutionTag.length());
                        final String[] tokens = line.split("\\s");
                        int last = tokens.length - 1;
                        for (int i = 0; i < last; i++) {
                            updateSolution(Integer.parseInt(tokens[i]));
                        }
                        int lit = Integer.parseInt(tokens[last]);
                        if (lit != 0) updateSolution(lit);
                        else if (sat!=null) break;
                    }
                    // not a solution line or a variable line, so ignore it.
                }
                try { out.close(); } catch (IOException ignored) { } // do nothing, we are done
                if (sat==null) {
                    throw new RuntimeException("invalid " + executable +
                                               " output (unknown result)");
                }
            }
            catch (IOException e) {
                throw new RuntimeException("SAT solver aborted: " + e, e);
            }
            catch (NumberFormatException e) {
                throw new RuntimeException("invalid " + executable + " output", e);
            }
            finally {
                if (proc != null) {
                    proc.destroy();
                    try { proc.waitFor(); } catch (InterruptedException ignored) { }
                }
                if (inpath != null) { inpath.toFile().delete(); }
                if (outpath != null) { outpath.toFile().delete(); }
            }
        }
        return sat;
    }    
    /**
     * @see kodkod.engine.satlab.SATSolver#valueOf(int)
     */
    public boolean valueOf(int variable) {
        if (!Boolean.TRUE.equals(sat))
            throw new IllegalStateException();
        if (variable < 1 || variable > vars)
            throw new IllegalArgumentException(variable + " !in [1.." + vars+"]");
        return solution.get(variable-1);
    }    
    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return executable + " " + options;
    }    
    public static final SATFactory satFactory(final String executable,
            final String tempInput, final String tempOutput,
            final String satTag, final String solTag, final String unsatTag,
            final String... options) {
        return new SATFactory() {
                    @Override
                    public SATSolver instance() {
                        return new ExternalSolverV2(
                            executable, tempOutput.length() != 0,
                            satTag, solTag, unsatTag, options);
                    }    
                    @Override
                    public boolean incremental() { return false; }
                };
    }
}    
