//  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
//
//  This file is part of the VNC system.
//
//  The VNC system is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
//  USA.
//
// If the source code for the VNC system is not available from the place 
// whence you received this file, check http://www.orl.co.uk/vnc or contact
// the authors on vnc@orl.co.uk for information on obtaining it.


#include "stdhdrs.h"

#include "vncviewer.h"
#include "omnithread.h"
#include "ClientConnection.h"
#include "SessionDialog.h"
#include "AuthDialog.h"
#include "AboutBox.h"

#include "Exception.h"
extern "C" {
	#include "vncauth.h"
}

#define INITIALNETBUFSIZE 4096
#define MAX_ENCODINGS 10
#define VWR_WND_CLASS_NAME "VNCviewer"

// read a pixel from the given address, and return a color value
#define COLOR_FROM_PIXEL8_ADDRESS(p) (RGB( \
                (int) (((*(CARD8 *)p >> rs) & rm) * 255 / rm), \
                (int) (((*(CARD8 *)p >> gs) & gm) * 255 / gm), \
                (int) (((*(CARD8 *)p >> bs) & bm) * 255 / bm) ))

#define COLOR_FROM_PIXEL16_ADDRESS(p) (RGB( \
                (int) ((( *(CARD16 *)p >> rs) & rm) * 255 / rm), \
                (int) ((( *(CARD16 *)p >> gs) & gm) * 255 / gm), \
                (int) ((( *(CARD16 *)p >> bs) & bm) * 255 / bm) ))

#define COLOR_FROM_PIXEL32_ADDRESS(p) (RGB( \
                (int) ((( *(CARD32 *)p >> rs) & rm) * 255 / rm), \
                (int) ((( *(CARD32 *)p >> gs) & gm) * 255 / gm), \
                (int) ((( *(CARD32 *)p >> bs) & bm) * 255 / bm) ))

#define COLOR_FROM_PIXEL8(p) (RGB( \
                (int) (((p >> rs) & rm) * 255 / rm), \
                (int) (((p >> gs) & gm) * 255 / gm), \
                (int) (((p >> bs) & bm) * 255 / bm) ))

#define COLOR_FROM_PIXEL16(p) (RGB( \
                (int) ((( p >> rs) & rm) * 255 / rm), \
                (int) ((( p >> gs) & gm) * 255 / gm), \
                (int) ((( p >> bs) & bm) * 255 / bm) ))

#define COLOR_FROM_PIXEL32(p) (RGB( \
                (int) (((p >> rs) & rm) * 255 / rm), \
                (int) (((p >> gs) & gm) * 255 / gm), \
                (int) (((p >> bs) & bm) * 255 / bm) ))

const rfbPixelFormat vnc8bitFormat = {8, 8, 1, 1, 7,7,3, 0,3,6,0,0};
const rfbPixelFormat vnc16bitFormat = {16, 16, 1, 1, 63, 31, 31, 0,6,11,0,0};

// Client connection *******************************************

static LRESULT CALLBACK ClientConnection::WndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);

ClientConnection::ClientConnection()
{
	Init();
	SessionDialog sessdlg(&opts);
    if (!sessdlg.DoDialog()) {
        SendMessage(m_hwnd, WM_CLOSE,0,0);
 		throw QuietException("User cancelled");
    }
	strncpy(m_host, sessdlg.m_host, 256);
	m_port = sessdlg.m_port;
}

ClientConnection::ClientConnection(SOCKET sock) 
{
	Init();
	m_sock = sock;
	struct sockaddr_in svraddr;
	int sasize = sizeof(svraddr);
	if (getpeername(sock, (struct sockaddr *) &svraddr, 
		&sasize) != SOCKET_ERROR) {
		sprintf(m_host, "%d.%d.%d.%d", 
			svraddr.sin_addr.S_un.S_un_b.s_b1, 
			svraddr.sin_addr.S_un.S_un_b.s_b2, 
			svraddr.sin_addr.S_un.S_un_b.s_b3, 
			svraddr.sin_addr.S_un.S_un_b.s_b4);
		m_port = svraddr.sin_port;
	} else {
		strcpy(m_host,"(unknown)");
		m_port = 0;
	};

}

ClientConnection::ClientConnection(char *host, int port)
{
	Init();
	strncpy(m_host, host, 256);
	m_port = port;
}

void ClientConnection::Init()
{
	m_hwnd = 0;
	desktopName = NULL;
	netbuf = NULL;
	netbufsize = 0;
	hwndNextViewer = NULL;	

	// We take the initial options from the application defaults:
	opts = appOptions;

	m_sock = 0;
	bKillThread = false;
	threadStarted = true;
	running = false;
	pendingFormatChange = false;

	// We know the offsets, but the size will depend
	// on the screen size.
	hScrollPos = 0; vScrollPos = 0;
	
	// Create the window
	WNDCLASSEX wndclass;

	wndclass.cbSize			= sizeof(wndclass);
	wndclass.style			= 0;
	wndclass.lpfnWndProc	= ClientConnection::WndProc;
	wndclass.cbClsExtra		= 0;
	wndclass.cbWndExtra		= 0;
	wndclass.hInstance		= hAppInstance;
	wndclass.hIcon			= LoadIcon(hAppInstance, MAKEINTRESOURCE(IDI_MAINICON));
	wndclass.hCursor		= LoadCursor(hAppInstance, MAKEINTRESOURCE(IDC_DOTCURSOR));
	wndclass.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
    wndclass.lpszMenuName	= (const char *) NULL;
	wndclass.lpszClassName	= VWR_WND_CLASS_NAME;
	wndclass.hIconSm		= LoadIcon(hAppInstance, MAKEINTRESOURCE(IDI_MAINICON));

	RegisterClassEx(&wndclass);

	m_hwnd = CreateWindow(VWR_WND_CLASS_NAME,
				"VNCviewer",
				WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX |
					WS_THICKFRAME | WS_VSCROLL | WS_HSCROLL,
				CW_USEDEFAULT,
				CW_USEDEFAULT,
				400, 300,
				NULL,
				NULL,
				hAppInstance,
				NULL);
	ShowWindow(m_hwnd, SW_HIDE);

	// record which client created this window
	SetWindowLong(m_hwnd, GWL_USERDATA, (LONG) this);

	// and create a buffer for various network operations
	CheckBufferSize(INITIALNETBUFSIZE);

	// Create a memory DC which we'll use for drawing to
	// the local framebuffer
	hBitmapDC = CreateCompatibleDC(NULL);
	hBitmap = NULL;

	// Add stuff to System menu
	HMENU hsysmenu = GetSystemMenu(m_hwnd, FALSE);
	AppendMenu(hsysmenu, MF_SEPARATOR, NULL, NULL);
	AppendMenu(hsysmenu, MF_STRING, IDC_OPTIONBUTTON, "Connection &options...");
	AppendMenu(hsysmenu, MF_STRING, ID_CONN_ABOUT, "Connection &info");
    AppendMenu(hsysmenu, MF_STRING, ID_CONN_CTLALTDEL, "&Send Ctrl-Alt-Del");
	AppendMenu(hsysmenu, MF_SEPARATOR, NULL, NULL);
    AppendMenu(hsysmenu, MF_STRING, ID_NEWCONN, "Ne&w connection...");
    AppendMenu(hsysmenu, MF_SEPARATOR, NULL, NULL);
	AppendMenu(hsysmenu, MF_STRING, IDD_APP_ABOUT, "&About VNCviewer...");
	if (appOptions.m_listening) {
		AppendMenu(hsysmenu, MF_SEPARATOR, NULL, NULL);
		AppendMenu(hsysmenu, MF_STRING, ID_CLOSEDAEMON, "Close listening &daemon");
	}
	DrawMenuBar(m_hwnd);

	UpdateWindow(m_hwnd);
}

// 
// Run() creates the connection if necessary and then
// starts the thread running which does the work.
//

void ClientConnection::Run()
{


	// The worker thread assumes we are already connected
	// so if the socket isn't there yet, we create it now.
	if (m_sock==0)
		Connect();
	
	SetSocketOptions();

	// This starts the worker thread.
	// The rest of the processing continues in run_undetached.
	start_undetached();
}

void ClientConnection::Connect()
{
	struct sockaddr_in thataddr;
	int res;
	
	m_sock = socket(PF_INET, SOCK_STREAM, 0);
	if (m_sock == INVALID_SOCKET) throw WarningException("Error creating socket");
	int one = 1;
	
	try {
		// The host may be specified as a dotted address "a.b.c.d"
		// Try that first
		thataddr.sin_addr.s_addr = inet_addr(m_host);
		
		// If it wasn't one of those, do gethostbyname
		if (thataddr.sin_addr.s_addr == INADDR_NONE) {
			LPHOSTENT lphost;
			lphost = gethostbyname(m_host);
			
			if (lphost == NULL) { 
				throw WarningException("Failed to get server address.\n\r"
                    "Did you type the host name correctly?"); 
			};
			thataddr.sin_addr.s_addr = ((LPIN_ADDR) lphost->h_addr)->s_addr;
		};
		
		thataddr.sin_family = AF_INET;
		thataddr.sin_port = htons(m_port);
		res = connect(m_sock, (LPSOCKADDR) &thataddr, sizeof(thataddr));
		if (res == SOCKET_ERROR) throw WarningException("Failed to connect to server");
		log.Print(0, _T("Connected to %s port %d\n"), m_host, m_port);

	} catch (...) {
		closesocket(m_sock);
		m_sock = 0;
		throw;
	}
}

void ClientConnection::SetSocketOptions() {
	// Disable Nagle's algorithm
	BOOL nodelayval = TRUE;
	if (setsockopt(m_sock, IPPROTO_TCP, TCP_NODELAY, (const char *) &nodelayval, sizeof(BOOL)))
		throw WarningException("Error disabling Nagle's algorithm");
}

// Close the socket, kill the thread, close the window
// unless otherwise specified. (We call this from within
// the CloseWindow handler).
void ClientConnection::Kill(bool closeWindow)
{
	bKillThread = true;
	running = false;

	if (m_sock != 0) {
		shutdown(m_sock, SD_BOTH);
		closesocket(m_sock);
	}

	if (closeWindow) 
		PostMessage(m_hwnd, WM_CLOSE, 0,0);
}


ClientConnection::~ClientConnection()
{
	if (desktopName != NULL) delete [] desktopName;
	delete [] netbuf;
	DeleteDC(hBitmapDC);
	if (hBitmap != NULL)
		DeleteObject(hBitmap);
}


// Process window messages
LRESULT CALLBACK ClientConnection::WndProc(HWND hwnd, UINT iMsg, 
										   WPARAM wParam, LPARAM lParam) {
	// This is a static method, so we don't know which instantiation we're 
	// dealing with. We use Allen Hadden's (ahadden@taratec.com) suggestion 
	// from a newsgroup to get the pseudo-this.
	ClientConnection *_this = (ClientConnection *) GetWindowLong(hwnd, GWL_USERDATA);

	switch (iMsg) {

	case WM_CREATE:
		return 0;

	case WM_SIZING:
		{
			// Don't allow sizing larger than framebuffer
			RECT *lprc = (LPRECT) lParam;
			switch (wParam) {
			case WMSZ_RIGHT:
				lprc->right = min(lprc->right, lprc->left + _this->fullwinwidth+1);
				break;
			case WMSZ_BOTTOM:
				lprc->bottom = min(lprc->bottom, lprc->top + _this->fullwinheight);
				break;
			case WMSZ_LEFT:
				lprc->left = max(lprc->left, lprc->right - _this->fullwinwidth);
				break;
			case WMSZ_TOP:
				lprc->top = max(lprc->top, lprc->bottom - _this->fullwinheight);
				break;
			case WMSZ_TOPLEFT:
				lprc->top = max(lprc->top, lprc->bottom - _this->fullwinheight);
				lprc->left = max(lprc->left, lprc->right - _this->fullwinwidth);
				break;
			case WMSZ_TOPRIGHT:
				lprc->top = max(lprc->top, lprc->bottom - _this->fullwinheight);
				lprc->right = min(lprc->right, lprc->left + _this->fullwinwidth);
				break;
			case WMSZ_BOTTOMLEFT:
				lprc->bottom = min(lprc->bottom, lprc->top + _this->fullwinheight);
				lprc->left = max(lprc->left, lprc->right - _this->fullwinwidth);
				break;
			case WMSZ_BOTTOMRIGHT:
				lprc->bottom = min(lprc->bottom, lprc->top + _this->fullwinheight);
				lprc->right = min(lprc->right, lprc->left + _this->fullwinwidth);
				break;

			}
			return 0;
		}
	case WM_SIZE:
		{
            /*
            // *** > JNW 24/04/98
			ULONG oldstyle = GetWindowLong(_this->m_hwnd, GWL_STYLE);
			ULONG newstyle = oldstyle;

			if (wParam != _this->m_cmdShow)
			{
				switch(wParam)
				{

					// The user has maximized the window!
				case SIZE_MAXIMIZED:
					// Disable some of the styles for full-screen mode!
					log.Print(3, "Setting window to borderless full-screen\n");
					newstyle = oldstyle & ~(WS_OVERLAPPED | WS_CAPTION |
						WS_THICKFRAME | WS_VSCROLL | WS_HSCROLL);
					break;

					// User has restored the normal window size
				case SIZE_RESTORED:
					// Restore the missing window styles :(
					log.Print(3, "Setting window to bordered\n");
					newstyle = oldstyle | (WS_OVERLAPPED |
						WS_CAPTION | WS_THICKFRAME | WS_VSCROLL | WS_HSCROLL);
					break;

				};
				_this->m_cmdShow = wParam;
			}

			// Update the window position
			if (oldstyle != newstyle)
			{
				SetWindowLong(_this->m_hwnd, GWL_STYLE, newstyle);
				SetWindowPos(_this->m_hwnd, NULL,
					0, 0, 0, 0,
					SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_FRAMECHANGED
					);
			}

			// *** < JNW 24/04/98
            */

			_this->cliwidth = LOWORD(lParam);
			_this->cliheight = HIWORD(lParam);
			
			// Calculate window dimensions
			RECT rect;
			GetWindowRect(hwnd, &rect);
			// update these for the record
			_this->winwidth = rect.right - rect.left;
			_this->winheight = rect.bottom - rect.top;

			// If the current window size would be large enough to hold the
			// whole screen without scrollbars, we turn them off.
			if (_this->winwidth >= _this->fullwinwidth  &&
				_this->winheight >= _this->fullwinheight ) {
				ShowScrollBar(hwnd, SB_HORZ, FALSE);
				ShowScrollBar(hwnd, SB_VERT, FALSE);
			} else {
				ShowScrollBar(hwnd, SB_HORZ, TRUE);
				ShowScrollBar(hwnd, SB_VERT, TRUE);
			}

            // update these for the record
			GetClientRect(hwnd, &rect);
			_this->cliwidth = rect.right - rect.left;
			_this->cliheight = rect.bottom - rect.top;
			
			_this->hScrollMax = _this->si.framebufferWidth;
			_this->vScrollMax = _this->si.framebufferHeight;
            
			int newhpos, newvpos;
			newhpos = max(0, min(_this->hScrollPos, _this->hScrollMax - max(_this->cliwidth, 0)));
			newvpos = max(0, min(_this->vScrollPos, _this->vScrollMax - max(_this->cliheight, 0)));

			ScrollWindowEx(hwnd, _this->hScrollPos-newhpos, _this->vScrollPos-newvpos,
				NULL, &rect, NULL, NULL,  SW_INVALIDATE);
			
			_this->hScrollPos = newhpos;
			_this->vScrollPos = newvpos;
           	_this->UpdateScrollbars();

			return 0;
		}

	case WM_PAINT:
		_this->DoBlit();
		return 0;

	case WM_LBUTTONDOWN:
	case WM_MBUTTONDOWN:
	case WM_RBUTTONDOWN:
	case WM_LBUTTONUP:
	case WM_MBUTTONUP:
	case WM_RBUTTONUP:
	case WM_MOUSEMOVE:
		{
			if (!_this->running) return 0;
			if (GetFocus() != hwnd) return 0;
			int x = LOWORD(lParam);
			int y = HIWORD(lParam);
			int mask;
			if (_this->opts.m_SwapMouse) {
				mask = ( ((wParam & MK_LBUTTON) ? rfbButton1Mask : 0) |
					((wParam & MK_MBUTTON) ? rfbButton3Mask : 0) |
					((wParam & MK_RBUTTON) ? rfbButton2Mask : 0)  );
			} else {
				mask = ( ((wParam & MK_LBUTTON) ? rfbButton1Mask : 0) |
					((wParam & MK_MBUTTON) ? rfbButton2Mask : 0) |
					((wParam & MK_RBUTTON) ? rfbButton3Mask : 0)  );
			}
			try {
				_this->SendPointerEvent(x + _this->hScrollPos,
										y + _this->vScrollPos,
										mask);
			} catch (Exception &e) {
				e.Report();
				DestroyWindow(hwnd);
			}
			return 0;
		}

	case WM_KEYDOWN:
	case WM_KEYUP:
	case WM_SYSKEYDOWN:
	case WM_SYSKEYUP:
		{
			if (!_this->running) return 0;
    
            _this->ProcessKeyEvent((int) wParam, (DWORD) lParam);
			return 0;
		}

	// Disable as much normal local keyboard handling as we can.
	case WM_CHAR:
	case WM_SYSCHAR:
    case WM_DEADCHAR:
    case WM_SYSDEADCHAR:
        {
            log.Print(4, _T("Got a char/syschar, and ignored it: w,l = 0x%x, 0x%x\n"),
                wParam, lParam);
            return 0;
        }

	case WM_CLOSE:
		{
            int numMyOtherWindows = _this->CountProcessOtherWindows();
            log.Print(1,"Closing window, %d others open\n", numMyOtherWindows);   

			// Close the connection as well
			_this->Kill(false);

			// Remove us from the clipboard viewer chain
			BOOL res = ChangeClipboardChain( hwnd, _this->hwndNextViewer);

			// We are currently in the main thread.
			// The worker thread should be about to finish if
			// it hasn't already. Wait for it.
			try {
				void *p;
				_this->join(&p);
			} catch (omni_thread_invalid& e) {
				// The thread probably hasn't been started yet,
			}

            // If this is the last window, and we're not running a daemon,
            // then close the app.
            if (numMyOtherWindows == 0 && !appOptions.m_listening) PostQuitMessage(0);

			break;
		}

	case WM_HSCROLL:
		{
			int dx = 0;
			int pos = HIWORD(wParam);
			switch (LOWORD(wParam)) {
			case SB_LINEUP:
				dx = -2; break;
			case SB_LINEDOWN:
				dx = 2; break;
			case SB_PAGEUP:
				dx = _this->cliwidth * -1/4; break;
			case SB_PAGEDOWN:
				dx = _this->cliwidth * 1/4; break;
			case SB_THUMBPOSITION:
				dx = pos - _this->hScrollPos;
			case SB_THUMBTRACK:
				dx = pos - _this->hScrollPos;
			}
			// We can't scroll outside the limits
			dx = max(dx, -_this->hScrollPos);
			dx = min(dx, _this->hScrollMax-(_this->cliwidth-1)-_this->hScrollPos);
			if (dx != 0) {
				_this->hScrollPos += dx;
				RECT clirect;
				GetClientRect(hwnd, &clirect);
				ScrollWindowEx(hwnd, -dx, 0, NULL, &clirect, NULL, NULL,  SW_INVALIDATE);
				//SetScrollPos(hwnd, SB_HORZ, _this->hScrollPos, TRUE);
				_this->UpdateScrollbars();
				UpdateWindow(hwnd);
			}
			return 0;
		}

	case WM_VSCROLL:
		{
			int dy = 0;
			int pos = HIWORD(wParam);
			switch (LOWORD(wParam)) {
			case SB_LINEUP:
				dy = -2; break;
			case SB_LINEDOWN:
				dy = 2; break;
			case SB_PAGEUP:
				dy = _this->cliheight * -1/4; break;
			case SB_PAGEDOWN:
				dy = _this->cliheight * 1/4; break;
			case SB_THUMBPOSITION:
				dy = pos - _this->vScrollPos;
			case SB_THUMBTRACK:
				dy = pos - _this->vScrollPos;
			}
			// We can't scroll outside the limits
			dy = max(dy, -_this->vScrollPos);
			dy = min(dy, _this->vScrollMax-(_this->cliheight-1)-_this->vScrollPos);
			if (dy != 0) {
				_this->vScrollPos += dy;
				RECT clirect;
				GetClientRect(hwnd, &clirect);
				ScrollWindowEx(hwnd, 0, -dy, NULL, &clirect, NULL, NULL,  SW_INVALIDATE);
				// SetScrollPos(hwnd, SB_VERT, _this->vScrollPos, TRUE);
				_this->UpdateScrollbars();
				UpdateWindow(hwnd);
			}
			return 0;
		}

	case WM_SYSCOMMAND:
		{
			switch (LOWORD(wParam)) {
            case ID_NEWCONN:
                {
                    // Add a new connection to the app
                    ClientConnection *pcc = NULL;
                    try {
                        pcc = new ClientConnection();
                        pcc->Run();
                    } catch (Exception &e) {
                        if (pcc != NULL)
                            pcc->Kill(true);
                        e.Report();				
                    } 
                    return 0;
                }
			case IDC_OPTIONBUTTON: 
				{
					if (_this->opts.DoDialog(true)) {
						_this->pendingFormatChange = true;
					};
					return 0;
				}
			case IDD_APP_ABOUT:
				ShowAboutBox();
				return 0;
			case ID_CONN_ABOUT:
				_this->ShowConnInfo();
				return 0;
            case ID_CONN_CTLALTDEL:
                _this->SendCtlAltDel();
                return 0;
			case ID_CLOSEDAEMON:
				if (MessageBox(NULL, "Are you sure you want to exit?", 
						"Closing VNCviewer", 
						MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2) == IDYES)
					PostQuitMessage(0);
				return 0;
			}
			break;
		}
	case WM_CHANGECBCHAIN:
		{
			HWND hWndRemove = (HWND) wParam;     // handle of window being removed 
			HWND hWndNext = (HWND) lParam;       // handle of next window in chain 
			// If the next window is closing, repair the chain.  
			if (hWndRemove == _this->hwndNextViewer) 
				_this->hwndNextViewer = hWndNext;  
			// Otherwise, pass the message to the next link.  
			else if (_this->hwndNextViewer != NULL) 
				::SendMessage(_this->hwndNextViewer, WM_CHANGECBCHAIN, 
				(WPARAM) hWndRemove,  (LPARAM) hWndNext );  
			return 0;
		}
	case WM_DRAWCLIPBOARD:
		{
			log.Print(2, _T("Clipboard changed\n"));
	
			
			HWND hOwner = GetClipboardOwner();
			if (hOwner == hwnd) {
				log.Print(2, _T("We changed it!\n"));
			} else {
                
                // The clipboard should not be modified by more than one thread at once
                omni_mutex_lock l(_this->clipMutex);

				if (OpenClipboard(hwnd)) { 
					HGLOBAL hglb = GetClipboardData(CF_TEXT); 
					if (hglb == NULL) {
						CloseClipboard();
                    } else {
					    LPSTR lpstr = (LPSTR) GlobalLock(hglb);  
						    
					    char *contents = new char[strlen(lpstr) + 1];
					    char *unixcontents = new char[strlen(lpstr) + 1];
					    strcpy(contents,lpstr);
					    GlobalUnlock(hglb); 
					    CloseClipboard();       		
				    
					    for (int i = 0, j = 0; contents[i] != '\0'; i++) {
						    if (contents[i] != '\x0d') {
							    unixcontents[j++] = contents[i];
						    }
					    }
					    unixcontents[j] = '\0';
					    try {
						    _this->SendClientCutText(unixcontents, strlen(unixcontents));
					    } catch (WarningException &e) {
                            log.Print(0, _T("Exception while sending clipboard text\n"));
						    DestroyWindow(hwnd);
					    }
					    delete [] contents; 
					    delete [] unixcontents;
                    }
				}
			}
			// Pass the message to the next window in clipboard 
			// viewer chain.  
			::SendMessage(_this->hwndNextViewer, WM_DRAWCLIPBOARD , 0,0); 
			return 0;
		}

	}

	return DefWindowProc(hwnd, iMsg, wParam, lParam);
	
	// We know about an unused variable here.
#pragma warning(disable : 4101)
}

#pragma warning(default : 4101)

// This bit is just for finding out which windows this viewer has open
// We should really keep a list of them in the app, but then everything would
// need to know about the app.

typedef struct {
    HWND ignore;
    unsigned int counter;
    DWORD pid;
} count_struct;

BOOL CALLBACK ClientConnEnumWindowsProc(  HWND hwnd,  LPARAM lParam) 
{
    char classbuf[256];
    DWORD wpid;
    count_struct *pcs = (count_struct *) lParam;
    // Do the quick things first
    // if this window isn't the one we know about
    if (hwnd != pcs->ignore) {
         // but it's owned by our application
        GetWindowThreadProcessId(  hwnd, &wpid ) ;
        if (wpid == pcs->pid) {
            // and it has the viewer class
            GetClassName(  hwnd,  classbuf, 255 );
            if (strcmp(classbuf, VWR_WND_CLASS_NAME) == 0)
                // then increment the counter
                (pcs->counter) ++;
        }
    }      
    return TRUE;
}

// Count the other viewer windows owned by this process
unsigned int ClientConnection::CountProcessOtherWindows()
{
    count_struct cs;
    cs.counter = 0;
    cs.pid = GetCurrentProcessId();
    cs.ignore = m_hwnd;
    EnumWindows((WNDENUMPROC) ClientConnEnumWindowsProc, (LPARAM) &cs);
    return cs.counter;
}



inline void ClientConnection::DoBlit() 
{
	if (hBitmap == NULL) return;
	omni_mutex_lock l(bitmapdcMutex);
				
	PAINTSTRUCT ps;

	
	HDC hdc = BeginPaint(m_hwnd, &ps);

	HBITMAP hOldBmp = SelectObject(hBitmapDC, hBitmap);
			
	if (opts.m_delay) {
		// Display the area to be updated for debugging purposes
		COLORREF oldbgcol = SetBkColor(hdc, RGB(0,0,0));
		::ExtTextOut(hdc, 0, 0, ETO_OPAQUE, &ps.rcPaint, NULL, 0, NULL);
		SetBkColor(hdc,oldbgcol);
		::GdiFlush();
		::Sleep(appOptions.m_delay);
	}
	
	if (!BitBlt(hdc, ps.rcPaint.left, ps.rcPaint.top, 
		ps.rcPaint.right-ps.rcPaint.left, ps.rcPaint.bottom-ps.rcPaint.top, 
		hBitmapDC, ps.rcPaint.left+hScrollPos, ps.rcPaint.top+vScrollPos, SRCCOPY)) 
	{
		throw WarningException("Error in blit!\n");
	}
	
	SelectObject(hBitmapDC, hOldBmp);
	EndPaint(m_hwnd, &ps);
}

inline void ClientConnection::UpdateScrollbars() 
{
	SCROLLINFO scri;
	scri.cbSize = sizeof(scri);
	scri.fMask = SIF_ALL;
	scri.nMin = 0;
	scri.nMax = hScrollMax;     
	scri.nPage= cliwidth;
	scri.nPos = hScrollPos; 
	
	SetScrollInfo(m_hwnd, SB_HORZ, &scri, TRUE);
	
	scri.cbSize = sizeof(scri);
	scri.fMask = SIF_ALL;
	scri.nMin = 0;
	scri.nMax = vScrollMax;     
	scri.nPage= cliheight;
	scri.nPos = vScrollPos; 
	
	SetScrollInfo(m_hwnd, SB_VERT, &scri, TRUE);
}


void ClientConnection::ShowConnInfo()
{
	char buf[2048];
	char kbdname[9];

	GetKeyboardLayoutName(kbdname);

	sprintf(
		buf,
		"Connected to: %s:\n\r"
		"Host: %s port: %d\n\r\n\r"
		"Desktop geometry: %d x %d x %d\n\r"
		"Using depth: %d\n\r"
		"Current protocol version: %d.%d\n\r\n\r"
		"Current keyboard name: %s\n\r",
		desktopName, m_host, m_port,
		si.framebufferWidth, si.framebufferHeight, si.format.depth,
		myFormat.depth,
		major, minor,
		kbdname
		);
	MessageBox(NULL, buf, "VNC connection info", MB_ICONINFORMATION | MB_OK);
}

// --------------------------------------------------------------------
//  Methods after this point are generally called by the worker thread
// --------------------------------------------------------------------


void* ClientConnection::run_undetached(void* arg) {

	threadStarted = true;

	try {

		NegotiateProtocolVersion();
		
		try {
			Authenticate();
		} catch (WarningException &e) {
			e.Report();
			throw;
		}
 
		SendClientInit();

		ReadServerInit();
		
		CreateLocalFramebuffer();

		SetupPixelFormat();
		
		SetFormatAndEncodings();
	
		// *** > JNW
		// ShowWindow(m_hwnd, SW_SHOW);
		// *** <
		UpdateWindow(m_hwnd);

		SendFullFramebufferUpdateRequest();

		// We want to know when the clipboard changes, so
		// insert ourselves in the viewer chain
		hwndNextViewer = SetClipboardViewer(m_hwnd); 
		
		running = true;
		
		while (!bKillThread) {
			
			// Look at the type of the message, but leave it in the buffer 
			CARD8 msgType;
			{
				omni_mutex_lock l(readMutex);
				int bytes = recv(m_sock, (char *) &msgType, 1, MSG_PEEK);
				if (bytes == 0) {
					Kill();
					break;
				}
				if (bytes < 0) {
					throw WarningException("Error while waiting for server message");
				}

					
			}
				
			switch (msgType) {
			case rfbFramebufferUpdate:
				ScreenUpdate();
				if (pendingFormatChange) {
					rfbPixelFormat oldFormat = myFormat;
					SetupPixelFormat();
					SetFormatAndEncodings();
					pendingFormatChange = false;
					// If the pixel format has changed, request whole screen
					if (memcmp(&myFormat, &oldFormat, sizeof(rfbPixelFormat)) != 0) {
						SendFullFramebufferUpdateRequest();
					} else {
						SendIncrementalFramebufferUpdateRequest();
					}
				} else {
					SendIncrementalFramebufferUpdateRequest();
				}
				break;
			case rfbSetColourMapEntries:
				throw WarningException("Unhandled SetColormap message type received!\n");
				break;
			case rfbBell:
				Bell();
				break;
			case rfbServerCutText:
				ServerCutText();
				break;
			default:
				log.Print(0, _T("Unhandled message type %d received!\n"), msgType);
				throw WarningException("Unhandled message type received!\n");
			}

		}

	} catch (WarningException &e) {
		// Here we just want to close the connection
        // e.Report();
		if (e.m_close) 
			PostMessage(m_hwnd, WM_CLOSE, 0, 0);
	} catch (QuietException &e) {
		e.Report();
	} 
	return this;
}


void ClientConnection::NegotiateProtocolVersion()
{
	rfbProtocolVersionMsg pv;

   /* if the connection is immediately closed, don't report anything, so
       that pmw's monitor can make test connections */

    try {
		ReadExact(pv, sz_rfbProtocolVersionMsg);
	} catch (Exception &c) {
		log.Print(0, _T("Error reading protocol version: %s\n"), c);
		throw QuietException(c.m_info);
	}

    pv[sz_rfbProtocolVersionMsg] = 0;

    if (sscanf(pv,rfbProtocolVersionFormat,&major,&minor) != 2) 
		throw WarningException("Not a valid RFB server");

    log.Print(0, _T("RFB server supports protocol version %d.%d\n"),
	    major,minor);

    if ((major == 3) && (minor < 3)) {
		
        /* if server is 3.2 we can't use the new authentication */
		log.Print(0, _T("Can't use IDEA authentication\n"));
        /* This will be reported later if authentication is requested*/

    } else {
		
        /* any other server version, just tell the server what we want */
		major = rfbProtocolMajorVersion;
		minor = rfbProtocolMinorVersion;

    }

    sprintf(pv,rfbProtocolVersionFormat,major,minor);

    WriteExact(pv, sz_rfbProtocolVersionMsg);

	log.Print(0, _T("Connected to RFB server, using protocol version %d.%d\n"),
		rfbProtocolMajorVersion, rfbProtocolMinorVersion);
}

void ClientConnection::Authenticate()
{
	CARD32 authScheme, reasonLen, authResult;
    CARD8 challenge[CHALLENGESIZE];
	
	ReadExact((char *)&authScheme, 4);
    authScheme = Swap32IfLE(authScheme);
	
    switch (authScheme) {
		
    case rfbConnFailed:
		ReadExact((char *)&reasonLen, 4);
		reasonLen = Swap32IfLE(reasonLen);
		
		CheckBufferSize(reasonLen+1);
		ReadString(netbuf, reasonLen);
		
		log.Print(0, _T("RFB connection failed, reason: %s\n"), netbuf);
		throw WarningException(netbuf);
        break;
		
    case rfbNoAuth:
		log.Print(0, _T("No authentication needed\n"));
		break;
		
    case rfbVncAuth:
		{
            if ((major == 3) && (minor < 3)) {
                /* if server is 3.2 we can't use the new authentication */
                log.Print(0, _T("Can't use IDEA authentication\n"));
                MessageBox(NULL, "Sorry - this server uses an older authentication scheme\n\r"
                    "which is no longer supported.", 
                    "Protocol Version error", MB_OK | MB_ICONSTOP | MB_SETFOREGROUND | MB_TOPMOST);
                throw WarningException("Can't use IDEA authentication any more!");
            }

			ReadExact((char *)challenge, CHALLENGESIZE);
			
			AuthDialog ad;
			ad.DoDialog();
			char passwd[256];
			strcpy(passwd, ad.m_passwd);

			if (strlen(passwd) == 0) {
				log.Print(0, _T("Password had zero length\n"));
				throw WarningException("Empty password");
			}
			if (strlen(passwd) > 8) {
				passwd[8] = '\0';
			}
			
			vncEncryptBytes(challenge, passwd);
			
			/* Lose the password from memory */
			for (int i=0; i< (int) strlen(passwd); i++) {
				passwd[i] = '\0';
			}
			
			WriteExact((char *) challenge, CHALLENGESIZE);
			ReadExact((char *) &authResult, 4);
			
			authResult = Swap32IfLE(authResult);
			
			switch (authResult) {
			case rfbVncAuthOK:
				log.Print(0, _T("VNC authentication succeeded\n"));
				break;
			case rfbVncAuthFailed:
				throw WarningException("VNC authentication failed!");
			case rfbVncAuthTooMany:
				throw WarningException(
					"VNC authentication failed - too many tries!");
			default:
				log.Print(0, _T("Unknown VNC authentication result: %d\n"),
					(int)authResult);
				throw ErrorException("Unknown VNC authentication result!");
			}
			break;
		}
		
	default:
		log.Print(0, _T("Unknown authentication scheme from RFB server: %d\n"),
			(int)authScheme);
		throw ErrorException("Unknown authentication scheme!");
    }
}


void ClientConnection::SendClientInit()
{
    rfbClientInitMsg ci;
	ci.shared = opts.m_Shared;

    WriteExact((char *)&ci, sz_rfbClientInitMsg);
}


void ClientConnection::ReadServerInit()
{
    ReadExact((char *)&si, sz_rfbServerInitMsg);
	
    si.framebufferWidth = Swap16IfLE(si.framebufferWidth);
    si.framebufferHeight = Swap16IfLE(si.framebufferHeight);
    si.format.redMax = Swap16IfLE(si.format.redMax);
    si.format.greenMax = Swap16IfLE(si.format.greenMax);
    si.format.blueMax = Swap16IfLE(si.format.blueMax);
    si.nameLength = Swap32IfLE(si.nameLength);
	
    desktopName = new char[si.nameLength + 1];
	
    ReadString(desktopName, si.nameLength);
	
    log.Print(0, _T("Desktop name \"%s\"\n"),desktopName);
    log.Print(1, _T("Geometry %d x %d depth %d\n"),
        si.framebufferWidth, si.framebufferHeight, si.format.depth );
	SetWindowText(m_hwnd, desktopName);	

	// Find how large the desktop work area is
	RECT workrect;
	SystemParametersInfo(SPI_GETWORKAREA, 0, &workrect, 0);
	int workwidth = workrect.right -  workrect.left;
	int workheight = workrect.bottom - workrect.top;

	// *** JNW 24/04/98
	// Should we just do full-screen?
    /*
	if ((si.framebufferWidth >= workwidth) && (si.framebufferHeight >= workheight))
	{
		SetWindowLong(m_hwnd, GWL_STYLE, GetWindowLong(m_hwnd, GWL_STYLE) | WS_MAXIMIZEBOX);
		ShowWindow(m_hwnd, SW_MAXIMIZE);
		ShowWindow(m_hwnd, SW_RESTORE);
	}
	else
		ShowWindow(m_hwnd, SW_RESTORE);
	*/
    // ***
    
	// Size the window.
	// Let's find out how big a window would be needed to display the
	// whole desktop (assuming no scrollbars).

	RECT fullwinrect;
	SetRect(&fullwinrect, 0, 0, si.framebufferWidth, si.framebufferHeight);
	AdjustWindowRect(&fullwinrect, 
		GetWindowLong(m_hwnd, GWL_STYLE) & ~WS_VSCROLL & ~WS_HSCROLL, FALSE);
	fullwinwidth = fullwinrect.right - fullwinrect.left;
	fullwinheight = fullwinrect.bottom - fullwinrect.top;

	winwidth = min(fullwinwidth, workwidth);
	winheight = min(fullwinheight, workheight);
	SetWindowPos(m_hwnd, HWND_TOP,
		workrect.left + (workwidth-winwidth) / 2,
		workrect.top + (workheight-winheight) / 2,
		winwidth, winheight, SWP_SHOWWINDOW);

	SetForegroundWindow(m_hwnd);
}

void ClientConnection::CreateLocalFramebuffer() {
	// We create a bitmap which has the same pixel characteristics as
	// the local display, in the hope that blitting will be faster.
	HDC hdc = ::GetDC(NULL);
	hBitmap = ::CreateCompatibleBitmap(hdc, 
									si.framebufferWidth, 
									si.framebufferHeight);
	::ReleaseDC(NULL, hdc);
	if (hBitmap == NULL)
		throw WarningException("Error creating local image of screen.");

	// Select bitmap into DC
	HBITMAP hOldBmp = SelectObject(hBitmapDC, hBitmap);
	if (hOldBmp == NULL) {
		// There has been an error, probably because the dc, which is 
		// compatible with the system display, is not compatible with the 
		// bitmap we wish to create.
		throw WarningException("The endpoint seems to be incompatible with the server.");
	};

	// Put a "please wait" message up initially
    RECT rect;
	SetRect(&rect, 0,0, si.framebufferWidth, si.framebufferHeight);
	COLORREF bgcol = RGB(0xcc, 0xcc, 0xcc);
	FillSolidRect(&rect, bgcol);
	
	COLORREF oldbgcol = SetBkColor(hBitmapDC, bgcol);
	COLORREF oldtxtcol = SetTextColor(hBitmapDC, RGB(0,0,64));
	// Put the text in the top-left corner
	rect.right = si.framebufferWidth / 2;
	rect.bottom = si.framebufferHeight / 2;
    DrawText (hBitmapDC, "Please wait - initial screen loading", -1, &rect,
                    DT_SINGLELINE | DT_CENTER | DT_VCENTER);
	SetBkColor(hBitmapDC, oldbgcol);
	SetTextColor(hBitmapDC, oldtxtcol);

	// Deselect the bitmap for now
	SelectObject(hBitmapDC, hOldBmp);

	InvalidateRect(m_hwnd, NULL, FALSE);
}

void ClientConnection::SetupPixelFormat() {
	// Have we requested a reduction to 8-bit?
    if (opts.m_Use8Bit) {
		
        myFormat = vnc8bitFormat;

    // We don't support colormaps so we'll ask the server to convert
    } else if (!si.format.trueColour) {
        
        // We'll just request a standard 16-bit truecolor
        myFormat = vnc16bitFormat;
        
    } else {

		// Normally we just use the sever's format suggestion
		myFormat = si.format;

		// It's silly requesting more bits than our current display has, but
		// in fact it doesn't usually amount to much on the network.
		// Windows doesn't support 8-bit truecolour.
		// If our display is palette-based, we want more than 8 bit anyway,
		// unless we're going to start doing palette stuff at the server.
		// So the main use would be a 24-bit true-colour desktop being viewed
		// on a 16-bit true-colour display, and unless you have lots of images
		// and hence lots of raw-encoded stuff, the size of the pixel is not
		// going to make much difference.
		//   We therefore don't bother with any restrictions, but here's the
		// start of the code if we wanted to do it.

		if (false) {
		
			// Get a DC for the root window
			HDC hrootdc = GetDC(NULL);
			int localBitsPerPixel = GetDeviceCaps(hrootdc, BITSPIXEL);
			int localRasterCaps	  = GetDeviceCaps(hrootdc, RASTERCAPS);
			log.Print(2, _T("Memory DC has depth of %d and %s pallete-based.\n"), 
				localBitsPerPixel, (localRasterCaps & RC_PALETTE) ? "is" : "is not");
			
			// If we're using truecolor, and the server has more bits than we do
			if ( (localBitsPerPixel > myFormat.depth) && 
				! (localRasterCaps & RC_PALETTE)) {
				myFormat.depth = localBitsPerPixel;

				// create a bitmap compatible with the current display
				// call GetDIBits twice to get the colour info.
				// set colour masks and shifts
				
			}

			ReleaseDC(NULL, hrootdc);
		}
	}

	// The endian will be set before sending
}

void ClientConnection::SetFormatAndEncodings()
{
	// Set pixel format to myFormat
    
	rfbSetPixelFormatMsg spf;

    spf.type = rfbSetPixelFormat;
    spf.format = myFormat;
    spf.format.redMax = Swap16IfLE(spf.format.redMax);
    spf.format.greenMax = Swap16IfLE(spf.format.greenMax);
    spf.format.blueMax = Swap16IfLE(spf.format.blueMax);
	spf.format.bigEndian = 0;

    WriteExact((char *)&spf, sz_rfbSetPixelFormatMsg);
	
	// shortcuts to speed things up
    rs = myFormat.redShift;   rm = myFormat.redMax;
    gs = myFormat.greenShift; gm = myFormat.greenMax;
    bs = myFormat.blueShift;  bm = myFormat.blueMax;

    // The number of bytes required to hold at least one pixel.
	minPixelBytes = (myFormat.bitsPerPixel + 7) >> 3;
	assert(minPixelBytes > 0 && minPixelBytes < 7);  // or something is very wrong!

	// Set encodings
    char buf[sz_rfbSetEncodingsMsg + MAX_ENCODINGS * 4];
    rfbSetEncodingsMsg *se = (rfbSetEncodingsMsg *)buf;
    CARD32 *encs = (CARD32 *)(&buf[sz_rfbSetEncodingsMsg]);
    int len = 0;
	
    se->type = rfbSetEncodings;
    se->nEncodings = 0;

	// Put the preferred encoding first, and change it if the
	// preferred encoding is not actually usable.
	for (int i = LASTENCODING; i >= rfbEncodingRaw; i--)
	{
		if (opts.m_PreferredEncoding == i) {
			if (opts.m_UseEnc[i]) {
				encs[se->nEncodings++] = Swap32IfLE(i);
			} else {
				opts.m_PreferredEncoding--;
			}
		}
	}

	// Now we go through and put in all the other encodings in order.
	// We do rather assume that the most recent encoding is the most
	// desirable!
	for (i = LASTENCODING; i >= rfbEncodingRaw; i--)
	{
		if ( (opts.m_PreferredEncoding != i) &&
			 (opts.m_UseEnc[i]))
		{
			encs[se->nEncodings++] = Swap32IfLE(i);
		}
	}

    len = sz_rfbSetEncodingsMsg + se->nEncodings * 4;
	
    se->nEncodings = Swap16IfLE(se->nEncodings);
	
    WriteExact((char *) buf, len);


}


inline void ClientConnection::SendIncrementalFramebufferUpdateRequest()
{
    SendFramebufferUpdateRequest(0, 0, si.framebufferWidth,
					si.framebufferHeight, true);
}

inline void ClientConnection::SendFullFramebufferUpdateRequest()
{
    SendFramebufferUpdateRequest(0, 0, si.framebufferWidth,
					si.framebufferHeight, false);
}

/*
 * SendFramebufferUpdateRequest.
 */

inline void
ClientConnection::SendFramebufferUpdateRequest(int x, int y, int w, int h, bool incremental)
{
    rfbFramebufferUpdateRequestMsg fur;

    fur.type = rfbFramebufferUpdateRequest;
    fur.incremental = incremental ? 1 : 0;
    fur.x = Swap16IfLE(x);
    fur.y = Swap16IfLE(y);
    fur.w = Swap16IfLE(w);
    fur.h = Swap16IfLE(h);

    WriteExact((char *)&fur, sz_rfbFramebufferUpdateRequestMsg);
}



// Screen updating ---------------------------------------------------------

void ClientConnection::ScreenUpdate() {

	rfbFramebufferUpdateMsg sut;
	ReadExact((char *) &sut, sz_rfbFramebufferUpdateMsg);
    sut.nRects = Swap16IfLE(sut.nRects);
	if (sut.nRects == 0) return;
	
	// No other threads can use DC
	omni_mutex_lock l(bitmapdcMutex);
	
	HBITMAP hOldBmp = SelectObject(hBitmapDC, hBitmap);

	// Find the bounding region of this batch of updates
	HRGN fullregion = NULL;
	
	for (UINT i=0; i < sut.nRects; i++) {
	
		rfbFramebufferUpdateRectHeader surh;
		ReadExact((char *) &surh, sz_rfbFramebufferUpdateRectHeader);
        surh.r.x = Swap16IfLE(surh.r.x);
        surh.r.y = Swap16IfLE(surh.r.y);
        surh.r.w = Swap16IfLE(surh.r.w);
        surh.r.h = Swap16IfLE(surh.r.h);
		surh.encoding = Swap32IfLE(surh.encoding);
		
		switch (surh.encoding) {
		case rfbEncodingRaw:
            RawRect(&surh);
			break;
		case rfbEncodingCopyRect:
			CopyRect(&surh);
			break;
		case rfbEncodingRRE:
			RRERect(&surh);
			break;
		case rfbEncodingCoRRE:
			CoRRERect(&surh);
			break;
		case rfbEncodingHextile:
			HextileRect(&surh);
			break;
		default:
			log.Print(0, _T("Unknown encoding %d - not supported!\n"), surh.encoding);
			break;
		}

		// Either we update the window when all rectangles have been received

		if (opts.m_batchmode) {

			if (fullregion == NULL) {
				fullregion = CreateRectRgn(  
					surh.r.x - hScrollPos, surh.r.y - vScrollPos,
					surh.r.x - hScrollPos + surh.r.w, 
					surh.r.y - vScrollPos + surh.r.h);
			} else {
				HRGN rectregion = CreateRectRgn(  
					surh.r.x - hScrollPos, surh.r.y - vScrollPos,
					surh.r.x - hScrollPos + surh.r.w, 
					surh.r.y - vScrollPos + surh.r.h);
				CombineRgn(fullregion, fullregion, rectregion, RGN_OR);
				DeleteObject(rectregion);
			}

		// or we do it for each rectangle

		} else {
			
			RECT rect;
			rect.left = surh.r.x - hScrollPos;
			rect.top = surh.r.y - vScrollPos;
			rect.right = rect.left+surh.r.w;
			rect.bottom = rect.top+surh.r.h;
			InvalidateRect(m_hwnd, &rect, FALSE);
		}
	}

	if (opts.m_batchmode) {
		InvalidateRgn(m_hwnd, fullregion, FALSE);
		DeleteObject(fullregion);
	}

	
	SelectObject(hBitmapDC, hOldBmp);
}


// ----------------- Here come the encodings ------------------------------

/*
 *  Raw encoding
 */

// This is fun for debugging purposes - it shows all the 'raw' pixels in
// inverse colour. This applies to bits drawn with the Raw encoding or with
// the raw subencoding of hextile.

// #define RAWCOLOURING

#ifdef RAWCOLOURING

#define SETPIXELS(bpp, x, y, w, h)															\
	{																			\
		CARD##bpp *p = (CARD##bpp *) netbuf;									\
		for (int k = y; k < y+h; k++) {										\
			for (int j = x; j < x+w; j++) {									\
                    CARD##bpp pix = *p;
					COLORREF c = COLOR_FROM_PIXEL##bpp##(pix);		\
					COLORREF neg = RGB(										\
						255-GetRValue(c),										\
						255-GetGValue(c),										\
						255-GetBValue(c));										\
					SetPixelV(hBitmapDC, j,k, neg);							\
					p++;														\
			}																	\
		}																		\
	}

#else   

#define SETPIXELS(bpp, x, y, w, h)															\
	{																			\
		CARD##bpp *p = (CARD##bpp *) netbuf;									\
        register CARD##bpp pix;                 \
		for (int k = y; k < y+h; k++) {			\
			for (int j = x; j < x+w; j++) {		\
                    pix = *p;          \
                    SetPixelV(hBitmapDC, j,k, COLOR_FROM_PIXEL##bpp##(pix));		\
					p++;														\
			}																	\
		}																		\
	}
 
#endif	 

void ClientConnection::RawRect(rfbFramebufferUpdateRectHeader *pfburh) {

	UINT numpixels = pfburh->r.w * pfburh->r.h;
    // this assumes at least one byte per pixel. Naughty.
	UINT numbytes = numpixels * minPixelBytes;
	// Read in the whole thing
    CheckBufferSize(numbytes);
	ReadExact(netbuf, numbytes);

	// This big switch is untidy but fast
	switch (myFormat.bitsPerPixel) {
	case 8:
		SETPIXELS(8, pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h)
		break;
	case 16:
		SETPIXELS(16, pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h)
		break;
	case 24:
	case 32:
		SETPIXELS(32, pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h)            
		break;
	default:
		log.Print(0, _T("Invalid number of bits per pixel: %d\n"), myFormat.bitsPerPixel);
		return;
	}
	
}

/*
 *  CopyRect Encoding
 */

void ClientConnection::CopyRect(rfbFramebufferUpdateRectHeader *pfburh) {
	rfbCopyRect cr;
	ReadExact((char *) &cr, sz_rfbCopyRect);
	cr.srcX = Swap16IfLE(cr.srcX); cr.srcY = Swap16IfLE(cr.srcY);
	
	if (!BitBlt(
		hBitmapDC,pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h,
		hBitmapDC, cr.srcX, cr.srcY, SRCCOPY)) {
		log.Print(0, _T("Error in blit in ClientConnection::CopyRect\n"));
	}
}

/*
 *  RRE (Rising Rectangle Encoding)
 */

void ClientConnection::RRERect(rfbFramebufferUpdateRectHeader *pfburh)
{
	// An RRE rect is always followed by a background color
	// For speed's sake we read them together into a buffer.
	char tmpbuf[sz_rfbRREHeader+4];			// biggest pixel is 4 bytes long
	assert(myFormat.bitsPerPixel <= 32);   // check that!
    rfbRREHeader *prreh = (rfbRREHeader *) tmpbuf;
	CARD8 *pcolor = (CARD8 *) tmpbuf + sz_rfbRREHeader;
	ReadExact(tmpbuf, sz_rfbRREHeader + minPixelBytes);

	prreh->nSubrects = Swap32IfLE(prreh->nSubrects);

    COLORREF color;
    switch (myFormat.bitsPerPixel) {
        case 8:
            color = COLOR_FROM_PIXEL8_ADDRESS(pcolor); break;
        case 16:
			color = COLOR_FROM_PIXEL16_ADDRESS(pcolor); break;
        case 24:
        case 32:
            color = COLOR_FROM_PIXEL32_ADDRESS(pcolor); break;
    }

    // Draw the background of the rectangle
    FillSolidRect(pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h, color);

    if (prreh->nSubrects == 0) return;

	// Draw the sub-rectangles
    rfbRectangle rect, *pRect;
	// The size of an RRE subrect including color info
	int subRectSize = minPixelBytes + sz_rfbRectangle;
    
	// Read subrects into the buffer 
	CheckBufferSize(subRectSize * prreh->nSubrects);
    ReadExact(netbuf, subRectSize * prreh->nSubrects);
	BYTE *p = (BYTE *) netbuf;

    for (CARD32 i = 0; i < prreh->nSubrects; i++) {
        pRect = (rfbRectangle *) (p + minPixelBytes);

		switch (myFormat.bitsPerPixel) {
        case 8:
            color = COLOR_FROM_PIXEL8_ADDRESS(p); break;
        case 16:
            color = COLOR_FROM_PIXEL16_ADDRESS(p); break;
        case 32:
            color = COLOR_FROM_PIXEL32_ADDRESS(p); break;
        };
		
		// color = COLOR_FROM_PIXEL8_ADDRESS(netbuf);
        rect.x = (CARD16) (Swap16IfLE(pRect->x) + pfburh->r.x);
        rect.y = (CARD16) (Swap16IfLE(pRect->y) + pfburh->r.y);
        rect.w = Swap16IfLE(pRect->w);
        rect.h = Swap16IfLE(pRect->h);
        FillSolidRect(rect.x, rect.y, rect.w, rect.h, color);
		p+=subRectSize;
    }
}

void ClientConnection::CoRRERect(rfbFramebufferUpdateRectHeader *pfburh)
{
	// An RRE rect is always followed by a background color
	// For speed's sake we read them together into a buffer.
	char tmpbuf[sz_rfbRREHeader+4];			// biggest pixel is 4 bytes long
	assert(myFormat.bitsPerPixel <= 32);   // check that!
    rfbRREHeader *prreh = (rfbRREHeader *) tmpbuf;
	CARD8 *pcolor = (CARD8 *) tmpbuf + sz_rfbRREHeader;
	ReadExact(tmpbuf, sz_rfbRREHeader + minPixelBytes);

	prreh->nSubrects = Swap32IfLE(prreh->nSubrects);

    COLORREF color;
    switch (myFormat.bitsPerPixel) {
        case 8:
            color = COLOR_FROM_PIXEL8_ADDRESS(pcolor); break;
        case 16:
			color = COLOR_FROM_PIXEL16_ADDRESS(pcolor); break;
        case 24:
        case 32:
            color = COLOR_FROM_PIXEL32_ADDRESS(pcolor); break;
    }

    // Draw the background of the rectangle
    FillSolidRect(pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h, color);

    if (prreh->nSubrects == 0) return;

	// Draw the sub-rectangles
    rfbCoRRERectangle *pRect;
	rfbRectangle rect;

	// The size of an CoRRE subrect including color info
	int subRectSize = minPixelBytes + sz_rfbCoRRERectangle;

	// Read subrects into the buffer 
	CheckBufferSize(subRectSize * prreh->nSubrects);
    ReadExact(netbuf, subRectSize * prreh->nSubrects);
	BYTE *p = (BYTE *) netbuf;

    for (CARD32 i = 0; i < prreh->nSubrects; i++) {
        pRect = (rfbCoRRERectangle *) (p + minPixelBytes);

		switch (myFormat.bitsPerPixel) {
        case 8:
            color = COLOR_FROM_PIXEL8_ADDRESS(p); break;
        case 16:
            color = COLOR_FROM_PIXEL16_ADDRESS(p); break;
        case 32:
            color = COLOR_FROM_PIXEL32_ADDRESS(p); break;
        };
		
		// color = COLOR_FROM_PIXEL8_ADDRESS(netbuf);
        rect.x = pRect->x + pfburh->r.x;
        rect.y = pRect->y + pfburh->r.y;
        rect.w = pRect->w;
        rect.h = pRect->h;
        FillSolidRect(rect.x, rect.y, rect.w, rect.h, color);
		p+=subRectSize;
    }
}

/*
 *  Hextile Encoding.
 */

void ClientConnection::HextileRect(rfbFramebufferUpdateRectHeader *pfburh)
{
	switch (myFormat.bitsPerPixel) {
	case 8:
		HandleHextileEncoding8(pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h);
		break;
	case 16:
		HandleHextileEncoding16(pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h);
		break;
	case 32:
		HandleHextileEncoding32(pfburh->r.x, pfburh->r.y, pfburh->r.w, pfburh->r.h);
		break;
	}
}


#define DEFINE_HEXTILE(bpp)                                                   \
void ClientConnection::HandleHextileEncoding##bpp(int rx, int ry, int rw, int rh)                    \
{                                                                             \
    CARD##bpp bg, fg;                                                         \
	COLORREF bgcolor, fgcolor;												  \
    int i;                                                                    \
    CARD8 *ptr;                                                               \
    int x, y, w, h;                                                           \
    int sx, sy, sw, sh;                                                       \
    CARD8 subencoding;                                                        \
    CARD8 nSubrects;                                                          \
																			 \
    CheckBufferSize( 16 * 16 * bpp );											  \
                                                                              \
    for (y = ry; y < ry+rh; y += 16) {                                        \
        for (x = rx; x < rx+rw; x += 16) {                                    \
            w = h = 16;                                                       \
            if (rx+rw - x < 16)                                               \
                w = rx+rw - x;                                                \
            if (ry+rh - y < 16)                                               \
                h = ry+rh - y;                                                \
                                                                              \
            ReadExact((char *)&subencoding, 1);                               \
                                                                              \
            if (subencoding & rfbHextileRaw) {                                \
                ReadExact(netbuf, w * h * (bpp / 8));                         \
                SETPIXELS(bpp, x,y,w,h)                                       \
                continue;                                                     \
            }                                                                 \
                                                                              \
		    if (subencoding & rfbHextileBackgroundSpecified) {                \
                ReadExact((char *)&bg, (bpp/8));                              \
				bgcolor = COLOR_FROM_PIXEL##bpp##_ADDRESS(&bg);  			  \
			}																  \
            FillSolidRect(x,y,w,h,bgcolor);                                   \
                                                                              \
            if (subencoding & rfbHextileForegroundSpecified)  {               \
                ReadExact((char *)&fg, (bpp/8));                              \
				fgcolor = COLOR_FROM_PIXEL##bpp##_ADDRESS(&fg);				  \
			}                                                                 \
                                                                              \
            if (!(subencoding & rfbHextileAnySubrects)) {                     \
                continue;                                                     \
            }                                                                 \
                                                                              \
            ReadExact( (char *)&nSubrects, 1) ;                              \
                                                                              \
            ptr = (CARD8 *)netbuf;                                            \
                                                                              \
            if (subencoding & rfbHextileSubrectsColoured) {                   \
				                                                              \
                ReadExact( netbuf, nSubrects * (2 + (bpp / 8)));              \
                                                                              \
                for (i = 0; i < nSubrects; i++) {                             \
                    fgcolor = COLOR_FROM_PIXEL##bpp##_ADDRESS(ptr);           \
					ptr += (bpp/8);                                           \
                    sx = *ptr >> 4;                                           \
                    sy = *ptr++ & 0x0f;                                       \
                    sw = (*ptr >> 4) + 1;                                     \
                    sh = (*ptr++ & 0x0f) + 1;                                 \
                    FillSolidRect(x+sx, y+sy, sw, sh, fgcolor);               \
                }                                                             \
                                                                              \
            } else {                                                          \
                ReadExact(netbuf, nSubrects * 2);                    \
                                                                              \
                for (i = 0; i < nSubrects; i++) {                             \
                    sx = *ptr >> 4;                                           \
                    sy = *ptr++ & 0x0f;                                       \
                    sw = (*ptr >> 4) + 1;                                     \
                    sh = (*ptr++ & 0x0f) + 1;                                 \
                    FillSolidRect(x+sx, y+sy, sw, sh, fgcolor);               \
                }                                                             \
            }                                                                 \
        }                                                                     \
    }                                                                         \
                                                                              \
}

DEFINE_HEXTILE(8)
DEFINE_HEXTILE(16)
DEFINE_HEXTILE(32)


// The server has copied some text to the clipboard - put it 
// in the Windows clipboard too.

void ClientConnection::ServerCutText() {
	rfbServerCutTextMsg sctm;
	ReadExact((char *) &sctm, sz_rfbServerCutTextMsg);
	int len = Swap32IfLE(sctm.length);
	
	CheckBufferSize(len);
	if (len == 0) {
		netbuf[0] = '\0';
	} else {
		ReadString(netbuf, len);
	}

	// Copy to wincontents replacing LF with CR-LF
	char *wincontents = new char[len * 2 + 1];
	for (int i = 0, j = 0; netbuf[i] != 0; i++, j++) {
        if (netbuf[i] == '\x0a') {
			wincontents[j++] = '\x0d';
            len++;
        }
		wincontents[j] = netbuf[i];
	}
	wincontents[j] = '\0';

    // The clipboard should not be modified by more than one thread at once
    {
        omni_mutex_lock l(clipMutex);

        if (!OpenClipboard(m_hwnd)) {
	        throw WarningException("Failed to open clipboard\n");
        }
        if (! ::EmptyClipboard()) {
	        throw WarningException("Failed to empty clipboard\n");
        }

        // Allocate a global memory object for the text. 
        HGLOBAL hglbCopy = GlobalAlloc(GMEM_DDESHARE, (len +1) * sizeof(TCHAR));
        if (hglbCopy != NULL) { 
	        // Lock the handle and copy the text to the buffer.  
	        LPTSTR lptstrCopy = (LPTSTR) GlobalLock(hglbCopy); 
	        memcpy(lptstrCopy, wincontents, len * sizeof(TCHAR)); 
	        lptstrCopy[len] = (TCHAR) 0;    // null character 
	        GlobalUnlock(hglbCopy);          // Place the handle on the clipboard.  
	        SetClipboardData(CF_TEXT, hglbCopy); 
        }

        delete [] wincontents;

        if (! ::CloseClipboard()) {
	        throw WarningException("Failed to close clipboard\n");
        }
    }
}

void ClientConnection::Bell() {
	rfbBellMsg bm;
	ReadExact((char *) &bm, sz_rfbBellMsg);

	if (! ::PlaySound("VNCViewerBell", NULL, 
		SND_APPLICATION | SND_ALIAS | SND_NODEFAULT | SND_ASYNC) ) {
		::Beep(440, 125);
	}
	if (opts.m_DeiconifyOnBell) {
		WINDOWPLACEMENT wp;
		GetWindowPlacement(m_hwnd, &wp);
		if (wp.showCmd == SW_MINIMIZE || wp.showCmd == SW_SHOWMINIMIZED) {
			ShowWindow(m_hwnd, SW_SHOWNORMAL);
		}
	}
}


// These draw a solid rectangle of colour on the bitmap
// They assume the bitmap is already selected into the DC, and the
// DC is locked if necessary.

inline void ClientConnection::FillSolidRect(RECT *pRect, COLORREF color) {
	COLORREF oldbgcol = SetBkColor(hBitmapDC, color);
	// This is the call MFC uses for FillSolidRect. Who am I to argue?
	::ExtTextOut(hBitmapDC, 0, 0, ETO_OPAQUE, pRect, NULL, 0, NULL);
	SetBkColor(hBitmapDC,oldbgcol);
}

inline void ClientConnection::FillSolidRect(int x, int y, int w, int h, 
											COLORREF color) {
	RECT r;
	r.left = x;		r.right = x + w;
	r.top = y;		r.bottom = y + h;
	FillSolidRect(&r, color);
}

//
// SendPointerEvent.
//

inline void
ClientConnection::SendPointerEvent(int x, int y, int buttonMask)
{
    rfbPointerEventMsg pe;

    pe.type = rfbPointerEvent;
    pe.buttonMask = buttonMask;
    if (x < 0) x = 0;
    if (y < 0) y = 0;
    pe.x = Swap16IfLE(x);
    pe.y = Swap16IfLE(y);
	try {
		WriteExact((char *)&pe, sz_rfbPointerEventMsg);
	} catch (WarningException &e) {
		e.Report();
		Kill();
	}
}

//
// ProcessKeyEvent
//
// Normally a single Windows key event will map onto a single RFB
// key message, but this is not always the case.  Much of the stuff
// here is to handle AltGr (=Ctrl-Alt) on international keyboards.
// Example cases:
//
//    We want Ctrl-F to be sent as:
//      Ctrl-Down, F-Down, F-Up, Ctrl-Up.
//    because there is no keysym for ctrl-f, and because the ctrl
//    will already have been sent by the time we get the F.
//
//    On German keyboards, @ is produced using AltGr-Q, which is
//    Ctrl-Alt-Q.  But @ is a valid keysym in its own right, and when
//    a German user types this combination, he doesn't mean Ctrl-@.
//    So for this we will send, in total:
//
//      Ctrl-Down, Alt-Down,   
//                 (when we get the AltGr pressed)
//
//      Alt-Up, Ctrl-Up, @-Down, Ctrl-Down, Alt-Down 
//                 (when we discover that this is @ being pressed)
//
//      Alt-Up, Ctrl-Up, @-Up, Ctrl-Down, Alt-Down
//                 (when we discover that this is @ being released)
//
//      Alt-Up, Ctrl-Up
//                 (when the AltGr is released)

inline void ClientConnection::ProcessKeyEvent(int virtkey, DWORD keyData)
{
    bool down = ((keyData & 0x80000000l) == 0);

    // if virtkey found in mapping table, send X equivalent
    // else
    //   try to convert directly to ascii
    //   if result is in range supported by X keysyms,
    //      raise any modifiers, send it, then restore mods
    //   else
    //      calculate what the ascii would be without mods
    //      send that

#ifdef _DEBUG
    char keyname[32];
    if (GetKeyNameText(  keyData,keyname, 31)) {
        log.Print(4, _T("Process key: %s (keyData %04x): "), keyname, keyData);
    };
#endif

    KeyActionSpec kas = keymap.PCtoX(virtkey, keyData);
    

    if (kas.releaseModifiers & KEYMAP_LCONTROL) {
        SendKeyEvent(XK_Control_L, false );
        log.Print(5, _T("fake L Ctrl raised\n"));
    }
    if (kas.releaseModifiers & KEYMAP_LALT) {
        SendKeyEvent(XK_Alt_L, false );
        log.Print(5, _T("fake L Alt raised\n"));
    }
    if (kas.releaseModifiers & KEYMAP_RCONTROL) {
        SendKeyEvent(XK_Control_R, false );
        log.Print(5, _T("fake R Ctrl raised\n"));
    }
    if (kas.releaseModifiers & KEYMAP_RALT) {
        SendKeyEvent(XK_Alt_R, false );
        log.Print(5, _T("fake R Alt raised\n"));
    }

    for (int i = 0; kas.keycodes[i] != XK_VoidSymbol && i < MaxKeysPerKey; i++) {
        SendKeyEvent(kas.keycodes[i], down );
        log.Print(4, _T("Sent keysym %04x (%s)\n"), 
            kas.keycodes[i], down ? _T("press") : _T("release"));
    }

    if (kas.releaseModifiers & KEYMAP_RALT) {
        SendKeyEvent(XK_Alt_R, true );
        log.Print(5, _T("fake R Alt pressed\n"));
    }
    if (kas.releaseModifiers & KEYMAP_RCONTROL) {
        SendKeyEvent(XK_Control_R, true );
        log.Print(5, _T("fake R Ctrl pressed\n"));
    }
    if (kas.releaseModifiers & KEYMAP_LALT) {
        SendKeyEvent(XK_Alt_L, false );
        log.Print(5, _T("fake L Alt pressed\n"));
    }
    if (kas.releaseModifiers & KEYMAP_LCONTROL) {
        SendKeyEvent(XK_Control_L, false );
        log.Print(5, _T("fake L Ctrl pressed\n"));
    }

}

//
// SendKeyEvent
//

inline void
ClientConnection::SendKeyEvent(CARD32 key, bool down)
{
    rfbKeyEventMsg ke;

    ke.type = rfbKeyEvent;
    ke.down = down ? 1 : 0;
    ke.key = Swap32IfLE(key);
    WriteExact((char *)&ke, sz_rfbKeyEventMsg);
    log.Print(6, _T("SendKeyEvent: key = x%04x status = %s\n"), key, 
        down ? _T("down") : _T("up"));
}

inline void ClientConnection::SendCtlAltDel() 
{
    SendKeyEvent(XK_Control_L, true);
    SendKeyEvent(XK_Alt_L, true);
    SendKeyEvent(XK_Delete, true);
    SendKeyEvent(XK_Delete, false);
    SendKeyEvent(XK_Alt_L, false);
    SendKeyEvent(XK_Control_L, false);
}


/*
 * SendClientCutText.
 */

inline void
ClientConnection::SendClientCutText(char *str, int len)
{
    rfbClientCutTextMsg cct;

    cct.type = rfbClientCutText;
    cct.length = Swap32IfLE(len);
    WriteExact((char *)&cct, sz_rfbClientCutTextMsg);
	WriteExact(str, len);
}


// Reads the number of bytes specified into the buffer given

inline void ClientConnection::ReadExact(char *inbuf, int wanted)
{
	assert(inbuf != NULL);

	omni_mutex_lock l(readMutex);
	// omni_mutex_lock l2(sockMutex);
	int offset = 0;
	
	while (wanted > 0) {

		int bytes = recv(m_sock, inbuf+offset, wanted, 0);
		if (bytes == 0) throw WarningException("Connection closed.");
		if (bytes == SOCKET_ERROR) {
			int err = ::GetLastError();
			log.Print(1, _T("Socket error %d\n"), err);
			running = false;
			throw WarningException("ReadExact: Socket error while reading.");
		}
		wanted -= bytes;
		offset += bytes;

	}
}

// Read the number of bytes and return them zero terminated in the buffer 
inline void ClientConnection::ReadString(char *buf, int length)
{
	assert(buf);
	if (length > 0)
		ReadExact(buf, length);
	buf[length] = '\0';
}


// Sends the number of bytes specified from the buffer
inline void ClientConnection::WriteExact(char *buf, int bytes)
{
	assert(buf != NULL);
	if (bytes == 0) return;
	
	omni_mutex_lock l(writeMutex);
	// omni_mutex_lock l2(sockMutex);

	int i = 0;
    int j;

    while (i < bytes) {

		j = send(m_sock, buf+i, bytes-i, 0);
		if (j == SOCKET_ERROR || j==0) {
			int err = ::GetLastError();
			log.Print(1, _T("Socket error %d\n"), err);
			running = false;
			throw WarningException("WriteExact: Socket error while writing.");
		}
		i += j;
    }
}

// Makes sure netbuf is at least as big as the specified size.
// Note that netbuf itself may change as a result of this call.
// Throws an exception on failure.
inline void ClientConnection::CheckBufferSize(int bufsize)
{
	if (netbufsize > bufsize) return;

	omni_mutex_lock l(bufferMutex);

	char *newbuf = new char[bufsize+256];;
	if (newbuf == NULL) {
		throw ErrorException("Insufficient memory to allocate network buffer.");
	}

	// Only if we're successful...

	if (netbuf != NULL)
		delete [] netbuf;
	netbuf = newbuf;
	netbufsize=bufsize + 256;
	log.Print(4, _T("bufsize expanded to %d\n"), netbufsize);
}

