//  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
//
//  This file is part of the VNC system.
//
//  The VNC system is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
//  USA.
//
// If the source code for the VNC system is not available from the place 
// whence you received this file, check http://www.orl.co.uk/vnc or contact
// the authors on vnc@orl.co.uk for information on obtaining it.


// vncProperties.cpp

// Implementation of the Properties dialog!

#include "stdhdrs.h"

#include "WinVNC.h"
#include "vncProperties.h"
#include "vncServer.h"
extern "C" {
#include "vncauth.h"
}

const char WINVNC_REGISTRY_KEY [] = "Software\\ORL\\WinVNC3";
const char NO_PASSWORD_WARN [] = "WARNING : Running WinVNC without setting a password is "
								"a dangerous security risk!\n"
								"Until you set a password, WinVNC will not accept incoming connections.";

// Constructor & Destructor
vncProperties::vncProperties()
{
	m_dlgvisible = FALSE;
}

vncProperties::~vncProperties()
{
}

// Initialisation
BOOL
vncProperties::Init(vncServer *server)
{
	// Save the server pointer
	m_server = server;
	
	// Load the settings from the registry
	Load();

	// If the password is empty then always show a dialog
	char passwd[MAXPWLEN+1];
	m_server->GetPassword(passwd);
	if (strlen(passwd) == 0)
		Show(TRUE);

	return TRUE;
}

// Dialog box handling functions
void
vncProperties::Show(BOOL show)
{
	if (show)
	{
		if (!m_dlgvisible)
		{
			for (;;)
			{
				// Do the dialog box
				int result = DialogBoxParam(hAppInstance,
											MAKEINTRESOURCE(IDD_PROPERTIES), 
											NULL,
											(DLGPROC) DialogProc,
											(LONG) this);
				if (result == -1)
				{
					// Dialog box failed, so quit
					PostQuitMessage(0);
					return;
				}

				// We're allowed to exit if the password is not empty
				char passwd[MAXPWLEN+1];
				m_server->GetPassword(passwd);
				if (strlen(passwd) != 0)
					return;

				// The password is empty, so if OK was used then redisplay the box,
				// otherwise, if CANCEL was used, close down WinVNC
				if (result == IDCANCEL)
				{
					PostQuitMessage(0);
					return;
				}

				// If we reached here then OK was used & there is no password!
				MessageBox(NULL, NO_PASSWORD_WARN, "WinVNC Warning", MB_OK | MB_ICONEXCLAMATION);
			}
		}
	}
}

BOOL CALLBACK
vncProperties::DialogProc(HWND hwnd,
						  UINT uMsg,
						  WPARAM wParam,
						  LPARAM lParam )
{
	// We use the dialog-box's USERDATA to store a _this pointer
	// This is set only once WM_INITDIALOG has been recieved, though!
	vncProperties *_this = (vncProperties *) GetWindowLong(hwnd, GWL_USERDATA);

	switch (uMsg)
	{

	case WM_INITDIALOG:
		{
			// Retrieve the Dialog box parameter and use it as a pointer
			// to the calling vncProperties object
			SetWindowLong(hwnd, GWL_USERDATA, lParam);
			_this = (vncProperties *) lParam;

			// Initialise the properties controls
			HWND hConnectSock = GetDlgItem(hwnd, IDC_CONNECT_SOCK);
			SendMessage(hConnectSock,
				BM_SETCHECK,
				_this->m_server->SockConnected(),
				0);

			HWND hConnectCorba = GetDlgItem(hwnd, IDC_CONNECT_CORBA);
			SendMessage(hConnectCorba,
				BM_SETCHECK,
				_this->m_server->CORBAConnected(),
				0);
#if(defined(_CORBA))
			EnableWindow(hConnectCorba, TRUE);
#else
			EnableWindow(hConnectCorba, FALSE);
#endif

			HWND hPortNo = GetDlgItem(hwnd, IDC_PORTNO);
			SetDlgItemInt(hwnd, IDC_PORTNO, _this->m_server->GetPort() - RFB_PORT_OFFSET, FALSE);
			EnableWindow(hPortNo, _this->m_server->SockConnected());
			
			HWND hPassword = GetDlgItem(hwnd, IDC_PASSWORD);
			EnableWindow(hPassword, _this->m_server->SockConnected());

			// Get the password
			char passwd[MAXPWLEN+1];
			_this->m_server->GetPassword(passwd);
			SetDlgItemText(hwnd, IDC_PASSWORD, (LPSTR) &passwd);

			// Remote input settings
			HWND hEnableInputs = GetDlgItem(hwnd, IDC_DISABLE_INPUTS);
			SendMessage(hEnableInputs,
				BM_SETCHECK,
				!(_this->m_server->InputsEnabled()),
				0);

			// Set the polling options
			HWND hPollFullScreen = GetDlgItem(hwnd, IDC_POLL_FULLSCREEN);
			SendMessage(hPollFullScreen,
				BM_SETCHECK,
				_this->m_server->PollFullScreen(),
				0);

			HWND hPollForeground = GetDlgItem(hwnd, IDC_POLL_FOREGROUND);
			SendMessage(hPollForeground,
				BM_SETCHECK,
				_this->m_server->PollForeground(),
				0);

			HWND hPollUnderCursor = GetDlgItem(hwnd, IDC_POLL_UNDER_CURSOR);
			SendMessage(hPollUnderCursor,
				BM_SETCHECK,
				_this->m_server->PollUnderCursor(),
				0);

			HWND hPollConsoleOnly = GetDlgItem(hwnd, IDC_CONSOLE_ONLY);
			SendMessage(hPollConsoleOnly,
				BM_SETCHECK,
				_this->m_server->PollConsoleOnly(),
				0);
			EnableWindow(hPollConsoleOnly,
				_this->m_server->PollUnderCursor() || _this->m_server->PollForeground()
				);

			HWND hPollOnEventOnly = GetDlgItem(hwnd, IDC_ONEVENT_ONLY);
			SendMessage(hPollOnEventOnly,
				BM_SETCHECK,
				_this->m_server->PollOnEventOnly(),
				0);
			EnableWindow(hPollOnEventOnly,
				_this->m_server->PollUnderCursor() || _this->m_server->PollForeground()
				);

			SetForegroundWindow(hwnd);

			_this->m_dlgvisible = TRUE;

			return TRUE;
		}

	case WM_COMMAND:
		switch (LOWORD(wParam))
		{

		case IDOK:
		case IDC_APPLY:
			{
				// Save the password
				char passwd[MAXPWLEN+1];
				if (GetDlgItemText(hwnd, IDC_PASSWORD, (LPSTR) &passwd, MAXPWLEN+1) == 0)
					_this->m_server->SetPassword("");
				else
					_this->m_server->SetPassword(passwd);

				// Save the new settings to the server
				BOOL success;
				UINT portno = GetDlgItemInt(hwnd, IDC_PORTNO, &success, TRUE);
				if (success)
					_this->m_server->SetPort(portno + RFB_PORT_OFFSET);
			
				HWND hConnectSock = GetDlgItem(hwnd, IDC_CONNECT_SOCK);
				_this->m_server->SockConnect(
					SendMessage(hConnectSock, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				HWND hConnectCorba = GetDlgItem(hwnd, IDC_CONNECT_CORBA);
				_this->m_server->CORBAConnect(
					SendMessage(hConnectCorba, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				// Remote input stuff
				HWND hEnableInputs = GetDlgItem(hwnd, IDC_DISABLE_INPUTS);
				_this->m_server->EnableInputs(
					SendMessage(hEnableInputs, BM_GETCHECK, 0, 0) != BST_CHECKED
					);

				// Handle the polling stuff
				HWND hPollFullScreen = GetDlgItem(hwnd, IDC_POLL_FULLSCREEN);
				_this->m_server->PollFullScreen(
					SendMessage(hPollFullScreen, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				HWND hPollForeground = GetDlgItem(hwnd, IDC_POLL_FOREGROUND);
				_this->m_server->PollForeground(
					SendMessage(hPollForeground, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				HWND hPollUnderCursor = GetDlgItem(hwnd, IDC_POLL_UNDER_CURSOR);
				_this->m_server->PollUnderCursor(
					SendMessage(hPollUnderCursor, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				HWND hPollConsoleOnly = GetDlgItem(hwnd, IDC_CONSOLE_ONLY);
				_this->m_server->PollConsoleOnly(
					SendMessage(hPollConsoleOnly, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				HWND hPollOnEventOnly = GetDlgItem(hwnd, IDC_ONEVENT_ONLY);
				_this->m_server->PollOnEventOnly(
					SendMessage(hPollOnEventOnly, BM_GETCHECK, 0, 0) == BST_CHECKED
					);

				// And to the registry
				_this->Save();

				// Was ok pressed?
				if (LOWORD(wParam) == IDOK)
				{
					// Yes, so close the dialog
					EndDialog(hwnd, IDOK);

					_this->m_dlgvisible = FALSE;
				}

				return TRUE;
			}

		case IDCANCEL:
			EndDialog(hwnd, IDCANCEL);
			_this->m_dlgvisible = FALSE;
			return TRUE;

		case IDC_CONNECT_SOCK:
			// The user has clicked on the socket connect tickbox
			{
				HWND hConnectSock = GetDlgItem(hwnd, IDC_CONNECT_SOCK);
				BOOL connectsockon =
					(SendMessage(hConnectSock, BM_GETCHECK, 0, 0) == BST_CHECKED);

				HWND hPortNo = GetDlgItem(hwnd, IDC_PORTNO);
				EnableWindow(hPortNo, connectsockon);
			
				HWND hPassword = GetDlgItem(hwnd, IDC_PASSWORD);
				EnableWindow(hPassword, connectsockon);
			}
			break;

		case IDC_POLL_FOREGROUND:
		case IDC_POLL_UNDER_CURSOR:
			// User has clicked on one of the polling mode buttons
			// affected by the pollconsole and pollonevent options
			{
				// Get the poll-mode buttons
				HWND hPollForeground = GetDlgItem(hwnd, IDC_POLL_FOREGROUND);
				HWND hPollUnderCursor = GetDlgItem(hwnd, IDC_POLL_UNDER_CURSOR);

				// Determine whether to enable the modifier options
				BOOL enabled = (SendMessage(hPollForeground, BM_GETCHECK, 0, 0) == BST_CHECKED) ||
					(SendMessage(hPollUnderCursor, BM_GETCHECK, 0, 0) == BST_CHECKED);

				HWND hPollConsoleOnly = GetDlgItem(hwnd, IDC_CONSOLE_ONLY);
				EnableWindow(hPollConsoleOnly, enabled);

				HWND hPollOnEventOnly = GetDlgItem(hwnd, IDC_ONEVENT_ONLY);
				EnableWindow(hPollOnEventOnly, enabled);
			}
			break;
		}

		break;

	case WM_DESTROY:
		EndDialog(hwnd, IDCANCEL);
		_this->m_dlgvisible = FALSE;
		return TRUE;

	}
	return 0;
}

// Functions to load & save the settings
LONG
vncProperties::LoadInt(HKEY key, LPCSTR valname, LONG defval)
{
	LONG pref;
	ULONG type = REG_DWORD;
	ULONG prefsize = sizeof(pref);

	if (RegQueryValueEx(key,
		valname,
		NULL,
		&type,
		(LPBYTE) &pref,
		&prefsize) != ERROR_SUCCESS)
		return defval;

	if (type != REG_DWORD)
		return defval;

	if (prefsize != sizeof(pref))
		return defval;

	return pref;
}

void
vncProperties::LoadPassword(char *buffer)
{
	HKEY key;
	DWORD type;
	int slen=MAXPWLEN+8;
	char inouttext[MAXPWLEN+8];
	char *passwd;

	// Get the registry key for this app
	if (RegCreateKey(HKEY_CURRENT_USER,
		WINVNC_REGISTRY_KEY,
		&key) != ERROR_SUCCESS)
	{
		strcpy((char *)buffer, "");
		return;
	}

	// Retrieve the encrypted password
	if (RegQueryValueEx(key,
		"Password",
		NULL,
		&type,
		(LPBYTE) &inouttext,
		(LPDWORD) &slen) != ERROR_SUCCESS)
	{
		RegCloseKey(key);
		strcpy(buffer, "");
		return;
	}

	RegCloseKey(key);

	// Check the type of the registry entry
	if (type != REG_BINARY)
		return;

	// Do the decryption
	passwd = vncDecryptPasswd(inouttext);
	if (passwd == NULL)
	{
		strcpy(buffer, "");
		return;
	}

	if (strlen(passwd) > MAXPWLEN)
		return;

	strcpy(buffer, passwd);
	free(passwd);
}

void
vncProperties::Load()
{
	HKEY appkey;

	// Get the registry key for this app
	if (RegCreateKey(HKEY_CURRENT_USER,
		WINVNC_REGISTRY_KEY,
		&appkey) != ERROR_SUCCESS)
		return;

	// Now load the preferences

	// Connection prefs
	m_server->SockConnect(
		LoadInt(appkey, "SocketConnect", m_server->SockConnected())
		);
	m_server->SetPort(
		LoadInt(appkey, "PortNumber", m_server->GetPort())
		);

	// Load the password
	char passwd[MAXPWLEN+1];
	LoadPassword(passwd);
	m_server->SetPassword(passwd);
	
	m_server->CORBAConnect(
		LoadInt(appkey, "CORBAConnect", m_server->CORBAConnected())
		);

	// Remote input prefs
	m_server->EnableInputs(
		LoadInt(appkey, "InputsEnabled", m_server->InputsEnabled())
		);

	// Polling prefs
	m_server->PollUnderCursor(
		LoadInt(appkey, "PollUnderCursor", m_server->PollUnderCursor())
		);
	m_server->PollForeground(
		LoadInt(appkey, "PollForeground", m_server->PollForeground())
		);
	m_server->PollFullScreen(
		LoadInt(appkey, "PollFullScreen", m_server->PollFullScreen())
		);

	m_server->PollConsoleOnly(
		LoadInt(appkey, "OnlyPollConsole", m_server->PollConsoleOnly())
		);
	m_server->PollOnEventOnly(
		LoadInt(appkey, "OnlyPollOnEvent", m_server->PollOnEventOnly())
		);

	RegCloseKey(appkey);
}

void
vncProperties::SaveInt(HKEY key, LPSTR valname, LONG val)
{
	RegSetValueEx(key, valname, 0, REG_DWORD, (LPBYTE) &val, sizeof(val));
}

void
vncProperties::SavePassword(char *buffer)
{
	HKEY key;
	char inouttext[MAXPWLEN+8];
	int len;

	// Get the registry key for this app
	if (RegCreateKey(HKEY_CURRENT_USER,
		WINVNC_REGISTRY_KEY,
		&key) != ERROR_SUCCESS)
		return;

	// Encrypt and save the key
	len = vncEncryptPasswd(buffer, inouttext);

	if (len != 0)
	{
		RegSetValueEx(key, "Password", 0, REG_BINARY, (LPBYTE) &inouttext, len);
	}

	RegCloseKey(key);
}

void
vncProperties::Save()
{
	HKEY appkey;

	// Get the registry key for this app
	if (RegCreateKey(HKEY_CURRENT_USER,
		WINVNC_REGISTRY_KEY,
		&appkey) != ERROR_SUCCESS)
		return;

	// Now save the preferences

	// Connection prefs
	SaveInt(appkey, "SocketConnect", m_server->SockConnected());
	SaveInt(appkey, "PortNumber", m_server->GetPort());
	SaveInt(appkey, "InputsEnabled", m_server->InputsEnabled());

	// Save the password
	char passwd[MAXPWLEN+1];
	m_server->GetPassword(passwd);
	SavePassword(passwd);

#if(defined(_CORBA))
	// Don't save the CORBA enabled flag if CORBA is not compiled in!
	SaveInt(appkey, "CORBAConnect", m_server->CORBAConnected());
#endif

	// Polling prefs
	SaveInt(appkey, "PollUnderCursor", m_server->PollUnderCursor());
	SaveInt(appkey, "PollForeground", m_server->PollForeground());
	SaveInt(appkey, "PollFullScreen", m_server->PollFullScreen());

	SaveInt(appkey, "OnlyPollConsole", m_server->PollConsoleOnly());
	SaveInt(appkey, "OnlyPollOnEvent", m_server->PollOnEventOnly());

	RegCloseKey(appkey);
}

