/*
 * tableinitcmtemplate.c - template for initialising lookup tables for
 * translation from a colour map to true colour.
 *
 * This file shouldn't be compiled.  It is included multiple times by
 * translate.c, each time with a different definition of the macro OUT.
 * For each value of OUT, this file defines a function which allocates an
 * appropriately sized lookup table and initialises it.
 *
 * I know this code isn't nice to read because of all the macros, but
 * efficiency is important here.
 */

#if !defined(OUT)
#error "This file shouldn't be compiled."
#error "It is included as part of translate.c"
#endif

#define OUT_T CONCAT2E(CARD,OUT)
#define SwapOUT(x) CONCAT2E(Swap,OUT) (x)
#define rfbInitColourMapSingleTableOUT \
				CONCAT2E(rfbInitColourMapSingleTable,OUT)

// THIS CODE HAS BEEN MODIFIED FROM THE ORIGINAL UNIX SOURCE
// TO WORK FOR WINVNC.  THE PALETTE SHOULD REALLY BE RETRIEVED
// FROM THE VNCDESKTOP OBJECT, RATHER THAN FROM THE OS DIRECTLY

static void
rfbInitColourMapSingleTableOUT (char **table,
								rfbPixelFormat *in,
								rfbPixelFormat *out)
{
	// COLOUR RETRIEVAL

	struct _BMInfo {
		BITMAPINFO		bmi;
		// Colormap info - comes straight after BITMAPINFO - **HACK**
		RGBQUAD			cmap[256];
	} bminfo;

	// Get the current output device
	HDC hrootdc = GetDC(NULL);
	if (hrootdc == NULL)
		return;

	// Create a compatible memory DC and a bitmap for it
	HDC hmemdc = CreateCompatibleDC(hrootdc);
	if (hmemdc == NULL)
	{
		ReleaseDC(NULL, hrootdc);
		return;
	}
	HBITMAP membitmap = CreateCompatibleBitmap(hrootdc, 1, 1);
	if (membitmap == NULL)
	{
		DeleteDC(hmemdc);
		ReleaseDC(NULL, hrootdc);
		return;
	}

	// Get the bitmap's format and colour details
	bminfo.bmi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bminfo.bmi.bmiHeader.biBitCount = 0;

	int result = GetDIBits(hmemdc, membitmap, 0, 1, NULL, &bminfo.bmi, DIB_RGB_COLORS);
	if (result == 0)
	{
		DeleteObject(membitmap);
		DeleteDC(hmemdc);
		ReleaseDC(NULL, hrootdc);
		return;
	}

	result = GetDIBits(hmemdc, membitmap,  0, 1, NULL, &bminfo.bmi, DIB_RGB_COLORS);
	if (result == 0)
	{
		DeleteObject(membitmap);
		DeleteDC(hmemdc);
		ReleaseDC(NULL, hrootdc);
		return;
	}

	// Free the Windows resources
	DeleteObject(membitmap);
	DeleteDC(hmemdc);
	ReleaseDC(NULL, hrootdc);

	// COLOUR TRANSLATION

	// We now have the colour table intact.  Map it into a translation table
    int i, r, g, b;
    OUT_T *t;
    int nEntries = 1 << in->bitsPerPixel;

	// Allocate the table
    if (*table) free(*table);
    *table = (char *)malloc(nEntries * sizeof(OUT_T));
    t = (OUT_T *)*table;

	_RPT2(_CRT_WARN, "vncTransColourMap : colours=%d (%d)\n", nEntries, 1 << bminfo.bmi.bmiHeader.biBitCount);

    for (i = 0; i < nEntries; i++)
	{
		// Split down the RGB quad
		r = bminfo.bmi.bmiColors[i].rgbRed;
		g = bminfo.bmi.bmiColors[i].rgbGreen;
		b = bminfo.bmi.bmiColors[i].rgbBlue;

		// Now translate it
		t[i] = ((((r * out->redMax + 127) / 255) << out->redShift) |
			(((g * out->greenMax + 127) / 255) << out->greenShift) |
			(((b * out->blueMax + 127) / 255) << out->blueShift));
#if (OUT != 8)
		if (out->bigEndian != in->bigEndian)
		{
			t[i] = SwapOUT(t[i]);
		}
#endif

		_RPT3(_CRT_WARN, "vncTransColourMap : colour %d=%x (%x)\n", i,
			*((int *) &bminfo.bmi.bmiColors[i]),
			t[i]);
    }
}

#undef OUT_T
#undef SwapOUT
#undef rfbInitColourMapSingleTableOUT
