//  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
//
//  This file is part of the VNC system.
//
//  The VNC system is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
//  USA.
//
// If the source code for the VNC system is not available from the place 
// whence you received this file, check http://www.orl.co.uk/vnc or contact
// the authors on vnc@orl.co.uk for information on obtaining it.


// Object used to encode data for RFB

#include "vncEncoder.h"
#include "vncBuffer.h"
// #include "rfb.h"

vncEncoder::vncEncoder()
{
	ZeroMemory(&m_remoteformat, sizeof(m_remoteformat));
	ZeroMemory(&m_localformat, sizeof(m_localformat));
	m_transtable = NULL;
	m_bytesPerRow = 0;
}

vncEncoder::~vncEncoder()
{
	if (m_transtable != NULL)
	{
		free(m_transtable);
		m_transtable = NULL;
	}
}

void
vncEncoder::Init()
{
}

UINT
vncEncoder::RequiredBuffSize(UINT width, UINT height)
{
	return sz_rfbFramebufferUpdateRectHeader +
		(width * height * m_remoteformat.bitsPerPixel)/8;
}

UINT
vncEncoder::NumCodedRects(RECT &rect)
{
	return 1;
}

// Translate a rectangle
inline void
vncEncoder::Translate(BYTE *source, BYTE *dest, const RECT &rect)
{
	// Calculate where in the source rectangle to read from
	BYTE *sourcepos = (BYTE *)(source + (m_bytesPerRow * rect.top)+(rect.left * (m_localformat.bitsPerPixel / 8)));

	// Call the translation function
	(*m_transfunc) (m_transtable,
					&m_localformat,
					&m_remoteformat,
					(char *)sourcepos,
					(char *)dest,
					m_bytesPerRow,
					rect.right-rect.left,
					rect.bottom-rect.top
					);
}

// Encode a rectangle
inline UINT
vncEncoder::EncodeRect(BYTE *source, BYTE *dest, const RECT &rect)
{
	// Create the header for the update in the destination area
	rfbFramebufferUpdateRectHeader *surh = (rfbFramebufferUpdateRectHeader *)dest;
	surh->r.x = (CARD16) rect.left;
	surh->r.y = (CARD16) rect.top;
	surh->r.w = (CARD16) (rect.right-rect.left);
	surh->r.h = (CARD16) (rect.bottom-rect.top);
	surh->r.x = Swap16IfLE(surh->r.x);
	surh->r.y = Swap16IfLE(surh->r.y);
	surh->r.w = Swap16IfLE(surh->r.w);
	surh->r.h = Swap16IfLE(surh->r.h);
	surh->encoding = Swap32IfLE(rfbEncodingRaw);

	// Translate the data in place in the output buffer
	Translate(source, dest + sz_rfbFramebufferUpdateRectHeader, rect);

	// Return the buffer size
	return sz_rfbFramebufferUpdateRectHeader +
		((rect.right-rect.left)*(rect.bottom-rect.top)*m_remoteformat.bitsPerPixel) / 8;
}

BOOL
vncEncoder::SetTranslateFunction()
{
	trace("vncEncoder : settranslatefunction called\n");

    // Check that bits per pixel values are valid

    if ((m_remoteformat.bitsPerPixel != 8) &&
		(m_remoteformat.bitsPerPixel != 16) &&
		(m_remoteformat.bitsPerPixel != 32))
    {
		trace("vncEncoder : only 8, 16 or 32 bits supported remotely - ");
		trace_int(m_remoteformat.bitsPerPixel);
		trace(" requested\n");

		return FALSE;
    }
	
    if ((m_localformat.bitsPerPixel != 8) &&
		(m_localformat.bitsPerPixel != 16) &&
		(m_localformat.bitsPerPixel != 32))
    {
		trace("vncEncoder : only 8, 16 or 32 bits supported locally - ");
		trace_int(m_remoteformat.bitsPerPixel);
		trace(" requested\n");

		return FALSE;
    }

	// Now choose the translation function to use

	// We don't do remote palettes
    if (!m_remoteformat.trueColour)
	{
		trace ("vncEncoder : remote palettized displays are not supported!\n");

		return FALSE;
	}

	// We only support 8 bit palette-based displays at the local end
	if (!m_localformat.trueColour && (m_localformat.bitsPerPixel > 8))
	{
		trace("vncEncoder : local colour map display only supported in 8-bit!\n");

		return FALSE;
    }

	// Are we doing colour map to truecolour?
	if (!m_localformat.trueColour)
	{
		// Yes, so pick the right translation function!
		trace("vncEncoder : using 8-bit colourmap to truecolour translation\n");

		m_transfunc = rfbTranslateWithSingleTableFns
			[m_localformat.bitsPerPixel / 16]
			[m_remoteformat.bitsPerPixel / 16];

		return SetClientColourMap();
	}

	// If we reach here then we're doing truecolour to truecolour

	// Are the formats identical?
    if (PF_EQ(m_remoteformat,m_localformat))
	{
		// Yes, so use the null translation function
		trace("vncEncoder : no translation required\n");

		m_transfunc = rfbTranslateNone;

		return TRUE;
    }

	// Is the local display a 16-bit one
    if (m_localformat.bitsPerPixel == 16)
	{
		// Yes, so use a single lookup-table
		trace("vncEncoder : single LUT used\n");

		m_transfunc = rfbTranslateWithSingleTableFns
			[m_localformat.bitsPerPixel / 16]
			[m_remoteformat.bitsPerPixel / 16];

		(*rfbInitTrueColourSingleTableFns[m_remoteformat.bitsPerPixel / 16])
			(&m_transtable, &m_localformat, &m_remoteformat);
    }
	else
	{
		// No, so use three tables - one for each of R, G, B.
		trace("vncEncoder : triple LUT used\n");

		m_transfunc = rfbTranslateWithRGBTablesFns
			[m_localformat.bitsPerPixel / 16]
			[m_remoteformat.bitsPerPixel / 16];

		(*rfbInitTrueColourRGBTablesFns[m_remoteformat.bitsPerPixel / 16])
			(&m_transtable, &m_localformat, &m_remoteformat);
    }

    return TRUE;
}

// Routine to deal with general palette-based displays
// (Currently only does local ones!)

BOOL
vncEncoder::SetClientColourMap()
{
	// If the local format is trueColour then we shouldn't be here!!!
    _ASSERT(!m_localformat.trueColour);

	// If remote format is truecolour then precalculate the remote palette settings
    if (m_remoteformat.trueColour)
	{
		(*rfbInitColourMapSingleTableFns[m_remoteformat.bitsPerPixel / 16])
			(&m_transtable, &m_localformat, &m_remoteformat);
		return TRUE;
    }

	// We don't do output to palette-based displays as yet...
	return FALSE;
}

BOOL
vncEncoder::SetLocalFormat(rfbPixelFormat &pixformat, int width, int height)
{
	// Work out the bytes per row at the local end - useful
	m_bytesPerRow = width * pixformat.bitsPerPixel/8;

	// Save the pixel format
	m_localformat = pixformat;
	return SetTranslateFunction();
}

BOOL
vncEncoder::SetRemoteFormat(rfbPixelFormat &pixformat)
{
	// Save the client pixel format
	m_remoteformat = pixformat;

	return SetTranslateFunction();
}
