//  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
//
//  This file is part of the VNC system.
//
//  The VNC system is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
//  USA.
//
// If the source code for the VNC system is not available from the place 
// whence you received this file, check http://www.orl.co.uk/vnc or contact
// the authors on vnc@orl.co.uk for information on obtaining it.


#include "stdhdrs.h"

#include "vncviewer.h"
#include "Daemon.h"
#include "Flasher.h"
#include "ClientConnection.h"
#include "Exception.h"

#include "omnithread.h"

HINSTANCE hAppInstance;
VNCOptions appOptions;
// All logging is done via this object
Log log;

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PSTR szCmdLine, int iCmdShow)
{

        hAppInstance = hInstance;

        // Read the command line
        appOptions.SetFromCommandLine(szCmdLine);

        // Logging info
        log.SetLevel(appOptions.m_logLevel);
        if (appOptions.m_logToConsole) {
            log.SetMode(Log::ToConsole | Log::ToDebug);
        }
        if (appOptions.m_logToFile) {
            log.SetFile(appOptions.m_logFilename);
        }

        // Initialise winsock
        WORD wVersionRequested = MAKEWORD(2, 0);
        WSADATA wsaData;
        if (WSAStartup(wVersionRequested, &wsaData) != 0) {
                MessageBox(NULL, "Error initialising sockets library", "VNC info", MB_OK | MB_ICONSTOP);
                exit(1);
        }
        log.Print(3, _T("Started and Winsock (v %d) initialised\n"), wsaData.wVersion);

        // Allow bell sound to be changed from control panel
        RegisterSounds();

        // Load a requested keyboard layout
        if (appOptions.m_kbdSpecified) {
                HKL hkl = LoadKeyboardLayout(  appOptions.m_kbdname, 
                        KLF_ACTIVATE | KLF_REPLACELANG | KLF_REORDER  );
                if (hkl == NULL) {
                        MessageBox(NULL, "Error loading specified keyboard layout", 
                                "VNC info", MB_OK | MB_ICONSTOP);
                        exit(1);
                }
        }
        
        Flasher *pflasher = NULL;
        Daemon *pdaemon = NULL;
        ClientConnection *pcc = NULL;

        MSG msg;
        
        // Start listening daemons if requested
        
        if (appOptions.m_listening) {
            log.Print(3, _T("In listening mode - staring daemons\n"));

            try {
                pflasher = new Flasher(FLASH_PORT_OFFSET);
                pdaemon = new Daemon(INCOMING_PORT_OFFSET);
            } catch (WarningException &e) {
                char msg[1024];
                sprintf(msg, "Error creating listening daemon:\n\r(%s)\n\r%s",
                    e.m_info, "Perhaps another VNCviewer is already running?");
                MessageBox(NULL, msg, "VNCviewer error", MB_OK | MB_ICONSTOP);
                exit(1);
            }
            
        } 
        

        try {

                // Start connection if specified on command line
                // or if not listening

                if (appOptions.m_connectionSpecified) {
                        
                        pcc = new ClientConnection(appOptions.m_host, appOptions.m_port);
                        pcc->Run();

                } else if (!appOptions.m_listening) {
                        
                        pcc = new ClientConnection();
                        pcc->Run();
                        
                }
                
                
                while ( GetMessage(&msg, NULL, 0,0) ) {
                        // TranslateMessage(&msg);  // convert key ups and downs to chars
                        DispatchMessage(&msg);
                }

        } catch (WarningException &e) {
                e.Report();
        } catch (QuietException &e) {
                e.Report();
        }

        // We don't need to clean up pcc if the thread has been joined.
        if (pdaemon != NULL) delete pdaemon;
        if (pflasher != NULL) delete pflasher;

        
        // Clean up winsock
        WSACleanup();
  
        log.Print(2, _T("VNC viewer closing down\n"));

        return msg.wParam;
}

// Register the Bell sound event


#define BELL_APPL_KEY_NAME "AppEvents\\Schemes\\Apps\\VNCviewer"
#define BELL_LABEL "VNCviewerBell"

void RegisterSounds() {
        HKEY hBellKey;
        char keybuf[256];

        sprintf(keybuf, "AppEvents\\EventLabels\\%s", BELL_LABEL);
        // First create a label for it
        if ( RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey)  == ERROR_SUCCESS ) {
                RegSetValue(hBellKey, NULL, REG_SZ, "Bell", 0);
                RegCloseKey(hBellKey);

                // Then put the detail in the app-specific area

                if ( RegCreateKey(HKEY_CURRENT_USER, BELL_APPL_KEY_NAME, &hBellKey)  == ERROR_SUCCESS ) {

                        sprintf(keybuf, "%s\\%s", BELL_APPL_KEY_NAME, BELL_LABEL);
                        RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey);
                        RegSetValue(hBellKey, NULL, REG_SZ, "Bell", 0);
                        RegCloseKey(hBellKey);

                        sprintf(keybuf, "%s\\%s\\.current", BELL_APPL_KEY_NAME, BELL_LABEL);
                        if (RegOpenKey(HKEY_CURRENT_USER, keybuf, &hBellKey) != ERROR_SUCCESS) {
                                RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey);
                                RegSetValue(hBellKey, NULL, REG_SZ, "ding.wav", 0);
                        }
                        RegCloseKey(hBellKey);

                        sprintf(keybuf, "%s\\%s\\.default", BELL_APPL_KEY_NAME, BELL_LABEL);
                        if (RegOpenKey(HKEY_CURRENT_USER, keybuf, &hBellKey) != ERROR_SUCCESS) {
                                RegCreateKey(HKEY_CURRENT_USER, keybuf, &hBellKey);
                                RegSetValue(hBellKey, NULL, REG_SZ, "ding.wav", 0);
                        }
                        RegCloseKey(hBellKey);
                }
                

        } 

}

// Move the given window to the centre of the screen
// and bring it to the top.
void CentreWindow(HWND hwnd)
{
        RECT winrect, workrect;
        
        // Find how large the desktop work area is
        SystemParametersInfo(SPI_GETWORKAREA, 0, &workrect, 0);
        int workwidth = workrect.right -  workrect.left;
        int workheight = workrect.bottom - workrect.top;
        
        // And how big the window is
        GetWindowRect(hwnd, &winrect);
        int winwidth = winrect.right - winrect.left;
        int winheight = winrect.bottom - winrect.top;
        // Make sure it's not bigger than the work area
        winwidth = min(winwidth, workwidth);
        winheight = min(winheight, workheight);

        // Now centre it
        SetWindowPos(hwnd, 
                HWND_TOP,
                workrect.left + (workwidth-winwidth) / 2,
                workrect.top + (workheight-winheight) / 2,
                winwidth, winheight, 
                SWP_SHOWWINDOW);
        SetForegroundWindow(hwnd);
}

// Convert "host:display" into host and port
// Returns true if valid format, false if not.
// Takes initial string, addresses of results and size of host buffer in wchars.
// If the display info passed in is longer than the size of the host buffer, it
// is assumed to be invalid, so false is returned.
bool ParseDisplay(LPTSTR display, LPTSTR phost, int hostlen, int *pport) 
{
    int tmpport;
    if (hostlen < (int) _tcslen(display))
        return false;
    TCHAR *colonpos = _tcschr( display, L':' );
    if (colonpos == NULL) 
        return false;
    _tcsncpy(phost, display, colonpos-display);
    phost[colonpos-display] = L'\0';
    if (_stscanf(colonpos+1, TEXT("%d"), &tmpport) != 1) 
        return false;
    if (tmpport < 100)
        tmpport += RFB_PORT_OFFSET;
    *pport = tmpport;
    return true;
}
