/*

Copyright (C) 1993-1996 Olivetti Research Limited, Cambridge, England.

THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE
LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR
OTHER PARTIES PROVIDE THIS SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND,
EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  NO
GUARANTEE IS MADE THAT THIS SOFTWARE IS FREE OF SOFTWARE VIRUSES.  THE ENTIRE
RISK AS TO THE QUALITY AND PERFORMANCE OF THIS SOFTWARE IS WITH YOU.  SHOULD
THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING WILL ANY
COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR REDISTRIBUTE THIS
SOFTWARE AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES, INCLUDING ANY
GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR
INABILITY TO USE THIS SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR
DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THIS SOFTWARE TO OPERATE WITH ANY OTHER SYSTEMS), EVEN IF SUCH
HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

*/

/*
 * tpviewport.c - X client to move the viewport around a teleport session
 * root window. Should work with any window manager where there is a top-level
 * window corresponding to each client window (i.e. it won't work when the
 * window manager creates a virtual root under which several client windows
 * are reparented).
 *
 * T J Richardson
 */

#include <stdio.h>
#include <malloc.h>
#include <sys/time.h>
#include <X11/Xmd.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>

#include "viewportCursor"
#include "viewportCursorMask"


static char *programName;
static Display *dpy;

static unsigned int buttonIds[] = { Button1, Button2, Button3, Button4, Button5 };
static unsigned int buttonMotionMasks[] = {
    Button1MotionMask, Button2MotionMask, Button3MotionMask,
    Button4MotionMask, Button5MotionMask
};

static Bool arrowKeys = False;
static unsigned int keyModifiers;

static Atom viewportSizeAtom, stickyNamesAtom, stickyAtom;
static Atom stickyWindowsAtom, windowOffsetListAtom;

static INT32 viewportX = 0, viewportY = 0, viewportWidth = 0, viewportHeight = 0;
static int rootWidth, rootHeight;
static int fromX = 0, fromY = 0;

static long rootEventMask = 0;


typedef struct stickyWindowListNode {
    Window window;	/* the top-level window, not the client's window */
    int x, y, width, height;
    struct timeval lastConfiguredTime;
    int noOfTimesQuicklyReconfigured;
    struct stickyWindowListNode *next, *prev;
} stickyWindowListNode;
stickyWindowListNode *stickyWindowListHead = 0, *stickyWindowListTail = 0;


typedef struct stickyNameListNode {
    char *name;
    struct stickyNameListNode *next, *prev;
} stickyNameListNode;
stickyNameListNode *stickyNameListHead = 0, *stickyNameListTail = 0;


typedef struct createdButUnmappedListNode {
    Window w;
    struct createdButUnmappedListNode *next, *prev;
} createdButUnmappedListNode;
createdButUnmappedListNode *createdButUnmappedListHead = 0;
createdButUnmappedListNode *createdButUnmappedListTail = 0;


static void ButtonPressEvent();
static void MotionNotifyEvent();
static void ButtonReleaseEvent();
static void KeyPressEvent();
static void PropertyNotifyEvent();
static void ConfigureNotifyEvent();
static void DestroyNotifyEvent();
static void ReparentNotifyEvent();
static void MappingNotifyEvent();
static void CreateNotifyEvent();
static void MapNotifyEvent();
static void GetViewportXY();
static void GetViewportSize();
static void SetViewportXY();
static void MoveViewport();
static void PutStickyWindows();
static void GetStickyWindows();
static void GetStickyNames();
static void RepositionStickyWindows();
static void RepositionStickyWindow();
static void AddToStickyList();
static void RemoveFromStickyList();
static Window FindClientWindow();
static Window TryInferiors();
static unsigned int ModifiersFromString();
static void OldArguments();

static void usage()
{
    fprintf(stderr,"usage: %s [-display dpy] [-arrowkeys <key-modifiers>] <button-modifiers> <button>\n",
	    programName);
    exit(1);
}

int main(argc, argv)
    int argc;
    char **argv;
{
    char *displayname = NULL;
    int i;
    Cursor cursor;
    XEvent event;
    unsigned int bestCursorWidth, bestCursorHeight;
    Pixmap src, msk;
    XColor fg, bg;
    unsigned int button, buttonMotionMask;
    unsigned int buttonModifiers;

    programName = argv[0];

    for (i = 1; i < argc; i++) {
	if (argv[i][0] != '-')
	    break;

	switch (argv[i][1]) {
	case 'd':			/* -display dpyname */
	    if (++i >= argc) usage();
	    displayname = argv[i];
	    break;

	case 'a':			/* -arrowkeys modifiers */
	    if (++i >= argc) usage();
	    arrowKeys = True;
	    keyModifiers = ModifiersFromString(argv[i]);
	    break;

	default:
	    usage();
	}
    }

    if ((argc - i) < 1)
	usage();

    if (strcmp(argv[i], "1") == 0) {
	OldArguments(argc, argv, i, &buttonModifiers, &button);
    } else {
	if ((argc - i) != 2)
	    usage();

	buttonModifiers = ModifiersFromString(argv[i]);
	if ((sscanf(argv[i+1], "%d", &button) < 1) || (button < 1) || (button > 5))
	    usage();
    }

    button = buttonIds[button - 1];
    buttonMotionMask = buttonMotionMasks[button - 1];

    if (!(dpy = XOpenDisplay(displayname))) {
	fprintf(stderr,"%s: unable to open display \"%s\"\r\n",
		programName, XDisplayName (displayname));
	exit(1);
    }


    /*
     * If this display can cope with a big enough cursor, use the cursor
     * defined by the bitmap "viewportCursor".  If not then use XC_sizing.
     */

    XQueryBestCursor(dpy, DefaultRootWindow(dpy), viewportCursor_width,
		     viewportCursor_height, &bestCursorWidth, &bestCursorHeight);

    if ((bestCursorWidth < viewportCursor_width)
	|| (bestCursorHeight < viewportCursor_height))
    {
	cursor = XCreateFontCursor(dpy, XC_sizing);
    }
    else
    {
	src = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), viewportCursor_bits,
				    viewportCursor_width, viewportCursor_height);
	msk = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), viewportCursorMask_bits,
				    viewportCursorMask_width, viewportCursorMask_height);
	XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "black", &fg, &fg);
	XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "white", &bg, &bg);

	cursor = XCreatePixmapCursor(dpy, src, msk, &fg, &bg, viewportCursor_x_hot,
				     viewportCursor_y_hot);
	XFreePixmap(dpy, src);
	XFreePixmap(dpy, msk);
    }

    /*
     * Select for appropriate events on the root window.  Also get a passive
     * grab on the given combination of mouse button and modifiers.
     */

    rootEventMask |= PropertyChangeMask | StructureNotifyMask | SubstructureNotifyMask;
    XSelectInput(dpy, DefaultRootWindow(dpy), rootEventMask);

    XGrabButton(dpy, button, buttonModifiers, DefaultRootWindow(dpy), False,
		buttonMotionMask | ButtonPressMask | ButtonReleaseMask,
		GrabModeAsync, GrabModeAsync, DefaultRootWindow(dpy), cursor);

    if (arrowKeys) {
	XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Left), keyModifiers,
		 DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Right), keyModifiers,
		 DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Up), keyModifiers,
		 DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Down), keyModifiers,
		 DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
    }

    rootWidth = DisplayWidth(dpy, DefaultScreen(dpy));
    rootHeight = DisplayHeight(dpy, DefaultScreen(dpy));

    /*
     * Get the initial state of each of the properties on the root window
     * we're interested in
     */

    viewportSizeAtom = XInternAtom(dpy, "_ORL_TP_VIEWPORT_SIZE", False);
    stickyNamesAtom = XInternAtom(dpy, "_ORL_TP_STICKY_NAMES", False);
    stickyAtom = XInternAtom(dpy, "_ORL_TP_STICKY", False);
    stickyWindowsAtom = XInternAtom(dpy, "_ORL_TP_STICKY_WINDOWS", False);
    windowOffsetListAtom = XInternAtom(dpy, "WINDOW_OFFSET_LIST", False);

    GetViewportXY();
    GetViewportSize();
    GetStickyWindows();
    GetStickyNames();

    /*
     * Now simply dispatch events.
     */

    while (1) {
        XNextEvent(dpy, &event);

        switch (event.type) {

	case ButtonPress:
	    ButtonPressEvent(&event);
	    break;

	case MotionNotify:
	    MotionNotifyEvent(&event);
	    break;

	case ButtonRelease:
	    ButtonReleaseEvent(&event);
	    break;

	case KeyPress:
	    KeyPressEvent(&event);
	    break;

	case PropertyNotify:
	    PropertyNotifyEvent(&event);
	    break;

	case MappingNotify:
	    MappingNotifyEvent(&event);
	    break;

	case ConfigureNotify:
	    ConfigureNotifyEvent(&event);
	    break;

	case DestroyNotify:
	    DestroyNotifyEvent(&event);
	    break;

	case ReparentNotify:
	    ReparentNotifyEvent(&event);
	    break;

	case CreateNotify:
	    CreateNotifyEvent(&event);
	    break;

	case MapNotify:
	    MapNotifyEvent(&event);
	    break;

	default:
            break;
        }
    }
}


/*
 * CatchErrorHandler is used to catch X errors.
 */

static Bool caughtError = 0;

static int CatchErrorHandler(dpy, error)
    Display *dpy;
    XErrorEvent *error;
{
    caughtError = True;
    return 0;
}


/*
 * When the user presses the right combination of modifiers & button, record
 * where it happened...
 */

static void ButtonPressEvent(event)
    XEvent *event;
{
    fromX = event->xbutton.x_root;
    fromY = event->xbutton.y_root;
}

/*
 * ...then as the mouse moves, move the viewport accordingly...
 */

static void MotionNotifyEvent(event)
    XEvent *event;
{
    Window dummyw;
    int dummyi;
    unsigned int keysButtons;
    int toX, toY, xChange, yChange;

    while (XCheckMaskEvent(dpy, PointerMotionMask, event))
	;	/* discard all motion notify events */

    XQueryPointer(dpy, DefaultRootWindow(dpy), &dummyw, &dummyw, &toX, &toY, &dummyi,
		  &dummyi, &keysButtons);

    if ((toX == fromX) && (toY == fromY))
	return;

    MoveViewport(toX - fromX, toY - fromY, &xChange, &yChange);
    XWarpPointer(dpy, None, None, 0, 0, 0, 0, fromX - toX, fromY - toY);

    fromX += xChange;
    fromY += yChange;
}

/*
 * ...then when the button is released, reposition the sticky windows.
 */

static void ButtonReleaseEvent(event)
    XEvent *event;
{
    RepositionStickyWindows();
}


/*
 * When a key is pressed, see which it is and scroll accordingly.
 */

static void KeyPressEvent(event)
    XEvent *event;
{
    int actualX, actualY;

    if (event->xkey.state != keyModifiers)
	return;

    switch(XLookupKeysym(&event->xkey, 0)) {

    case XK_Left:
	MoveViewport(-viewportWidth, 0, &actualX, &actualY);
	break;

    case XK_Right:
	MoveViewport(viewportWidth, 0, &actualX, &actualY);
	break;

    case XK_Up:
	MoveViewport(0, -viewportHeight, &actualX, &actualY);
	break;

    case XK_Down:
	MoveViewport(0, viewportHeight, &actualX, &actualY);
	break;
    }

    RepositionStickyWindows();
}


/*
 * If a property we're interested in changes, go and read the new value.
 */

static void PropertyNotifyEvent(event)
    XEvent *event;
{
    if (event->xproperty.atom == viewportSizeAtom) {
	GetViewportSize();
	if ((viewportX + viewportWidth) > rootWidth)
	    viewportX = rootWidth - viewportWidth;
	if ((viewportY + viewportHeight) > rootHeight)
	    viewportY = rootHeight - viewportHeight;
	SetViewportXY();
	RepositionStickyWindows();
    }
    else if (event->xproperty.atom == stickyWindowsAtom) {
	GetStickyWindows();
    }
    else if (event->xproperty.atom == stickyNamesAtom) {
	GetStickyNames();
    }
}


/*
 * Keep keyboard mapping up to date, and regrab keys.
 */

static void MappingNotifyEvent(event)
    XEvent *event;
{
    if (event->xmapping.request == MappingKeyboard) {
	XRefreshKeyboardMapping(&event->xmapping);

	if (arrowKeys) {
	    XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Left), keyModifiers,
		     DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	    XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Right), keyModifiers,
		     DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	    XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Up), keyModifiers,
		     DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	    XGrabKey(dpy, XKeysymToKeycode(dpy, XK_Down), keyModifiers,
		     DefaultRootWindow(dpy), False, GrabModeAsync, GrabModeAsync);
	}
    }
}


/*
 * ConfigureNotify - if a sticky window is reconfigured, take note of any
 * resizing, but don't allow the window to move. If it has moved, then move it
 * back to where it "should" be. If this happens lots of times quickly then
 * we're fighting against the window manager, so remove its stickyness and just
 * let it go.
 */

static void ConfigureNotifyEvent(event)
    XEvent *event;
{
    stickyWindowListNode *node;

    for (node = stickyWindowListHead; node; node = node->next) {
	if (node->window == event->xconfigure.window) {
	    int change = 0;
	    struct timeval tv;
	    long secDiff, usDiff;

	    gettimeofday(&tv, NULL);

	    usDiff = tv.tv_usec - node->lastConfiguredTime.tv_usec;
	    secDiff = tv.tv_sec - node->lastConfiguredTime.tv_sec;
	    if (usDiff < 0) {
		usDiff += 1000000;
		secDiff--;
	    }

	    if (secDiff < 1) {	/* "quickly" is < 1 second */
		node->noOfTimesQuicklyReconfigured++;
		if (node->noOfTimesQuicklyReconfigured > 10) {
		    fprintf(stderr,"%s: window %x reconfigured quickly too often - stickiness removed\n",
			    programName, (unsigned int)node->window);
		    RemoveFromStickyList(node);
		    PutStickyWindows();
		    break;
		}
	    } else {
		node->noOfTimesQuicklyReconfigured = 0;
	    }

	    node->lastConfiguredTime = tv;

	    if (node->width != event->xconfigure.width + 2 * event->xconfigure.border_width) {
		node->width = event->xconfigure.width + 2 * event->xconfigure.border_width;
		change = 1;
	    }
	    if (node->height != event->xconfigure.height + 2 * event->xconfigure.border_width) {
		node->height = event->xconfigure.height + 2 * event->xconfigure.border_width;
		change = 1;
	    }
	    if (change) {
		PutStickyWindows();
	    }
	    RepositionStickyWindow(node);
	    break;
	}
    }
}


/*
 * Keep a list of all top-level windows which have been created but are
 * unmapped.
 */

static void CreateNotifyEvent(event)
    XEvent *event;
{
    createdButUnmappedListNode *node;

    node = (createdButUnmappedListNode *)malloc(sizeof(createdButUnmappedListNode));
    node->w = event->xcreatewindow.window;
    node->next = (createdButUnmappedListNode *)0;
    node->prev = createdButUnmappedListTail;

    if (createdButUnmappedListTail)
	createdButUnmappedListTail->next = node;
    else
	createdButUnmappedListHead = node;

    createdButUnmappedListTail = node;
}


/*
 * When a top-level window is first mapped, find the name on the client window
 * under it, and if it's one of the "sticky names" then mark the window as
 * sticky.
 */

static void MapNotifyEvent(event)
    XEvent *event;
{
    createdButUnmappedListNode *cnode;
    stickyNameListNode *snode;
    stickyWindowListNode *wnode;
    int x, y, centreX, centreY, displayWidth, displayHeight;
    unsigned int width, height, borderWidth, dummyi;
    Window cltWin, dummyw;
    char *name;
    int normalHints;
    XSizeHints hints;
    Atom type = None;
    int format;
    unsigned long nitems, after;
    unsigned char *data;

    for (cnode = createdButUnmappedListHead;
	 cnode && (cnode->w != event->xmap.window);
	 cnode = cnode->next)
	;

    if (!cnode)
	return;	/* not a top-level window being mapped for the first time */

    if (cnode->prev)
	cnode->prev->next = cnode->next;
    else
	createdButUnmappedListHead = cnode->next;
    if (cnode->next)
	cnode->next->prev = cnode->prev;
    else
	createdButUnmappedListTail = cnode->prev;
    free((char *)cnode);

    for (wnode = stickyWindowListHead; wnode; wnode = wnode->next) {
	if (wnode->window == event->xmap.window) {
	    return;	/* already sticky */
	}
    }

    cltWin = FindClientWindow(event->xmap.window);

    if (!cltWin)
	return;		/* no client window under here */

    if (!XFetchName(dpy, cltWin, &name))
	return;		/* client window not named */

    for (snode = stickyNameListHead;
	 snode && (strcmp(snode->name, name) != 0);
	 snode = snode->next)
	;

    XFree(name);

    if (!snode)
	return;		/* not a sticky name */

    /*
     * Now we have found a client window with a sticky name. Mark the top-level
     * window as sticky.
     */

    displayWidth = DisplayWidth(dpy,DefaultScreen(dpy));
    displayHeight = DisplayHeight(dpy,DefaultScreen(dpy));

    XGetGeometry(dpy, event->xmap.window, &dummyw, &x, &y, &width,
		 &height, &borderWidth, &dummyi);
    width += 2 * borderWidth;
    height += 2 * borderWidth;

    /*
     * In order to get "-geometry -x-y" (or -x+y or +x-y) to work properly we
     * have to be careful with "sticky name" windows.  The problem is the
     * "-x-y" gets translated to "+x'+y'". where x' = displaywidth - x, y' =
     * displayheight - y.  We want to get back to the "-x-y" so that we know
     * how to position the window relative to the nearest corner of the
     * viewport.
     *
     * There are 3 different cases we have to consider here:
     *
     * (a) The window is placed interactively by the user via the window
     *     manager. We can distinguish this case because the WM_NORMAL_HINTS
     *     will not have the USPosition flag set.  In this case we calculate
     *     the window's position relative to the current position and size of
     *     the viewport.
     *
     * (b) The window is placed via a "-geometry" specification. In this case
     *     we want to find the window's position relative to the whole root
     *     window.  We can distinguish this case from (a) by the USPosition
     *     flag (or in the case of the TWM icon manager the WM_NORMAL_HINTS
     *     property does not exist).
     *
     * (c) The window was previously placed as in (b) but the window manager
     *     has now been restarted, so the top-level window corresponding to
     *     the client's window is being mapped for the first time.  This case
     *     can only be distinguished from (b) by the fact that we have
     *     previously marked the client's window with an _ORL_TP_STICKY marker.
     *     In this case we need to calculate the window's position relative to
     *     the viewport, not the root window.
     */

    normalHints = XGetNormalHints(dpy, cltWin, &hints);

    XGetWindowProperty(dpy, cltWin, stickyAtom, 0, 0, False, AnyPropertyType,
		       &type, &format, &nitems, &after, &data);
    if (type != None)
	XFree(data);

    if ((normalHints && !(hints.flags & USPosition)) || (type != None))
    {					/* cases (a) & (c) */
	x -= viewportX;
	y -= viewportY;

	displayWidth = viewportWidth;
	displayHeight = viewportHeight;
    }

    /*
     * Find which corner the window is nearest to and record its position
     * relative to that corner.
     */

    centreX = x + width / 2;
    centreY = y + height / 2;

    if (x < 0) x = 0;
    if (y < 0) y = 0;

    if (centreX > (displayWidth / 2))
	x -= displayWidth;

    if (centreY > (displayHeight / 2))
	y -= displayHeight;

    XChangeProperty(dpy, cltWin, stickyAtom, stickyAtom, 8, PropModeReplace, "", 1);

    AddToStickyList(event->xmap.window, x, y, width, height);
    PutStickyWindows();
    RepositionStickyWindows();
}


/*
 * Remove information on windows which are destroyed.
 */

static void DestroyNotifyEvent(event)
    XEvent *event;
{
    stickyWindowListNode *snode;
    createdButUnmappedListNode *cnode;

    for (snode = stickyWindowListHead; snode; snode = snode->next) {
	if (snode->window == event->xdestroywindow.window) {
	    RemoveFromStickyList(snode);
	    PutStickyWindows();
	    break;
	}
    }

    for (cnode = createdButUnmappedListHead; cnode; cnode = cnode->next) {
	if (cnode->w == event->xdestroywindow.window) {
	    if (cnode->prev)
		cnode->prev->next = cnode->next;
	    else
		createdButUnmappedListHead = cnode->next;
	    if (cnode->next)
		cnode->next->prev = cnode->prev;
	    else
		createdButUnmappedListTail = cnode->prev;
	    free((char *)cnode);
	    break;
	}
    }
}


/*
 * If a sticky window is reparented, get the new top-level window it is under.
 */

static void ReparentNotifyEvent(event)
    XEvent *event;
{
    stickyWindowListNode *snode;
    createdButUnmappedListNode *cnode;

    for (snode = stickyWindowListHead; snode; snode = snode->next) {
	if (snode->window == event->xreparent.window) {
	    GetStickyWindows();
	    break;
	}
    }

    for (cnode = createdButUnmappedListHead; cnode; cnode = cnode->next) {
	if (cnode->w == event->xreparent.window) {
	    if (cnode->prev)
		cnode->prev->next = cnode->next;
	    else
		createdButUnmappedListHead = cnode->next;
	    if (cnode->next)
		cnode->next->prev = cnode->prev;
	    else
		createdButUnmappedListTail = cnode->prev;
	    free((char *)cnode);
	    break;
	}
    }
}


/*
 * Get the x,y position of the viewport window.
 */

static void GetViewportXY()
{
    Window dummyw;
    int x, y;
    unsigned int dummyi;

    XGetGeometry(dpy, DefaultRootWindow(dpy), &dummyw, &x, &y,
		 &dummyi, &dummyi, &dummyi, &dummyi);
    viewportX = -x;
    viewportY = -y;
}


/*
 * Get the size of the viewport window.
 */

static void GetViewportSize()
{
    Atom actualType;
    int format;
    unsigned long nitems, bytesAfter;
    long *data;

    XGetWindowProperty(dpy, DefaultRootWindow(dpy), viewportSizeAtom, 0, 2, False,
		       XA_INTEGER, &actualType, &format, &nitems, &bytesAfter,
		       (unsigned char **)&data);
    if ((actualType == XA_INTEGER) && (format == 32) && (nitems == 2)) {
	viewportWidth = data[0];
	viewportHeight = data[1];
	XFree(data);
    }
}


/*
 * Set the x,y position of the viewport window. This is actually achieved by
 * configuring the position of the root window (the viewport is the parent
 * of the root window of a teleport session).
 */

static void SetViewportXY()
{
    XWindowChanges chg;

    chg.x = -viewportX;
    chg.y = -viewportY;
    XConfigureWindow(dpy, DefaultRootWindow(dpy), CWX | CWY, &chg);
    XSync(dpy, False);
}


/*
 * Move the viewport by a given (relative) amount.  Returns the actual
 * change (which may be less if we're at the edge of the root window).
 */

static void MoveViewport(xChange, yChange, actualXChange, actualYChange)
    int xChange, yChange;
    int *actualXChange, *actualYChange;
{
    if ((viewportX + xChange) < 0)
	xChange = -viewportX;
    if ((viewportX + xChange + viewportWidth) > rootWidth)
	xChange = rootWidth - viewportWidth - viewportX;
    *actualXChange = xChange;
    viewportX += xChange;

    if ((viewportY + yChange) < 0)
	yChange = -viewportY;
    if ((viewportY + yChange + viewportHeight) > rootHeight)
	yChange = rootHeight - viewportHeight - viewportY;
    *actualYChange = yChange;
    viewportY += yChange;

    SetViewportXY();
}


/*
 * Put the current value of the sticky list into the corresponding property
 * on the root window.
 */

static void PutStickyWindows()
{
    stickyWindowListNode *node;
    int i, nWindows = 0;
    long *windowOffsetList;
    
    for (node = stickyWindowListHead; node; node = node->next)
	nWindows++;

    windowOffsetList = (long *)malloc(sizeof(*windowOffsetList) * 3 * nWindows);

    for (i = 0, node = stickyWindowListHead; node; i++, node = node->next) {
	windowOffsetList[i*3] = (long)node->window;
	windowOffsetList[i*3+1] = (long)node->x;
	windowOffsetList[i*3+2] = (long)node->y;
    }

    /*
     * Make sure we don't get a PropertyNotify from this change, and that
     * no other client (i.e. tpsticky) changes it in the meantime.
     */

    XGrabServer(dpy);
    rootEventMask &= ~PropertyChangeMask;
    XSelectInput(dpy, DefaultRootWindow(dpy), rootEventMask);

    XChangeProperty(dpy, DefaultRootWindow(dpy), stickyWindowsAtom, windowOffsetListAtom,
		    32, PropModeReplace, (unsigned char *)windowOffsetList, i * 3);

    rootEventMask |= PropertyChangeMask;
    XSelectInput(dpy, DefaultRootWindow(dpy), rootEventMask);
    XUngrabServer(dpy);

    free((char *)windowOffsetList);
}


/*
 * GetStickyWindows builds the sticky list from the corresponding property on
 * the root window. Any bad windows are removed from the list, and the property
 * updated accordingly.
 */

static void GetStickyWindows()
{
    Atom actualType;
    int format;
    unsigned long nitems, bytesAfter;
    long *data;
    stickyWindowListNode *node, *nextNode;
    Window window, dummyw, parent, *children;
    unsigned int nchildren, width, height, borderWidth, dummyui;
    int i, dummyi;
    INT32 x, y;
    Bool badStickyWindow = False;

    /*
     * First empty the sticky list.
     */

    for (node = stickyWindowListHead; node; node = nextNode) {
	nextNode = node->next;
	RemoveFromStickyList(node);
    }

    /*
     * Get the value of the property.
     */

    XGetWindowProperty(dpy, DefaultRootWindow(dpy), stickyWindowsAtom, 0, 1024, False,
		       windowOffsetListAtom, &actualType, &format, &nitems, &bytesAfter,
		       (unsigned char **)&data);

    if ((actualType != windowOffsetListAtom) && (format != 32))
	return;

    for (i = 0; i < nitems / 3; i++) {
	window = (Window)(data[i*3]);
	x = (INT32)(data[i*3+1]);
	y = (INT32)(data[i*3+2]);

	caughtError = False;
	XSetErrorHandler(CatchErrorHandler);
	XQueryTree(dpy, window, &dummyw, &parent, &children, &nchildren);
	XFree(children);
	XSetErrorHandler(0);

	if (caughtError || (parent != DefaultRootWindow(dpy))) {
	    badStickyWindow = True;
	    continue;
	}

	XGetGeometry(dpy, window, &dummyw, &dummyi, &dummyi, &width, &height,
		     &borderWidth, &dummyui);

	AddToStickyList(window, x, y, width + 2 * borderWidth, height + 2 * borderWidth);
    }

    XFree(data);

    if (badStickyWindow)
	PutStickyWindows();

    RepositionStickyWindows();
}


/*
 * Get the list of sticky names from the property on the root window.
 */

static void GetStickyNames()
{
    Atom actualType;
    int i, len, format;
    unsigned long nItems, bytesAfter;
    char *data;
    stickyNameListNode *node, *nextNode;

    /*
     * First empty the existing list.
     */

    for (node = stickyNameListHead; node; node = nextNode) {
	nextNode = node->next;
	free(node->name);
	free((char *)node);
    }
    stickyNameListHead = stickyNameListTail = 0;

    /*
     * Get the property.
     */

    XGetWindowProperty(dpy, DefaultRootWindow(dpy), stickyNamesAtom, 0, 1024, False,
		       XA_STRING, &actualType, &format, &nItems, &bytesAfter,
		       (unsigned char **)&data);

    if ((actualType != XA_STRING) || (format != 8))
	return;

    /*
     * Pick out each string from the property and put it on the list.
     */

    for (i = 0; i < nItems; i += len+1) {
	len = strlen(&data[i]);
	if ((i + len + 1) > nItems)
	    break;

	node = (stickyNameListNode *)malloc(sizeof(stickyNameListNode));
	node->name = (char *)malloc(len + 1);
	strcpy(node->name, &data[i]);
	node->next = (stickyNameListNode *)0;
	node->prev = stickyNameListTail;

	if (stickyNameListTail)
	    stickyNameListTail->next = node;
	else
	    stickyNameListHead = node;

	stickyNameListTail = node;
    }

    XFree(data);
}


/*
 * Reposition all the sticky windows.
 */

static void RepositionStickyWindows()
{
    stickyWindowListNode *node;

    for (node = stickyWindowListHead; node; node = node->next) {
	RepositionStickyWindow(node);
    }
}


/*
 * Reposition a given sticky window.
 */

static void RepositionStickyWindow(node)
    stickyWindowListNode *node;
{
    XWindowChanges chg;
    XWindowAttributes attr;
    XSetWindowAttributes sattr;
    Window parent, dummyw, *children;
    unsigned int nchildren;

    chg.x = node->x;
    chg.y = node->y;
    if (chg.x < 0)
	chg.x = viewportWidth + chg.x;
    if (chg.y < 0)
	chg.y = viewportHeight + chg.y;
    if (chg.x + node->width > viewportWidth)
	chg.x = viewportWidth - node->width;
    if (chg.x < 0)
	chg.x = 0;
    if (chg.y + node->height > viewportHeight)
	chg.y = viewportHeight - node->height;
    if (chg.y < 0)
	chg.y = 0;

    chg.x += viewportX;
    chg.y += viewportY;

    /*
     * Basically here we want to reconfigure the sticky window but without:
     * (a) the window manager interfering with it,
     * (b) having a configure notify event sent to us as a result (otherwise
     *     it's very difficult to tell which configure notify events are due
     *     to the user moving the window and which are due to us moving it).
     * We also want to make sure that the window is definitely a direct child
     * of the root. If it's not then it's probably been reparented and we
     * haven't received/dealt with the reparent notify event yet.
     *
     * So to achieve this we:
     * (a) grab the server to ensure no other activity on the server (such
     *     as someone else reconfiguring the window)
     * (b) check that this window's parent is the root window.
     * (c) set the window's override redirect attribute to stop the window
     *     manager interfering
     * (d) remove our "SubstructureNotifyMask" selection on the root window
     * (e) configure the window
     * (f) undo the steps in (d), (c) and (a)
     */

    XGrabServer(dpy);

    XQueryTree(dpy, node->window, &dummyw, &parent, &children, &nchildren);
    XFree(children);

    if (parent == DefaultRootWindow(dpy))
    {
	XGetWindowAttributes(dpy, node->window, &attr);
	if (!attr.override_redirect) {
	    sattr.override_redirect = 1;
	    XChangeWindowAttributes(dpy, node->window, CWOverrideRedirect, &sattr);
	}

	rootEventMask &= ~SubstructureNotifyMask;
	XSelectInput(dpy, DefaultRootWindow(dpy), rootEventMask);

	XConfigureWindow(dpy, node->window, CWX | CWY, &chg);

	rootEventMask |= SubstructureNotifyMask;
	XSelectInput(dpy, DefaultRootWindow(dpy), rootEventMask);

	if (!attr.override_redirect) {
	    sattr.override_redirect = 0;
	    XChangeWindowAttributes(dpy, node->window, CWOverrideRedirect, &sattr);
	}
    }

    XUngrabServer(dpy);
}


/*
 * Add a window to the sticky list, recording its width, height and position
 * relative to the nearest corner.
 */

static void AddToStickyList(window, x, y, width, height)
    Window window;
    int x, y, width, height;
{
    stickyWindowListNode *node;

    node = (stickyWindowListNode *)malloc(sizeof(stickyWindowListNode));
    node->window = window;
    node->x = x;
    node->y = y;
    node->width = width;
    node->height = height;
    node->noOfTimesQuicklyReconfigured = 0;
    gettimeofday(&node->lastConfiguredTime, NULL);
    node->next = (stickyWindowListNode *)0;
    node->prev = stickyWindowListTail;

    if (stickyWindowListTail)
	stickyWindowListTail->next = node;
    else
	stickyWindowListHead = node;

    stickyWindowListTail = node;
}


/*
 * Remove a window from the sticky list.
 */

static void RemoveFromStickyList(node)
    stickyWindowListNode *node;
{
    if (node->prev)
	node->prev->next = node->next;
    else
	stickyWindowListHead = node->next;
    if (node->next)
	node->next->prev = node->prev;
    else
	stickyWindowListTail = node->prev;
    free((char *)node);
}


/*
 * Find a window with WM_STATE, else return win itself, as per ICCCM. Returns
 * None in the event of an error.
 */

static Window FindClientWindow(win)
    Window win;
{
    Atom WM_STATE;
    Atom type = None;
    int format;
    unsigned long nitems, after;
    unsigned char *data;
    Window inferior;

    WM_STATE = XInternAtom(dpy, "WM_STATE", True);
    if (!WM_STATE)
	return win;

    caughtError = False;
    XSetErrorHandler(CatchErrorHandler);
    XGetWindowProperty(dpy, win, WM_STATE, 0, 0, False, AnyPropertyType,
		       &type, &format, &nitems, &after, &data);
    if (caughtError) {
	XSetErrorHandler(0);
	return None;
    }

    if (type != None) {
	XFree(data);
	XSetErrorHandler(0);
	return win;
    }

    inferior = TryInferiors(dpy, win, WM_STATE);

    XSetErrorHandler(0);

    if (caughtError)
	return None;

    if (inferior != None)
	return inferior;

    return win;
}


/*
 * TryInferiors recursively searches down the window tree, returning the first
 * window if finds with a WM_STATE property, or None if no such window can be
 * found. It assumes that CatchErrorHandler has already been installed.
 */

static Window TryInferiors(dpy, win, WM_STATE)
    Display *dpy;
    Window win;
    Atom WM_STATE;
{
    Window root, parent;
    Window *children;
    unsigned int nchildren;
    unsigned int i;
    Atom type = None;
    int format;
    unsigned long nitems, after;
    unsigned char *data;
    Window inferior = None;

    caughtError = False;

    XQueryTree(dpy, win, &root, &parent, &children, &nchildren);

    if (caughtError)
	return None;

    for (i = 0; (inferior == None) && (i < nchildren); i++)
    {
	XGetWindowProperty(dpy, children[i], WM_STATE, 0, 0, False,
			   AnyPropertyType, &type, &format, &nitems,
			   &after, &data);
	if (caughtError)
	    return None;

	if (type != None)
	    inferior = children[i];
    }

    for (i = 0; (inferior == None) && (i < nchildren); i++)
    {
	inferior = TryInferiors(dpy, children[i], WM_STATE);

	if (caughtError)
	    return None;
    }

    if (children) XFree((caddr_t)children);
    return inferior;
}


/*
 * ModifiersFromString converts a string into a modifer mask.
 * The string contains zero or more of:
 * "c" for control, "s" for shift, "l" for lock,
 * "m1" to "m5" for modifiers 1 to 5.  Also a single "m" counts as "m1".
 */

static unsigned int ModifiersFromString(str)
    char *str;
{
    unsigned int modifiers = 0;

    if (strchr(str, 'c'))
	modifiers |= ControlMask;
    if (strchr(str, 's'))
	modifiers |= ShiftMask;
    if (strchr(str, 'l'))
	modifiers |= LockMask;
    if (strstr(str, "m1"))
	modifiers |= Mod1Mask;
    if (strstr(str, "m2"))
	modifiers |= Mod2Mask;
    if (strstr(str, "m3"))
	modifiers |= Mod3Mask;
    if (strstr(str, "m4"))
	modifiers |= Mod4Mask;
    if (strstr(str, "m5"))
	modifiers |= Mod5Mask;

    while ((str = strchr(str, 'm'))) {
	if ((str[1] < '1') || (str[1] > '5')) {
	    modifiers |= Mod1Mask;
	    break;
	}
	str++;
    }

    return modifiers;
}

static void OldArguments(argc, argv, i, modifiersP, buttonP)
    int argc;
    char **argv;
    int i;
    unsigned int *modifiersP;
    unsigned int *buttonP;
{
    if ((argc - i) < 2)
	usage();

    if ((sscanf(argv[i], "%d", buttonP) < 1) || (*buttonP < 1) || (*buttonP > 5))
	usage();

    *modifiersP = 0;
    for (i++; i < argc; i++) {
	if (strcmp(argv[i], "shift") == 0)
	    *modifiersP |= ShiftMask;
	else if (strcmp(argv[i], "lock") == 0)
	    *modifiersP |= LockMask;
	else if (strcmp(argv[i], "control") == 0)
	    *modifiersP |= ControlMask;
	else if (strcmp(argv[i], "mod1") == 0)
	    *modifiersP |= Mod1Mask;
	else if (strcmp(argv[i], "mod2") == 0)
	    *modifiersP |= Mod2Mask;
	else if (strcmp(argv[i], "mod3") == 0)
	    *modifiersP |= Mod3Mask;
	else if (strcmp(argv[i], "mod4") == 0)
	    *modifiersP |= Mod4Mask;
	else if (strcmp(argv[i], "mod5") == 0)
	    *modifiersP |= Mod5Mask;
	else
	    usage();
    }
}
