/*

Copyright (C) 1993-1996 Olivetti Research Limited, Cambridge, England.

THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE
LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR
OTHER PARTIES PROVIDE THIS SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND,
EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  NO
GUARANTEE IS MADE THAT THIS SOFTWARE IS FREE OF SOFTWARE VIRUSES.  THE ENTIRE
RISK AS TO THE QUALITY AND PERFORMANCE OF THIS SOFTWARE IS WITH YOU.  SHOULD
THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING WILL ANY
COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR REDISTRIBUTE THIS
SOFTWARE AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES, INCLUDING ANY
GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR
INABILITY TO USE THIS SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR
DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THIS SOFTWARE TO OPERATE WITH ANY OTHER SYSTEMS), EVEN IF SUCH
HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

*/

/*
 * tpsticky.c
 *
 * T J Richardson
 */

#include <stdio.h>
#include <malloc.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

char *programName;
Display *dpy;

static void SetNames();
static void SetWindowIds();
static Window SelectWindow();
static void GetViewportGeometry();

static void usage()
{
    fprintf(stderr,"usage: %s [-display dpy] [+/- | +/-id <window> | +/-name <name>]\n",
	    programName);
    exit(1);
}

int main(argc, argv)
    int argc;
    char **argv;
{
    char *displayname = NULL;
    int i;
    int singleMinus = 0, singlePlus = 0;
    char *addNames[256], *remNames[256];
    int nAddNames = 0, nRemNames = 0, addNamesLen = 0;
    Window addWindows[256], remWindows[256];
    int nAddWindows = 0, nRemWindows = 0;

    programName = argv[0];

    for (i = 1; i < argc; i++) {
	switch (argv[i][0]) {

	case '+':
	    switch (argv[i][1]) {
	    case 'n':			/* +name <window-name> */
		if (++i >= argc)
		    usage();
		addNames[nAddNames++] = argv[i];
		addNamesLen += strlen(argv[i]) + 1;
		break;
	    case 'i':			/* +id <window-id> */
		if (++i >= argc)
		    usage();
		if ((sscanf(argv[i], "0x%lx", &addWindows[nAddWindows]) != 1) &&
		    (sscanf(argv[i], "%ld", &addWindows[nAddWindows]) != 1)) {
		    fprintf(stderr,"%s: invalid window id format: %s\n", programName, argv[i]);
		    exit(1);
		}
		nAddWindows++;
		break;
	    case '\0':			/* + */
		if ((nAddWindows > 0) || (nRemWindows > 0) || (nAddNames > 0)
		    || (nRemNames > 0) || (i+1 < argc))
		    usage();
		singlePlus = 1;
		break;
	    default:
		usage();
	    }
	    break;

	case '-':
	    switch (argv[i][1]) {
	    case 'd':			/* -display dpyname */
		if (++i >= argc)
		    usage();
		displayname = argv[i];
		break;
	    case 'n':			/* -name <window-name> */
		if (++i >= argc)
		    usage();
		remNames[nRemNames++] = argv[i];
		break;
	    case 'i':			/* -id <window-id> */
		if (++i >= argc)
		    usage();
		if ((sscanf(argv[i], "0x%lx", &remWindows[nRemWindows]) != 1) &&
		    (sscanf(argv[i], "%ld", &remWindows[nRemWindows]) != 1)) {
		    fprintf(stderr,"%s: invalid window id format: %s\n", programName, argv[i]);
		    exit(1);
		}
		nRemWindows++;
		break;
	    case '\0':			/* - */
		if ((nAddWindows > 0) || (nRemWindows > 0) || (nAddNames > 0)
		    || (nRemNames > 0) || (i+1 < argc))
		    usage();
		singleMinus = 1;
		break;
	    default:
		usage();
	    }
	    break;

	default:
	    usage();
	}
    }

    if (argc == 1)
	singlePlus = 1;		/* the default if no args */

    if (!(dpy = XOpenDisplay(displayname))) {
	fprintf(stderr,"%s: unable to open display \"%s\"\n",
		programName, XDisplayName(displayname));
	exit(1);
    }

    SetNames(addNames, remNames, nAddNames, nRemNames, addNamesLen);

    if (singlePlus) {
	addWindows[nAddWindows] = SelectWindow();
	if (!addWindows[nAddWindows]) {
	    XCloseDisplay(dpy);
	    exit(0);
	}
	nAddWindows++;
    }
    else if (singleMinus) {
	remWindows[nRemWindows] = SelectWindow();
	if (!remWindows[nRemWindows]) {
	    XCloseDisplay(dpy);
	    exit(0);
	}
	nRemWindows++;
    }

    SetWindowIds(addWindows, remWindows, nAddWindows, nRemWindows, singlePlus);

    XCloseDisplay(dpy);
    return 0;
}


/*
 * Set a property containing a list of names of windows which should
 * automatically become sticky upon creation. The property contains a
 * sequence of null-terminated strings.
 */

static void SetNames(add, rem, nAdd, nRem, addLen)
    char **add, **rem;
    int nAdd, nRem, addLen;
{
    Atom stickyNamesAtom, actualType;
    int i, j, k, len, format, remove;
    unsigned long nItems, bytesAfter;
    char *oldNames, *newNames;

    stickyNamesAtom = XInternAtom(dpy, "_ORL_TP_STICKY_NAMES", False);

    XGetWindowProperty(dpy, DefaultRootWindow(dpy), stickyNamesAtom, 0, 1024, False,
		       XA_STRING, &actualType, &format, &nItems, &bytesAfter,
		       (unsigned char **)&oldNames);

    if ((actualType != XA_STRING) || (format != 8)) {
	nItems = 0;
    }
    
    newNames = (char *)malloc(nItems + addLen);

    i = j = 0;
    while (i < nItems) {
	len = strlen(&oldNames[i]);
	if ((i + len + 1) > nItems)
	    break;
	remove = 0;
	for (k = 0; k < nRem; k++) {
	    if (strcmp(&oldNames[i], rem[k]) == 0) {
		remove = 1;
		break;
	    }
	}
	for (k = 0; k < nAdd; k++) {
	    if (strcmp(&oldNames[i], add[k]) == 0) {
		remove = 1;
		break;
	    }
	}
	if (!remove) {
	    strcpy(&newNames[j], &oldNames[i]);
	    j += len + 1;
	}
	i += len + 1;
    }

    for (k = 0; k < nAdd; k++) {
	strcpy(&newNames[j], add[k]);
	j += strlen(add[k]) + 1;
    }

    XChangeProperty(dpy, DefaultRootWindow(dpy), stickyNamesAtom, XA_STRING, 8,
		    PropModeReplace, (unsigned char *)newNames, j);

    XFree(oldNames);
    free(newNames);
}


/*
 * Set a property containing a list of ids of windows which should be sticky.
 * The list is actually a list of triples of (window, X offset, Y offset).
 * We assume that a window can be represented by a "long". Unfortunately this
 * seems to be the only way to get it to work on alphas as well as 32-bit
 * architectures, since the alpha Xlib automatically translates 32-bit
 * properties into 64-bit longs.
 */

static void SetWindowIds(add, rem, nAdd, nRem, singlePlus)
    Window *add, *rem;
    int nAdd, nRem;
    int singlePlus;
{
    Atom stickyWindowsAtom, windowOffsetListAtom, actualType;
    int actualFormat;
    unsigned long nItems, bytesAfter;
    Window dummyw;
    long *oldWindows, *newWindows;
    int i, j, k;
    int x, y, centreX, centreY;
    unsigned int w, h, b, d, nchildren;
    Window parent, *children;
    int viewportX, viewportY, viewportWidth, viewportHeight;

    stickyWindowsAtom = XInternAtom(dpy, "_ORL_TP_STICKY_WINDOWS", False);
    windowOffsetListAtom = XInternAtom(dpy, "WINDOW_OFFSET_LIST", False);

    XGetWindowProperty(dpy, DefaultRootWindow(dpy), stickyWindowsAtom, 0, 1024, False,
		       windowOffsetListAtom, &actualType, &actualFormat, &nItems, &bytesAfter,
		       (unsigned char **)&oldWindows);

    nItems /= 3;

    if ((actualType != windowOffsetListAtom) || (actualFormat != 32)) {
	nItems = 0;
    }

    newWindows = (long *)malloc(sizeof(long) * 3 * (nItems + nAdd));

    k = 0;
    for (i = 0; i < nItems; i++)
    {
	int remove = 0;
	for (j = 0; j < nRem; j++) {
	    if ((Window)oldWindows[i*3] == rem[j]) {
		remove = 1;
		break;
	    }
	}
	for (j = 0; j < nAdd; j++) {
	    if ((Window)oldWindows[i*3] == add[j]) {
		remove = 1;
		break;
	    }
	}
	if (!remove) {
	    newWindows[k*3] = oldWindows[i*3];
	    newWindows[k*3+1] = oldWindows[i*3+1];
	    newWindows[k*3+2] = oldWindows[i*3+2];
	    k++;
	}
    }

    GetViewportGeometry(&viewportX,&viewportY,&viewportWidth,&viewportHeight);

    for (j = 0; j < nAdd; j++)
    {
	XQueryTree(dpy, add[j], &dummyw, &parent, &children, &nchildren);
	XFree(children);

	if (parent != DefaultRootWindow(dpy)) {
	    fprintf(stderr,"%s: window 0x%lx is not a top-level window\n",
		    programName,add[j]);
	    continue;
	}

	XGetGeometry(dpy, add[j], &dummyw, &x, &y, &w, &h, &b, &d);

	x -= viewportX;
	y -= viewportY;

	centreX = x + b + w / 2;
	centreY = y + b + h / 2;

	if (x < 0) x = 0;
	if (y < 0) y = 0;

	if (centreX > (viewportWidth / 2))
	    x -= viewportWidth;

	if (centreY > (viewportHeight / 2))
	    y -= viewportHeight;

	newWindows[k*3] = (long)add[j];
	newWindows[k*3+1] = x;
	newWindows[k*3+2] = y;
	k++;
    }

    XChangeProperty(dpy, DefaultRootWindow(dpy), stickyWindowsAtom,
		    windowOffsetListAtom, 32, PropModeReplace,
		    (unsigned char *)newWindows, k * 3);

    XFree((char *)oldWindows);
    free((char *)newWindows);
}


/*
 * Routine to let user select a window using the mouse
 */

static Window SelectWindow()
{
    Cursor cursor;
    XEvent event;
    Window window = 0;

    cursor = XCreateFontCursor(dpy, XC_crosshair);

    XGrabPointer(dpy, DefaultRootWindow(dpy), False, ButtonPressMask, GrabModeSync,
		 GrabModeAsync, DefaultRootWindow(dpy), cursor, CurrentTime);

    while (!window) {
	XAllowEvents(dpy, SyncPointer, CurrentTime);
	XWindowEvent(dpy, DefaultRootWindow(dpy), ButtonPressMask, &event);
	if (event.type == ButtonPress) {
	    window = event.xbutton.subwindow; /* window selected */
	    break;
	}
    }

    XUngrabPointer(dpy, CurrentTime);      /* Done with pointer */

    return window;
}


/*
 * Get the x,y position of the viewport window.
 */

static void GetViewportGeometry(viewportXP, viewportYP, viewportWidthP, viewportHeightP)
    int *viewportXP;
    int *viewportYP;
    int *viewportWidthP;
    int *viewportHeightP;
{
    Window dummyw;
    int x, y;
    unsigned int dummyi;
    Atom actualType;
    int format;
    unsigned long nitems, bytesAfter;
    long *data;
    Atom viewportSizeAtom = XInternAtom(dpy, "_ORL_TP_VIEWPORT_SIZE", False);

    XGetGeometry(dpy, DefaultRootWindow(dpy), &dummyw, &x, &y,
		 &dummyi, &dummyi, &dummyi, &dummyi);
    *viewportXP = -x;
    *viewportYP = -y;

    XGetWindowProperty(dpy, DefaultRootWindow(dpy), viewportSizeAtom, 0, 2, False,
		       XA_INTEGER, &actualType, &format, &nitems, &bytesAfter,
		       (unsigned char **)&data);
    if ((actualType != XA_INTEGER) || (format != 32) || (nitems != 2)) {
	fprintf(stderr,"%s: not in a teleport session\n",programName);
	exit(1);
    }

    *viewportWidthP = data[0];
    *viewportHeightP = data[1];
    XFree(data);
}
