// -*- Mode: C++; -*-
//                            Package   : omniORBpy
// pyContext.cc               Created on: 2002/01/17
//                            Author    : Duncan Grisby (dpg1)
//
//    Copyright (C) 2002 AT&T Laboratories Cambridge
//
//    This file is part of the omniORBpy library
//
//    The omniORBpy library is free software; you can redistribute it
//    and/or modify it under the terms of the GNU Lesser General
//    Public License as published by the Free Software Foundation;
//    either version 2.1 of the License, or (at your option) any later
//    version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
//    MA 02111-1307, USA
//
//
// Description:
//    Context support

// $Id: pyContext.cc,v 1.1.2.1 2002/01/18 15:49:44 dpg1 Exp $
// $Log: pyContext.cc,v $
// Revision 1.1.2.1  2002/01/18 15:49:44  dpg1
// Context support. New system exception construction. Fix None call problem.
//

#include <omnipy.h>

OMNI_USING_NAMESPACE(omni)


static inline
void marshalRawPyString(cdrStream& stream, PyObject* pystring)
{
  CORBA::ULong slen = PyString_GET_SIZE(pystring) + 1;
  slen >>= stream;
  char* str = PyString_AS_STRING(pystring);
  stream.put_octet_array((const CORBA::Octet*)((const char*)str), slen);
}

static inline PyObject*
unmarshalRawPyString(cdrStream& stream)
{
  CORBA::ULong len; len <<= stream;

  if (!stream.checkInputOverrun(1, len))
    OMNIORB_THROW(MARSHAL, MARSHAL_PassEndOfMessage,
		  (CORBA::CompletionStatus)stream.completion());

  PyObject* pystring = PyString_FromStringAndSize(0, len - 1);

  stream.get_octet_array((_CORBA_Octet*)PyString_AS_STRING(pystring), len);
  return pystring;
}


void
omniPy::validateContext(PyObject* c_o, CORBA::CompletionStatus compstatus)
{
  if (!PyInstance_Check(c_o))
    OMNIORB_THROW(BAD_PARAM, BAD_PARAM_WrongPythonType, compstatus);

  PyObject* cdict = ((PyInstanceObject*)c_o)->in_dict;

  PyObject* t_o = PyDict_GetItemString(cdict, (char*)"_NP_RepositoryId");
  if (!(t_o && PyString_Check(t_o)))
    OMNIORB_THROW(BAD_PARAM, BAD_PARAM_WrongPythonType, compstatus);

  if (!omni::strMatch(PyString_AS_STRING(t_o),
		      "IDL:omg.org/CORBA/Context:1.0"))
    OMNIORB_THROW(BAD_PARAM, BAD_PARAM_WrongPythonType, compstatus);
}

void
omniPy::marshalContext(cdrStream& stream, PyObject* p_o, PyObject* c_o)
{
  PyObject* values = PyObject_CallMethod(c_o, "_get_values", "O", p_o);
  if (values) {
    PyObject* items = PyDict_Items(values);
    CORBA::ULong count = PyList_GET_SIZE(items);
    CORBA::ULong mlen  = count * 2;
    mlen >>= stream;

    for (CORBA::ULong i=0; i < count; i++) {
      PyObject* item = PyList_GET_ITEM(items, i);
      marshalRawPyString(stream, PyTuple_GET_ITEM(item, 0));
      marshalRawPyString(stream, PyTuple_GET_ITEM(item, 1));
    }
    Py_DECREF(values);
  }
  else {
    if (omniORB::trace(1)) {
      {
	omniORB::logger l;
	l << "Exception trying to get Context values:\n";
      }
      PyErr_Print();
    }
    else
      PyErr_Clear();
    
    OMNIORB_THROW(TRANSIENT, TRANSIENT_PythonExceptionInORB,
		  CORBA::COMPLETED_NO);
  }
}

PyObject*
omniPy::unmarshalContext(cdrStream& stream)
{
  PyObject* dict = PyDict_New();
  CORBA::ULong mlen;
  mlen <<= stream;

  if (mlen % 2)
    OMNIORB_THROW(MARSHAL,
		  MARSHAL_InvalidContextList,
		  CORBA::COMPLETED_MAYBE);

  CORBA::ULong count = mlen / 2;

  for (CORBA::ULong i=0; i < count; i++) {
    PyObject* k = unmarshalRawPyString(stream);
    PyObject* v = unmarshalRawPyString(stream);
    PyDict_SetItem(dict, k, v);
    Py_DECREF(k);
    Py_DECREF(v);
  }
  PyObject* r = PyObject_CallFunction(omniPy::pyCORBAContextClass,
				      (char*)"sON", "", Py_None, dict);
  if (!r) {
    if (omniORB::trace(1)) {
      {
	omniORB::logger l;
	l << "Exception trying to construct Context:\n";
      }
      PyErr_Print();
    }
    else
      PyErr_Clear();

    OMNIORB_THROW(TRANSIENT, TRANSIENT_PythonExceptionInORB,
		  CORBA::COMPLETED_NO);
  }
  return r;
}


PyObject*
omniPy::filterContext(PyObject* p_o, PyObject* c_o)
{
  PyObject* values = PyObject_CallMethod(c_o, "_get_values", "O", p_o);

  if (values) {
    PyObject* r = PyObject_CallFunction(omniPy::pyCORBAContextClass,
					(char*)"sON", "", Py_None, values);
    if (r) return r;
  }
  if (omniORB::trace(1)) {
    {
      omniORB::logger l;
      l << "Exception trying to filter Context:\n";
    }
    PyErr_Print();
  }
  else
    PyErr_Clear();

  OMNIORB_THROW(TRANSIENT, TRANSIENT_PythonExceptionInORB,
		CORBA::COMPLETED_NO);
  return 0;
}
