// -*- Mode: C++; -*-
//                            Package   : omniEvents
// eventc.cc                  Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//    Client to the event channel factory. Requests creation of an event
//    channel and registers it with the Naming service.
//	

/*
  $Log: eventc.cc,v $
  Revision 1.4  2000/03/16 05:34:30  naderp
  Added stdlib.h for solaris getopt()

  Revision 1.3  2000/03/16 02:44:13  naderp
  Added iostream and signal headers.

  Revision 1.2  2000/03/06 13:23:50  naderp
  Using util getRootNamingContext function.
  Using stub headers.

  Revision 1.1  1999/11/01 20:37:42  naderp
  Updated usage statement.

Revision 1.0  99/11/01  17:05:13  17:05:13  naderp (Paul Nader)
omniEvents 2.0.
Added -m switch to support MaxEventsPerConsumer criteria.

Revision 0.6  99/08/27  11:48:22  11:48:22  naderp (Paul Nader)
Partitioned EventChannelFactory_i from CosEvent_i.

Revision 0.5  99/05/10  11:27:50  11:27:50  naderp (Paul Nader)
Initialised rootContext.

Revision 0.4  99/04/23  16:02:22  16:02:22  naderp (Paul Nader)
gcc port.

Revision 0.3  99/04/23  09:32:58  09:32:58  naderp (Paul Nader)
Windows Port.

Revision 0.2  99/04/21  18:06:23  18:06:23  naderp (Paul Nader)
*** empty log message ***

Revision 0.1.1.1  98/11/27  17:01:51  17:01:51  naderp (Paul Nader)
Enclosed supports call in try block to avoid core dump.
Added information messages for exceptions.

Revision 0.1  98/11/25  14:07:22  14:07:22  naderp (Paul Nader)
Initial Revision

*/

//
//
#include <CosEventChannelAdmin.hh>
#include <EventChannelAdmin.hh>
#ifdef __WIN32__
#include <getopt.h>
#include <iostream>
#else 
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include <signal.h>
#endif

static void usage();
extern CosNaming::NamingContext_ptr getRootNamingContext(CORBA::ORB_ptr orb);

int
main(int argc, char **argv)
{
  //
  // Start orb and boa.
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB2");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB2_BOA");

  // Process Options
  int c;
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";
  char *factoryName = (char *) "EventChannelFactory";
  char *factoryKind = (char *) "EventChannelFactory";
  CORBA::ULong pullRetryPeriod = 1;
  CORBA::ULong maxEventsPerConsumer = 0;

  while ((c = getopt(argc,argv,"n:k:N:K:r:m:h")) != EOF)
  {
     switch (c)
     {
        case 'n': channelName = optarg;
                  break;

        case 'k': channelKind = optarg;
                  break;

        case 'N': factoryName = optarg;
                  break;

        case 'K': factoryKind = optarg;
                  break;

        case 'r': pullRetryPeriod = atol(optarg);
                  break;

        case 'm': maxEventsPerConsumer = atol(optarg);
                  break;

        case 'h':
        default : usage();
                  exit(-1);
                  break;
     }
  }


  //
  // Get Name Service root context.
  CosNaming::NamingContext_ptr rootContext;
  rootContext = getRootNamingContext(orb);

  //
  // Obtain reference to the Event Channel Factory implementation.
  EventChannelAdmin::EventChannelFactory_ptr factory;
  CosNaming::Name name;
  name.length (1);
  name[0].id = CORBA::string_dup (factoryName);
  name[0].kind = CORBA::string_dup (factoryKind);

  try {
    CORBA::Object_var obj = rootContext->resolve(name);
    factory = EventChannelAdmin::EventChannelFactory::_narrow(obj);
    if (CORBA::is_nil(factory))
    {
       cerr << "Failed to narrow Event Channel Factory reference." << endl;
       exit(1);
    }

  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the "
          << "naming service." << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot find event channel factory ! [\""
          << factoryName << "\", \"" << factoryKind << "\"]"
          << endl;
     exit (1);
  }

  // Check that the factory is of the right type
  CosLifeCycle::Key key;
  key.length (1);
  key[0].id = CORBA::string_dup ("EventChannel");
  key[0].kind = CORBA::string_dup ("object interface");
  try {
     if (! factory->supports(key))
     {
        cerr << "Factory does not support Event Channel Interface ! [\""
             << factoryName << "\", \"" << factoryKind << "\"]"
             << endl;
        exit (1);
     }
  }
  catch (...)
  {
     cerr << "Failure contacting Event Channel Factory" << endl;
     exit (1);
  }  

  //
  // Create Event Channel Object.
  CosEventChannelAdmin::EventChannel_var channel;
  try {
     CORBA::Object_var channelObj;
     CosLifeCycle::Criteria criteria;
     criteria.length (2);
     criteria[0].name = CORBA::string_dup ("PullRetryPeriod");
     criteria[0].value <<= (CORBA::ULong) pullRetryPeriod;
     criteria[1].name = CORBA::string_dup ("MaxEventsPerConsumer");
     criteria[1].value <<= (CORBA::ULong) maxEventsPerConsumer;

     channelObj = factory->create_object(key, criteria);
     if (CORBA::is_nil(channelObj))
     {
        cerr << "Channel Factory returned nil reference. ! [\""
             << channelName << "\", \"" << channelKind << "\"]"
             << endl;
        exit(1);
     }

     // Narrow object returned to an Event Channel
     channel = CosEventChannelAdmin::EventChannel::_narrow(channelObj);
     if (CORBA::is_nil(channel))
     {
        cerr << "Failed to narrow Event Channel ! [\""
             << channelName << "\", \"" << channelKind << "\"]"
             << endl;
        exit(1);
     }
  }
  catch (CosLifeCycle::NoFactory& ex) {
     cerr << "Failed to create Event Channel : "
          << "Interface not supported "
          << endl;
     exit(1);
  }
  catch (CosLifeCycle::CannotMeetCriteria& ex) {
     cerr << "Failed to create Event Channel : "
          << "Cannot meet Criteria "
          << endl;
     exit(1);
  }
  catch (...) {
     cerr << "Failed to create Event Channel ! [\""
          << channelName << "\", \"" << channelKind << "\"]"
          << endl;
     exit(1);
  }

  //
  // Register event channel with naming service
  name.length (1);
  name[0].id = CORBA::string_dup (channelName);
  name[0].kind = CORBA::string_dup (channelKind);
  try {
    rootContext->bind (name,
              CosEventChannelAdmin::EventChannel::_duplicate(channel));
  }
  catch(CosNaming::NamingContext::AlreadyBound& ex) {
    rootContext->rebind(name,
              CosEventChannelAdmin::EventChannel::_duplicate(channel));
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the "
          << "naming service." << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot register event channel ! [\""
          << channelName << "\", \"" << channelKind << "\"]"
          << endl;
     exit (1);
  }

  return 0;
}

static void
usage()
{
   cerr << "\nusage: eventc [-n name] [-k kind] [-N name] [-K kind] [-r sec] [-m events] [-h]\n" << endl;
   cerr << "         -n name specify channel name [\"EventChannel\"]" << endl;
   cerr << "         -k kind specify channel kind [\"EventChannel\"]" << endl;
   cerr << "         -N name specify factory name [\"EventChannelFactory\"]" << endl;
   cerr << "         -K kind specify factory kind [\"EventChannelFactory\"]" << endl;
   cerr << "         -r Pull retry Period (in seconds) [1 sec]" << endl;
   cerr << "         -m Maximum Number of Events queued per Consumer [0 = unlimited]" << endl;
   cerr << "         -h     display usage" << endl;
   cerr << endl;
}

