// $Id: Clock.java,v 1.2 1998/01/21 17:30:13 bjh Exp $

/*
$Log: Clock.java,v $
# Revision 1.2  1998/01/21  17:30:13  bjh
# moved into UK.co.orl.weather.
#
# Revision 1.1  1997/10/17  16:59:28  bjh
# Initial revision
#
*/

package UK.co.orl.weather;

import java.awt.*;
import java.util.Date;
import java.util.Calendar;
import java.util.TimeZone;
import java.text.DateFormat;

public class Clock extends Dial {
  static {
    if (System.getProperty("java.version").equals("1.1.4") &&
	TimeZone.getDefault().getID().equals("GMT")) {
      // Workaround for wrong default UK time zone in 1.1.4 (bug #4085909)
      TimeZone.setDefault(new java.util.SimpleTimeZone(0, "GB-Eire",
						       Calendar.MARCH, -1,
						       Calendar.SUNDAY,
						       3600000,
						       Calendar.OCTOBER, -1,
						       Calendar.SUNDAY,
						       2*3600000));
    }
  }

  Date currentValue;

  public Clock(Color handColor) {
    this(new Date(), handColor);
  }
  public Clock(Date date, Color handColor) {
    this.pointerAngle = new float[2];
    this.setValue(date);
    this.pointerColor = new Color[] { handColor, handColor };
    this.pointerStyle = new Pointer[] {new TrianglePointer((float)0.7),
					  new TrianglePointer() };
  }
  
  public void setValue(Date newValue) {
    if (!newValue.equals(currentValue)) {
      currentValue = newValue;
      this.pointerAngle[0] = hourAngle(newValue);
      this.pointerAngle[1] = minuteAngle(newValue);
      repaint();
    }
  }

  protected float hourAngle(Date date) {
    int millis;
    try {
      Calendar cal = Calendar.getInstance();
      cal.setTime(date);
      millis = (cal.get(Calendar.HOUR) * (60*60*1000) +
		cal.get(Calendar.MINUTE) * (60*1000) +
		cal.get(Calendar.SECOND) * 1000 +
		cal.get(Calendar.MILLISECOND));
    } catch (NoClassDefFoundError e) { // JDK 1.0
      millis = (date.getHours() * 3600000 +
		date.getMinutes() * 60000 +
		date.getSeconds() * 1000);
      }
    return hourAngle(millis);
  }

  protected float hourAngle(int millis) {
    return -(float)millis * ((float)Math.PI/(12*60*60*1000/2)) +
      (float)Math.PI/2; // get 0 to be up.
  }

  protected float minuteAngle(Date date) {
    int millis;
    try {
      Calendar cal = Calendar.getInstance();
      cal.setTime(date);
      millis = (cal.get(Calendar.MINUTE) * (60*1000) +
		cal.get(Calendar.SECOND) * 1000 +
		cal.get(Calendar.MILLISECOND));
    } catch (NoClassDefFoundError e) { // JDK 1.0
      millis = (date.getMinutes() * (60*1000) +
		date.getSeconds() * 1000);
      }
    return minuteAngle(millis);
  }

  protected float minuteAngle (int millis) {
    return -(float)millis * ((float)Math.PI/(60*60*1000/2)) +
      (float)Math.PI/2; // get 0 to be up.
  }   

  public void paint(Graphics g) {
    super.paint(g);
    this.ticks(g);
    centreText(g, formatDate(currentValue), new Point(0, (int)(radius * 0.5)));
  }
 
  protected void ticks(Graphics g) {
    // Make all sorts of hoorible non-internationalisable assumptions
    // about the number of minutes in an hour and suchlike.
    for (int h = 1; h <= 12; h++) {
      float angle = this.hourAngle(h * (60*60*1000));
      this.tick(g, this.radius, angle, 5);
      if (h % 3 == 0)
	centreText(g, "" + h, polar((int)(radius * 0.8), angle));
    }
    for (int m = 0; m < 60; m++) {
      this.tick(g, this.radius, this.minuteAngle(m * (60*1000)), 3);
    }
  }

  String formatDate(Date date) {
    try {
      DateFormat df = DateFormat.getTimeInstance(DateFormat.MEDIUM);
      // Workaround for wrong default timezone (safe on versions which behave
      // correctly) (bug #4071776).
      df.setTimeZone(TimeZone.getDefault());
      return df.format(date);
    } catch (NoClassDefFoundError e) { // JDK 1.0
      return date.getHours() + ":" + date.getMinutes();
    }
  }
}

