// -*- Mode: C++; -*-
//                            Package   : omniORB2
// tcpSocketMTSocket.cc       Created on: 18/3/96
//                            Author    : Sai Lai Lo (sll)
//
//    Copyright (C) 1996, 1997 Olivetti & Oracle Research Laboratory
//
//    This file is part of the omniORB library
//
//    The omniORB library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//	Implementation of the Strand using TCP/IP and BSD socket interface
//	

/*
  $Log: tcpSocketMTfactory.cc,v $
  Revision 1.9  1998/04/22 16:39:50  sll
  Added try-catch loop to guard against exception raised by the thread library
  when it cannot create a new thread for tcpSocketWorker.

  Revision 1.8  1998/04/08 16:06:49  sll
  Added support for Reliant UNIX 5.43

  Revision 1.7  1998/04/07 19:39:40  sll
  Replace cerr with omniORB::log.

  Revision 1.6  1998/03/19 19:53:14  sll
  Delay connect to the remote address space until the first send or recv.
  Previously, connect was made inside the ctor of tcpSocketStrand.

  Revision 1.5  1998/03/04 14:44:36  sll
  Updated to use omniORB::giopServerThreadWrapper.

// Revision 1.4  1998/01/20  17:32:38  sll
// Added support for OpenVMS.
//
  Revision 1.3  1997/12/18 17:27:39  sll
  Updated to work under glibc-2.0.

  Revision 1.2  1997/12/12 18:44:11  sll
  Added call to gateKeeper.

  Revision 1.1  1997/12/09 18:43:18  sll
  Initial revision

  */

#include <omniORB2/CORBA.h>
#include <omniORB2/QOS.h>
#include <omniORB2/ConnectionInfo.h>
#include <ropeFactory.h>
#include <inetSocketFactory.h>
#include <inetSocket.h>
#include <tcpMTSocket.h>
#include <gatekeeper.h>

#include <netinet/tcp.h>

/////////////////////////////////////////////////////////////////////////////

tcpSocketIncomingRope::tcpSocketIncomingRope(inetSocketMTincomingFactory* f,
					     unsigned int maxStrands,
					     inetSocketEndpoint *e,
					     CORBA::Boolean export)
  : inetSocketIncomingRope(f, maxStrands, e, export), rendezvouser(0)
{
  struct sockaddr_in myaddr;

  // For the moment, we do not impose a restriction on the maximum
  // no. of strands that can be accepted. In other words, <maxStrands> is 
  // ignored.

  if ((pd_rendezvous.sock = socket(INETSOCKET,SOCK_STREAM,0)) == RC_INVALID_SOCKET) {
    throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
  }
  myaddr.sin_family = INETSOCKET;
  myaddr.sin_addr.s_addr = INADDR_ANY;
  myaddr.sin_port = htons(e->tcpPort());

  if (myaddr.sin_port) {
    int valtrue = 1;
    if (setsockopt(pd_rendezvous.sock,SOL_SOCKET,
		   SO_REUSEADDR,(char*)&valtrue,sizeof(int)) == RC_SOCKET_ERROR)
      {
	CLOSEINETSOCKET(pd_rendezvous);
	throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
      }
  }

  if (bind(pd_rendezvous.sock,(struct sockaddr *)&myaddr,
	   sizeof(struct sockaddr_in)) == RC_SOCKET_ERROR) 
  {
    CLOSEINETSOCKET(pd_rendezvous);
    throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
  }

  // Make it a passive socket
  if (listen(pd_rendezvous.sock,5) == RC_SOCKET_ERROR) {
    CLOSEINETSOCKET(pd_rendezvous);
    throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
  }
  
  {
#if (defined(__GLIBC__) && __GLIBC__ >= 2)
    // GNU C library uses socklen_t * instead of int* in getsockname().
    // This is suppose to be compatible with the upcoming POSIX standard.
    socklen_t l;
#elif defined(__aix__) || defined(__VMS) || defined(__SINIX__)
    size_t l;
# else
    int l;
# endif


    l = sizeof(struct sockaddr_in);
    if (getsockname(pd_rendezvous.sock,
		    (struct sockaddr *)&myaddr,&l) == RC_SOCKET_ERROR) {
      CLOSEINETSOCKET(pd_rendezvous);
      throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
    }

    e->tcpPort(ntohs(myaddr.sin_port));
    if (e->inetHost() == 0 || strlen((const char*)e->inetHost()) == 0) {

      char self[64];
      if (gethostname(&self[0],64) == RC_SOCKET_ERROR) {
	throw omniORB::fatalException(__FILE__,__LINE__,
				      "Cannot get the name of this host");
      }

      LibcWrapper::hostent_var h;
      int rc;

      if (LibcWrapper::gethostbyname(self,h,rc) < 0) {
	throw omniORB::fatalException(__FILE__,__LINE__,
				      "Cannot get the address of this host");
      }
      memcpy((void *)&myaddr.sin_addr,
	     (void *)h.hostent()->h_addr_list[0],
	     sizeof(myaddr.sin_addr));
      char ipaddr[16];
      // To prevent purify from generating UMR warnings, use the following temp
      // variables to store the IP address fields.
      int ip1 = (int)((ntohl(myaddr.sin_addr.s_addr) & 0xff000000) >> 24);
      int ip2 = (int)((ntohl(myaddr.sin_addr.s_addr) & 0x00ff0000) >> 16);
      int ip3 = (int)((ntohl(myaddr.sin_addr.s_addr) & 0x0000ff00) >> 8);
      int ip4 = (int)(ntohl(myaddr.sin_addr.s_addr) & 0x000000ff);
      sprintf(ipaddr,"%d.%d.%d.%d",ip1,ip2,ip3,ip4);
      e->inetHost((const CORBA::Char *) ipaddr);

    }
    else {
      // The caller has already specified the host name, we are not going to
      // override it here. However, it may be possible that the host name does
      // not resolve to one of the IP addresses that identified the network
      // interfaces of this machine. There is no way to guard against this
      // mistake.
      // Do nothing here.
    }
  }

  me = new inetSocketEndpoint(e);
}

tcpSocketIncomingRope::~tcpSocketIncomingRope()
{
  if (omniORB::traceLevel >= 15) {
    cerr << "tcpSocketIncomingRope::~tcpSocketIncomingRope: called." << endl;
  }
  if (me) {
    delete me;
    me = 0;
  }
  if (pd_rendezvous.sock != RC_INVALID_SOCKET) {
    CLOSEINETSOCKET(pd_rendezvous);
    pd_rendezvous.sock = RC_INVALID_SOCKET;
  }
}

void
tcpSocketIncomingRope::cancelThreads()
{
  if (rendezvouser) {
      pd_lock.lock();
      pd_shutdown = SHUTDOWN;
      pd_lock.unlock();
  }
  CutStrands();
  
  if (rendezvouser) {
    // Unblock the rendezvouser from its accept() call.
    // Cannot shutdown the rendezvous socket because we want to keep it open
    // and could be served later by another rendezvouser.
    // 
    // Instead, unblock the rendezvouser by making a connect() call.

    struct sockaddr_in myaddr;

    {
      Endpoint* e = 0;
      this_is(e);
      inetSocketEndpoint* te = (inetSocketEndpoint*)e;
      myaddr.sin_family = INETSOCKET;
      myaddr.sin_port   = htons(te->tcpPort());
      if (LibcWrapper::isipaddr((char*)te->inetHost())) 
	{
	  CORBA::ULong ip_p = inet_addr((char*) te->inetHost());
	  memcpy((void*) &myaddr.sin_addr, (void*) &ip_p, sizeof(myaddr.sin_addr));
	}
      else
	{
	  LibcWrapper::hostent_var h;
	  int  rc;
	  LibcWrapper::gethostbyname((char*)te->inetHost(),h,rc);
	  memcpy((void*)&myaddr.sin_addr,
		 (void*)h.hostent()->h_addr_list[0],
		 sizeof(myaddr.sin_addr));
	}
      delete te;
    }

    inetSocketHandle_t  tmp_sock;

    tmp_sock.ssl = NULL;
    if ((tmp_sock.sock = socket(INETSOCKET,SOCK_STREAM,0)) == RC_INVALID_SOCKET) 
      {
	// If we cannot create a socket, we cannot shutdown the rendezvouser
	if (omniORB::traceLevel > 0) {
	  cerr << "tcpSocketIncomingRope::cancelThreads() cannot create a socket to connect to the rendezvous socket.\n"
		       << "The rendezvous thread may or may not have exited.\n"
		       << "If this is temporary resource exhaustion, try again later.\n";
	  cerr <<  endl;
	}
	return;
      }
    if (connect(tmp_sock.sock,(struct sockaddr *)&myaddr,
		sizeof(struct sockaddr_in)) == RC_SOCKET_ERROR) 
      {
	// OK, so we cannot connect to the rendezvouser, it would have
	// unblock from accept() anyway. That is fine as well.
      }
    else 
      {
	// Now we have unblock the rendezvouser, just close the socket.
	CLOSEINETSOCKET(tmp_sock);
      }

    // Now we have got the rendezvouser's attention. We acknowlege that
    // we have seen it and instructs the rendezvouser to exit.
    pd_lock.lock();
    pd_shutdown = NO_THREAD;
    pd_lock.unlock();

    if (omniORB::traceLevel >= 15) {
      cerr << "tcpSocketMTincomingFactory::stopIncoming: Waiting for tcpSocketMT Rendezvouser to exit..." << endl;
    }
    rendezvouser->join(0); // Wait till the rendezvouser to come back
    if (omniORB::traceLevel >= 15) {
      cerr << "tcpSocketMTincomingFactory::stopIncoming: tcpSocketMT Rendezvouser has exited" << endl;
    }
    rendezvouser = 0;
  }
}

/////////////////////////////////////////////////////////////////////////////


tcpSocketOutgoingRope::tcpSocketOutgoingRope(inetSocketMToutgoingFactory* f,
					     unsigned int maxStrands,
					     inetSocketEndpoint *e)
  : inetSocketOutgoingRope(f, maxStrands, e)
{
  tcpContext::initContext();
}

Strand *
tcpSocketOutgoingRope::newStrand()
{
  return new tcpSocketStrand(this,remote,1);
}

/////////////////////////////////////////////////////////////////////////////


// Size of transmit and receive buffers
const
unsigned int
tcpSocketStrand::buffer_size = 8192 + (int)omni::max_alignment;

tcpSocketStrand::tcpSocketStrand(tcpSocketOutgoingRope *rope,
				 inetSocketEndpoint   *r,
				 CORBA::Boolean heapAllocated)
  : inetSocketStrand(rope, r, tcpSocketStrand::buffer_size, heapAllocated)
{
  // Do not try to connect to the remote host in this ctor.
  // This is to avoid holding the mutex on rope->pd_lock while the connect
  // is in progress. Holding the mutex for an extended period is bad as this 
  // can have ***serious*** side effect. 
  // One immediate consequence of holding the rope->pd_lock is that the
  // outScavenger will be blocked on rope->pd_lock when it is scanning
  // for idle strands. This in turn blockout any thread trying to lock
  // rope->pd_anchor->pd_lock. This is really bad because no new rope
  // can be added to the anchor.

  pd_socket.sock = RC_INVALID_SOCKET;
  pd_delay_connect = new inetSocketEndpoint(r);

  DefaultConnectionInterceptorFactory *factory = new DefaultConnectionInterceptorFactory();
  pd_connection_interceptor = factory->create();

  // Do the connect on first call to ll_recv or ll_send.
}

tcpSocketStrand::tcpSocketStrand(tcpSocketIncomingRope *r,
				 inetSocketHandle_t sock,
				 CORBA::Boolean heapAllocated)
  : inetSocketStrand(r, sock, tcpSocketStrand::buffer_size, heapAllocated)
{
}


tcpSocketStrand::~tcpSocketStrand() 
{
  if (omniORB::traceLevel >= 5) {
    cerr << "tcpSocketStrand::~Strand() close socket no. " << pd_socket.sock << endl;
  }
  if (pd_socket.sock != RC_INVALID_SOCKET)
    CLOSEINETSOCKET(pd_socket);
  pd_socket.sock = RC_INVALID_SOCKET;
  if (pd_delay_connect)
    delete pd_delay_connect;
  pd_delay_connect = 0;
  if (pd_connect)
    delete pd_connect;
  pd_connect = 0;
  if (pd_connection_interceptor)
    delete pd_connection_interceptor;
  pd_connection_interceptor = 0;
}

size_t
tcpSocketStrand::ll_recv(void* buf, size_t sz)
{
  if (pd_delay_connect) {
    // We have not connect to the remote host yet. Do the connect now.
    // Note: May block on connect for sometime if the remote host is down
    //
    IIOPConnectionState iiop_cs(this);

    pd_connection_interceptor->beforeConnect(iiop_cs);

    pd_socket = tcpRealConnect(this, pd_delay_connect);
    if (pd_socket.sock == RC_INVALID_SOCKET) {
      _setStrandIsDying();
      throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
    }
    if(pd_connection_interceptor->afterConnect(iiop_cs) == Reject) {
      CLOSEINETSOCKET(pd_socket);
      _setStrandIsDying();
      throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
    }
    pd_connect = pd_delay_connect;
    pd_delay_connect = 0;
  }

  int rx;
  while (1) {
    if (omniORB::traceLevel >= 15) {
      cerr << "normal read" << endl;
    }
    if ((rx = ::recv(pd_socket.sock,(char*)buf,sz,0)) == RC_SOCKET_ERROR) {
      if (errno == EINTR)
	continue;
      else
	{
	  _setStrandIsDying();
	  throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_MAYBE);
	}
    }
    else
      if (rx == 0) {
	_setStrandIsDying();
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_MAYBE);
      }
    break;
  }
  return (size_t)rx;
}

void
tcpSocketStrand::ll_send(void* buf,size_t sz) 
{

try_reconnect:

  if (pd_delay_connect) {
    // We have not connect to the remote host yet. Do the connect now.
    // Note: May block on connect for sometime if the remote host is down
    //
    IIOPConnectionState iiop_cs(this);

    pd_connection_interceptor->beforeConnect(iiop_cs);

    pd_socket = tcpRealConnect(this, pd_delay_connect);
    if (pd_socket.sock == RC_INVALID_SOCKET) {
      _setStrandIsDying();
      throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
    }
    if(pd_connection_interceptor->afterConnect(iiop_cs) == Reject) {
      CLOSEINETSOCKET(pd_socket);
      _setStrandIsDying();
      throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
    }
    pd_connect = pd_delay_connect;
    pd_delay_connect = 0;
  }

  int tx;
  char *p = (char *)buf;
  while (sz) {
    if (omniORB::traceLevel >= 15) {
      cerr << "normal send" << endl;
    }
    if ((tx = ::send(pd_socket.sock,p,sz,0)) == RC_SOCKET_ERROR) {
      if (errno == EINTR)
	continue;
      else {
        if(pd_connect) {
          //
          // Before raising an exception, try to reconenct to the server.
          //
          shutdown();

          pd_delay_connect = pd_connect;
          pd_connect = 0;

          goto try_reconnect;
        } else {
          _setStrandIsDying();
          throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_MAYBE);
        }
      }
    }
    else
      if (tx == 0) {
	_setStrandIsDying();
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_MAYBE);
      }
    sz -= tx;
    p += tx;
  }
  return;
}

CORBA::Boolean
tcpSocketStrand::ll_connect() 
{
  if (pd_delay_connect) {
    // We have not connect to the remote host yet. Do the connect now.
    // Note: May block on connect for sometime if the remote host is down
    //
    IIOPConnectionState iiop_cs(this);

    pd_connection_interceptor->beforeConnect(iiop_cs);

    pd_socket = tcpRealConnect(this, pd_delay_connect);
    if (pd_socket.sock == RC_INVALID_SOCKET) {
      _setStrandIsDying();
      throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
    }
    if(pd_connection_interceptor->afterConnect(iiop_cs) == Reject) {
      CLOSEINETSOCKET(pd_socket);
      _setStrandIsDying();
      return 0;
    }
    pd_connect = pd_delay_connect;
    pd_delay_connect = 0;

    return 1;
  } else {
    return 0;
  }
}

CORBA::Boolean
tcpSocketStrand::ll_shutdown()
{
  IIOPConnectionState iiop_cs(this);

  if((pd_connect == NULL) && (pd_delay_connect == NULL)) {
    //
    // The case for servers.
    //
    if(pd_connection_interceptor->beforeDisconnect(iiop_cs) == Reject) {
      return 0;
    }

    shutdown();
    pd_connection_interceptor->afterDisconnect(iiop_cs);

    return 1;
  } else {
    //
    // The case for clients.
    //
    if(pd_connect) {

      if(pd_connection_interceptor->beforeDisconnect(iiop_cs) == Reject) {
        return 0;
      }

      shutdown();

      pd_connection_interceptor->afterDisconnect(iiop_cs);

      pd_delay_connect = pd_connect;
      pd_connect = 0;

      return 1;
    } else {
      return 0;
    }
  }
}
void
tcpSocketStrand::shutdown()
{
  if (pd_send_giop_closeConnection)
    {
      // XXX This is a bit ugly. GIOP requires us to send a CloseConnection 
      // message before shutdown. Ideally, we do not want this abstraction to 
      // do any GIOP dependent stuff. If this a problem in future, we should 
      // perhap make it a parameter to decide whether or what to send on
      // shutdown.
      size_t sz = sizeof(GIOP_Basetypes::MessageHeader::CloseConnection);
      char*  p = (char*)&GIOP_Basetypes::MessageHeader::CloseConnection;
      while (sz) {
	fd_set wrfds;
	FD_ZERO(&wrfds);
	FD_SET(pd_socket.sock,&wrfds);
	struct timeval t = { 5,0};
	int rc;
	if ((rc = select(pd_socket.sock+1,0,&wrfds,0,&t)) <= 0) {
	  // Any timeout or error, we just don't border sending the message.
	  break;
	}
	int tx;
	if ((tx = ::send(pd_socket.sock,p,sz,0)) == RC_SOCKET_ERROR) {
	  // Any error we just don't border sending anymore.
	  break;
	}
	sz -= tx;
	p += tx;
      }
    }
  _setStrandIsDying();

  SHUTDOWNINETSOCKET(pd_socket);

  return;
}

CORBA::Boolean
tcpSocketStrand::controlStrand(command_type type, char* buf, int size)
{
  ConnectionInfo info;
  IIOPConnectionState *iiop_cs;
  CORBA::Boolean flag;
  int one_or_zero, in_size, rc;
  struct sockaddr_in raddr;
  char ipaddr[16];

  switch(type) {

  case GetConnectionState:
    if(size != sizeof(iiop_cs)) {
      return 0;
    }

    iiop_cs = new IIOPConnectionState(this);
    memcpy(buf, (char *) &iiop_cs, sizeof(iiop_cs));

    break;

  case GetConnectionInfo:

    if(size != sizeof(info)) {
      return 0;
    }

    info.con_request = pd_request;
    info.con_info = pd_info;
    memcpy(buf, (char *) &info, sizeof(info));

    break;

  case GetConnectionStatus:

    if(size != sizeof(flag)) {
      return 0;
    }

    flag = (pd_connect) ? 1 : 0;
    memcpy(buf, (char *) &flag, sizeof(flag));

    break;

  case CheckPeerAddress:

    if(size != sizeof(flag)) {
      return 0;
    }

    flag = gateKeeper::checkConnect(this);
    memcpy(buf, (char *) &flag, sizeof(flag)); 

    break;

  case GetPeerAddress:
    size = sizeof(raddr);
    rc = getpeername(pd_socket.sock, (struct sockaddr *)&raddr, &size);
    // To prevent purify from generating UMR warnings, use the following temp
    // variables to store the IP address fields.
    {
      int ip1 = (int)((ntohl(raddr.sin_addr.s_addr) & 0xff000000) >> 24);
      int ip2 = (int)((ntohl(raddr.sin_addr.s_addr) & 0x00ff0000) >> 16);
      int ip3 = (int)((ntohl(raddr.sin_addr.s_addr) & 0x0000ff00) >> 8);
      int ip4 = (int)(ntohl(raddr.sin_addr.s_addr) & 0x000000ff);
      sprintf(ipaddr,"%d.%d.%d.%d",ip1,ip2,ip3,ip4);
    }
    memcpy(buf, ipaddr, strlen(ipaddr) + 1);

    break;

  case SetConnectionInterceptor:

    if(size != sizeof(pd_connection_interceptor)) {
      return 0;
    }

    memcpy((char *) &pd_connection_interceptor,
			buf, sizeof(pd_connection_interceptor));

    break;

  case SetConnectionInfo:

    if(size != sizeof(info)) {
      return 0;
    }

    memcpy((char *) &info, buf, sizeof(info));

    pd_request = info.con_request;
    pd_info = info.con_info;

    break;

  case SetDelayedAck:

    if(size != sizeof(one_or_zero)) {
      return 0;
    }

    memcpy((char *) &one_or_zero, buf, sizeof(one_or_zero));    

    if(setsockopt(pd_socket.sock, IPPROTO_TCP, TCP_NODELAY,
                        (char *)&one_or_zero, sizeof(one_or_zero)) < 0) {
      return 0;
    }

    break;

  case NotifyChangeTransport:
    {
       ConnectionState *cs;
       memcpy(&cs, buf, sizeof(cs));

       pd_connection_interceptor->changeTransport(*cs);
    }
    break;

  default:
     buf[0] = NULL;
     return 0;
  }

  return 1;
}

inetSocketHandle_t
tcpRealConnect(tcpSocketStrand* s, inetSocketEndpoint* r)
{
  static struct sockaddr_in raddr; 
  LibcWrapper::hostent_var h;
  int  rc;
  int  len;
  inetSocketHandle_t sock;

  sock.ssl = NULL;
  if (! LibcWrapper::isipaddr( (char*) r->inetHost()))
    {
      if (LibcWrapper::gethostbyname((char *)r->inetHost(),h,rc) < 0) 
	{
	  // XXX look at rc to decide what to do or if to give up what errno
	  // XXX to return EINVAL.
 	  
	  //
 	  sock.sock = RC_INVALID_SOCKET;
	  return sock;
	}
      // We just pick the first address in the list, may be we should go
      // through the list and if possible pick the one that is on the same
      // subnet.
      memcpy((void*)&raddr.sin_addr,
	     (void*)h.hostent()->h_addr_list[0],
	     sizeof(raddr.sin_addr));
    }
  else
    {
      // The machine name is already an IP address
      CORBA::ULong ip_p;
      if ( (ip_p = inet_addr( (char*) r->inetHost() )) == RC_INADDR_NONE)
	{
	  sock.sock = RC_INVALID_SOCKET;
	  return sock;
	}
      memcpy((void*) &raddr.sin_addr, (void*) &ip_p, sizeof(raddr.sin_addr));
    }

  raddr.sin_family = INETSOCKET;
  raddr.sin_port = htons(r->tcpPort());

  if ((sock.sock = socket(INETSOCKET,SOCK_STREAM,0)) == RC_INVALID_SOCKET) {
    sock.sock = RC_INVALID_SOCKET;
    return sock;
  }

#if defined(__sunos__) && defined(__sparc__) && __OSVERSION__ >= 5
  // Use non-blocking connect.
  int fl = O_NONBLOCK;
  if (fcntl(sock.sock,F_SETFL,fl) < RC_SOCKET_ERROR) {
    CLOSEINETSOCKET(sock);
    sock.sock = RC_INVALID_SOCKET;
    return sock;
  }
  if (connect(sock.sock,(struct sockaddr *)&raddr,
	      sizeof(struct sockaddr_in)) == RC_SOCKET_ERROR) 
  {
    if(omniORB::traceLevel >= 15) {
      cerr << "Connection cannot be set up " << errno << endl;
    }
    if (errno != EINPROGRESS) {
      CLOSEINETSOCKET(sock);
      sock.sock = RC_INVALID_SOCKET;
      return sock;
    }
    fd_set wrfds;
    FD_ZERO(&wrfds);
    FD_SET(sock.sock,&wrfds);
    struct timeval t = { 30,0 };
    int rc;
    if ((rc = select(sock.sock+1,0,&wrfds,0,&t)) <= 0) {
      // Timeout, do not bother trying again.
      CLOSEINETSOCKET(sock);
      sock.sock = RC_INVALID_SOCKET;
      return sock;
    }
    // Set the socket back to blocking
    fl = 0;
    if (fcntl(sock.sock,F_SETFL,fl) < RC_SOCKET_ERROR) {
      CLOSEINETSOCKET(sock);
      sock.sock = RC_INVALID_SOCKET;
      return sock;
    }
  }

#else
  if (connect(sock.sock,(struct sockaddr *)&raddr,
	      sizeof(struct sockaddr_in)) == RC_SOCKET_ERROR) 
  {
    CLOSEINETSOCKET(sock);
    sock.sock = RC_INVALID_SOCKET;
    return sock;
  }
#endif

  sock.ssl = NULL;

  return sock;
}

/////////////////////////////////////////////////////////////////////////////

#if defined(__sunos__) && defined(__sparc__) && __OSVERSION__ >= 5

#include <signal.h>
extern "C" void abort()
{
  kill (getpid(),SIGABRT);
  while (1) {
    sleep(1000000);
  }
}

typedef void (*PFV)();
extern PFV set_terminate(PFV);

#endif

/////////////////////////////////////////////////////////////////////////////

void*
tcpSocketRendezvouser::run_undetached(void *arg)
{
  tcpSocketIncomingRope* r = (tcpSocketIncomingRope*) arg;

#if defined(__sunos__) && defined(__sparc__) && __OSVERSION__ >= 5
  set_terminate(abort);
#endif
  if (omniORB::traceLevel >= 5) {
    cerr << "tcpSocketMT Rendezvouser thread: starts." << endl;
  }

  tcpSocketStrand *newSt = 0;
  tcpSocketWorker *newthr = 0;
  CORBA::Boolean   die = 0;
  ConnectionInterceptor *connectionInterceptor;

  tcpContext::initContext();

  while (r->pd_shutdown == inetSocketIncomingRope::ACTIVE  && !die) {

    try {

      inetSocketHandle_t new_sock;
      struct sockaddr_in raddr;
 
#if (defined(__GLIBC__) && __GLIBC__ >= 2)
      // GNU C library uses socklen_t * instead of int* in accept ().
      // This is suppose to be compatible with the upcoming POSIX standard.
      socklen_t l;
#elif defined(__aix__) || defined(__VMS) || defined(__SINIX__)
    size_t l;
#else
    int l;
#endif

      l = sizeof(struct sockaddr_in);

      if (omniORB::traceLevel >= 15) {
	cerr << "tcpSocketMT Rendezvouser thread: block on accept()." << endl;
      }

      connectionInterceptor = tcpContext::getContext()
			->getConnectionInterceptorFactory()->create();
      connectionInterceptor->beforeAccept();
      delete connectionInterceptor;

      new_sock.ssl = NULL;
      if ((new_sock.sock = ::accept(r->pd_rendezvous.sock,
		(struct sockaddr *)&raddr,&l)) == RC_INVALID_SOCKET) {
        delete connectionInterceptor;
	throw CORBA::COMM_FAILURE(errno,CORBA::COMPLETED_NO);
      }

      if (omniORB::traceLevel >= 15) {
	cerr << "tcpSocketMT Rendezvouser thread: unblock from accept()." << endl;
      }

      {
	omni_mutex_lock sync(r->pd_lock);

	if (r->pd_shutdown != tcpSocketIncomingRope::ACTIVE) {
	  // It has been indicated that this thread should stop
	  // accepting connection request.
	  CLOSEINETSOCKET(new_sock);
	  continue;
	}

	newSt = new tcpSocketStrand(r,new_sock,1);
	newSt->incrRefCount(1);
      }

      if (omniORB::traceLevel >= 5) {
	cerr << "tcpSocketMT Rendezvouser thread: accept new strand." << endl;
      }

      IIOPConnectionState *iiop_cs = new IIOPConnectionState(newSt);

      connectionInterceptor = tcpContext::getContext()
			->getConnectionInterceptorFactory()->create();

      newSt->controlStrand(SetConnectionInterceptor,
	(char *) &connectionInterceptor, sizeof(ConnectionInterceptor *));

      userRequest_t req = connectionInterceptor->afterAccept(*iiop_cs);

      delete iiop_cs;

      if(req != Reject) {
        try {
	  newthr = new tcpSocketWorker(newSt);
        }
        catch(...) {
	  newthr = 0;
	}
      } else {
	newthr = 0;
      }
      if (!newthr) {
	// Cannot create a new thread to serve the strand
	// We have no choice but to shutdown the strand.
	// The long term solutions are:  start multiplexing the new strand
	// and the rendezvous; close down idle connections; reasign
	// threads to strands; etc.
	newSt->decrRefCount();
	newSt->shutdown();
      }
    }
    catch(const CORBA::COMM_FAILURE &) {
      // XXX accepts failed. The probable cause is that the number of
      //     file descriptors opened has exceeded the limit.
      //     On unix, the value of this limit can be set and get using the
      //              ulimit command.
      //     On NT, if this is part of a DLL, the limit is 256(?)
      //            else the limit is 16(?)
      // The following is a temporary fix, this thread just wait for a while
      // and tries again. Hopfully, some connections might be freed by then.
      if (omniORB::traceLevel >= 5) {
	cerr << "tcpSocketMT Rendezvouser thread: accept fails. Too many file descriptors opened?" << endl;
      }
      omni_thread::sleep(1,0);
      continue;
    }
    catch(const omniORB::fatalException &ex) {
      if (omniORB::traceLevel > 0) {
	cerr << "#### You have caught an omniORB2 bug, details are as follows:\n"
		     << ex.file() << " " << ex.line() << ":" << ex.errmsg()
		     << "\n"
	     << "tcpSocketMT Rendezvouser thread will not accept new connection." << endl;
      }
      die = 1;
    }
    catch(...) {
      if (omniORB::traceLevel > 0) {
	cerr << "######## Unexpected exception caught by tcpSocketMT Rendezvouser\n"
	     << "tcpSocketMT Rendezvouser thread will not accept new connection." << endl;
      }
      die = 1;
    }
    if (die && newSt) {
      newSt->decrRefCount();
      newSt->shutdown();
      if (!newthr) {
	if (omniORB::traceLevel >= 5) {
	  cerr << "tcpSocketMT Rendezvouser thread cannot spawn a new server thread." << endl;
	}
      }
    }
  }
  if (die) {
    // Something very wrong has happened, before we quit, we have to
    // make sure that a future call by another thread to
    // tcpSocketIncomingRope::cancelThreads() would not wait indefinitely
    // on this thread to response to its connect() request to the
    // rendezous socket. 
    // Shutdown (not close) the rendezvous socket to make sure that the
    // connect() in cancelThreads() would fail.
    SHUTDOWNINETSOCKET(r->pd_rendezvous);
  }

  while (r->pd_shutdown != inetSocketIncomingRope::NO_THREAD) {

    // We keep on accepting connection requests but close the new sockets
    // immediately until the state of pd_shutdown changes to NO_THREAD.
    // This is to make sure that the thread calling cancelThreads()
    // will be unblocked from the connect() call.

    {
      fd_set rdfds;
      FD_ZERO(&rdfds);
      FD_SET(r->pd_rendezvous.sock,&rdfds);
      struct timeval t = { 1,0};
      int rc;
      if ((rc = select(r->pd_rendezvous.sock+1,&rdfds,0,0,&t)) <= 0) {
	if (rc < 0 && errno != EINTR) {
	  die = 1;
	}
	if (omniORB::traceLevel >= 15) {
	  cerr << "tcpSocketMT Rendezvouser thread: waiting on shutdown state to change to NO_THREAD." << endl;
	}
	continue;
      }
    }

    inetSocketHandle_t new_sock;
    struct sockaddr_in raddr;
#if (defined(__GLIBC__) && __GLIBC__ >= 2)
    // GNU C library uses socklen_t * instead of int* in accept ().
    // This is suppose to be compatible with the upcoming POSIX standard.
    socklen_t l;
#elif defined(__aix__) || defined(__VMS) || defined(__SINIX__)
    size_t l;
#else
    int l;
#endif

    new_sock.ssl = NULL;
    l = sizeof(struct sockaddr_in);
    if ((new_sock.sock = ::accept(r->pd_rendezvous.sock,(struct sockaddr *)&raddr,&l)) 
	      == RC_INVALID_SOCKET) 
      {
	die = 1;
	continue;
      }
    CLOSEINETSOCKET(new_sock);
  }

  if (omniORB::traceLevel >= 5) {
    cerr << "tcpSocketMT Rendezvouser thread: exits." << endl;
  }
  return 0;
}

void
tcpSocketWorker::run(void *arg)
{
  omniORB::giopServerThreadWrapper::
        getGiopServerThreadWrapper()->run(tcpSocketWorker::_realRun,arg);
  // the wrapper run() method will pass back control to tcpSocketWorker
  // by calling  _realRun(arg) when it is ready.
}

void
tcpSocketWorker::_realRun(void *arg)
{
  tcpSocketStrand* s = (tcpSocketStrand*)arg;

#if defined(__sunos__) && defined(__sparc__) && __OSVERSION__ >= 5
  set_terminate(abort);
#endif
  
  if (omniORB::traceLevel >= 5) {
    cerr << "tcpSocketMT Worker thread: starts." << endl;
  }

  if (!gateKeeper::checkConnect(s)) {
    s->ll_shutdown();
  }
  else {
    while (1) {
      try {
	GIOP_S::dispatcher(s);
      }
      catch (const CORBA::COMM_FAILURE &) {
	if (omniORB::traceLevel >= 5) {
	  cerr << "#### Communication failure. Connection closed." << endl;
	}
	break;
      }
      catch(const omniORB::fatalException &ex) {
	if (omniORB::traceLevel > 0) {
	  cerr << "#### You have caught an omniORB2 bug, details are as follows:\n"
		       << ex.file() << " " << ex.line() << ":" << ex.errmsg() << endl;
	}
	break;
      }
      catch (...) {
	if (omniORB::traceLevel > 0) {
	  cerr << "#### A system exception has occured and was caught by tcpSocketMT Worker thread." <<endl;
	}
	break;
      }
    }
  }
  if (omniORB::traceLevel >= 5) {
    cerr << "tcpSocketMT Worker thread: exits." << endl;
  }
}
