// -*- Mode: C++; -*-
//                            Package   : omniORB2
// inetSocketFactory.cc       Created on: 18/3/96
//                            Author    : Sai Lai Lo (sll)
//                                      : Tatsuo Nakajima (tatsuo)
//
//    Copyright (C) 1996, 1997 Olivetti & Oracle Research Laboratory
//
//    This file is part of the omniORB library
//
//    The omniORB library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//	

/*
  $Log: tcpSocket.cc,v $
  Revision 1.4  1998/04/07 19:39:24  sll
  Replace cerr with omniORB::log.

  Revision 1.3  1998/03/13 17:34:16  sll
  Bug fix. No longer reject an IOR with zero length object key.

  Revision 1.2  1997/12/12 18:45:33  sll
  Added call to print out the version of gatekeeper.

  Revision 1.1  1997/12/09 18:43:12  sll
  Initial revision

  */

#include <omniORB2/CORBA.h>
#include <omniORB2/QOS.h>
#include <ropeFactory.h>
#include <objectManager.h>
#include <inetSocket.h>
#include <inetSocketFactory.h>
#include <tcpMTSocket.h>
#include <sslMTSocket.h>
#include <gatekeeper.h>

#define ATM_HOST_LEN      40

//////////////////////////////////////////////////////////////////////

#ifndef Swap16
#define Swap16(s) ((((s) & 0xff) << 8) | (((s) >> 8) & 0xff))
#else
#error "Swap16 has already been defined"
#endif

#ifndef Swap32
#define Swap32(l) ((((l) & 0xff000000) >> 24) | \
		   (((l) & 0x00ff0000) >> 8)  | \
		   (((l) & 0x0000ff00) << 8)  | \
		   (((l) & 0x000000ff) << 24))
#else
#error "Swap32 has already been defined"
#endif

inetSocketFactoryType* inetSocketFactoryType::singleton = 0;

const ComponentId TAG_SSL_SEC_TRANS	= 20;
const ComponentId TAG_ATM_TRANS		= 21;

void
inetSocketFactoryType::init()
{
  if (singleton) return;
  singleton = new inetSocketFactoryType;
  singleton->next = ropeFactoryTypeList;
  ropeFactoryTypeList = singleton;

  if (omniORB::traceLevel >= 2) {
    cerr << "omniORB2 gateKeeper is " << gateKeeper::version() 
	 << endl;
  }
}

inetSocketFactoryType::inetSocketFactoryType()
{
}

CORBA::Boolean
inetSocketFactoryType::is_IOPprofileId(IOP::ProfileId tag) const
{
  return (tag == IOP::TAG_INTERNET_IOP) ? 1 : 0;
}

CORBA::Boolean
inetSocketFactoryType::is_protocol(const char* name) const
{
  return (strcmp(name,inetSocketEndpoint::protocol_name) == 0) ? 1 : 0;
}

CORBA::Boolean
inetSocketFactoryType::decodeIOPprofile(const IOP::TaggedProfile& profile,
				       Endpoint*&     addr,
				       CORBA::Octet*& objkey,
				       size_t&        objkeysize) const
{
  if (profile.tag != IOP::TAG_INTERNET_IOP)
    return 0;

  CORBA::Char*  inethost;
  CORBA::Char   atmhost[ATM_HOST_LEN + 1];
  CORBA::UShort tcpPort = 0, sslPort = 0, atmPort = 0;
  CORBA::ULong throughput;
  CORBA::ULong begin = 0;
  CORBA::ULong end = 0;
  ConnectionType ts = TRANS_IIOP;
  inetSocketEndpoint *inet_addr;

  // profile.profile_data[0] - byteorder
  end += 1;
  if (profile.profile_data.length() <= end)
    throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);

  CORBA::Boolean byteswap = ((profile.profile_data[begin] == 
			      omni::myByteOrder) ? 0 : 1);

  // profile.profile_data[1] - iiop_version.major
  // profile.profile_data[2] - iiop_version.minor
  begin = end;
  end = begin + 2;
  if (profile.profile_data.length() <= end)
    throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
  if (((profile.profile_data[begin]  != (CORBA::Octet)IIOP::current_major) ||
      (profile.profile_data[begin+1] > 1 /*(CORBA::Octet)IIOP::current_minor*/))) {
    throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
  }

  // profile.profile_data[3] - padding
  // profile.profile_data[4] - profile.profile_data[7] host string length
  begin = end + 1;
  end = begin + 4;
  if (profile.profile_data.length() <= end)
    throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
  {
    CORBA::ULong len;
    if (!byteswap) {
      len = ((CORBA::ULong &) profile.profile_data[begin]);
    }
    else {
      CORBA::ULong t = ((CORBA::ULong &) profile.profile_data[begin]);
      len = Swap32(t);
    }

    // profile.profile_data[8] - profile.profile_data[8+len-1] host string
    begin = end;
    end = begin + len;
    if (profile.profile_data.length() <= end)
      throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);

    // Is this string null terminated?
    if (((char)profile.profile_data[end-1]) != '\0')
      throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);

    inethost = (CORBA::Char*)&profile.profile_data[begin];

  }
    
  // align to CORBA::UShort
  begin = (end + 1) & ~(1);
  // profile.profile_data[begin] port number
  end = begin + 2;
  if (profile.profile_data.length() <= end)
    throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
  if (!byteswap) {
    tcpPort = ((CORBA::UShort &) profile.profile_data[begin]);
  }
  else {
    CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
    tcpPort = Swap16(t);
  }

  // align to CORBA::ULong
  begin = (end + 3) & ~(3);
  // profile.profile_data[begin]  object key length
  end = begin + 4;
  if (profile.profile_data.length() < end)
    throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);

  if (profile.profile_data.length() == end) {
    objkeysize = 0;
    objkey = new CORBA::Octet[1];
  }
  else {
    CORBA::ULong len;
    if (!byteswap) {
      len = ((CORBA::ULong &) profile.profile_data[begin]);
    }
    else {
      CORBA::ULong t = ((CORBA::ULong &) profile.profile_data[begin]);
      len = Swap32(t);
    }

    begin = end;
    end = begin + len;
    if (profile.profile_data.length() < end)
      throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);

    // extract object key
    objkeysize = len;
    objkey = new CORBA::Octet[objkeysize];
    memcpy((void *)objkey,(void *)&(profile.profile_data[begin]),objkeysize);
  }

  if (profile.profile_data.length() != end) { // There are IOP components.
    CORBA::ULong num, len;

    // align to CORBA::ULong
    begin = (end + 3) & ~(3);
    end = begin + 4;

    if(profile.profile_data.length() <= end) {
      throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
    }

    if (!byteswap) {
      num = ((CORBA::ULong &) profile.profile_data[begin]);
    } else {
      CORBA::ULong t = ((CORBA::ULong &) profile.profile_data[begin]);
      num = Swap32(t);
    }

    while(num-- != 0) {
      CORBA::ULong len;
      ComponentId  tag;

      begin = (end + 3) & ~(3);
      end = begin + 4;

      if(profile.profile_data.length() <= end) {
        throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
      }

      if (!byteswap) {
        tag = ((ComponentId &) profile.profile_data[begin]);
      } else {
        CORBA::ULong t = ((ComponentId &) profile.profile_data[begin]);
        tag = Swap32(t);
      }

      begin = (end + 3) & ~(3);
      end = begin + 4;

      if(profile.profile_data.length() <= end) {
        throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
      }

      if (!byteswap) {
        len = ((CORBA::ULong &) profile.profile_data[begin]);
      } else {
        CORBA::ULong t = ((CORBA::ULong &) profile.profile_data[begin]);
        len = Swap32(t);
      }

      if(tag == TAG_SSL_SEC_TRANS) {

	CORBA::UShort target;
	CORBA::UShort require;
	CORBA::UShort padding;

        if(len != 2+2+2+2) { // target + require + align + ssl port
          throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
        } 

 	begin = end;		// for Target
	end = begin +2;

        if (!byteswap) {
          target = ((CORBA::UShort &) profile.profile_data[begin]);
        } else {
          CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
          target = Swap16(t);
        }

 	begin = end;		// for require
	end = begin +2;

        if (!byteswap) {
          require = ((CORBA::UShort &) profile.profile_data[begin]);
        } else {
          CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
          require = Swap16(t);
        }

	begin = end;		// for Padding
        end = begin + 2; 

        if (!byteswap) {
          padding = ((CORBA::UShort &) profile.profile_data[begin]);
        } else {
          CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
          padding = Swap16(t);
        }

        begin = end;		// for Padding
        end = begin + 2; 

        if (!byteswap) {
          sslPort = ((CORBA::UShort &) profile.profile_data[begin]);
        } else {
          CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
          sslPort = Swap16(t);
        }

	if(ts == TRANS_ATM) {
          ts = TRANS_SSL_ATM;
	} else {
	  ts = TRANS_SSL;
	}

      } else if(tag == TAG_ATM_TRANS) {

	CORBA::UShort padding;

        if(len != ATM_HOST_LEN+2+2+4) { // host name + padding
				      // + atm port + throughput
          throw CORBA::MARSHAL(0,CORBA::COMPLETED_NO);
        } 

 	begin = end;		// for ATM Host
	end = begin + ATM_HOST_LEN;

	memcpy(atmhost, &profile.profile_data[begin], ATM_HOST_LEN);
        atmhost[ATM_HOST_LEN] = '\0';

	begin = end;		// for Padding
        end = begin + 2; 

        if (!byteswap) {
          padding = ((CORBA::UShort &) profile.profile_data[begin]);
        } else {
          CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
          padding = Swap16(t);
        }

        begin = end;		// for ATM port
        end = begin + 2; 

        if (!byteswap) {
          atmPort = ((CORBA::UShort &) profile.profile_data[begin]);
        } else {
          CORBA::UShort t = ((CORBA::UShort &) profile.profile_data[begin]);
          atmPort = Swap16(t);
        }

        begin = end;		// for ATM throughput
        end = begin + 4; 

        if (!byteswap) {
          throughput = ((CORBA::ULong &) profile.profile_data[begin]);
        } else {
          CORBA::ULong l = ((CORBA::ULong &) profile.profile_data[begin]);
          throughput = Swap32(l);
        }

	if(ts == TRANS_SSL) {
          ts = TRANS_SSL_ATM;
	} else {
	  ts = TRANS_ATM;
	}

      } else {
        begin = end;
      }
    }
  }
 
  inet_addr = new inetSocketEndpoint();
  inet_addr->inetHost(inethost);
  inet_addr->tcpPort(tcpPort);

  switch(ts) {
  case TRANS_SSL:
    inet_addr->sslPort(sslPort);
    break;

  case TRANS_ATM:
    inet_addr->atmHost(atmhost);
    inet_addr->atmPort(atmPort);
    inet_addr->throughput(throughput);
    break;

  case TRANS_SSL_ATM:
    inet_addr->sslPort(sslPort);
    inet_addr->atmHost(atmhost);
    inet_addr->atmPort(atmPort);
    break;
    
  default:
    break;
  }

  inet_addr->status(ts);
  addr = (Endpoint *)inet_addr;

  return 1;
}

void
inetSocketFactoryType::encodeIOPprofile(const Endpoint* addr,
				       const CORBA::Octet* objkey,
				       const size_t objkeysize,
				       IOP::TaggedProfile& profile) const
{
  inetSocketEndpoint* inetaddr = inetSocketEndpoint::castup(addr);
  if (!inetaddr)
    throw omniORB::fatalException(__FILE__,__LINE__,
				  "Endpoint is not tcpSocket");
  profile.tag = IOP::TAG_INTERNET_IOP;

  CORBA::ULong hlen = strlen((const char *)inetaddr->inetHost()) + 1;
  {
    // calculate the total size of the encapsulated stream
    CORBA::ULong total = 8 + hlen;        // first 4 bytes + aligned host
    total = ((total + 1) & ~(1)) + 2;     // aligned port value
    total = ((total + 3) & ~(3)) + 4 +	// aligned object key
      objkeysize;

    switch(inetaddr->status()) {

    case TRANS_SSL:
      total = ((total + 3) & ~(3)) + 4 + 4 + 4 + 2 + 2 + 2 + 2;
		// component num + SSL component tag + component length
 		// + target support + target require + align + SSL port
      break;

    case TRANS_ATM:
      total = ((total + 3) & ~(3)) + 4 + 4 + 4 + ATM_HOST_LEN + 2 + 2 + 4;
		// component num + SSL component tag + component length
 		// + atm addr + paddind + atm port
      break;

    case TRANS_SSL_ATM:
      total = ((total + 3) & ~(3)) + 4 + 4 + 4 + 2 + 2 + 2 + 2;
		// component num + SSL component tag + component length
 		// + target support + target require + align + SSL port

      total = ((total + 3) & ~(3)) + 4 + 4 + ATM_HOST_LEN + 2 + 2 + 4;
		// ATM component tag + component length
 		// + atm addr + paddind + atm port
      break;

    case TRANS_IIOP:
      break;

    default:
      if(omniORB::traceLevel > 15) {    
        cerr << "unknown Endpoint Status in Encode IOP Profile" << endl;
      }
      break;
    }
    
    profile.profile_data.length(total);
  }

  profile.profile_data[0] = omni::myByteOrder;
  profile.profile_data[1] = IIOP::current_major;
  profile.profile_data[2] = 1; // IIOP::current_minor; (need to fix later!!)
  profile.profile_data[3] = 0;

  {
    CORBA::ULong &l = (CORBA::ULong &) profile.profile_data[4];
    l = hlen;
  }
  memcpy((void *)&(profile.profile_data[8]),(void *)inetaddr->inetHost(),hlen);
  CORBA::ULong idx = ((8 + hlen) + 1) & ~(1);
  {
    CORBA::UShort &l = (CORBA::UShort &) profile.profile_data[idx];
    l = inetaddr->tcpPort();
  }
  idx = ((idx + 2) + 3) & ~(3);
  {
    CORBA::ULong &l = (CORBA::ULong &) profile.profile_data[idx];
    l = objkeysize;
  }
  if (objkeysize) {
    idx += 4;
    memcpy((void *)&profile.profile_data[idx],
	   (void *)objkey,objkeysize);
  }
  if(omniORB::traceLevel > 15) {    
    cerr << "Object Keysize = " << objkeysize << endl;
  }

  switch(inetaddr->status()) {
    
  case TRANS_SSL:
    {
      idx += objkeysize;
      idx = (idx + 3) & ~(3);
      CORBA::ULong &l1 = (CORBA::ULong &) profile.profile_data[idx];
      l1 = 1;	// component number
      idx += 4;
      ComponentId &l2 = (ComponentId &) profile.profile_data[idx];
      l2 = TAG_SSL_SEC_TRANS; 		// tag for SSL
      idx += 4;
      CORBA::ULong &l3 = (CORBA::ULong &) profile.profile_data[idx];
      l3 = 8;
      idx += 4;
      CORBA::UShort &s1 = (CORBA::UShort &) profile.profile_data[idx];
      s1 = 0;				// target supports
      idx += 2;
      CORBA::UShort &s2 = (CORBA::UShort &) profile.profile_data[idx];
      s2 = 32;				// target requires
      idx += 2;
      CORBA::UShort &s3 = (CORBA::UShort &) profile.profile_data[idx];
      s3 = 0;
      idx += 2;
      CORBA::UShort &s4 = (CORBA::UShort &) profile.profile_data[idx];
      s4 = inetaddr->sslPort();
      idx += 2;
      break;
    }

  case TRANS_ATM:
    {
      idx += objkeysize;
      idx = (idx + 3) & ~(3);
      CORBA::ULong &l1 = (CORBA::ULong &) profile.profile_data[idx];
      l1 = 1;	// component number
      idx += 4;
      ComponentId &l2 = (ComponentId &) profile.profile_data[idx];
      l2 = TAG_ATM_TRANS; 		// tag for ATM
      idx += 4;
      CORBA::ULong &l3 = (CORBA::ULong &) profile.profile_data[idx];
      l3 = 48;
      idx += 4;
      memcpy((void *)&profile.profile_data[idx],
			(void *)inetaddr->atmHost(), ATM_HOST_LEN);
      idx += ATM_HOST_LEN;
      CORBA::UShort &s1 = (CORBA::UShort &) profile.profile_data[idx];
      s1 = 0;
      idx += 2;
      CORBA::UShort &s2 = (CORBA::UShort &) profile.profile_data[idx];
      s2 = inetaddr->atmPort();
      idx += 2;
      CORBA::ULong &l4 = (CORBA::ULong &) profile.profile_data[idx];
      l4 = inetaddr->throughput();
      idx += 4;
      break;
    }

  case TRANS_SSL_ATM:
    {
      idx += objkeysize;
      idx = (idx + 3) & ~(3);
      CORBA::ULong &l1 = (CORBA::ULong &) profile.profile_data[idx];
      l1 = 2;	// component number
      idx += 4;

      // Encode SSL Tag
      ComponentId &l2 = (ComponentId &) profile.profile_data[idx];
      l2 = TAG_SSL_SEC_TRANS; 		// tag for SSL
      idx += 4;
      CORBA::ULong &l3 = (CORBA::ULong &) profile.profile_data[idx];
      l3 = 8;
      idx += 4;
      CORBA::UShort &s1 = (CORBA::UShort &) profile.profile_data[idx];
      s1 = 0;				// target supports
      idx += 2;
      CORBA::UShort &s2 = (CORBA::UShort &) profile.profile_data[idx];
      s2 = 32;				// target requires
      idx += 2;
      CORBA::UShort &s3 = (CORBA::UShort &) profile.profile_data[idx];
      s3 = 0;
      idx += 2;
      CORBA::UShort &s4 = (CORBA::UShort &) profile.profile_data[idx];
      s4 = inetaddr->sslPort();
      idx += 2;

      // Encode ATM Tag
      idx = (idx + 3) & ~(3);
      ComponentId &l4 = (ComponentId &) profile.profile_data[idx];
      l4 = TAG_ATM_TRANS; 		// tag for ATM
      idx += 4;
      CORBA::ULong &l5 = (CORBA::ULong &) profile.profile_data[idx];
      l5 = 48;
      idx += 4;
      memcpy((void *)&profile.profile_data[idx],
			(void *)inetaddr->atmHost(), ATM_HOST_LEN);
      idx += ATM_HOST_LEN;
      CORBA::UShort &s5 = (CORBA::UShort &) profile.profile_data[idx];
      s5 = 0;
      idx += 2;
      CORBA::UShort &s6 = (CORBA::UShort &) profile.profile_data[idx];
      s6 = inetaddr->atmPort();
      idx += 2;
      CORBA::ULong &l6 = (CORBA::ULong &) profile.profile_data[idx];
      l6 = inetaddr->throughput();
      idx += 4;

      break;
    }

  case TRANS_IIOP:
    profile.profile_data[2] = 0; // IIOP minor version
    break;

  default:
    cerr << "Unknown Status" << endl;
    break;
  }
}

#undef Swap16
#undef Swap32

///////////////////////////////////////////////////////////////////////

inetSocketMTincomingFactory::inetSocketMTincomingFactory() : pd_state(IDLE)
{
  inetSocketFactoryType::init();
}
 
CORBA::Boolean
inetSocketMTincomingFactory::isIncoming(Endpoint* addr) const
{
  Rope_iterator next_rope(&pd_anchor);
  Rope* r;
  while ((r = next_rope()))
    {
      if (r->this_is(addr))
	return 1;
    }
  return 0;
}

void
inetSocketMTincomingFactory::instantiateIncoming(Endpoint* addr,
				CORBA::Boolean export)
{
  inetSocketEndpoint* te = inetSocketEndpoint::castup(addr);
  inetSocketIncomingRope* r;

  if (!te)
    throw omniORB::fatalException(__FILE__,__LINE__,
				  "wrong endpoint argument type");

  Rope_iterator rope_lock(&pd_anchor);

  if (pd_state == ZOMBIE) {
    throw omniORB::fatalException(__FILE__,__LINE__,
			    "cannot instantiate incoming in ZOMBIE state");
  }

  switch(te->status()) {

  case TRANS_SSL:
    r = new sslSocketIncomingRope(this,0,te,export);
    break;

  case TRANS_IIOP:
    r = new tcpSocketIncomingRope(this,0,te,export);
    break;

  default:
    if(omniORB::traceLevel > 15) {    
      cerr << "unknown Endpoint Status in instantiate incoming" << endl;
    }
    break;
  }
  r->incrRefCount(1);

  if (pd_state == ACTIVE) {
    switch(te->status()) {

    case TRANS_SSL:
      ((sslSocketIncomingRope *)r)->rendezvouser 
		= new sslSocketRendezvouser((sslSocketIncomingRope *)r);
      break;
 
    case TRANS_IIOP:
      ((tcpSocketIncomingRope *)r)->rendezvouser
		= new tcpSocketRendezvouser((tcpSocketIncomingRope *)r);
      break;

    default:
      break;
    }
  }
}

void
inetSocketMTincomingFactory::startIncoming()
{
  Rope_iterator next_rope(&pd_anchor);
  inetSocketIncomingRope* r;

  switch (pd_state) {
  case IDLE:
    try {
      pd_state = ACTIVE;
      while ((r = (inetSocketIncomingRope*)next_rope())) {
	if (r->pd_shutdown == inetSocketIncomingRope::NO_THREAD) {
	  r->pd_shutdown = inetSocketIncomingRope::ACTIVE;
          switch(r->getConnectionType()) {

          case TRANS_SSL:
	    ((sslSocketIncomingRope *)r)->rendezvouser
		= (sslSocketRendezvouser *)
		    new sslSocketRendezvouser((sslSocketIncomingRope *)r);
            break;

	  case TRANS_IIOP:
	    ((tcpSocketIncomingRope *)r)->rendezvouser
		= (tcpSocketRendezvouser *)
		    new tcpSocketRendezvouser((tcpSocketIncomingRope *)r);
 	    break;

	  default:
            if(omniORB::traceLevel > 15) {    
              cerr << "unknown Endpoint Status in start incoming" << endl;
            }
	    break;
	  }
	}
      }
    }
    catch(...) { }
    break;
  default:
    break;
  }
}

void
inetSocketMTincomingFactory::stopIncoming()
{
  Rope_iterator next_rope(&pd_anchor);
  inetSocketIncomingRope* r;

  switch (pd_state) {
  case ACTIVE:
    try {
      while ((r = (inetSocketIncomingRope*)next_rope())) {
	r->cancelThreads();
      }
      pd_state = IDLE;
    }
    catch(...) {}
    break;
  default:
    break;
  }
}

void
inetSocketMTincomingFactory::removeIncoming()
{
  Rope_iterator next_rope(&pd_anchor);
  inetSocketIncomingRope* r;

  switch (pd_state) {
  case ACTIVE:
  case IDLE:
    try {
      while ((r = (inetSocketIncomingRope*)next_rope())) {
	r->cancelThreads();
	if (r->pd_shutdown != inetSocketIncomingRope::NO_THREAD) {
	  // rendezvouser has not been shutdown properly
	  continue;
	}
	if (r->pd_rendezvous.sock != RC_INVALID_SOCKET) {
	  CLOSEINETSOCKET(r->pd_rendezvous);
	  r->pd_rendezvous.sock = RC_INVALID_SOCKET;
	  r->decrRefCount(1);
	}
      }
      pd_state = ZOMBIE;
    }
    catch(...) {}
    break;
  default:
    break;
  }
}

Rope*
inetSocketMTincomingFactory::findIncoming(Endpoint* addr) const
{
  inetSocketEndpoint* te = inetSocketEndpoint::castup(addr);
  if (!te) return 0;

  Rope_iterator next_rope(&pd_anchor);
  Rope* r;
  while ((r = next_rope()))
    {
      if ((r->getConnectionType() == TRANS_IIOP) && r->this_is(addr)) {
	r->incrRefCount(1);
	return r;
      }
    }
  return 0;
}

void 
inetSocketMTincomingFactory::getIncomingIOPprofiles(const CORBA::Octet* objkey,
						   const size_t objkeysize,
			      IOP::TaggedProfileList& profilelist) const
{
  Rope_iterator next_rope(&pd_anchor);
  inetSocketIncomingRope* r;
  inetSocketEndpoint* e = NULL;

  //
  // If SSL is used, two ropes are created for accepting both normal TCP
  // and SSL. Thus, for creating IOP, a normal tcp port and a SSL port
  // should be extracted from respective ropes.
  // In this function, first, a tcpSocketEndpoint is created, and fill
  // both normal tcp port and SSL port from two ropes, and makes IOP.
  //
  while ((r = (inetSocketIncomingRope*) next_rope()) && r->pd_export) {
    inetSocketEndpoint *re = r->me;

    switch(r->getConnectionType()) {
    case TRANS_IIOP:
      if(e == NULL) {
        e = new inetSocketEndpoint(re);
      } else {
        e->tcpPort(re->tcpPort());
        e->inetHost(re->inetHost());
      }
      break;

    case TRANS_SSL:
      if(e == NULL) {
        e = new inetSocketEndpoint(re);
      } else {
        e->sslPort(re->sslPort());
	if(e->status() == TRANS_ATM) {
  	  e->status(TRANS_SSL_ATM);
	} else {
          e->status(TRANS_SSL);
	}
      }
      break;

    case TRANS_ATM:
      if(e == NULL) {
        e = new inetSocketEndpoint(re);
      } else {
        e->atmPort(re->atmPort());
        e->atmHost(re->atmHost());
	if(e->status() == TRANS_SSL) {
  	  e->status(TRANS_SSL_ATM);
	} else {
          e->status(TRANS_ATM);
	}
      }
      break;

    default:
      break;
    }
  }
  
  CORBA::ULong index = profilelist.length();
  profilelist.length(index+1);
  inetSocketFactoryType::singleton->encodeIOPprofile(e, objkey, objkeysize,
						      profilelist[index]);
  delete e;

  if (omniORB::traceLevel >= 15) {
    cerr << "getIncomingIOPprofiles return" << endl;
  }
}

//
// XXX inetSocketMTincomingFactory::getIncomingEndpoint() and
// XXX inetSocketMTincomingFactory::getIOPprofiles are used to
// XXX generate of IOR for dynamic transport selection. 
// XXX Introducing these methods is ugly, and should be find
// XXX a more general solution in future.
//
Endpoint *
inetSocketMTincomingFactory::getIncomingEndpoint()
{
  Rope_iterator next_rope(&pd_anchor);
  inetSocketIncomingRope* r;
  inetSocketEndpoint* e = NULL;

  //
  // If SSL is used, two ropes are created for accepting both normal TCP
  // and SSL. Thus, for creating IOP, a normal tcp port and a SSL port
  // should be extracted from respective ropes.
  // In this function, first, a tcpSocketEndpoint is created, and fill
  // both normal tcp port and SSL port from two ropes, and makes IOP.
  //
  while ((r = (inetSocketIncomingRope*) next_rope()) && r->pd_export) {
    inetSocketEndpoint *re = r->me;

    switch(r->getConnectionType()) {
    case TRANS_IIOP:
      if(e == NULL) {
        e = new inetSocketEndpoint(re);
      } else {
        e->tcpPort(re->tcpPort());
        e->inetHost(re->inetHost());
      }
      break;

    case TRANS_SSL:
      if(e == NULL) {
        e = new inetSocketEndpoint(re);
      } else {
        e->sslPort(re->sslPort());
	if(e->status() == TRANS_ATM) {
  	  e->status(TRANS_SSL_ATM);
	} else {
          e->status(TRANS_SSL);
	}
      }
      break;

    case TRANS_ATM:
      if(e == NULL) {
        e = new inetSocketEndpoint(re);
      } else {
        e->atmPort(re->atmPort());
        e->atmHost(re->atmHost());
	if(e->status() == TRANS_SSL) {
  	  e->status(TRANS_SSL_ATM);
	} else {
          e->status(TRANS_ATM);
	}
      }
      break;

    default:
      break;
    }
  }

  return (Endpoint *)e;
}

void 
inetSocketMTincomingFactory::getIOPprofiles(const CORBA::Octet* objkey,
						   const size_t objkeysize,
			      IOP::TaggedProfileList& profilelist,
			      Endpoint *e) 
{
  CORBA::ULong index = profilelist.length();
  profilelist.length(index+1);
  inetSocketFactoryType::singleton->encodeIOPprofile(e, objkey, objkeysize,
						      profilelist[index]);
}

inetSocketMToutgoingFactory::inetSocketMToutgoingFactory()
{
  inetSocketFactoryType::init();
}

CORBA::Boolean
inetSocketMToutgoingFactory::isOutgoing(Endpoint* addr) const
{
  Rope_iterator next_rope(&pd_anchor);
  Rope* r;
  while ((r = next_rope()))
    {
      if (r->remote_is(addr))
	return 1;
    }
  return 0;
}

Rope*
inetSocketMToutgoingFactory::findOrCreateOutgoing(Endpoint* addr)
{
  inetSocketEndpoint* te = inetSocketEndpoint::castup(addr);
  if (!te) {
    return 0;
  }
  ConnectionType status = te->status();

  ConnectionSet *rh = new ConnectionSet();
  tcpSocketOutgoingRope *tcp_r;
  sslSocketOutgoingRope *ssl_r;

  if(tcpContext::checkIIOPOnly()) {
    tcp_r = new tcpSocketOutgoingRope(this,5,te);
    rh->insertRope(tcp_r, TRANS_IIOP);
    tcp_r->setConnectionSet(rh);
  } else {

    if(((status == TRANS_SSL) || (status == TRANS_SSL_ATM))
			&& sslContext::checkUseSSL()) {
      ssl_r = new sslSocketOutgoingRope(this,5,te);
      rh->insertRope(ssl_r, TRANS_SSL);
      ssl_r->setConnectionSet(rh);

      ssl_r->incrRefCount(1);
      return ssl_r;

    } else {
      tcp_r = new tcpSocketOutgoingRope(this,5,te);
      rh->insertRope(tcp_r, TRANS_IIOP);
      tcp_r->setConnectionSet(rh);

      switch (status) {
      case TRANS_SSL_ATM:
      case TRANS_SSL:
        ssl_r = new sslSocketOutgoingRope(this,5,te);
        rh->insertRope(ssl_r, TRANS_SSL);
        ssl_r->setConnectionSet(rh);

        ssl_r->incrRefCount(1);

	if(te->status() == TRANS_SSL) {
	  break;
	}

      default:
	break;
      }
    }
  }
  tcp_r->incrRefCount(1);
  return tcp_r;
}

