// -*- Mode: C++; -*-
//                            Package   : omniORB2
// inetSocket.h               Created on: 24/11/97
//                            Author    : Ben Harris (bjh)
//
//    Copyright (C) 1996, 1997 Olivetti & Oracle Research Laboratory
//
//    This file is part of the omniORB library
//
//    The omniORB library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:

#ifndef _INETSOCKET_H_
#define	_INETSOCKET_H_

#include <relStream.h>
#include <inetSocketFactory.h>

class inetSocketEndpoint;
class inetSocketIncomingRope;
class inetSocketOutgoingRope;

class inetSocketStrand : public reliableStreamStrand
{
public:

  inetSocketStrand(inetSocketOutgoingRope *r,
		  inetSocketEndpoint *remote,
		  size_t buffer_size,	  
		  _CORBA_Boolean heapAllocated = 0);
  // Concurrency Control:
  //    MUTEX = r->pd_lock
  // Pre-condition:
  //	  Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit, even if an exception is raised

  inetSocketStrand(inetSocketIncomingRope *r,
		  inetSocketHandle_t sock,
		  size_t buffer_size,
		  _CORBA_Boolean heapAllocated = 0);
  // Concurrency Control:
  //    MUTEX = r->pd_lock
  // Pre-condition:
  //	  Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit, even if an exception is raised

  virtual ~inetSocketStrand() {}
  // MUTEX:
  //    pd_rope->pd_lock
  // Pre-condition:
  //    Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit

  virtual void shutdown() = 0;
  virtual void ll_send(void* buf,size_t sz) = 0;
  virtual size_t ll_recv(void* buf,size_t sz) = 0;
  virtual CORBA::Boolean ll_connect() = 0;
  virtual CORBA::Boolean ll_shutdown() = 0;

  inline inetSocketHandle_t handle() const {
    return pd_socket;
  }

protected:

  inetSocketHandle_t  pd_socket;
  CORBA::Boolean      pd_send_giop_closeConnection;
  inetSocketEndpoint* pd_delay_connect;
  inetSocketEndpoint* pd_connect;
  ConnectionInterceptor*      pd_connection_interceptor;
  int pd_request;
  int pd_info;
};

class inetSocketIncomingRope : public Rope {
public:

  virtual ~inetSocketIncomingRope() {}
  // Concurrency Control:
  //    MUTEX = pd_anchor->pd_lock
  // Pre-condition:
  //    Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit

  CORBA::Boolean is_incoming() const { return 1; }
  CORBA::Boolean is_outgoing() const { return 0; }
  CORBA::Boolean remote_is(Endpoint *&e) { return 0; }
  virtual CORBA::Boolean this_is(Endpoint *&e);

  ConnectionType getConnectionType();

protected:

  inetSocketIncomingRope(inetSocketMTincomingFactory* f,
			unsigned int maxStrands,
			inetSocketEndpoint *e,
			CORBA::Boolean export);
  // Create a tcpSocket incoming Rope.
  //      Creates a passive socket. e->port() specifies the port number to
  //      bind to or 0 if an arbitary port number can be assigned). 
  //      e->host() can either be the host's fully qualified domain name 
  //      (FQDN) or a 0 length string. If it is the latter, the constructor
  //      will initialise the host field with the host's IP address.
  //      This is a dot separated numeric string of the form "xxx.xxx.xxx.xxx".
  // Concurrency Control:
  //    MUTEX = f->anchor().pd_lock
  // Pre-condition:
  //    Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit, even if an exception is raised

  virtual void cancelThreads() = 0;
  // When this call returns, the rendezvouser thread would have been
  // cancelled and all the sockets instantiated by the accept() call on
  // the pd_rendezvous socket would have been shutdown.
  // The worker threads that are serving the sockets would notice the shutdown
  // and exit eventually. However, when will the worker threads see the
  // the shutdown is completely asynchronous to this call.
  // Concurrency Control:
  //    MUTEX = f->anchor().pd_lock
  // Pre-condition:
  //    Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit, even if an exception is raised

  virtual Strand *newStrand();

  inetSocketEndpoint *me;
  inetSocketHandle_t  pd_rendezvous;
  CORBA::Boolean     pd_export;
  enum { ACTIVE, SHUTDOWN, NO_THREAD } pd_shutdown;
};

class inetSocketOutgoingRope : public Rope {
public:

  ~inetSocketOutgoingRope();

  // Concurrency Control:
  //    MUTEX = pd_anchor->pd_lock
  // Pre-condition:
  //    Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit

  CORBA::Boolean is_incoming() const { return 0; }
  CORBA::Boolean is_outgoing() const { return 1; }
  virtual CORBA::Boolean remote_is(Endpoint *&e);
  CORBA::Boolean this_is(Endpoint *&e) { return 0; }
  
  virtual Strand *newStrand() = 0;

  ConnectionType getConnectionType();

  CORBA::ULong getThroughput();
  void setThroughput(CORBA::ULong throughput);

  void setTerminate() { pd_terminate = 1; }

protected:

  inetSocketOutgoingRope(inetSocketMToutgoingFactory* f,
			unsigned int maxStrands,
			inetSocketEndpoint *e);
  // Create a tcpSocket outgoing Rope.
  //      The endpoint <e> is the remote endpoint and should contain the
  //      host name in FQDN form or as a dot separeted numeric string.
  // Concurrency Control:
  //    MUTEX = a->pd_lock
  // Pre-condition:
  //    Must hold <MUTEX> on entry
  // Post-condition:
  //    Still hold <MUTEX> on exit, even if an exception is raised

  inetSocketEndpoint *remote;
  int pd_terminate;
};

class inetSocketEndpoint : public Endpoint {
public:
  inetSocketEndpoint();
  inetSocketEndpoint(const inetSocketEndpoint *e);
  inetSocketEndpoint &operator=(const inetSocketEndpoint &e);
  CORBA::Boolean operator==(const inetSocketEndpoint *e);
  virtual ~inetSocketEndpoint();
  CORBA::Char * inetHost() const;
  void inetHost(const CORBA::Char *p);
  CORBA::Char * atmHost() const;
  void atmHost(const CORBA::Char *p);
  CORBA::UShort tcpPort() const;
  void tcpPort(const CORBA::UShort p);
  CORBA::UShort sslPort() const;
  void sslPort(const CORBA::UShort p);
  CORBA::UShort atmPort() const;
  void atmPort(const CORBA::UShort p);
  ConnectionType status() const;
  void status(const ConnectionType ts);
  CORBA::ULong throughput() const;
  void throughput(const CORBA::ULong throughput);
  static inetSocketEndpoint *castup(const Endpoint* e);

  static const char* protocol_name;

private:
  CORBA::Char  *pd_inet_host;
  CORBA::Char  *pd_atm_host;
  CORBA::UShort pd_tcp_port;
  CORBA::UShort pd_ssl_port;
  CORBA::UShort pd_atm_port;
  CORBA::ULong  pd_throughput;

  ConnectionType pd_status;
};

#endif _INETSOCKET_H_
