// eg3_impl.cc - This is the source code of example 3 used in Chapter 2
//               "The Basics" of the omniORB2 user guide.
//
//               This is the object implementation.
//
// Usage: eg3_impl
//
//        On startup, the object reference is registered with the 
//        COS naming service. The client uses the naming service to
//        locate this object.
//
//        The name which the object is bound to is as follows:
//              root  [context]
//               |
//              text  [context] kind [my_context]
//               |
//              Echo  [object]  kind [Object]
//

#include <iostream.h>
#include "omnithread.h"
#include "echo.hh"

#include <omniORB2/QOS.h>

#include "echo_i.cc"

const int silent = 0;

class MyConnectionInterceptorForIIOP : ConnectionInterceptor {
public:
  virtual void beforeConnect(ConnectionState& cs);
  virtual userRequest_t afterConnect(ConnectionState& cs);
  virtual void beforeAccept();
  virtual userRequest_t afterAccept(ConnectionState& cs);
  virtual userRequest_t beforeDisconnect(ConnectionState& cs);
  virtual void afterDisconnect(ConnectionState& cs);
};

class MyConnectionInterceptorForSSL : ConnectionInterceptor {
public:
  virtual void beforeConnect(ConnectionState& cs);
  virtual userRequest_t afterConnect(ConnectionState& cs);
  virtual void beforeAccept();
  virtual userRequest_t afterAccept(ConnectionState& cs);
  virtual userRequest_t beforeDisconnect(ConnectionState& cs);
  virtual void afterDisconnect(ConnectionState& cs);
};

class MyConnectionInterceptorFactoryForIIOP : ConnectionInterceptorFactory {
public:
  virtual ConnectionInterceptor *create();
};

class MyConnectionInterceptorFactoryForSSL : ConnectionInterceptorFactory {
public:
  virtual ConnectionInterceptor *create();
};

ConnectionInterceptor *
MyConnectionInterceptorFactoryForIIOP::create()
{
  cerr << "create MyConnectionInterceptorFactoryForIIOP" << endl;
  return((ConnectionInterceptor *)new MyConnectionInterceptorForIIOP());
}

ConnectionInterceptor *
MyConnectionInterceptorFactoryForSSL::create()
{
  cerr << "create MyConnectionInterceptorFactoryForSSL" << endl;
  return((ConnectionInterceptor *)new MyConnectionInterceptorForSSL());
}

void
MyConnectionInterceptorForIIOP::beforeConnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::beforeConnect is called" << endl;
  }
}

userRequest_t
MyConnectionInterceptorForIIOP::afterConnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::afterConnect is called" << endl;
  }

  return Accept;
}

void
MyConnectionInterceptorForIIOP::beforeAccept()
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::beforeAccept is called" << endl;
  }
}

userRequest_t
MyConnectionInterceptorForIIOP::afterAccept(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::afterAccept is called" << endl;
  }

  return Accept;
}

userRequest_t
MyConnectionInterceptorForIIOP::beforeDisconnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::beforeDisconnect is called" << endl;
  }

  return Accept;
}

void
MyConnectionInterceptorForIIOP::afterDisconnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::afterDisconnect is called" << endl;
  }
}

void
MyConnectionInterceptorForSSL::beforeConnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::beforeConnect is called" << endl;
  }
}

userRequest_t
MyConnectionInterceptorForSSL::afterConnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::afterConnect is called" << endl;
  }

  return Accept;
}

void
MyConnectionInterceptorForSSL::beforeAccept()
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::beforeAccept is called" << endl;
  }
}

userRequest_t
MyConnectionInterceptorForSSL::afterAccept(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::afterAccept is called" << endl;
  }

  return Accept;
}

userRequest_t
MyConnectionInterceptorForSSL::beforeDisconnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::beforeDisconnect is called" << endl;
  }

  return Accept;
}

void
MyConnectionInterceptorForSSL::afterDisconnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::afterDisconnect is called" << endl;
  }
}

static CORBA::Boolean bindObjectToName(CORBA::ORB_ptr,CORBA::Object_ptr);

int
main(int argc, char **argv)
{
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB2");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB2_BOA");

  tcpContext::initContext(argc, argv);
  tcpContext *tcp_context = tcpContext::getContext();
  tcp_context->setConnectionInterceptorFactory((ConnectionInterceptorFactory *) 
			new MyConnectionInterceptorFactoryForIIOP());

  sslContext::initContext(argc, argv);
  sslContext *ssl_context = sslContext::getContext();
  ssl_context->setConnectionInterceptorFactory((ConnectionInterceptorFactory *) 
			new MyConnectionInterceptorFactoryForSSL());
  ssl_context->setCertificateFile("/home/tatsuo/cert/cert.pem");
  ssl_context->setKeyFile("/home/tatsuo/cert/key.pem");

  Echo_i *myobj = new Echo_i();
  myobj->_obj_is_ready(boa);

  {
    Echo_var myobjRef = myobj->_this();
    if (!bindObjectToName(orb,myobjRef)) {
      return 1;
    }
  }

  boa->impl_is_ready();
  // Tell the BOA we are ready. The BOA's default behaviour is to block
  // on this call indefinitely.

  return 0;
}


static
CORBA::Boolean
bindObjectToName(CORBA::ORB_ptr orb,CORBA::Object_ptr obj)
{
  CosNaming::NamingContext_var rootContext;
  
  try {
    // Obtain a reference to the root context of the Name service:
    CORBA::Object_var initServ;
    initServ = orb->resolve_initial_references("NameService");

    // Narrow the object returned by resolve_initial_references()
    // to a CosNaming::NamingContext object:
    rootContext = CosNaming::NamingContext::_narrow(initServ);
    if (CORBA::is_nil(rootContext)) 
      {
        cerr << "Failed to narrow naming context." << endl;
        return 0;
      }
  }
  catch(CORBA::ORB::InvalidName& ex) {
    cerr << "Service required is invalid [does not exist]." << endl;
    return 0;
  }


  try {
    // Bind a context called "test" to the root context:

    CosNaming::Name contextName;
    contextName.length(1);
    contextName[0].id   = (const char*) "test";    // string copied
    contextName[0].kind = (const char*) "my_context"; // string copied    
    // Note on kind: The kind field is used to indicate the type
    // of the object. This is to avoid conventions such as that used
    // by files (name.type -- e.g. test.ps = postscript etc.)

    CosNaming::NamingContext_var testContext;
    try {
      // Bind the context to root, and assign testContext to it:
      testContext = rootContext->bind_new_context(contextName);
    }
    catch(CosNaming::NamingContext::AlreadyBound& ex) {
      // If the context already exists, this exception will be raised.
      // In this case, just resolve the name and assign testContext
      // to the object returned:
      CORBA::Object_var tmpobj;
      tmpobj = rootContext->resolve(contextName);
      testContext = CosNaming::NamingContext::_narrow(tmpobj);
      if (CORBA::is_nil(testContext)) {
        cerr << "Failed to narrow naming context." << endl;
        return 0;
      }
    } 

    // Bind the object (obj) to testContext, naming it Echo:
    CosNaming::Name objectName;
    objectName.length(1);
    objectName[0].id   = (const char*) "Echo";   // string copied
    objectName[0].kind = (const char*) "Object"; // string copied


    // Bind obj with name Echo to the testContext:
    try {
      testContext->bind(objectName,obj);
    }
    catch(CosNaming::NamingContext::AlreadyBound& ex) {
      testContext->rebind(objectName,obj);
    }
    // Note: Using rebind() will overwrite any Object previously bound 
    //       to /test/Echo with obj.
    //       Alternatively, bind() can be used, which will raise a
    //       CosNaming::NamingContext::AlreadyBound exception if the name
    //       supplied is already bound to an object.

    // Amendment: When using OrbixNames, it is necessary to first try bind
    // and then rebind, as rebind on it's own will throw a NotFoundexception if
    // the Name has not already been bound. [This is incorrect behaviour -
    // it should just bind].
  }
  catch (CORBA::COMM_FAILURE& ex) {
    cerr << "Caught system exception COMM_FAILURE, unable to contact the "
         << "naming service." << endl;
    return 0;
  }
  catch (omniORB::fatalException& ex) {
    throw;
  }
  catch (...) {
    cerr << "Caught a system exception while using the naming service."<< endl;
    return 0;
  }
  return 1;
}
