// eg2_impl.cc - This is the source code of example 2 used in Chapter 2
//               "The Basics" of the omniORB2 user guide.
//
//               This is the object implementation.
//
// Usage: eg2_impl
//
//        On startup, the object reference is printed to cerr as a
//        stringified IOR. This string should be used as the argument to 
//        eg2_clt.
//
#include <iostream.h>

#include "echo.hh"
#include "omnithread.h"

#include <omniORB2/QOS.h>
#include <omniORB2/ConnectionInfo.h>

#include "echo_i.cc"

const int silent = 0;
const int reject_ssl = 0;

class MyConnectionInterceptorForIIOP : ConnectionInterceptor {
public:
  virtual userRequest_t afterAccept(ConnectionState& cs);
};

class MyConnectionInterceptorForSSL : ConnectionInterceptor {
public:
  virtual userRequest_t afterAccept(ConnectionState& cs);
};

class MyConnectionInterceptor : ConnectionInterceptor {
public:
  virtual userRequest_t beforeDisconnect(ConnectionState& cs);
};

class MyConnectionInterceptorFactoryForIIOP : ConnectionInterceptorFactory {
public:
  virtual ConnectionInterceptor *create();
};

class MyConnectionInterceptorFactoryForSSL : ConnectionInterceptorFactory {
public:
  virtual ConnectionInterceptor *create();
};

ConnectionInterceptor *
MyConnectionInterceptorFactoryForIIOP::create()
{
  return((ConnectionInterceptor *)new MyConnectionInterceptorForIIOP());
}

ConnectionInterceptor *
MyConnectionInterceptorFactoryForSSL::create()
{
  return((ConnectionInterceptor *)new MyConnectionInterceptorForSSL());
}

userRequest_t
MyConnectionInterceptor::beforeDisconnect(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptor::beforeDisconnect for the session is called" << endl;
  }

  return Accept;
}

userRequest_t
MyConnectionInterceptorForIIOP::afterAccept(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForIIOP::afterAccept is called" << endl;

    ConnectionControl *cc = cs.getConnectionControl();
    if(cc->getConnectionType() != TRANS_IIOP) {
      cerr << "Connection State is not for IIOP" << endl;

      return Reject;
    }

    IIOPConnectionControl *iiop_cc = (IIOPConnectionControl *)cc;
    char *ipaddr = iiop_cc->getPeerAddress();
    cerr << "Client's IP Address : " << ipaddr << endl;
    CORBA::string_free(ipaddr);

    MyConnectionInterceptor *interceptor
                = new MyConnectionInterceptor();
    iiop_cc->insertConnectionInterceptor((ConnectionInterceptor *)interceptor);
  }

  return Accept;
}

userRequest_t
MyConnectionInterceptorForSSL::afterAccept(ConnectionState& cs)
{
  if(!silent) {
    cout << "MyConnectionInterceptorForSSL::afterAccept is called" << endl;

    ConnectionControl *cc = cs.getConnectionControl();
    if(cc->getConnectionType() != TRANS_SSL) {
      cerr << "Connection State is not for SSL" << endl;

      return Reject;
    }

    char *str;

    SSLConnectionState *ssl_cs = (SSLConnectionState *) &cs;
    SSLConnectionControl *ssl_cc = (SSLConnectionControl *)cc;
    char *ipaddr = ssl_cc->getPeerAddress();
    cerr << "Client's IP Address : " << ipaddr << endl;
    CORBA::string_free(ipaddr);

    str = ssl_cs->getChipher();
    cout << "Chipher : " << str << endl;
    CORBA::string_free(str);

    MyConnectionInterceptor *interceptor
                = new MyConnectionInterceptor();
    ssl_cc->insertConnectionInterceptor((ConnectionInterceptor *)interceptor);
  }

  if(reject_ssl) {
    return Reject;
  } else {
    return Accept;
  }
}

int
main(int argc, char **argv)
{
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB2");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB2_BOA");

  tcpContext::initContext(argc, argv);
  tcpContext *tcp_context = tcpContext::getContext();
  tcp_context->setConnectionInterceptorFactory((ConnectionInterceptorFactory *) 
			new MyConnectionInterceptorFactoryForIIOP());

  sslContext::initContext(argc, argv);
  sslContext *ssl_context = sslContext::getContext();
  ssl_context->setConnectionInterceptorFactory((ConnectionInterceptorFactory *) 
			new MyConnectionInterceptorFactoryForSSL());
  ssl_context->setCertificateFile("cert.pem");
  ssl_context->setKeyFile("key.pem");

  Echo_i *myobj = new Echo_i();
  myobj->_obj_is_ready(boa);

  {
    Echo_var myobjRef = myobj->_this();
    CORBA::String_var p = orb->object_to_string(myobjRef);
    cerr << "'" << (char*)p << "'" << endl;
  }

  boa->impl_is_ready();
  // Tell the BOA we are ready. The BOA's default behaviour is to block
  // on this call indefinitely.

  return 0;
}
