// -*- Mode: C++; -*-
//                            Package   : omniEvents
//  eventd.cc                 Created on: 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniOEvents application
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//    Event Services Channel Factory implementation. The factory registers
//    itself with the naming service. Clients withing to create event
//    channels should either use the factory by resolving its name with the
//    naming service or create in-process channels.
//	

/*
  $Log:	eventd.cc,v $
Revision 1.1  98/04/02  16:15:37  16:15:37  naderp (Paul Nader)
Initial revision

 */

#include <iostream.h>
#include "CosEvent_i.h"
static void usage();

int
main(int argc, char **argv)
{
  //
  // Start orb and boa.
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB2");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB2_BOA");

  // Process Options
  int c;
  char *factoryName = (char *) "EventChannelFactory";
  char *factoryKind = (char *) "EventChannelFactory";

  while ((c = getopt(argc,argv,"N:K:h")) != EOF)
  {
     switch (c)
     {
        case 'N': factoryName = optarg;
                  break;

        case 'K': factoryKind = optarg;
                  break;

        case 'h':
        default : usage();
                  exit(-1);
                  break;
     }
  }

  //
  // Obtain and narrow reference to Name Service.
  CosNaming::NamingContext_ptr rootContext;
  try {

     // Get initial reference.
     CORBA::Object_var initServ;
     initServ = orb->resolve_initial_references("NameService");

     // Narrow the object returned by resolve_initial_references()
     // to a CosNaming::NamingContext object:
     rootContext = CosNaming::NamingContext::_narrow(initServ);
     if (CORBA::is_nil(rootContext))
     {
        cerr << "Failed to narrow naming context." << endl;
        exit(1);
     }
  }
  catch(CORBA::ORB::InvalidName& ex) {
     cerr << "Service required is invalid [does not exist]." << endl;
     exit(1);
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE."
          << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Caught a system exception while resolving the naming service."
          << endl;
     exit(1);
  }

  //
  // Create Event Channel Factory.
  EventChannelFactory_i *factory = new EventChannelFactory_i();
  //
  // Register Event Channel Factory with the Naming Service.
  CosNaming::Name name;
  name.length (1);
  name[0].id = CORBA::string_dup (factoryName);
  name[0].kind = CORBA::string_dup (factoryKind);

  try {
    rootContext->bind(name, factory);
  }
  catch(CosNaming::NamingContext::AlreadyBound& ex) {
    // overwrite previously bound object
    rootContext->rebind(name, factory);
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the "
          << "naming service." << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot register event channel factory ! [\""
          << factoryName << "\", \"" << factoryKind << "\"]"
          << endl;
     exit (1);
  }

  factory->_obj_is_ready(boa);

  // Start implementation, blocking.
  boa->impl_is_ready();

  return 0;
}

static void
usage()
{
   cerr << "\nusage: eventd [-N name] [-K kind] [-h]\n" << endl;
   cerr << "         -N name specify factory name [\"EventChannelFactory\"]" << endl;
   cerr << "         -K kind specify factory kind [\"EventChannelFactory\"]" << endl;
   cerr << "         -h     display usage" << endl;
   cerr << endl;
}

