// -*- Mode: C++; -*-
//                            Package   : omniEvents
// eventc.cc                  Created on: 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application 
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//    Client to the event channel factory. Requests creation of an event
//    channel and registers it with the Naming service.
//	

/*
  $Log:	eventc.cc,v $
Revision 1.1  98/04/02  16:15:37  16:15:37  naderp (Paul Nader)
Initial revision

 */

//
//
#include <iostream.h>
#include "CosEvent_i.h"

static void usage();

int
main(int argc, char **argv)
{
  //
  // Start orb and boa.
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB2");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB2_BOA");

  // Process Options
  int c;
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";
  char *factoryName = (char *) "EventChannelFactory";
  char *factoryKind = (char *) "EventChannelFactory";
  CORBA::ULong pullRetryPeriod = 1;

  while ((c = getopt(argc,argv,"n:k:N:K:r:h")) != EOF)
  {
     switch (c)
     {
        case 'n': channelName = optarg;
                  break;

        case 'k': channelKind = optarg;
                  break;

        case 'N': factoryName = optarg;
                  break;

        case 'K': factoryKind = optarg;
                  break;

        case 'r': pullRetryPeriod = atol(optarg);
                  break;

        case 'h':
        default : usage();
                  exit(-1);
                  break;
     }
  }


  //
  // Obtain and narrow reference to Name Service.
  CosNaming::NamingContext_ptr rootContext;
  try {

     // Get initial reference.
     CORBA::Object_var initServ;
     initServ = orb->resolve_initial_references("NameService");
     if (CORBA::is_nil(initServ))
     {
        cerr << "Failed to resolve NameService initial reference" << endl;
        exit(1);
     }

     // Narrow the object returned by resolve_initial_references()
     // to a CosNaming::NamingContext object:
     rootContext = CosNaming::NamingContext::_narrow(initServ);
     if (CORBA::is_nil(rootContext))
     {
        cerr << "Failed to narrow naming context." << endl;
        exit(1);
     }
  }
  catch(CORBA::ORB::InvalidName& ex) {
     cerr << "Service required is invalid [does not exist]." << endl;
     exit(1);
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE."
          << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Caught a system exception while resolving the naming service."
          << endl;
     exit(1);
  }

  //
  // Obtain reference to the Event Channel Factory implementation.
  EventChannelAdmin::EventChannelFactory_ptr factory;
  CosNaming::Name name;
  name.length (1);
  name[0].id = CORBA::string_dup (factoryName);
  name[0].kind = CORBA::string_dup (factoryKind);

  try {
    CORBA::Object_var obj = rootContext->resolve(name);
    factory = EventChannelAdmin::EventChannelFactory::_narrow(obj);
    if (CORBA::is_nil(factory))
    {
       cerr << "Failed to narrow Event Channel Factory reference." << endl;
       exit(1);
    }

  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the "
          << "naming service." << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot find event channel factory ! [\""
          << factoryName << "\", \"" << factoryKind << "\"]"
          << endl;
     exit (1);
  }

  // Check that the factory is of the right type
  CosLifeCycle::Key key;
  key.length (1);
  key[0].id = CORBA::string_dup ("EventChannel");
  key[0].kind = CORBA::string_dup ("object interface");
  if (! factory->supports(key))
  {
     cerr << "Factory does not support Event Channel Interface ! [\""
          << factoryName << "\", \"" << factoryKind << "\"]"
          << endl;
     exit (1);
  }
  

  //
  // Create Event Channel Object.
  CosEventChannelAdmin::EventChannel_var channel;
  try {
     CORBA::Object_var channelObj;
     CosLifeCycle::Criteria criteria;
     criteria.length (1);
     criteria[0].name = CORBA::string_dup ("pullRetryPeriod");
     criteria[0].value <<= (CORBA::ULong) pullRetryPeriod;

     channelObj = factory->create_object(key, criteria);
     if (CORBA::is_nil(channelObj))
     {
        cerr << "Channel Factory returned nil reference. ! [\""
             << channelName << "\", \"" << channelKind << "\"]"
             << endl;
        exit(1);
     }

     // Narrow object returned to an Event Channel
     channel = CosEventChannelAdmin::EventChannel::_narrow(channelObj);
     if (CORBA::is_nil(channel))
     {
        cerr << "Failed to narrow Event Channel ! [\""
             << channelName << "\", \"" << channelKind << "\"]"
             << endl;
        exit(1);
     }
  }
  catch (...) {
     cerr << "Failed to create Event Channel ! [\""
          << channelName << "\", \"" << channelKind << "\"]"
          << endl;
     exit(1);
  }

  //
  // Register event channel with naming service
  name.length (1);
  name[0].id = CORBA::string_dup (channelName);
  name[0].kind = CORBA::string_dup (channelKind);
  try {
    rootContext->bind (name,
              CosEventChannelAdmin::EventChannel::_duplicate(channel));
  }
  catch(CosNaming::NamingContext::AlreadyBound& ex) {
    rootContext->rebind(name,
              CosEventChannelAdmin::EventChannel::_duplicate(channel));
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the "
          << "naming service." << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot register event channel ! [\""
          << channelName << "\", \"" << channelKind << "\"]"
          << endl;
     exit (1);
  }

//  boa->impl_is_ready(0,0);

  return 0;
}

static void
usage()
{
   cerr << "\nusage: eventc [-n name] [-k kind] [-N name] [-K kind] [-r sec] [-h]\n" << endl;
   cerr << "         -n name specify channel name [\"EventChannel\"]" << endl;
   cerr << "         -k kind specify channel kind [\"EventChannel\"]" << endl;
   cerr << "         -N name specify factory name [\"EventChannelFactory\"]" << endl;
   cerr << "         -K kind specify factory kind [\"EventChannelFactory\"]" << endl;
   cerr << "         -r Pull retry Period (in seconds) [1 sec]]" << endl;
   cerr << "         -h     display usage" << endl;
   cerr << endl;
}

