// -*- Mode: C++; -*-
//                            Package   : omniProperties
// propclt.cc                 Created on: 2/5/1999
//                            Author    : Ted Byrd (tbyrd@idiom.com)
//
//    Copyright (C) 1999 Ted Byrd
//    All Rights Reserved.
//
//    This file is part of omniProperties
//
//    omniProperties is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//
//    Simple COSS Property Service client.
//
//
// Revision History:
//
// $Log: propclt.cc,v $
// Revision 1.3  1999/02/06 05:37:59  tbyrd
// Added try block around execCommand() to trap rogue exceptions
//
// Revision 1.2  1999/02/06 04:30:03  tbyrd
// Completed Linux port
//
// Revision 1.1  1999/02/06 03:58:24  tbyrd
// Initial check in
//
//
//	

#include <stdio.h>
#include <iostream.h>
#include <fstream.h>

#include <string>
#include <vector>
#include <iterator>

#include <CosPropertyService.hh>
#include <CosLifeCycle.hh>

#ifdef __WIN32__
int getopt(int argc, char** argv, char* opts);
extern char* optarg;
extern int optind;
#else
#define stricmp strcasecmp
#endif

// list of commands support by program

enum CommandType {
  cmd_create,
  cmd_destroy,
  cmd_add,
  cmd_update,
  cmd_remove,
  cmd_mode,
  cmd_find,
  cmd_list,
  cmd_import,
  cmd_export,
  cmd_help,
  cmd_invalid
};

// list of command strings/types to make parsing easier

struct command_string {
  char*       command;
  CommandType type;
};

static const command_string commands[] = {
  { "create",   cmd_create },
  { "destroy",  cmd_destroy },
  { "add",      cmd_add },
  { "update",   cmd_update },
  { "remove",   cmd_remove },
  { "mode",     cmd_mode },
  { "find",     cmd_find },
  { "list",     cmd_list },
  { "import",   cmd_import },
  { "export",   cmd_export },
  { "help",     cmd_help },
  { NULL,       cmd_invalid }
};

struct command {
  CommandType type;
  char*       factoryName;
  char*       factoryKind;
  char*       setName;
  char*       setKind;
  char*       name;
  char*       value;
};

static const char* modeNames[] = {
  "normal",
  "read_only",
  "fixed_normal",
  "fixed_readonly",
  "undefined",
  NULL
};

static const char* listModeNames[] = {
  "--n-",
  "---r",
  "-fn-",
  "-f-r",
  "u---",
  NULL
};

struct imported_property {
  std::string name;
  CORBA::Any value;
  CosPropertyService::PropertyModeType mode;
};

static CosNaming::NamingContext_ptr g_rootContext;

void usageMessage();
void execCommand( command& cmd );

int main(int argc, char** argv)
{
  //
  // Initialize command structure to default values
  //

  command cmd;
  memset(&cmd, 0, sizeof(command));

  cmd.factoryName = "PropertySetDefFactory";
  cmd.factoryKind = "PropertySetDefFactory";

  cmd.setName = "PropertySetDef";
  cmd.setKind = "PropertySetDef";

  //
  // Parse command line for user defined options
  //

  if ( argc < 2 )
  {
    usageMessage();
    return 1;
  }

  cmd.type = cmd_invalid;

  for ( const command_string* p = commands; p->command; p++ )
  {
    if ( stricmp(argv[1], p->command) == 0 )
    {
      cmd.type = p->type;
      break;
    }
  }

  if ( cmd.type == cmd_invalid )
  {
    cerr << "Unknown command '" << argv[1] << "'" << endl;
    usageMessage();
    return 1;
  }

  int opt;
  optind = 2;

  while ( (opt = getopt(argc, argv, "N:K:s:k:n:v:m:f:")) != -1 )
  {
    switch ( opt )
    {
    case 'N':
      cmd.factoryName = optarg;
      break;
    case 'K':
      cmd.factoryKind = optarg;
      break;
    case 's':
      cmd.setName = optarg;
      break;
    case 'k':
      cmd.setKind = optarg;
      break;
    case 'n':
      cmd.name = optarg;
      break;
    case 'v':
      cmd.value = optarg;
      break;
    case 'm':
      cmd.value = optarg;
      break;
    case 'f':
      cmd.value = optarg;
      break;
    }
  }

  if ( cmd.type == cmd_help || optind < argc )
  {
    usageMessage();
    return 1;
  }

  //
  // Initialize the ORB and BOA
  //

  CORBA::ORB_ptr orb;
  CORBA::BOA_ptr boa;

  try
  {
    orb = CORBA::ORB_init(argc, argv, "omniORB2");
    boa = orb->BOA_init(argc, argv, "omniORB2_BOA");
  }
  catch (...)
  {
    cerr << "Exception caught while initializing ORB and BOA" << endl;
    return 1;
  }

  //
  // Obtain and narrow reference to Name Service.
  //

  try
  {
    // Get initial reference.
    CORBA::Object_var initServ;
    initServ = orb->resolve_initial_references("NameService");
    if (CORBA::is_nil(initServ))
    {
      cerr << "Failed to resolve NameService initial reference!" << endl;
      return 1;
    }

    // Narrow the object returned by resolve_initial_references()
    // to a CosNaming::NamingContext object:
    g_rootContext = CosNaming::NamingContext::_narrow(initServ);
    if (CORBA::is_nil(g_rootContext))
    {
      cerr << "Failed to narrow naming context!" << endl;
      return 1;
    }
  }
  catch(CORBA::ORB::InvalidName&) {
    cerr << "Naming service is unavailable!" << endl;
    return 1;
  }
  catch (CORBA::COMM_FAILURE&) {
    cerr << "Can't contact the naming service!" << endl;
    return 1;
  }
  catch (CORBA::SystemException&) {
    cerr << "CORBA error while resovling the naming service!" << endl;
    return 1;
  }
  catch (omniORB::fatalException&) {
    cerr << "Caught omniORB Fatal Exception!" << endl;
    return 1;
  }
  catch (...) {
    cerr << "Unknown error while resolving the naming service!" << endl;
    return 1;
  }

  //
  // Execute the user's command
  //

  try
  {
    execCommand( cmd );
  }
  catch (...)
  {
    cerr << "Unknown error while processing command!" << endl;
    return 1;
  }

  return 0;
}

void usageMessage()
{
  cout << endl
       << "Usage: propclt command [options]" << endl
       << endl
       << "Where 'command' is one of the following:" << endl
       << endl
       << "  create     Create a new property set" << endl
       << "  destroy    Destroy an existing property set" << endl
       << "  add        Add a property to a set" << endl
       << "  remove     Remove a property from a set" << endl
       << "  update     Update a property in a set (same as 'add')" << endl
       << "  find       Find a property in a set" << endl
       << "  list       List all properties in a set" << endl
       << "  mode       Set property mode" << endl
       << "  import     Import properties from a file" << endl
       << "  export     Export properties to a file" << endl
       << "  help       Display this usage message" << endl
       << endl
       << "And 'options' are one or more of the following:" << endl
       << endl
       << "  [-N name]  Factory name for create and destroy (def: PropertySetDefFactory)" << endl
       << "  [-K kind]  Factory kind for create and destroy (def: PropertySetDefFactory)" << endl
       << "  [-s name]  PropertySet name for all operations (def: PropertySetDef)" << endl
       << "  [-k kind]  PropertySet kind for all operations (def: PropertySetDef)" << endl
       << "  [-n name]  Property name for add, remove, update, mode, and find" << endl
       << "  [-v value] Property value for add and update" << endl
       << "  [-m mode]  Property mode: normal, read_only, fixed_normal, or fixed_readonly" << endl
       << "  [-f file]  File name for import and export" << endl
       << endl;
}

CosPropertyService::PropertySetDef_ptr
lookupPropertySet(const char* pszSetName, const char* pszSetKind)
{
  //
  // Obtain PropertySet Object.
  //

  CosNaming::Name name;

  name.length (1);
  name[0].id = CORBA::string_dup (pszSetName);
  name[0].kind = CORBA::string_dup (pszSetKind);

  try
  {
    CORBA::Object_var obj = g_rootContext->resolve(name);
    CosPropertyService::PropertySetDef_ptr propertySetDef;
    propertySetDef = CosPropertyService::PropertySetDef::_narrow(obj);

    if ( ! CORBA::is_nil(propertySetDef) )
      return propertySetDef;

    cerr << "Failed to narrow '" << pszSetName 
         << "' to PropertySetDef reference!" << endl;
  }
  catch (CORBA::ORB::InvalidName&) {
     cerr << "Property set '" << pszSetName << "' isn't registered in the name service" << endl;
  }
  catch (CORBA::COMM_FAILURE&) {
     cerr << "Unable to contact property set '" << pszSetName << "'!" << endl;
  }
  catch (omniORB::fatalException&) {
     cerr << "Caught omniORB Fatal Exception!" << endl;
  }
  catch (...) {
     cerr << "Can't find property set '" << pszSetName << "'" << endl;
  }

  exit(1);

  return CosPropertyService::PropertySetDef::_nil();  // keep compiler smiling
}

int printAny(ostream& os, CORBA::Any& any)
{
  char szBuf[1024];

  switch ( any.type()->kind() )
  {
  case CORBA::tk_string:
    {
      char* s;
      any >>= s;
      strcpy(szBuf, s);
    }
    break;
  case CORBA::tk_short:
    {
      CORBA::Short s;
      any >>= s;
      sprintf(szBuf, "%d", s);
    }
    break;
  case CORBA::tk_long:
    {
      CORBA::Long l;
      any >>= l;
      sprintf(szBuf, "%ld", l);
    }
    break;
  case CORBA::tk_void:
    {
      strcpy(szBuf, "(undefined)");
    }
    break;
  case CORBA::tk_null:
    {
      strcpy(szBuf, "(null)");
    }
    break;
  default:
    strcpy(szBuf, "(unsupported type)");
    break;
  }

  os << szBuf;

  return strlen(szBuf);
}

int createPropertySet(const char* pszFactoryName, const char* pszFactoryKind, const char* pszSetName, const char* pszSetKind)
{
  //
  // Obtain PropertySetDefFactory Object.
  //

  CosNaming::Name name;

  name.length (1);
  name[0].id = CORBA::string_dup (pszFactoryName);
  name[0].kind = CORBA::string_dup (pszFactoryKind);

  CosPropertyService::PropertySetDefFactory_var propertySetDefFactory;

  try
  {
    CORBA::Object_var obj = g_rootContext->resolve(name);
    propertySetDefFactory = CosPropertyService::PropertySetDefFactory::_narrow(obj);
    if (CORBA::is_nil(propertySetDefFactory))
    {
       cerr << "Factory '" << pszFactoryName << "' is not a CosPropertyService::PropertySetDefFactory!" << endl;
       return 0;
    }
  }
  catch (CORBA::ORB::InvalidName&) {
     cerr << "Property set factory '" << pszFactoryName << "' isn't registered in the name service" << endl;
     return 0;
  }
  catch (CORBA::COMM_FAILURE&) {
     cerr << "Unable to contact the property service!" << endl;
     return 0;
  }
  catch (omniORB::fatalException&) {
     cerr << "Caught omniORB Fatal Exception!" << endl;
     return 0;
  }
  catch (...) {
     cerr << "Can't find property service '" << pszFactoryName << "'" << endl;
     return 0;
  }

  //
  // Create the property set (def)
  //

  CosPropertyService::PropertySetDef_var propertySetDef;

  propertySetDef = propertySetDefFactory->create_propertysetdef();

  //
  // Bind the property set to a name in the naming service
  //

  CosNaming::Name propertySetName;
  propertySetName.length(1);
  propertySetName[0].id = pszSetName;
  propertySetName[0].kind = pszSetKind;

  try {
    g_rootContext->bind(propertySetName, propertySetDef);
  }
  catch (CosNaming::NamingContext::AlreadyBound&) {
    g_rootContext->rebind(propertySetName, propertySetDef);
  }
  catch (CORBA::COMM_FAILURE&) {
    cerr << "The name service is unavailable" << endl;
    return 0;
  }
  catch (...) {
    cerr << "System exception while binding PropertySetDef to '" << pszSetName << "'" << endl;
    return 0;
  }

  cout << "Property set '" << pszSetName << "' created with kind '" << pszSetKind << "'" << endl;

  return 1;
}

int destroyPropertySet(const char* pszSetName, const char* pszSetKind)
{
  //
  // Obtain LifeCycleObject.
  //

  CosNaming::Name name;

  name.length (1);
  name[0].id = CORBA::string_dup (pszSetName);
  name[0].kind = CORBA::string_dup (pszSetKind);

  try
  {
    CORBA::Object_var obj = g_rootContext->resolve(name);

    CosLifeCycle::LifeCycleObject_var lifeCycleObject;
    lifeCycleObject = CosLifeCycle::LifeCycleObject::_narrow(obj);

    if (CORBA::is_nil(lifeCycleObject))
    {
       cerr << "Failed to narrow LifeCycleObject reference for '" << pszSetName << "'!" << endl;
       return 0;
    }

    lifeCycleObject->remove();
  }
  catch (CORBA::ORB::InvalidName&) {
     cerr << "Property set '" << pszSetName << "' isn't registered in the name service" << endl;
     return 0;
  }
  catch (CORBA::COMM_FAILURE&) {
     cerr << "Unable to contact property set '" << pszSetName << "'!" << endl;
     return 0;
  }
  catch (omniORB::fatalException&) {
     cerr << "Caught omniORB Fatal Exception!" << endl;
     return 0;
  }
  catch (...) {
     cerr << "Can't find property set '" << pszSetName << "'" << endl;
     return 0;
  }

  cout << "Property set '" << pszSetName << "' destroyed" << endl;

  return 1;
}

static inline void
fill(ostream& os, int pad)
{
  while ( pad-- > 0 )
    os << " ";
}
  
int listProperties(const char* pszSetName, const char* pszSetKind)
{
  CosPropertyService::PropertySetDef_ptr set;
  set = lookupPropertySet( pszSetName, pszSetKind );

  int cProperties = set->get_number_of_properties();
  CosPropertyService::Properties_var properties;
  CosPropertyService::PropertiesIterator_var rest;

  set->get_all_properties( cProperties, properties, rest );

  cout << "Contents of property set '" << pszSetName << "'" << endl
       << endl;

  cout << "Property";
  fill(cout, 8);
  cout << "Value";
  fill(cout, 53);
  cout << "Mode" << endl;

  cout << "--------------- ";
  cout << "--------------------------------------------------------- ";
  cout << "----" << endl;

  for ( size_t i = 0; i < properties->length(); i++ )
  {
    cout << properties[i].property_name;
    fill(cout, 16 - strlen(properties[i].property_name));

    int len = printAny(cout, properties[i].property_value);
    fill(cout, 58 - len );

    cout << listModeNames[set->get_property_mode(properties[i].property_name)]
         << endl;
  }

  cout << endl;

  return 1;
}

int defineProperty(const char* pszSetName, const char* pszSetKind, const char* pszName, const char* pszValue)
{
  CosPropertyService::PropertySet_ptr set;
  set = lookupPropertySet( pszSetName, pszSetKind );
  if ( set == CosPropertyService::PropertySet::_nil() )
  {
    return 0;
  }

  CORBA::Any any;
  any <<= pszValue;

  try
  {
    set->define_property(pszName, any);
  }
  catch (CosPropertyService::InvalidPropertyName&)
  {
    cerr << "Property '" << pszName << "' is not a valid name" << endl;
    return 0;
  }
  catch (CosPropertyService::ConflictingProperty&)
  {
    cerr << "Property '" << pszName << "' already exists with another type, try deleting first" << endl;
    return 0;
  }
  catch (CosPropertyService::UnsupportedTypeCode&)
  {
    cerr << "Unsupported data type (string) for property '" << pszName << "'" << endl;
    return 0;
  }
  catch (CosPropertyService::UnsupportedProperty&)
  {
    cerr << "Unsupported property '" << pszName << "'" << endl;
    return 0;
  }
  catch (CosPropertyService::ReadOnlyProperty&)
  {
    cerr << "Property '" << pszName << "' is read-only, try deleting first" << endl;
    return 0;
  }
  catch (CORBA::SystemException&)
  {
    cerr << "CORBA exception while adding property '" << pszName << "'" << endl;
    return 0;
  }
  catch (...)
  {
    cerr << "Unable to contact property set '" << pszSetName << "'" << endl;
    return 0;
  }

  cout << "Property '" << pszName << "' added to set '" << pszSetName << "' with value '" << pszValue << "'" << endl;

  return 1;
}

int deleteProperty(const char* pszSetName, const char* pszSetKind, const char* pszName)
{
  CosPropertyService::PropertySet_ptr set;
  set = lookupPropertySet( pszSetName, pszSetKind );
  if ( set == CosPropertyService::PropertySet::_nil() )
  {
    return 0;
  }

  try
  {
    set->delete_property(pszName);
  }
  catch (CosPropertyService::PropertyNotFound&)
  {
    cerr << "Property '" << pszName << "' not found in set '" << pszSetName << "'" << endl;
    return 0;
  }
  catch (CosPropertyService::InvalidPropertyName&)
  {
    cerr << "Property '" << pszName << "' is invalid" << endl;
    return 0;
  }
  catch (CosPropertyService::FixedProperty&)
  {
    cerr << "Property '" << pszName << "' is fixed, can't delete" << endl;
    return 0;
  }
  catch (...)
  {
    throw;
  }

  cout << "Property '" << pszName << "' deleted" << endl;

  return 1;
}

int findProperty(const char* pszSetName, const char* pszSetKind, const char* pszName)
{
  CosPropertyService::PropertySetDef_ptr set;
  set = lookupPropertySet( pszSetName, pszSetKind );
  if ( set == CosPropertyService::PropertySet::_nil() )
  {
    return 0;
  }

  try
  {
    CORBA::Any_var any;
    any = set->get_property_value(pszName);
    printAny(cout, any);
    cout << endl;
  }
  catch (CosPropertyService::PropertyNotFound&)
  {
    cerr << "Property '" << pszName << "' not found in set '" << pszSetName << "'" << endl;
    return 0;
  }
  catch (CosPropertyService::InvalidPropertyName&)
  {
    cerr << "Property '" << pszName << "' is invalid" << endl;
    return 0;
  }
  catch (CosPropertyService::FixedProperty&)
  {
    cerr << "Property '" << pszName << "' is fixed" << endl;
    return 0;
  }
  catch (...)
  {
    throw;
  }

  return 1;
}

int changeMode(const char* pszSetName, const char* pszSetKind, const char* pszName, const char* pszMode)
{
  CosPropertyService::PropertySetDef_ptr set;
  set = lookupPropertySet( pszSetName, pszSetKind );
  if ( set == CosPropertyService::PropertySet::_nil() )
  {
    return 0;
  }

  CosPropertyService::PropertyModeType mode;

  if ( stricmp(pszMode, "normal") == 0 )
    mode = CosPropertyService::normal;
  else if ( stricmp(pszMode, "read_only") == 0 )
    mode = CosPropertyService::read_only;
  else if ( stricmp( pszMode, "fixed_normal") == 0 )
    mode = CosPropertyService::fixed_normal;
  else if ( stricmp( pszMode, "fixed_readonly") == 0 )
    mode = CosPropertyService::fixed_readonly;
  else
  {
    usageMessage();
    return 0;
  }
  
  try
  {
    set->set_property_mode(pszName, mode);
  }
  catch (CosPropertyService::PropertyNotFound&)
  {
    cerr << "Property '" << pszName << "' not found in set '" << pszSetName << "'" << endl;
    return 0;
  }
  catch (CosPropertyService::InvalidPropertyName&)
  {
    cerr << "Property '" << pszName << "' is invalid" << endl;
    return 0;
  }
  catch (...)
  {
    throw;
  }

  cout << "Property '" << pszName << "' mode changed to '" << pszMode << "'" << endl;

  return 1;
}

int importPropertySet(const char* pszSetName, const char* pszSetKind, const char* pszFileName)
{
  ifstream ifs(pszFileName, ios::in);

  if ( ! ifs.is_open() )
  {
    cerr << "Can't open file '" << pszFileName << "' for import" << endl;
    return 0;
  }

  std::vector<imported_property> propList;

  while ( ! ifs.eof() )
  {
    imported_property prop;
    char szBuf[512];

    ifs >> ws;

    if ( ! ifs.getline(szBuf, sizeof(szBuf), '|') )
      break;

    prop.name = szBuf;

    if ( ! ifs.getline(szBuf, sizeof(szBuf), '|') )
      break;

    prop.value <<= szBuf;

    if ( ! ifs.getline(szBuf, sizeof(szBuf), '\n') )
      break;

    int c;
    const char** psz;

    for ( c = 0, psz = modeNames; *psz; psz++, c++ )
    {
      if ( stricmp(*psz, szBuf) == 0 )
        break;
    }

    if ( *psz )
      prop.mode = (CosPropertyService::PropertyModeType) c;
    else
    {
      cerr << "Invalid property mode '" << szBuf << "' specified in import file '" << pszFileName << "'" << endl;
      ifs.close();
      return 0;
    }

    propList.insert(propList.end(), prop);
  }

  if ( propList.size() )
  {
    try
    {
      CosPropertyService::PropertyDefs propertyDefs(propList.size());
      propertyDefs.length(propList.size());

      std::vector<imported_property>::iterator iter;
      size_t i;

      for ( i = 0, iter = propList.begin(); iter != propList.end(); i++, iter++ )
      {
        propertyDefs[i].property_name = (*iter).name.data();
        propertyDefs[i].property_value = (*iter).value;
        propertyDefs[i].property_mode = (*iter).mode;
      }

      CosPropertyService::PropertySetDef_ptr set;
      set = lookupPropertySet( pszSetName, pszSetKind );

      set->define_properties_with_modes(propertyDefs);
    }
    catch (CosPropertyService::MultipleExceptions& ex)
    {
      for ( size_t i = 0; i < ex.exceptions.length(); i++ )
      {
        switch ( ex.exceptions[i].reason )
        {
        case CosPropertyService::invalid_property_name:
          cerr << "Property '" << ex.exceptions[i].failing_property_name << "' is not a valid name" << endl;
          break;
        case CosPropertyService::conflicting_property:
          cerr << "Property '" << ex.exceptions[i].failing_property_name << "' already exists with another type, try deleting first" << endl;
          break;
        case CosPropertyService::unsupported_type_code:
          cerr << "Unsupported data type (string) for property '" << ex.exceptions[i].failing_property_name << "'" << endl;
          break;
        case CosPropertyService::unsupported_property:
          cerr << "Unsupported property '" << ex.exceptions[i].failing_property_name << "'" << endl;
          break;
        case CosPropertyService::unsupported_mode:
          cerr << "Unsupported mode for property '" << ex.exceptions[i].failing_property_name << "'" << endl;
          break;
        case CosPropertyService::read_only_property:
          cerr << "Property '" << ex.exceptions[i].failing_property_name << "' is read-only, try deleting first" << endl;
          break;
	default:
	  break;
        }
      }

      ifs.close();
      return 0;
    }
    catch (CORBA::SystemException&)
    {
      cerr << "CORBA exception while importing properties" << endl;
      ifs.close();
      return 0;
    }
    catch (...)
    {
      cerr << "Unable to contact property set '" << pszSetName << "'" << endl;
      ifs.close();
      return 0;
    }
  }

  ifs.close();

  cout << "Import file '" << pszFileName << "' loaded into property set '" << pszSetName << "'" << endl;

  return 1;
}

int exportPropertySet(const char* pszSetName, const char* pszSetKind, const char* pszFileName)
{
  ofstream ofs(pszFileName, ios::out|ios::trunc);

  if ( ! ofs.is_open() )
  {
    cerr << "Can't open file '" << pszFileName << "' for export" << endl;
    return 0;
  }

  CosPropertyService::PropertySetDef_ptr set;
  set = lookupPropertySet( pszSetName, pszSetKind );

  int cProperties = set->get_number_of_properties();

  CosPropertyService::Properties_var properties;
  CosPropertyService::PropertiesIterator_var rest;
  set->get_all_properties( cProperties, properties, rest );

  for ( size_t i = 0; i < properties->length(); i++ )
  {
    ofs << properties[i].property_name << "|";
    printAny(ofs, properties[i].property_value);
    ofs << "|" << modeNames[set->get_property_mode(properties[i].property_name)] << endl;
  }

  ofs.close();

  cout << "Property set '" << pszSetName << "' exported to '" << pszFileName << "'" << endl;

  return 1;
}

void execCommand( command& cmd )
{
  switch ( cmd.type )
  {
  case cmd_destroy:
    destroyPropertySet(cmd.setName, cmd.setKind);
    break;

  case cmd_create:
    createPropertySet(cmd.factoryName, cmd.factoryKind, cmd.setName, cmd.setKind );
    break;

  case cmd_add:
  case cmd_update:
    if ( ! cmd.name || ! cmd.value )
    {
      usageMessage();
      break;
    }
    defineProperty(cmd.setName, cmd.setKind, cmd.name, cmd.value);
    break;

  case cmd_remove:
    if ( ! cmd.name )
    {
      usageMessage();
      break;
    }
    deleteProperty(cmd.setName, cmd.setKind, cmd.name);
    break;

  case cmd_mode:
    if ( ! cmd.value )
    {
      usageMessage();
      break;
    }
    changeMode(cmd.setName, cmd.setKind, cmd.name, cmd.value);
    break;

  case cmd_find:
    if ( ! cmd.name )
    {
      usageMessage();
      break;
    }
    findProperty(cmd.setName, cmd.setKind, cmd.name);
    break;

  case cmd_list:
    listProperties(cmd.setName, cmd.setKind);
    break;

  case cmd_import:
    if ( ! cmd.value )
    {
      usageMessage();
      break;
    }
    importPropertySet(cmd.setName, cmd.setKind, cmd.value);
    break;

  case cmd_export:
    if ( ! cmd.value )
    {
      usageMessage();
      break;
    }
    exportPropertySet(cmd.setName, cmd.setKind, cmd.value);
    break;

  default:
    usageMessage();
    break;
  }
}
