// -*- Mode: C++; -*-
//                            Package   : omniProperties
// CosPropertyService_.cc     Created on: 2/5/1999
//                            Author    : Ted Byrd (tbyrd@idiom.com)
//
//    Copyright (C) 1999 Ted Byrd
//    All Rights Reserved.
//
//    This file is part of omniProperties
//
//    omniProperties is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//
//    Implementation of the COSS Property Service. 
//	
//
// Revision History:
//
// $Log: CosPropertyService_i.cc,v $
// Revision 1.2  1999/02/06 04:30:03  tbyrd
// Completed Linux port
//
// Revision 1.1  1999/02/06 03:58:23  tbyrd
// Initial check in
//
//

#include "CosPropertyService_i.h"

//
// PropertySetFactory Implementation
//

PropertySetFactory_i::PropertySetFactory_i()
{
}

PropertySetFactory_i::~PropertySetFactory_i()
{
}

CosPropertyService::PropertySet_ptr
PropertySetFactory_i::create_propertyset()
{
  PropertySet_i* pps = new PropertySet_i();
  pps->_obj_is_ready(_boa());
  return pps->_this();
}

CosPropertyService::PropertySet_ptr
PropertySetFactory_i::create_constrained_propertyset(
  const CosPropertyService::PropertyTypes & allowed_property_types,
  const CosPropertyService::Properties & allowed_properties)
{
  PropertySet_i* pps = new PropertySet_i( allowed_property_types, allowed_properties );
  pps->_obj_is_ready(_boa());
  return pps->_this();
}

CosPropertyService::PropertySet_ptr
PropertySetFactory_i::create_initial_propertyset(
  const CosPropertyService::Properties & initial_properties)
{
  PropertySet_i* pps = new PropertySet_i( initial_properties );
  pps->_obj_is_ready(_boa());
  return pps->_this();
}

CosLifeCycle::LifeCycleObject_ptr
PropertySetFactory_i::copy(CosLifeCycle::FactoryFinder_ptr  there, const CosLifeCycle::Criteria & the_criteria)
{
  throw CosLifeCycle::NotCopyable();
  return CosLifeCycle::LifeCycleObject::_nil();
}

void
PropertySetFactory_i::move(CosLifeCycle::FactoryFinder_ptr  there, const CosLifeCycle::Criteria & the_criteria)
{
  throw CosLifeCycle::NotMovable();
}

void
PropertySetFactory_i::remove()
{
  _dispose();
}

//
// PropertySetDefFactory Implementation
//

PropertySetDefFactory_i::PropertySetDefFactory_i()
{
}

PropertySetDefFactory_i::~PropertySetDefFactory_i()
{
}

CosPropertyService::PropertySetDef_ptr
PropertySetDefFactory_i::create_propertysetdef()
{
  PropertySetDef_i* ppsd = new PropertySetDef_i();
  ppsd->_obj_is_ready(_boa());
  return ppsd->_this();
}

CosPropertyService::PropertySetDef_ptr
PropertySetDefFactory_i::create_constrained_propertysetdef(
  const CosPropertyService::PropertyTypes & allowed_property_types,
  const CosPropertyService::PropertyDefs & allowed_property_defs)
{
  PropertySetDef_i* ppsd = new PropertySetDef_i(allowed_property_types, allowed_property_defs);
  ppsd->_obj_is_ready(_boa());
  return ppsd->_this();
}

CosPropertyService::PropertySetDef_ptr
PropertySetDefFactory_i::create_initial_propertysetdef(
  const CosPropertyService::PropertyDefs & initial_property_defs)
{
  PropertySetDef_i* ppsd = new PropertySetDef_i(initial_property_defs);
  ppsd->_obj_is_ready(_boa());
  return ppsd->_this();
}

CosLifeCycle::LifeCycleObject_ptr
PropertySetDefFactory_i::copy(CosLifeCycle::FactoryFinder_ptr  there, const CosLifeCycle::Criteria & the_criteria)
{
  throw CosLifeCycle::NotCopyable();
  return CosLifeCycle::LifeCycleObject::_nil();
}

void
PropertySetDefFactory_i::move(CosLifeCycle::FactoryFinder_ptr  there, const CosLifeCycle::Criteria & the_criteria)
{
  throw CosLifeCycle::NotMovable();
}

void
PropertySetDefFactory_i::remove()
{
  _dispose();
}

//
// PropertySet Implementation
//

PropertySet_i::PropertySet_i()
{
}

PropertySet_i::PropertySet_i(
  const CosPropertyService::PropertyTypes& allowed_property_types,
  const CosPropertyService::Properties& allowed_properties)
{
  omni_mutex_lock l(m_lock);
  size_t i;

  for (i = 0; i < allowed_property_types.length(); i++)
  {
    m_allowedTypes.insert(m_allowedTypes.end(), allowed_property_types[i]->kind());
  }

  for (i = 0; i < allowed_properties.length(); i++)
  {
    property_t p;

    p.name = allowed_properties[i].property_name;
    p.mode = CosPropertyService::normal;
    p.value <<= 0L;

    m_allowedProperties.insert(m_allowedProperties.end(), p);
  }
}

PropertySet_i::PropertySet_i(const CosPropertyService::Properties& initial_properties)
{
  omni_mutex_lock l(m_lock);
  size_t i;

  for (i = 0; i < initial_properties.length(); i++)
  {
    property_t p;

    p.name = initial_properties[i].property_name;
    p.mode = CosPropertyService::normal;
    p.value = initial_properties[i].property_value;

    m_propmap[initial_properties[i].property_name] = p;
  }
}

PropertySet_i::~PropertySet_i()
{
}

property_map_iterator_t
PropertySet_i::find_property( const char* key )
{
  property_map_iterator_t i;

  for ( i = m_propmap.begin(); i != m_propmap.end(); i++ )
  {
    if ( (*i).second.name == key )
      return i;
  }

  return m_propmap.end();
}

void
PropertySet_i::define_property(const char* property_name, const CORBA::Any& property_value)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  omni_mutex_lock l(m_lock);
  
  property_map_iterator_t iter = find_property(property_name);

  if ( iter != m_propmap.end() )
  {
    if ( (*iter).second.value.type()->kind() != property_value.type()->kind() )
      throw CosPropertyService::ConflictingProperty();

    switch ( (*iter).second.mode )
    {
    case CosPropertyService::read_only:
    case CosPropertyService::fixed_readonly:
      throw CosPropertyService::ReadOnlyProperty();
    default:
      break;
    }

    (*iter).second.value = property_value;
  }
  else
  {
    property_t p;

    p.name = property_name;
    p.mode = CosPropertyService::normal;
    p.value = property_value;

    if ( m_allowedProperties.size() )
    {
      properties_iterator_t iProps;
      bool bPropertyAllowed = false;

      for ( iProps = m_allowedProperties.begin(); iProps != m_allowedProperties.end(); iProps++ )
      {
        if ( (*iProps).name == property_name )
        {
          bPropertyAllowed = true;
          break;
        }
      }

      if ( ! bPropertyAllowed )
      {
        throw CosPropertyService::UnsupportedProperty();
      }
    }

    if ( m_allowedTypes.size() )
    {
      property_types_iterator_t iTypes;
      bool bTypeAllowed = false;

      for (iTypes = m_allowedTypes.begin(); iTypes != m_allowedTypes.end(); iTypes++)
      {
        if ( (*iTypes) == property_value.type()->kind() )
        {
          bTypeAllowed = true;
          break;
        }
      }

      if ( ! bTypeAllowed )
      {
        throw CosPropertyService::UnsupportedTypeCode();
      }
    }

    m_propmap[property_name] = p;
  }
}

void
PropertySet_i::define_properties(const CosPropertyService::Properties& nproperties)
{
  size_t i;
  property_exceptions_t pe;

  for (i = 0; i < nproperties.length(); i++)
  {
    try
    {
      define_property(nproperties[i].property_name, nproperties[i].property_value);
    }
    catch (CosPropertyService::InvalidPropertyName&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::invalid_property_name;
      e.failing_property_name = CORBA::string_dup(nproperties[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::ConflictingProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::conflicting_property;
      e.failing_property_name = CORBA::string_dup(nproperties[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::UnsupportedTypeCode&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::unsupported_type_code;
      e.failing_property_name = CORBA::string_dup(nproperties[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::UnsupportedProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::unsupported_property;
      e.failing_property_name = CORBA::string_dup(nproperties[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::ReadOnlyProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::read_only_property;
      e.failing_property_name = CORBA::string_dup(nproperties[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (...)
    {
      throw;
    }
  }

  if ( pe.size() )
  {
    CosPropertyService::MultipleExceptions me;
    property_exceptions_iterator_t iter;

    me.exceptions.length(pe.size());

    for ( i = 0, iter = pe.begin(); iter != pe.end(); i++, iter++ )
    {
      me.exceptions[i].reason = (*iter).reason;
      me.exceptions[i].failing_property_name = (*iter).failing_property_name;
    }

    throw me;
  }
}

CORBA::ULong
PropertySet_i::get_number_of_properties()
{
  return m_propmap.size();
}

void
PropertySet_i::get_all_property_names(
  CORBA::ULong how_many,
  CosPropertyService::PropertyNames*& property_names,
  CosPropertyService::PropertyNamesIterator_ptr& rest)
{
  int i;
  int c;
  property_map_iterator_t iter;

  omni_mutex_lock l(m_lock);
  
  if ( m_propmap.size() < how_many )
  {
    rest = CosPropertyService::PropertyNamesIterator::_nil();
  }

  property_names = new CosPropertyService::PropertyNames();

  if ( ! m_propmap.size() )
  {
    property_names->length(0);
    return;
  }

  c = min(m_propmap.size(), how_many);
  property_names->length(c);

  iter = m_propmap.begin();

  for (i = 0; i < c && iter != m_propmap.end(); i++, iter++)
  {
    (*property_names)[i] = CORBA::string_dup( (*iter).second.name.data() );
  }

  if ( how_many < m_propmap.size() )
  {
    PropertyNamesIterator_i* restIter;
    restIter = new PropertyNamesIterator_i(iter, m_propmap.begin(), m_propmap.end());
    restIter->_obj_is_ready(_boa());
    rest = restIter->_this();
  }
}

CORBA::Any*
PropertySet_i::get_property_value(const char*  property_name)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  omni_mutex_lock l(m_lock);
  
  property_map_iterator_t i = find_property(property_name);

  if ( i == m_propmap.end() )
    throw CosPropertyService::PropertyNotFound();

  return new CORBA::Any((*i).second.value);
}

CORBA::Boolean
PropertySet_i::get_properties(
  const CosPropertyService::PropertyNames& property_names,
  CosPropertyService::Properties*& nproperties)
{
  CORBA::Boolean bFoundAll = TRUE;
  size_t i;

  nproperties = new CosPropertyService::Properties();
  nproperties->length(property_names.length());

  for (i = 0; i < property_names.length(); i++)
  {
    property_map_iterator_t iter = find_property(property_names[i]);

    if ( iter != m_propmap.end() )
    {
      (*nproperties)[i].property_name = CORBA::string_dup((*iter).second.name.data());
      (*nproperties)[i].property_value = (*iter).second.value;
    }
    else
    {
      CORBA::Any any(CORBA::_tc_void, NULL);
      (*nproperties)[i].property_name = CORBA::string_dup(property_names[i]);
      (*nproperties)[i].property_value = any;
      bFoundAll = FALSE;
    }
  }

  return bFoundAll;
}

void
PropertySet_i::get_all_properties(
  CORBA::ULong how_many,
  CosPropertyService::Properties*& nproperties,
  CosPropertyService::PropertiesIterator_ptr& rest)
{
  int i;
  int c;
  property_map_iterator_t iter;

  nproperties = new CosPropertyService::Properties();

  if ( m_propmap.size() < how_many )
  {
    rest = CosPropertyService::PropertiesIterator::_nil();
  }

  if ( ! m_propmap.size() )
  {
    nproperties->length(0);
    return;
  }

  c = min(m_propmap.size(), how_many);
  nproperties->length(c);

  iter = m_propmap.begin();

  for (i = 0; i < c && iter != m_propmap.end(); i++, iter++)
  {
    (*nproperties)[i].property_name = CORBA::string_dup( (*iter).second.name.data() );
    (*nproperties)[i].property_value = (*iter).second.value;
  }

  if ( how_many < m_propmap.size() )
  {
    PropertiesIterator_i* restIter;
    restIter = new PropertiesIterator_i(iter, m_propmap.begin(), m_propmap.end());
    restIter->_obj_is_ready(_boa());
    rest = restIter->_this();
  }
}

void
PropertySet_i::delete_property(const char* property_name)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  omni_mutex_lock l(m_lock);
  
  property_map_iterator_t i = find_property(property_name);

  if ( i == m_propmap.end() )
    throw CosPropertyService::PropertyNotFound();

  if ( (*i).second.mode == CosPropertyService::fixed_readonly ||
       (*i).second.mode == CosPropertyService::fixed_normal )
    throw CosPropertyService::FixedProperty();

  m_propmap.erase(i);
}

void
PropertySet_i::delete_properties(const CosPropertyService::PropertyNames &property_names)
{
  size_t i;
  property_exceptions_t pe;

  for ( i = 0; i < property_names.length(); i++ )
  {
    try
    {
      delete_property(property_names[i]);
    }
    catch (CosPropertyService::InvalidPropertyName&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::invalid_property_name;
      e.failing_property_name = CORBA::string_dup(property_names[i]);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::PropertyNotFound&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::property_not_found;
      e.failing_property_name = CORBA::string_dup(property_names[i]);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::FixedProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::fixed_property;
      e.failing_property_name = CORBA::string_dup(property_names[i]);
      pe.insert(pe.end(), e);
    }
    catch (...)
    {
      throw;
    }
  }

  if ( pe.size() )
  {
    CosPropertyService::MultipleExceptions me;
    property_exceptions_iterator_t iter;

    me.exceptions.length(pe.size());

    for ( i = 0, iter = pe.begin(); iter != pe.end(); i++, iter++)
    {
      me.exceptions[i].reason = (*iter).reason;
      me.exceptions[i].failing_property_name = (*iter).failing_property_name;
    }

    throw me;
  }
}

CORBA::Boolean
PropertySet_i::delete_all_properties()
{
  CORBA::Boolean bDeletedAll = TRUE;

  property_map_iterator_t i;

  omni_mutex_lock l(m_lock);
  
  for ( i = m_propmap.begin(); i != m_propmap.end(); i++ )
  {
    if ( (*i).second.mode == CosPropertyService::fixed_readonly ||
         (*i).second.mode == CosPropertyService::fixed_normal )
      bDeletedAll = FALSE;
    else
      m_propmap.erase(i);
  }

  return bDeletedAll;
}

CORBA::Boolean
PropertySet_i::is_property_defined(const char* property_name)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  if ( find_property(property_name) != m_propmap.end() )
    return TRUE;

  return FALSE;
}

CosLifeCycle::LifeCycleObject_ptr
PropertySet_i::copy(CosLifeCycle::FactoryFinder_ptr  there, const CosLifeCycle::Criteria & the_criteria)
{
  throw CosLifeCycle::NotCopyable();
  return CosLifeCycle::LifeCycleObject::_nil();
}

void
PropertySet_i::move(CosLifeCycle::FactoryFinder_ptr  there, const CosLifeCycle::Criteria & the_criteria)
{
  throw CosLifeCycle::NotMovable();
}

void
PropertySet_i::remove()
{
  _dispose();
}

//
// PropertySetDef Implementation
//

PropertySetDef_i::PropertySetDef_i()
: PropertySet_i()
{
}

PropertySetDef_i::PropertySetDef_i( 
  const CosPropertyService::PropertyTypes& allowed_property_types,
  const CosPropertyService::PropertyDefs& allowed_property_defs)
: PropertySet_i()
{
  size_t i;

  omni_mutex_lock l(m_lock);
  
  for (i = 0; i < allowed_property_types.length(); i++)
  {
    m_allowedTypes.insert(m_allowedTypes.end(), allowed_property_types[i]->kind());
  }

  for (i = 0; i < allowed_property_defs.length(); i++)
  {
    property_t p;

    p.name = allowed_property_defs[i].property_name;
    p.mode = allowed_property_defs[i].property_mode;
    p.value <<= 0L;

    m_allowedProperties.insert(m_allowedProperties.end(), p);
  }
}

PropertySetDef_i::PropertySetDef_i(const CosPropertyService::PropertyDefs& initial_property_defs)
: PropertySet_i()
{
  size_t i;

  omni_mutex_lock l(m_lock);
  
  for (i = 0; i < initial_property_defs.length(); i++)
  {
    property_t p;

    p.name = initial_property_defs[i].property_name;
    p.mode = initial_property_defs[i].property_mode;
    p.value = initial_property_defs[i].property_value;

    m_propmap[initial_property_defs[i].property_name] = p;
  }
}

PropertySetDef_i::~PropertySetDef_i()
{
}

void
PropertySetDef_i::get_allowed_property_types(CosPropertyService::PropertyTypes*& property_types)
{
  property_types_iterator_t iter;
  int i;

  property_types = new CosPropertyService::PropertyTypes();
  property_types->length(m_allowedProperties.size());

  for (i = 0, iter = m_allowedTypes.begin(); iter != m_allowedTypes.end(); i++, iter++)
  {
    switch ( (*iter) )
    {
    case CORBA::tk_void:
      (*property_types)[i] = CORBA::_tc_void;
      break;
    case CORBA::tk_short:
      (*property_types)[i] = CORBA::_tc_short;
      break;
    case CORBA::tk_long:
      (*property_types)[i] = CORBA::_tc_long;
      break;
    case CORBA::tk_ushort:
      (*property_types)[i] = CORBA::_tc_ushort;
      break;
    case CORBA::tk_ulong:
      (*property_types)[i] = CORBA::_tc_ulong;
      break;
    case CORBA::tk_float:
      (*property_types)[i] = CORBA::_tc_float;
      break;
    case CORBA::tk_double:
      (*property_types)[i] = CORBA::_tc_double;
      break;
    case CORBA::tk_boolean:
      (*property_types)[i] = CORBA::_tc_boolean;
      break;
    case CORBA::tk_char:
      (*property_types)[i] = CORBA::_tc_char;
      break;
    case CORBA::tk_octet:
      (*property_types)[i] = CORBA::_tc_octet;
      break;
    case CORBA::tk_any:
      (*property_types)[i] = CORBA::_tc_any;
      break;
    case CORBA::tk_TypeCode:
      (*property_types)[i] = CORBA::_tc_TypeCode;
      break;
    case CORBA::tk_Principal:
      (*property_types)[i] = CORBA::_tc_Principal;
      break;
    case CORBA::tk_objref:
      (*property_types)[i] = CORBA::_tc_Object;
      break;
    case CORBA::tk_string:
      (*property_types)[i] = CORBA::_tc_string;
      break;
    case CORBA::tk_null:
    default:
      (*property_types)[i] = CORBA::_tc_null;
      break;
    }
  }
}

void
PropertySetDef_i::get_allowed_properties(CosPropertyService::PropertyDefs*& property_defs)
{
  properties_iterator_t iter;
  int i;

  property_defs = new CosPropertyService::PropertyDefs();
  property_defs->length(m_allowedProperties.size());

  for (i=0, iter=m_allowedProperties.begin(); iter != m_allowedProperties.end(); i++, iter++)
  {
    (*property_defs)[i].property_name = CORBA::string_dup((*iter).name.data());
    (*property_defs)[i].property_mode = (*iter).mode;
    (*property_defs)[i].property_value = (*iter).value;
  }
}

void
PropertySetDef_i::define_property_with_mode(
  const char* property_name,
  const CORBA::Any& property_value,
  CosPropertyService::PropertyModeType property_mode)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  omni_mutex_lock l(m_lock);
  
  property_map_iterator_t iter = find_property(property_name);

  if ( iter != m_propmap.end() )
  {
    if ( (*iter).second.value.type()->kind() != property_value.type()->kind() )
      throw CosPropertyService::ConflictingProperty();

    if ( (*iter).second.mode != property_mode )
      throw CosPropertyService::ConflictingProperty();

    switch ( (*iter).second.mode )
    {
    case CosPropertyService::read_only:
    case CosPropertyService::fixed_readonly:
      throw CosPropertyService::ReadOnlyProperty();
    default:
      break;
    }

    (*iter).second.value = property_value;
  }
  else
  {
    property_t p;

    p.name = property_name;
    p.mode = property_mode;
    p.value = property_value;

    if ( m_allowedProperties.size() )
    {
      properties_iterator_t iProps;
      bool bPropertyAllowed = false;

      for ( iProps = m_allowedProperties.begin(); iProps != m_allowedProperties.end(); iProps++ )
      {
        if ( (*iProps).name == property_name )
        {
          bPropertyAllowed = true;
          break;
        }
      }

      if ( ! bPropertyAllowed )
      {
        throw CosPropertyService::UnsupportedProperty();
      }
    }

    if ( m_allowedTypes.size() )
    {
      property_types_iterator_t iTypes;
      bool bTypeAllowed = false;

      for (iTypes = m_allowedTypes.begin(); iTypes != m_allowedTypes.end(); iTypes++)
      {
        if ( (*iTypes) == property_value.type()->kind() )
        {
          bTypeAllowed = true;
          break;
        }
      }

      if ( ! bTypeAllowed )
      {
        throw CosPropertyService::UnsupportedTypeCode();
      }
    }

    m_propmap[property_name] = p;
  }
}

void
PropertySetDef_i::define_properties_with_modes(const CosPropertyService::PropertyDefs& property_defs)
{
  size_t i;
  property_exceptions_t pe;

  for (i = 0; i < property_defs.length(); i++)
  {
    try
    {
      define_property_with_mode(
        property_defs[i].property_name,
        property_defs[i].property_value,
        property_defs[i].property_mode
      );
    }
    catch (CosPropertyService::InvalidPropertyName&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::invalid_property_name;
      e.failing_property_name = CORBA::string_dup(property_defs[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::ConflictingProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::conflicting_property;
      e.failing_property_name = CORBA::string_dup(property_defs[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::UnsupportedTypeCode&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::unsupported_type_code;
      e.failing_property_name = CORBA::string_dup(property_defs[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::UnsupportedProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::unsupported_property;
      e.failing_property_name = CORBA::string_dup(property_defs[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::UnsupportedMode&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::unsupported_mode;
      e.failing_property_name = CORBA::string_dup(property_defs[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::ReadOnlyProperty&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::read_only_property;
      e.failing_property_name = CORBA::string_dup(property_defs[i].property_name);
      pe.insert(pe.end(), e);
    }
  }

  if ( pe.size() )
  {
    CosPropertyService::MultipleExceptions me;
    property_exceptions_iterator_t iter;

    me.exceptions.length(pe.size());

    for ( i = 0, iter = pe.begin(); iter != pe.end(); i++, iter++ )
    {
      me.exceptions[i].reason = (*iter).reason;
      me.exceptions[i].failing_property_name = (*iter).failing_property_name;
    }

    throw me;
  }
}

CosPropertyService::PropertyModeType
PropertySetDef_i::get_property_mode(const char*  property_name)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  property_map_iterator_t iter = find_property(property_name);

  if ( iter == m_propmap.end() )
    throw CosPropertyService::PropertyNotFound();

  return (*iter).second.mode;
}

CORBA::Boolean
PropertySetDef_i::get_property_modes(const CosPropertyService::PropertyNames& property_names, CosPropertyService::PropertyModes*& property_modes)
{
  CORBA::Boolean bFoundAll = TRUE;
  size_t i;

  property_modes = new CosPropertyService::PropertyModes();
  property_modes->length(property_names.length());

  for ( i = 0; i < property_names.length(); i++ )
  {
    (*property_modes)[i].property_name = CORBA::string_dup(property_names[i]);

    property_map_iterator_t iter = find_property(property_names[i]);

    if ( iter == m_propmap.end() )
    {
      (*property_modes)[i].property_mode = CosPropertyService::undefined;
      bFoundAll = FALSE;
    }
    else
    {
      (*property_modes)[i].property_mode = (*iter).second.mode;
    }
  }

  return bFoundAll;
}

void
PropertySetDef_i::set_property_mode(const char*  property_name, CosPropertyService::PropertyModeType  property_mode)
{
  if ( ! property_name || ! *property_name )
    throw CosPropertyService::InvalidPropertyName();

  omni_mutex_lock l(m_lock);
  
  property_map_iterator_t iter = find_property(property_name);

  if ( iter == m_propmap.end() )
    throw CosPropertyService::PropertyNotFound();

  (*iter).second.mode = property_mode;
}

void
PropertySetDef_i::set_property_modes(const CosPropertyService::PropertyModes& property_modes)
{
  property_exceptions_t pe;
  size_t i;

  for ( i = 0; i < property_modes.length(); i++ )
  {
    try
    {
      set_property_mode( property_modes[i].property_name, property_modes[i].property_mode );
    }
    catch (CosPropertyService::InvalidPropertyName&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::invalid_property_name;
      e.failing_property_name = CORBA::string_dup(property_modes[i].property_name);
      pe.insert(pe.end(), e);
    }
    catch (CosPropertyService::PropertyNotFound&)
    {
      CosPropertyService::PropertyException e;
      e.reason = CosPropertyService::property_not_found;
      e.failing_property_name = CORBA::string_dup(property_modes[i].property_name);
      pe.insert(pe.end(), e);
    }
  }

  if ( pe.size() )
  {
    CosPropertyService::MultipleExceptions me;
    property_exceptions_iterator_t iter;

    me.exceptions.length(pe.size());

    for ( i = 0, iter = pe.begin(); iter != pe.end(); i++, iter++ )
    {
      me.exceptions[i].reason = (*iter).reason;
      me.exceptions[i].failing_property_name = (*iter).failing_property_name;
    }

    throw me;
  }
}

//
// PropertyNamesIterator Implementation
//

PropertyNamesIterator_i::PropertyNamesIterator_i(
  const property_map_iterator_t& iterCur,
  const property_map_iterator_t& iterBegin,
  const property_map_iterator_t& iterEnd)
: m_iterCur(iterCur),
  m_iterFirst(iterBegin),
  m_iterLast(iterEnd)
{
}

PropertyNamesIterator_i::~PropertyNamesIterator_i()
{
}

void PropertyNamesIterator_i::reset()
{
  omni_mutex_lock l(m_lock);
  
  m_iterCur = m_iterFirst;
}

CORBA::Boolean
PropertyNamesIterator_i::next_one(char*& property_name)
{
  CORBA::Boolean bItemPresent = TRUE;

  omni_mutex_lock l(m_lock);
  
  if ( m_iterCur == m_iterLast )
  {
    property_name = CORBA::string_dup("");
    bItemPresent = FALSE;
  }
  else
  {
    property_name = CORBA::string_dup( (*m_iterCur).second.name.data() );
    m_iterCur++;
  }

  return bItemPresent;
}

CORBA::Boolean
PropertyNamesIterator_i::next_n(
  CORBA::ULong how_many,
  CosPropertyService::PropertyNames*& property_names)
{
  CORBA::Boolean bItemPresent = TRUE;
  property_map_iterator_t iter;
  size_t i;
  size_t c;

  omni_mutex_lock l(m_lock);
  
  for ( c = 0, iter = m_iterCur; iter != m_iterLast && c < how_many; iter++ )
    c++;

  property_names = new CosPropertyService::PropertyNames();
  property_names->length(c);

  if ( how_many > 0 && c == 0 )
  {
    bItemPresent = FALSE;
  }
  else
  {
    for ( i = 0; i < c; i++ )
    {
      (*property_names)[i] = CORBA::string_dup( (*m_iterCur).second.name.data() );
      m_iterCur++;
    }
  }

  return bItemPresent;
}

void PropertyNamesIterator_i::destroy()
{
  omni_mutex_lock l(m_lock);
  
  _dispose();
}

//
// PropertiesIterator Implementation
//

PropertiesIterator_i::PropertiesIterator_i(
  const property_map_iterator_t& iterCur,
  const property_map_iterator_t& iterBegin,
  const property_map_iterator_t& iterEnd)
: m_iterCur(iterCur),
  m_iterFirst(iterBegin),
  m_iterLast(iterEnd)
{
}

PropertiesIterator_i::~PropertiesIterator_i()
{
}

void PropertiesIterator_i::reset()
{
  omni_mutex_lock l(m_lock);
  
  m_iterCur = m_iterFirst;
}

CORBA::Boolean
PropertiesIterator_i::next_one(CosPropertyService::Property*& aproperty)
{
  CORBA::Boolean bItemPresent = TRUE;

  omni_mutex_lock l(m_lock);
  
  aproperty = new CosPropertyService::Property();

  if ( m_iterCur == m_iterLast )
  {
    bItemPresent = FALSE;
  }
  else
  {
    aproperty->property_name = CORBA::string_dup( (*m_iterCur).second.name.data() );
    aproperty->property_value = (*m_iterCur).second.value;
    m_iterCur++;
  }

  return bItemPresent;
}

CORBA::Boolean
PropertiesIterator_i::next_n(CORBA::ULong how_many, CosPropertyService::Properties*& nproperties)
{
  CORBA::Boolean bItemPresent = TRUE;
  property_map_iterator_t iter;
  size_t i;
  size_t c;

  omni_mutex_lock l(m_lock);
  
  for ( c = 0, iter = m_iterCur; iter != m_iterLast && c < how_many; iter++ )
    c++;

  nproperties = new CosPropertyService::Properties();
  nproperties->length(c);

  if ( how_many > 0 && c == 0 )
  {
    bItemPresent = FALSE;
  }
  else
  {
    for ( i = 0; i < c; i++ )
    {
      (*nproperties)[i].property_name = CORBA::string_dup( (*m_iterCur).second.name.data() );
      (*nproperties)[i].property_value = (*m_iterCur).second.value;
      m_iterCur++;
    }
  }

  return bItemPresent;
}

void PropertiesIterator_i::destroy()
{
  omni_mutex_lock l(m_lock);
  
  _dispose();
}
