/* --------------------------------------------------
 * Porting to Borland's tools (omniORB 2.6.0)
 * 27.11.1998 -- Mikhail Soukhanov <mailto:mik@astra.relline.ru.>
 *
 * genwrapper.c - generic wrapper around "CC/AR/LINK" programs to perform
 * file name translation (from unix-like to dos-like names).
 *
 * Both GNUWIN32 and OpenNT file names are supported - the first argument
 * to this program specifies which.
 *
 * Insert #include "genwrapper.c" at the top of a wrapper source and
 * implement the WrapArgs() function specific to the program.
 *
 * --------------------------------------------------*/

/*
 *    Copyright (C) 1996, 1997 Olivetti & Oracle Research Laboratory
 *
 *    This file is part of the omniORB library
 *
 *    The omniORB library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Library General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Library General Public License for more details.
 *
 *    You should have received a copy of the GNU Library General Public
 *    License along with this library; if not, write to the Free
 *    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *    02111-1307, USA
 *
 */

#include <windows.h>
#include <stdio.h>
#include <process.h>
#include <string.h>

#ifndef _P_WAIT
#define _P_WAIT P_WAIT
#endif/*_P_WAIT*/

int WrapArgs(int argc, char **argv, char ***wargv1, char ***wargv2);

#define MAX_MOUNTS 256

void GetGnuwin32Mounts(void);
void GetOpenNTMounts(void);
char *TranslateFileName(char *in, int offset);
char *EscapeDoubleQuotes(char *in);
int gnuwin32 = 0;
int opennt = 0;

int main(int argc, char **argv)
{
  int rc;
  int i;

  char **wargv, **wargv_after=NULL;
  int silent;

  if (argc > 1 && strcmp(argv[1], "-gnuwin32") == 0) {
    GetGnuwin32Mounts();
    gnuwin32 = 1;
  } else if (argc > 1 && strcmp(argv[1], "-opennt") == 0) {
    GetOpenNTMounts();
    opennt = 1;
  } else {
    fprintf(stderr,
	    "%s: must specify either -gnuwin32 or -opennt as first argument",
	    argv[0]);
    return 1;
  }

  silent = WrapArgs(argc-2, argv+2, &wargv, &wargv_after);

  if (!silent) {
    printf(wargv[0]);

    for (i = 1; wargv[i]; i++) {
      printf(" %s", wargv[i]);
    }

    printf("\n");
  }

  rc = spawnvp(_P_WAIT,wargv[0],wargv);
  if (rc == -1) {
    fprintf(stderr,"%s: ",wargv[0]);
    perror("");
    return 1;
  }

  if (wargv_after) {
	wargv = wargv_after;

	if (!silent) {
      printf(wargv[0]);

      for (i = 1; wargv[i]; i++) {
        printf(" %s", wargv[i]);
      }

      printf("\n");
    }

    rc = spawnvp(_P_WAIT,wargv[0],wargv);
    if (rc == -1) {
      fprintf(stderr,"%s: ",wargv[0]);
      perror("");
      return 1;
    }
  }

  return rc;
}


char *EscapeDoubleQuotes(char *in)
{
  int i, j;
  char *out = malloc(strlen(in) * 2 + 1);

  j = 0;
  for (i = 0; i < strlen(in); i++) {
    if (in[i] == '"') {
      out[j++] = '\\';
    }
    out[j++] = in[i];
  }

  out[j] = '\0';

  return out;
}


char *dos[MAX_MOUNTS];
char *unix[MAX_MOUNTS];
int nmounts;

char *TranslateFileName(char *in, int offset)
{
  int i;
  char *out = NULL;

  for (i = 0; i < nmounts; i++) {
    if (strncmp(unix[i], &in[offset], strlen(unix[i])) == 0) {
      out = malloc(strlen(in) - strlen(unix[i]) + strlen(dos[i]) + 1);
      strncpy(out, in, offset);
      strcat(out, dos[i]);
      strcat(out, &in[offset + strlen(unix[i])]);
      break;
    }
  }

  if (!out) {
    out = malloc(strlen(in) + 1);
    strcpy(out, in);
  }

  for (i = offset; i < strlen(out); i++) {
    if (out[i] == '/') {
      out[i] = '\\';
    }
  }

  return out;
}

void GetOpenNTMounts(void)
{
  char c;

  nmounts = 0;

  for (c = 'A'; c <= 'Z'; c++) {
    unix[nmounts] = (char *)malloc(5);
    sprintf(unix[nmounts], "//%c/", c);
    dos[nmounts] = (char *)malloc(4);
    sprintf(dos[nmounts], "%c:/", c);
    nmounts++;
  }

  for (c = 'a'; c <= 'z'; c++) {
    unix[nmounts] = (char *)malloc(5);
    sprintf(unix[nmounts], "//%c/", c);
    dos[nmounts] = (char *)malloc(4);
    sprintf(dos[nmounts], "%c:/", c);
    nmounts++;
  }
}

void GetGnuwin32Mounts(void)
{
  HKEY hkey;
  LONG rc;
  int len;
  char key[256];

  for (nmounts = 0; nmounts < MAX_MOUNTS; nmounts++) {
    sprintf(key,
            "Software\\Cygnus Solutions\\CYGWIN.DLL setup\\b15.0\\mounts\\%02X"
            ,nmounts);

    if (RegOpenKeyEx(HKEY_CURRENT_USER,
                     key, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      if (nmounts == 0) {
        printf("Couldn't find registry key '%s' - RegOpenKey error %d\n",
               key, GetLastError());
        exit(1);
      }
      break;
    }

    if (RegQueryValueEx(hkey,"unix",NULL,NULL,NULL,&len) != ERROR_SUCCESS) {
      printf("RegQueryValueEx failed - error %d\n",GetLastError());
      exit(1);
    }
    unix[nmounts] = (char *)malloc(len+1);
    RegQueryValueEx(hkey, "unix", NULL, NULL, unix[nmounts], &len);

    if (RegQueryValueEx(hkey,"native",NULL,NULL,NULL,&len) != ERROR_SUCCESS) {
      printf("RegQueryValueEx failed - error %d\n",GetLastError());
      exit(1);
    }
    if (strcmp(unix[nmounts], "/") == 0) {
      dos[nmounts] = (char *)malloc(len+2);
      RegQueryValueEx(hkey, "native", NULL, NULL, dos[nmounts], &len);
      dos[nmounts][len-1] = '\\';
      dos[nmounts][len] = 0;
    } else {
      dos[nmounts] = (char *)malloc(len+1);
      RegQueryValueEx(hkey, "native", NULL, NULL, dos[nmounts], &len);
    }
  }
}
