/* 
   Copyright (c) 1993-1999 Tumbleweed Software Corp. All Rights Reserved.
*/
//
// $Workfile: $
//
// Description:
//    
//    Interface of the HTTP protocol - Http1.0, 1.1 - rfc 2068.
//
// 
//     $Log: HttpConnection.h,v $
//     Revision 1.16  1999/06/08 01:40:50  edumas
//     	Review:
//     	-> remove TW tabulations,
//     	-> remove French comments. Because this is External, coding
//     	guideline is not allowded ;-).
//
//     Revision 1.15  1999/05/13 00:40:14  jimmy
//     Export label.
//
//     Revision 1.14  1999/05/04 02:15:48  edumas
//     	Add Base64 convertors.
//
//     Revision 1.13  1999/04/28 23:10:00  edumas
//     	HTTP over SSL.
//
//     Revision 1.12  1999/04/23 19:27:05  rshoup
//     Change enum names, enum value names
//
//     Revision 1.11  1999/04/20 17:06:11  edumas
//            Change the constants values GET, POST, etc. in order to
//     compile under Windows, and to prevent some problems with libCore.
//
//     Revision 1.10  1999/04/16 23:51:26  edumas
//     	- Code review modification.
//
//     Revision 1.9  1999/04/14 18:34:27  abonafos
//     Code review
//
//     Revision 1.8  1999/04/14 00:21:44  cgalerne
//     Code Review
//
//     Revision 1.7  1999/04/09 04:52:12  dmitry
//     Code Review
//
//     Revision 1.6  1999/04/09 00:37:08  rshoup
//     Code Review
//
//     Revision 1.5  1999/04/07 17:39:46  edumas
//     	Make Constructor protected.
//
//     Revision 1.4  1999/03/26 01:32:18  edumas
//     	Add SingleConnectionFactoryC.
//
//     Revision 1.3  1999/03/23 03:39:23  edumas
//     	Randy's Review modifications.
//
//     Revision 1.2  1999/03/23 02:06:41  edumas
//     	- Rewrite the interface to allow streaming.
//     	- Use a pure virtual class.
//     	- C++ interface.
//
//     Revision 1.1  1999/02/13 01:52:16  edumas
//         Add the HttpConnection Interface.
//
//     Revision 1.1  1999/02/10 01:54:21  edumas
//         External interfaces for IOP_HTTP.
//
// $Id: HttpConnection.h,v 1.16 1999/06/08 01:40:50 edumas Exp $
//
// Add new header
//
// $NoKeywords: $
//

#ifndef HttpConnectionInterface_h
#define HttpConnectionInterface_h

#ifdef WIN32
#ifdef HTTPCONNECTION_DLL
#define HTTPCONNECTION_LIB_DECL __declspec(dllexport)       // Export from DLL
#else   
#define HTTPCONNECTION_LIB_DECL __declspec(dllimport)       // Import from DLL
#endif  
#else
#define HTTPCONNECTION_LIB_DECL
#endif

#ifdef HAS_SSL

#ifndef SSLCONECTION_TK_H
#include <SSLConnection_tk.h>
#endif
#ifndef SSLCIPHER_TK_H
#include <SSLCipher_tk.h>
#endif

#ifndef false
#define false 0
#endif
#ifndef true
#define true 1
#endif

#endif

class  HTTPCONNECTION_LIB_DECL HttpConnectionInterfaceI
{
 protected:
#ifdef HAS_SSL
  HttpConnectionInterfaceI(const char *host, 
			   unsigned int port = 80,
			   unsigned char ssl=false,
			   const char 	*rootCertificates = NULL,
			   SSLConnectionInterfaceI::ProtocolVersionT ssl_version = SSLConnectionInterfaceI::TK_SSL_Version_2_0,
			   SSLCipherManagerInterfaceI::SSLCipherT *cipher_array = NULL,
			   unsigned int num_cipers = 0){};
#else
  HttpConnectionInterfaceI(const char *host, 
			   unsigned int port = 80){};
#endif

 private:
  // Hide dangerous methods.
  HttpConnectionInterfaceI(const HttpConnectionInterfaceI &other);
  HttpConnectionInterfaceI &operator=(const HttpConnectionInterfaceI &other);

 public:
  virtual ~HttpConnectionInterfaceI();

 public:
	
  // Enum of the methods as defined in RFC 2068
  enum HttpMethodT {
    HTTP_METHOD_UNKNOWN = 0,
    HTTP_METHOD_OPTIONS,
    HTTP_METHOD_GET,
    HTTP_METHOD_HEAD,
    HTTP_METHOD_POST,
    HTTP_METHOD_PUT,
    HTTP_METHOD_DELETE,
    HTTP_METHOD_TRACE
  };

  // This enum contains all the error codes. 
  enum HTTPReturnValueT {
    HTTP_RETURN_VALUE_UNKNOWN = 0,     				// Unknown state.
    HTTP_RETURN_CONNECTION_NOT_IMPLEMENTED = 1,		// The method is not implemented.

    HTTP_RETURN_CONNECTION_BAD_VALUE       = 10,

    /* RFC 2068 status code for HTTP1.1 / 1.0 protocol. */
    HTTP_RETURN_CONTINUE            			= 100,
    HTTP_RETURN_SWITCHING_PROTOCOLS 			= 101,
		
    HTTP_RETURN_OK					= 200,
    HTTP_RETURN_CREATED 				= 201,
    HTTP_RETURN_ACCEPTED 				= 202,
    HTTP_RETURN_NON_AUTHORITATIVE_INFORMATION 	        = 203,
    HTTP_RETURN_NO_CONTENT 				= 204,
    HTTP_RETURN_RESET_CONTENT 				= 205,
    HTTP_RETURN_PARTIAL_CONTENT 			= 206,
		
    HTTP_RETURN_MULTIPLE_CHOICES 			= 300,
    HTTP_RETURN_MOVED_PERMANENTLY 			= 301,
    HTTP_RETURN_MOVED_TEMPORARILY 			= 302,
    HTTP_RETURN_SEE_OTHER 				= 303,
    HTTP_RETURN_NOT_MODIFIED 				= 304,
    HTTP_RETURN_USE_PROXY				= 305,

    HTTP_RETURN_BAD_REQUEST 				= 400,
    HTTP_RETURN_UNAUTHORIZED 				= 401,
    HTTP_RETURN_PAYMENT_REQUIRED 			= 402,
    HTTP_RETURN_FORBIDDEN 				= 403,
    HTTP_RETURN_NOT_FOUND 				= 404,
    HTTP_RETURN_METHOD_NOT_ALLOWED 			= 405,
    HTTP_RETURN_NOT_ACCEPTABLE 				= 406,
    HTTP_RETURN_PROXY_AUTHENTICATION_REQUIRED 		= 407,
    HTTP_RETURN_REQUEST_TIME_OUT 			= 408,
    HTTP_RETURN_CONFLICT 				= 409,
    HTTP_RETURN_GONE 					= 410,
    HTTP_RETURN_LENGTH_REQUIRED 			= 411,
    HTTP_RETURN_PRECONDITION_FAILED 			= 412,
    HTTP_RETURN_REQUEST_ENTITY_TOO_LARGE 		= 413,
    HTTP_RETURN_REQUEST_URI_TOO_LARGE 			= 414,
    HTTP_RETURN_UNSUPPORTED_MEDIA_TYPE 			= 415,

    HTTP_RETURN_INTERNAL_SERVER_ERROR 			= 500,
    HTTP_RETURN_NOT_IMPLEMENTED 			= 501,
    HTTP_RETURN_BAD_GATEWAY 				= 502,
    HTTP_RETURN_SERVICE_UNABAILABLE 			= 503,
    HTTP_RETURN_GATEWAY_TIME_OUT 			= 504,
    HTTP_RETURN_HTTP_VERSION_NOT_SUPPORTED 		= 505		
  };

  /*
    How to use this interface?  Each call has to be done in
    a specific order:
    1- set the header fields,
    2- validate this header and send it using SendRequestHeader
    3- write the body (Begin, Write, End...) (if any - POST for example)
    4- Get the answer header (ReceiveResponseHeader)
    5- Read the inputs (Begin, Read, End...) (if any - GET for example)
  */

  // Header.
  virtual HTTPReturnValueT SetHeader(const char *header, const char *value) = 0;
  virtual HTTPReturnValueT SetHeaderContentLength(unsigned int sz) = 0;
  virtual HTTPReturnValueT SetHeaderContentType(const char *mimetype) = 0;

  /* Flush the header and send the data to the Web Server, using:
     path: to build the full URL,
     method: the HTTP packet method to be used.
  */
  virtual HTTPReturnValueT    SendRequestHeader(const char *path, 
						HttpMethodT method = HTTP_METHOD_GET)= 0;

  /* Streaming Usage: The way of reading/writing raw data (body
     data) is done using a "stream" interface. For example, if you
     want to write in the BodyOutput, you have to following the
     following steps:
     BeginBodyOutput()
     while(data) {
     WriteBodyOutput(data, sizeof(data));
     ...
     put new informations in data.
     }
     EndBodyOutput();

  */

  // The EndBodyOutput perform a complete flush of the data.
  virtual HTTPReturnValueT BeginBodyOutput() = 0;
  virtual HTTPReturnValueT WriteBodyOutput(const char *buf, unsigned int sz,
					   unsigned int &sz_write) = 0;
  virtual HTTPReturnValueT EndBodyOutput() = 0;
	
  // To use only after the EndBodyOutput();
  virtual HTTPReturnValueT ReceiveResponseHeader() = 0;

  /* 
     Return the data or NULL if the header does not exist.
     The user must free the resource.
  */
  virtual char  *GetHeader(const char *header) = 0;
  virtual unsigned int GetHeaderContentLength() = 0;
	
  // The EndBodyInput() perform a flush of the HTTP packet: the
  // HTTP operation is finished.
  virtual HTTPReturnValueT BeginBodyInput() = 0;
  virtual HTTPReturnValueT ReadBodyInput(char *buf, unsigned int sz, 
					 unsigned int &sz_read) = 0;
  virtual HTTPReturnValueT EndBodyInput() = 0;	

  /* Base64 Encoding/Deencoding  - you must free the returned string. */
  virtual char *EncodeBase64(const char *s) = 0;
  virtual char *UnEncodeBase64(const char *s) = 0;
};


//  The owner has to perform an explicit delete on the object created.
class HTTPCONNECTION_LIB_DECL HttpConnectionInterfaceFactoryC {
 protected:
  HttpConnectionInterfaceFactoryC() {};
	
 public:
#ifdef HAS_SSL
  virtual HttpConnectionInterfaceI *CreateHttpConnection(const char *host, 
							 unsigned int port = 443,
							 unsigned char ssl= false,
							 const char *rootCertificates = NULL,
							 SSLConnectionInterfaceI::ProtocolVersionT ssl_version = SSLConnectionInterfaceI::TK_SSL_Version_2_0,
							 SSLCipherManagerInterfaceI::SSLCipherT *cipher_array = NULL,
							 unsigned int num_cipers = 0) = 0;	
#else
  virtual HttpConnectionInterfaceI *CreateHttpConnection(const char *host, 
							 unsigned int port = 80) = 0;	
#endif

  virtual ~HttpConnectionInterfaceFactoryC() {};
 public:	
  // Do not use directly
  static HttpConnectionInterfaceFactoryC *m_ptr;
	
 private:
  // Hide dangerous methods.
  HttpConnectionInterfaceFactoryC(const HttpConnectionInterfaceFactoryC &other);
  HttpConnectionInterfaceFactoryC &operator=(const HttpConnectionInterfaceFactoryC &other);
	
};	
 
extern HTTPCONNECTION_LIB_DECL HttpConnectionInterfaceFactoryC 
*GetHttpConnectionInterfaceManager();

#endif
