//
//    Copyright (C)  1993-1999 Tumbleweed Software Corp.
//
//    This file is part of the omniORB library
//
//    The omniORB library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA

//
// $Workfile: $
//
// Description:
//    
//    Ip Mapping file management.
//
//     $Log: ip_mapping_file.cc,v $
//     Revision 1.9  1999/05/19 15:51:53  edumas
//     	GPL licences.
//
//     Revision 1.8  1999/04/29 22:03:13  edumas
//     	Add some traces.
//
//     Revision 1.7  1999/04/14 02:00:03  edumas
//     	- Bug fix: tabulations in the config file.
//     	- Add traces based on the ORBtraceLevel.
//
//     Revision 1.6  1999/03/26 01:34:54  edumas
//     	Randy's comments added.
//
//     Revision 1.5  1999/03/25 02:06:55  edumas
//     	First set of Randy's comments modifications:
//     	 First set of Randy's comments modifications:
//             - version(), get_default*() are made const when returning const char*
//             - get_port_binded() -> get_port_bound()
//     	- debug code removed,
//     	- cursor removed.
//
//     Revision 1.4  1999/02/19 02:32:26  edumas
//
//     	- Comments update,
//     	- Modify the relaying policies,
//     	- The thread-proxy is now created *only* if we really need to
//     		use it, aka, only if relaying is necessary,
//
//     Revision 1.3  1999/02/16 01:08:05  edumas
//     *** empty log message ***
//
//     Revision 1.2  1999/02/12 03:36:34  edumas
//         IP mapping working integration. Better Corba type handling.
//
//     Revision 1.1  1999/02/11 02:33:33  edumas
//         ip mapping file added.
//
// Add new header
//
// $NoKeywords: $
//
#ifdef BUILD_VERSION 
const static char _rcsid[] = "$Id: ip_mapping_file.cc,v 1.9 1999/05/19 15:51:53 edumas Exp $ for build: " BUILD_VERSION ;
#else
const static char _rcsid[] = "$Id: ip_mapping_file.cc,v 1.9 1999/05/19 15:51:53 edumas Exp $" ;
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <omniORB2/CORBA.h>

#include <IOP_HTTP/IOP_HTTP_external.h>
#include <IOP_HTTP/ip_mapping_file.h>

#define MAX_LINE_SIZE 200

/* */
static inline char ip_match_address(const char *ip_mask, const char *ip);

/* Define the IP mapping file we are going to read. This is a simple
 linked list. */
struct ip_file_mapping_s {
	char         *ip;         /* IP address to be matched, i.e. 12.4.5.* */
	char         *host;       /* Hostname to send data */
	unsigned int  port_http;  /* port of the http server */
	unsigned int  port_https; /* port of the https server */
	char         *pathname;   /* Pathname to reach the remotegateway */
	char         *mimetype;   /* MimeType to use */
	SSL_State     ssl_policy; /* Type SSL_Policy. */

	ip_file_mapping_t next;  
};

ip_mapping_file::ip_mapping_file() 
    :
    pd_ip_file_data(NULL),
    pd_filename(NULL)
{
}

ip_mapping_file::ip_mapping_file(const char *filename_to_load) 
    :
    pd_ip_file_data(NULL)
{
	if (filename_to_load != NULL) {
		pd_filename = CORBA::string_dup(filename_to_load);
	}
}

ip_mapping_file::~ip_mapping_file() {

	delete_data();  

	if (pd_filename != NULL) {
		CORBA::string_free(pd_filename);
	}
}


void 
ip_mapping_file::delete_data() {

	if (pd_ip_file_data != NULL) {
		delete_data_recursive(pd_ip_file_data);
		pd_ip_file_data = NULL;
	}
}

/* *Never* expose this method to the public. */
void  
ip_mapping_file::delete_data_recursive(ip_file_mapping_t ipmap) 
{
  if (ipmap == NULL) {
    return;
  }
  
  
  if (ipmap->next != NULL) {
    delete_data_recursive(ipmap->next);
  }
  
  CORBA::string_free(ipmap->ip);
  CORBA::string_free(ipmap->host);
  CORBA::string_free(ipmap->pathname);
  CORBA::string_free(ipmap->mimetype);
  
  delete ipmap;
}

unsigned char
ip_mapping_file::is_mapping_feature() {
	return  pd_ip_file_data != NULL;
}

unsigned int
ip_mapping_file::load_file(const char *filename_to_load) {
	char  buf[MAX_LINE_SIZE];
	FILE *fp = NULL;
	
	/* NOTE: 
	   The argument (filename_to_load) is optional. If it omited 
	   or if it is NULL, we are going to try to reload the file.
	   This is a very convenient way to sync data. */
	if (filename_to_load != NULL) {
		if (pd_filename != NULL) {
			CORBA::string_free(pd_filename);
		}
		pd_filename = CORBA::string_dup(filename_to_load);
	}
	
	if ((fp = fopen(pd_filename, "r")) == NULL) {
		if (omniORB::traceLevel > 5) {
			omniORB::log << " [IOP_HTTP]: Cannot load ip_mapping file: " << pd_filename << "\n";
			omniORB::log.flush();
		}
		return 0;
	}
	
	/* Clean the existing data structure */
	if (pd_ip_file_data != NULL) {
		delete_data();
	}
	unsigned int nb_item = 0;
  
	/* Read the text file line by line, and parse it. This is portable
       Nt/Unix. */
	while (fgets(buf, MAX_LINE_SIZE, fp) != NULL) {

		/* Create a new item. */
		ip_file_mapping_t fm = new struct ip_file_mapping_s;
		memset(fm, 0, sizeof(struct ip_file_mapping_s));
		size_t pos=0;
		/* Read each component */
		fm->ip         = read_word(buf, &pos);
		fm->host       = read_word(buf, &pos);
		fm->port_http  = read_digit(buf, &pos);
		fm->port_https = read_digit(buf, &pos);
		fm->pathname   = read_word(buf, &pos);
		fm->mimetype   = read_word(buf, &pos);
		fm->ssl_policy = (SSL_State) read_digit(buf, &pos);

		/* End of parsing of the line. Let's check if data are valids. If
		   this is not the case, we ignore the line. */
		
		if ((fm->ip == NULL)                || (fm->host == NULL) || 
		    (!fm->port_http)                || (!fm->port_https) ||
		    (fm->pathname == NULL)          || (fm->mimetype == NULL) ||
		    (fm->ssl_policy == SSL_UNKNOWN) || (fm->ssl_policy >= SSL_LAST_VALUE)) {
		  
			/* Always delete data allocated. */
			delete_data_recursive(fm);
			fm = NULL;
			if (omniORB::traceLevel >= 5) {
				omniORB::log << " [IOP_HTTP]: Problem in the IP_Mapping file while checking data... Maybe a comment?\n";
				omniORB::log.flush();
			}
		} else {
		  
		  /* Add the new ip_mapping in the list. Standard head insert. */
		  if (pd_ip_file_data == NULL) {
		    pd_ip_file_data = fm;
		  } else {
		    fm->next=pd_ip_file_data -> next;
		    pd_ip_file_data -> next = fm;
		  }
		  /* Increase the number of valid item stored in the list */
		  ++nb_item;
		}
	} /* End of while */
	fclose(fp);
	
	return nb_item;
}

/* Extract a word from a string. */
char *
ip_mapping_file::read_word(const char *ptr, size_t *start)
{
	if ((ptr == NULL) || (start == NULL)) {
		return NULL;
	}
	

	/* Start at the good offset */
	if (*start > strlen(ptr)) {
		return NULL;
	}
  
	char *buf = (char *) &ptr[*start];
	/* Remove the begining spaces and tabulations. */
	while (((*buf == ' ') || (*buf == '\t')) && (*buf != '\0')) {
		++(*start);
		++buf;
	}
  
	/* Count the number of chars before the end of the word. 
	   Note: remember that # is the comment sign. */
	char *buf_count = buf;
	size_t       sz = 0;  
	while ((*buf_count != ' ')  && (*buf_count != '\t') && (*buf_count != '\0') && 
		   (*buf_count != '\n') && (*buf_count != '\r') &&
		   (*buf_count != '#')) {
		++sz;
		++(*start);
		++buf_count;
	}

	/* Nothing to consume. */
	if (sz == 0) {
		return NULL;
	}
  
	// string_alloc allocate sz+1
	char *return_buffer = CORBA::string_alloc(sz);
	strncpy(return_buffer, buf, sz);
	return_buffer[sz] = '\0';

	return return_buffer;
}

        
unsigned int
ip_mapping_file::read_digit(const char *ptr, size_t *start) {
	char *buf = read_word(ptr, start);
	int   ret;
  
	if (buf == NULL) {
		return 0;
	}
  
	ret = atoi(buf);  
	CORBA::string_free(buf);
  
	return ret;
}

/* Access methods for ipf data. */
const char   *
ip_mapping_file::get_ip(ip_file_mapping_t ipf) 
{
	if (ipf==NULL) {
		return NULL;  
	}
		
	return ipf->ip;
}

const char   *
ip_mapping_file::get_host(ip_file_mapping_t ipf)
{
	if (ipf==NULL) {
		return NULL;  
	}
  
	return ipf->host;
}


unsigned int  
ip_mapping_file::get_port_http(ip_file_mapping_t ipf)
{
	if (ipf==NULL) {
		return NULL;  
	}
	
	return ipf->port_http;
}

unsigned int  
ip_mapping_file::get_port_https(ip_file_mapping_t ipf)
{
	if (ipf==NULL) {
		return NULL;  
	}
	return ipf->port_https;
}

const char *
ip_mapping_file::get_pathname(ip_file_mapping_t ipf)
{
	if (ipf==NULL) {
		return NULL;
	}
	
	return ipf->pathname;
}

const char   *
ip_mapping_file::get_mimetype(ip_file_mapping_t ipf)
{
	if (ipf==NULL) {
		return NULL;
	}
	
	return ipf->mimetype;
}

SSL_State     
ip_mapping_file::get_ssl_policy (ip_file_mapping_t ipf)
{
	if (ipf==NULL) {
		return SSL_UNKNOWN; 
	}
	return ipf->ssl_policy;
}

/* Try to find a match. */
const ip_file_mapping_t 
ip_mapping_file::ip_match(const char *ip) {
  
	if (ip==NULL) {
		return NULL;
	}

	/* Scan the current ip_mapping table */
	for (ip_file_mapping_t ipf = pd_ip_file_data ; ipf != NULL ; ipf = ipf->next) {
		if (ip_match_address(ipf->ip, ip)) {
			return ipf;
		}
	}
	return NULL;
}

/* Very simple parser - the only intersting chars to check are the
   '*' and '.'.

   Note that an IP address can be a.b.c.d where a,b,c,d may be * for
   user convenience */

static inline char 
ip_match_address(const char *ip_mask, const char *ip) {
	char         *ptr_mask;
	char         *ptr_ip;
	unsigned int  dot;

	if ((ip_mask == NULL) || (ip == NULL)) {
		return 0;
	}
  
	dot      = 0;
	ptr_mask = (char *)ip_mask;
	ptr_ip   = (char *)ip;
    
	while ((*ptr_mask != '\0') && (*ptr_ip != '\0')) {
    
		if (*ptr_mask == '*') {
			/* Special case for the third dot: a.b.c.* 
			   We have already matched a.b.c. => it matches */
			if (dot == 3)
				return 1;
      
			/* Move ptr_ip to the next '.' */
			while ((*ptr_ip != '\0') && (*ptr_ip != '.')) {
				++ptr_ip;
			}
			
			if (ptr_ip == '\0') { /* Missing dot - bad ip address */ 
				return 0;
			}

		} else {
			/* Standard char comparaison */
			if (*ptr_mask != *ptr_ip) {
				return 0;
			}
      
			++ptr_ip;
      
			if (*ptr_mask == '.') {
				++dot;
			}
		}
		++ptr_mask;
	}

	/* For bad addresses (a.b is not valid) */
	if (dot < 3) {
		return 0;
	}
	return 1;
}
