//
//    Copyright (C)  1993-1999 Tumbleweed Software Corp.
//
//    This file is part of the omniORB library
//
//    The omniORB library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA

//
// $Workfile: $
//
// Description:
//    
//    IOP/HTTP convertor.
//
//      NOTE: because of conflics between TW toolkit/objecspace and ORB, 
//    we must not use both of them in the same file.
//
//     $Log: IOP_HTTP.cc,v $
//     Revision 1.55  1999/06/10 03:21:44  edumas
//     	Increase the level of ORBtraceLevel for all the messages to
//     	be displayed, except for extreme critical problems (bad config
//     	file,etc.).
//
//     Revision 1.54  1999/06/08 20:27:59  edumas
//     	Bug #3389.
//
//     	If a HTTP proxy failes while we are reading the BodyInput,
//     	we may receive a set of "0 byte" read. Add a retry system.
//
//     Revision 1.53  1999/06/08 18:43:19  edumas
//     	Misc error checks.
//
//     Revision 1.52  1999/06/08 03:06:58  edumas
//     	Some code cleanup.
//
//     Revision 1.51  1999/06/04 16:13:16  edumas
//     	Check the return value of the Read operation on the HTTP connection
//     	in order to prevent looping.
//
//     Revision 1.50  1999/05/27 23:34:33  edumas
//     	Do not use proxy when SSL is activated.
//
//     Revision 1.49  1999/05/18 18:53:38  edumas
//     	Fix in the ReadBodyInput operation.
//
//     Revision 1.48  1999/05/18 01:36:33  edumas
//     	Misc.
//
//     	- In the case the proxy/web server return the Connection: close
//     	field, we will close and reopen the connection,
//     	- Fix some problems with path/proxy/ssl stuff.
//
//     Revision 1.47  1999/05/17 22:42:19  edumas
//     	- Fix a typo bug for proxy (http/https)
//     	- Optimisation in network i/o.
//
//     Revision 1.46  1999/05/12 07:32:05  jimmy
//     SSL is working on NT!
//
//     Revision 1.45  1999/05/11 21:47:56  edumas
//     	NT portability: errno -> GETERRNO()
//
//     Revision 1.44  1999/05/11 21:06:14  edumas
//     	Detect the ORB closed its internal socket to ower socket entry
//     	earlier - this way, we should be quicker in communications.
//
//     Revision 1.43  1999/05/11 00:30:43  rshoup
//     NT compilation
//
//     Revision 1.42  1999/05/06 01:22:00  edumas
//     	IOP/SSL/HTTP
//
//     Revision 1.41  1999/05/05 21:41:53  edumas
//     	- Add the Root Certificate file reference.
//     	- Cleanup of the SSL configuration.
//     	- Add the SSL specific code between HAS_SSL defines...
//     	- The library can be compiled without SSL. In this case an
//     	exception is thrown, and an error message sent.
//     	- Check of the SSL/non SSL parameters.
//
//     Revision 1.40  1999/05/04 02:49:10  edumas
//     	Proxy with Authentification part1.
//
//     Revision 1.39  1999/04/30 22:21:33  edumas
//     	-> Add two news entries in the config file in order to
//     	take care of the "login/password" proxies.
//     	-> increase the limit of the number of items availables in
//     	the omniORB.cfg config file.
//
//     Revision 1.38  1999/04/29 22:03:13  edumas
//     	Add some traces.
//
//     Revision 1.37  1999/04/29 18:17:00  edumas
//
//     	Proxy - step 1 :
//     	-> add entries in omniORB.cfg (no authentification),
//     	-> forward socket connection to the proxy instead of the
//     		Web server,
//     	-> Non-Authentification proxy implementation.
//     		(Tested with Squid proxy - Unix cache/proxy).
//
//     Revision 1.36  1999/04/29 00:53:50  rshoup
//     IIOP/HTTP NT compilation
//
//     Revision 1.35  1999/04/29 00:35:57  edumas
//     	NT great portability:
//     		- close() -> CLOSESOCKET()
//     		- ENOTCONN -> WSAENOTCONN
//
//     Revision 1.34  1999/04/26 18:37:35  edumas
//     	-> TCP Wrapper modification in order to take a socket descriptor
//     		as parameter
//     	-> Add the TCP/Wrapper checks for the IOP_HTTP proxy feature.
//
//     Revision 1.33  1999/04/26 18:03:10  rshoup
//     HTTPReturnValuesT -> HTTPReturnValueT
//     HTTPMethodsT -> HttpMethodT
//
//     Revision 1.32  1999/04/20 17:06:29  edumas
//            Change the constants values GET, POST, etc. in order to
//     compile under Windows, and to prevent some problems with libCore.
//
//     Revision 1.31  1999/04/17 00:00:12  edumas
//     	-> update of the code with the interfaces modifications
//     	-> create a dummystub library to be used by people who do not
//     		want the HTTP encapsulation (omniNames, etc.)
//     	-> changes in the httpconnection directory in order to
//     		manage multiple libraries.
//
//     Revision 1.30  1999/04/14 04:06:04  edumas
//     	Remove ugly warning when the ORB disconnect the socket.
//
//     Revision 1.29  1999/04/07 17:44:20  edumas
//      Encapsultate the global_iop_http object as a static member of IOP_HTTP.
//     		(Randy's comment).
//
//     Revision 1.28  1999/04/06 00:22:49  edumas
//
//     	Add IOP_HTTPS_PORT config file field.
//
//     Revision 1.27  1999/04/01 03:20:34  edumas
//     	- Thread locking.
//     	- Memory management
//     	- Next set of Randy's comments.
//
//     Revision 1.26  1999/03/26 19:49:52  edumas
//     	Build fix.
//
//     Revision 1.25  1999/03/26 19:13:07  edumas
//     	- add stop_proxy() method,
//     	- Comments.
//
//     Revision 1.24  1999/03/26 01:34:54  edumas
//     	Randy's comments added.
//
//     Revision 1.23  1999/03/25 02:00:59  edumas
//     	First set of Randy's comments modification:
//     	- version() is made const.
//
//     Revision 1.22  1999/03/25 00:08:26  edumas
//     	Update the changes done in IOP_HTTP_handshake.h.
//
//     Revision 1.21  1999/03/24 02:02:56  edumas
//     	- Fix the build for non TW plateforms.
//     	- Allow the specification of an other HTTP toolkit.
//     	- Modify the linux build configuration.
//
//     Revision 1.20  1999/03/23 03:47:18  edumas
//     	- Adaption from the interfaces changes,
//     	- Fix an error recovery.
//
//     Revision 1.19  1999/03/23 02:10:18  edumas
//     	Use the new HTTP interface. It allows :
//     		- streaming of input/output on http buffers,
//     		- *one* http connection by IOP/HTTP thread.
//     		It should really speed up the connection... and should
//     		make SSL quicker and easier to implement (derivation?).
//     	Update of the makefiles.
//
//     Revision 1.18  1999/03/22 22:54:09  edumas
//     	Memory leak fix.
//
//     Revision 1.17  1999/03/22 20:43:21  edumas
//     	- Fix the chunck problem. We are now using the GIOP packet header
//     	and we reconstruct the "original" packet when it is supposed to
//     	be send in multiple chunchs. This way, it solves the answer/no
//     	answer.
//     	- Simplification of the code.
//     	- Cleanup.
//
//     Revision 1.16  1999/03/18 02:14:25  edumas
//     	- Better socket error management
//     	- tmp fix.
//
//     Revision 1.15  1999/03/17 04:31:09  edumas
//     	- Change the size of the buffers,
//     	- memory management optimisation.
//
//     Revision 1.14  1999/03/16 02:56:46  edumas
//     	Fix the way we get the binded port of the thread. In the case
//     the port is already used, jump to the next one.
//
//     Revision 1.13  1999/03/13 01:26:42  edumas
//     	Temporary fix...
//
//     Revision 1.12  1999/03/12 23:03:01  edumas
//     	Use omniORB trace facilities.
//
//     Revision 1.11  1999/03/10 00:49:48  edumas
//     	Handshaking is now done in a portable way.
//
//     Revision 1.10  1999/02/22 19:52:52  edumas
//        Handshake with the RemoteGateway proxy using different endian model fixed.
//
//     Revision 1.9  1999/02/19 02:32:26  edumas
//
//     	- Comments update,
//     	- Modify the relaying policies,
//     	- The thread-proxy is now created *only* if we really need to
//     		use it, aka, only if relaying is necessary,
//
//     Revision 1.8  1999/02/17 22:04:06  edumas
//     	NT compilation of libomniORB2 + IOP/HTTP.
//
//     Revision 1.7  1999/02/16 19:14:46  edumas
//         - Comments updates.
//
//     Revision 1.6  1999/02/16 01:08:05  edumas
//     *** empty log message ***
//
//     Revision 1.5  1999/02/13 02:02:59  edumas
//         - CORBA:: exceptions,
//         - errors are loged, using the ORB log features.
//
//     Revision 1.4  1999/02/12 03:36:34  edumas
//         IP mapping working integration. Better Corba type handling.
//
//     Revision 1.3  1999/02/11 02:33:33  edumas
//         ip mapping file added.
//
//     Revision 1.2  1999/02/10 01:58:44  edumas
//         IOP_HTTP implementation. This first commit is only a "security" in
//     order to keep track of this version. Should not be used.
//
//     Revision 1.1.1.1  1999/02/09 16:30:17  edumas
//     omniORB 2.6.1 with IOP/HTTP
//
//
// Add new header
//
// $NoKeywords: $
//

#ifdef BUILD_VERSION 
const static char _rcsid[] = "$Id: IOP_HTTP.cc,v 1.55 1999/06/10 03:21:44 edumas Exp $ for build: " BUILD_VERSION ;
#else
const static char _rcsid[] = "$Id: IOP_HTTP.cc,v 1.55 1999/06/10 03:21:44 edumas Exp $" ;
#endif

#include <omniORB2/CORBA.h>
#include <omnithread.h>

/* Portability headers - come from std ORB code source */
#if defined(__WIN32__)

#include <winsock.h>
/* Windows is wonderful - specific W$ errno */
#define GETERRNO()				WSAGetLastError()
#define EADDRINUSE				WSAEADDRINUSE
#define ENOTCONN				WSAENOTCONN
// !!!TODO:  Make this define portable!!!
#define MSG_WAITALL				0
#include <sys/types.h>

#define RC_INADDR_NONE			INADDR_NONE
#define RC_INVALID_SOCKET		INVALID_SOCKET
#define RC_SOCKET_ERROR			SOCKET_ERROR
#define INETSOCKET				PF_INET
#define CLOSESOCKET(sock)		closesocket(sock)
#define SHUTDOWNSOCKET(sock)	::shutdown(sock,2)
#else

#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#if defined(__sunos__) && defined(__sparc__) && __OSVERSION__ >= 5
#include <sys/types.h>
#include <fcntl.h>
#endif

#define GETERRNO() errno
#define RC_INADDR_NONE     ((CORBA::ULong)-1)
#define RC_INVALID_SOCKET  (-1)
#define RC_SOCKET_ERROR    (-1)
#define INETSOCKET         AF_INET
#define CLOSESOCKET(sock)   close(sock)
#if defined(__sunsos__) && defined(__sparc__) && __OSVERSION__ >= 5
#define SHUTDOWNSOCKET(sock)  ::shutdown(sock,2)
#elif defined(__osf1__) && defined(__alpha__)
#define SHUTDOWNSOCKET(sock)  ::shutdown(sock,2)
#else
  // XXX none of the above, calling shutdown() may not have the
  // desired effect.
#define SHUTDOWNSOCKET(sock)  ::shutdown(sock,2)
#endif
#endif

#include <sys/types.h>
#include <errno.h>
#include <stdio.h>

// Wrappers
#include <libcWrapper.h>
#include <gatekeeper.h>

#include <HttpConnection.h>
/* IOP_HTTP includes */
#include <IOP_HTTP/IOP_HTTP_handshake.h>
#include <IOP_HTTP/IOP_HTTP_external.h>
#include <IOP_HTTP/ip_mapping_file.h>
#include "IOP_HTTP.h"

static const char *IOP_HTTP_VERSION="IOP_HTTP version 1.0 - Tumbleweed Software";

enum { DEFAULT_PORT_TO_BIND = 5678 }; /* Default port */
enum { LISTENQSIZE          = 150 };  /* Maximum socket connection */
enum { MAX_BIND_RETRY       = 10 };
enum { MAX_LOOP_RETRY       = 1024 };

// This thread is *unique*
static omni_thread   *pd_proxy_thread = NULL;
static omni_mutex     proxy_thread_lock;

#define HTTP_DUMMY_HOSTNAME "dummy"
#define HTTP_DUMMY_PATH      "/"
#define HTTP_DUMMY_MIME_TYPE ""

enum { HTTP_DUMMY_PORT = 0 };
enum { HTTPS_DUMMY_PORT = 0 };

/* This is the global_iop_http object - used by the ORB internaly. */
iop_http *iop_http::global_iop_http = NULL;

iop_http &IopHttpManager()
{
    if (iop_http::global_iop_http == NULL) {
	iop_http::global_iop_http = new iop_http();
    }
    
    return *iop_http::global_iop_http;
}

/* */
iop_http::iop_http() 
  :
  pd_default_port(HTTP_DUMMY_PORT),
  pd_SSL_requiered(SSL_UNKNOWN),
  pd_port_to_bind(DEFAULT_PORT_TO_BIND),
  pd_default_hostname(CORBA::string_dup(HTTP_DUMMY_HOSTNAME)),
  pd_default_path(CORBA::string_dup(HTTP_DUMMY_PATH)),
  pd_default_mime_type(CORBA::string_dup(HTTP_DUMMY_MIME_TYPE)),
  pd_default_port_https(HTTPS_DUMMY_PORT),
  pd_remote_proxy_hostname(NULL),
  pd_remote_proxy_login(NULL),
  pd_remote_proxy_password(NULL),
  pd_remote_proxy_auth(NULL),
  pd_SSL_root_certificates(NULL)
{ 
  if (omniORB::traceLevel >= 5) {
    omniORB::log << version() << "\n";
    omniORB::log.flush();
  }

}

iop_http::~iop_http() {
  // Stop the proxy first.
  if (pd_proxy_thread != NULL) {
    stop_proxy();
  }

  CORBA::string_free(pd_default_hostname);
  CORBA::string_free(pd_default_path);
  CORBA::string_free(pd_default_mime_type);

  CORBA::string_free(pd_remote_proxy_hostname);
  CORBA::string_free(pd_remote_proxy_login);
  CORBA::string_free(pd_remote_proxy_password);
  CORBA::string_free(pd_remote_proxy_auth);
  
  CORBA::string_free(pd_SSL_root_certificates);
}

/* Check if default values have been redefined. */
CORBA::Boolean
iop_http::have_to_relay() {

  if ((!strcmp(pd_default_hostname, HTTP_DUMMY_HOSTNAME)) ||
      (pd_default_port == HTTP_DUMMY_PORT) ||
      (!strcmp(pd_default_path, HTTP_DUMMY_PATH)) ||
      (!strcmp(pd_default_mime_type, HTTP_DUMMY_MIME_TYPE)))
    return 0;

  return 1;
}

/* This method is supposed to be in a separate thread, or can be use
   in a standalone software. As main loop. */
int
iop_http::thread_listen_bind_socket(void *incoming_fd) {
  int                listenfd;
  struct sockaddr_in cliaddr;
  int                addrlen = sizeof(cliaddr);
  int                connfd;
  
  if (incoming_fd == NULL) {
    return -1;
  }

  listenfd = (int)incoming_fd;

  /* For each new socket connection, create a new thread which is
     going to deal with the read/write data.*/
  while(1) {
    if ((connfd = ::accept(listenfd, (struct sockaddr *)&cliaddr, &addrlen)) == RC_INVALID_SOCKET) {
      if (omniORB::traceLevel >= 5) {
	omniORB::log << " [IOP_HTTP]: Cannot accept new socket \n";
	omniORB::log << " [IOP_HTTP]: accept() " << strerror(GETERRNO()) << "\n.";
	omniORB::log.flush();
      }
      proxy_thread_lock.lock();
      pd_proxy_thread = NULL;
      proxy_thread_lock.unlock();

      throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
    }
    
    // Check if the remote host is allowed to speack with us.
    if (!gateKeeper::checkConnect(connfd)) {
      if (omniORB::traceLevel >= 5) {
	omniORB::log << " [IOP_HTTP]: Rejecting connection!\n";
	omniORB::log.flush();
      }
      CLOSESOCKET(connfd);
    } else {
      omni_thread::create(iop_http_proxy_s, (void *)connfd, omni_thread::PRIORITY_NORMAL);  
    }
    
  }
  
  return 0;
}

void
iop_http::stop_proxy()
{
  if (pd_proxy_thread == NULL) {
    return;
  }
  proxy_thread_lock.lock();

  // Why this f... is not working
  //pd_proxy_thread->exit();

  pd_proxy_thread = NULL;

  proxy_thread_lock.unlock();
}

/* Create socket, bind it and create a new thread */
void
iop_http::start_proxy() {  

  /* Start the proxy only if necessary */
  proxy_thread_lock.lock();

  if (pd_proxy_thread != NULL) {
    proxy_thread_lock.unlock();
    return;
  }
  
  int listenfd;  
  /* Create the socket, bind it, listen it! */
  if ((listenfd = ::socket(AF_INET, SOCK_STREAM, 0)) == RC_INVALID_SOCKET) {
    if (omniORB::traceLevel >= 5) {
      omniORB::log << " [IOP_HTTP]: Cannot create a new socket \n";
      omniORB::log << " [IOP_HTTP]: socket() " << strerror(GETERRNO()) << "\n";
      omniORB::log.flush();
    }
    proxy_thread_lock.unlock();    
    throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
  }

  int    retry_bind = 0;
  while (retry_bind != MAX_BIND_RETRY) {
    /* Configure the type of socket binding. */
    struct sockaddr_in servaddr;
    servaddr.sin_family      = AF_INET;
    servaddr.sin_addr.s_addr = htonl(INADDR_ANY);
    servaddr.sin_port        = htons(pd_port_to_bind);
  
    if (::bind(listenfd, (const struct sockaddr*)&servaddr, 
             sizeof(servaddr))== RC_SOCKET_ERROR) {
      /* We have not been able to bind the socket. Test if
         the address is already used. In this case, we 
         try an other one. */
	if (omniORB::traceLevel >= 5) {
	  omniORB::log << " [IOP_HTTP]: Received error " << GETERRNO() << " trying to bind a new socket on port " << pd_port_to_bind << "\n.";
	  omniORB::log.flush();
	}
	
      if ((GETERRNO() == EADDRINUSE) && (retry_bind != MAX_BIND_RETRY)) {
	if (omniORB::traceLevel >= 5) {
	  omniORB::log << " [IOP_HTTP]: Cannot bind a new socket on port " << pd_port_to_bind << "\n.";
	  omniORB::log << " [IOP_HTTP]: Retrying on " << pd_port_to_bind + 1 << "\n.";
	  omniORB::log.flush();
	}
	
        ++retry_bind;
        ++pd_port_to_bind;
      } else {
        /* Serious error, or retry period is over. */
        retry_bind == -1;
        break;
      }
    } else {
      /* We succeed to bind the socket */
      retry_bind = MAX_BIND_RETRY;
    }
  }

  /* Impossible to bind the socket. Exception to be throwned */
  if (retry_bind == -1) {
    CLOSESOCKET(listenfd);

    if (omniORB::traceLevel >= 5) {
      omniORB::log << " [IOP_HTTP]: Cannot bind the socket.\n";
      omniORB::log << " [IOP_HTTP]: bind() " << strerror(GETERRNO()) << "\n.";
      omniORB::log.flush();
    }
    proxy_thread_lock.unlock();
    throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
  }
  
  /* Listen for the connection on this port */
  if (::listen(listenfd, LISTENQSIZE) == RC_SOCKET_ERROR) {
    if (omniORB::traceLevel >= 5) {
      omniORB::log << " [IOP_HTTP]: Cannot listen the socket.\n";
      omniORB::log << " [IOP_HTTP]: listen() " << strerror(GETERRNO()) << "\n.";
      omniORB::log.flush();
    }
    
    CLOSESOCKET(listenfd);
    proxy_thread_lock.unlock();
    throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
  } 
  
  /* We are now ready to accept any connections on this socket. 
     In order to perform the loop to accept multiple connection,
     we create a separate thread. */
  pd_proxy_thread = omni_thread::create(listen_bind_socket_s, (void *)listenfd, omni_thread::PRIORITY_NORMAL);
  proxy_thread_lock.unlock();
}

const char *
iop_http::version() const {
  return (const char*)IOP_HTTP_VERSION;
}   

const CORBA::Boolean
iop_http::is_valid_port(unsigned int p) {
  if ((!p) || (p >= 65536))
    return 0;
  
  return 1;
}

/* Can be called from : the init-file object, or can be used
   through a cover from the client side. */
int 
iop_http::init_file_value(const char *entryname, const char *data)
{
  // This lock is used to protect the modifications of the data.
  omni_mutex data_lock;
  int check_me;

  data_lock.lock();

  if (!strcmp(entryname, IOP_HTTP_PORT)) {
    sscanf(data,"%d", &check_me);

    if (!is_valid_port(check_me)) {
      if (omniORB::traceLevel > 0) {
	omniORB::log << " [IOP_HTTP]: Bad IOP_HTTP_PORT value in config file!.\n";
	omniORB::log.flush();
      }
      
      data_lock.unlock();
      return -1;
    }
    else
      pd_default_port = check_me;

  } else if (!strcmp(entryname, IOP_HTTPS_PORT)) {
    sscanf(data,"%d", &check_me);

    if (!is_valid_port(check_me)) {
      if (omniORB::traceLevel > 0) {
	omniORB::log << " [IOP_HTTP]: Bad IOP_HTTPS_PORT value in config file!.\n";
	omniORB::log.flush();
      }
      
      data_lock.unlock();
      return -1;
    }
    else
      pd_default_port_https = check_me;

  } else if (!strcmp(entryname, IOP_HTTP_BIND_PORT)) {
    sscanf(data,"%d", &check_me);
    
    if (!is_valid_port(check_me)) {
      if (omniORB::traceLevel > 0) {
	omniORB::log << " [IOP_HTTP]: Bad IOP_HTTP_BIND_PORT value in config file!.\n";
	omniORB::log.flush();
      }
      
      data_lock.unlock();
      return -1;
    } else {
      pd_port_to_bind = check_me;
    }    

  } else if (!strcmp(entryname, IOP_HTTP_HOST)) {
    pd_default_hostname = CORBA::string_dup(data);
    
  } else if (!strcmp(entryname, IOP_HTTP_REMOTE_PROXY_HOSTNAME)) {
    pd_remote_proxy_hostname = CORBA::string_dup(data);
    
  } else if (!strcmp(entryname, IOP_HTTP_REMOTE_PROXY_LOGIN)) {
    pd_remote_proxy_login = CORBA::string_dup(data);
    
  } else if (!strcmp(entryname, IOP_HTTP_REMOTE_PROXY_PASSWORD)) {
    pd_remote_proxy_password = CORBA::string_dup(data);
    
  } else if (!strcmp(entryname, IOP_HTTP_SSL_ROOTCERT_FILE)) {
    pd_SSL_root_certificates = CORBA::string_dup(data);
    
  } else if (!strcmp(entryname, IOP_HTTP_REMOTE_PROXY_PORT)) {
    sscanf(data,"%d", &check_me);
    if (!is_valid_port(check_me)) {
      if (omniORB::traceLevel > 0) {
	omniORB::log << " [IOP_HTTP]: Bad IOP_HTTP_REMOTE_PROXY_PORT value in config file!.\n";
	omniORB::log.flush();
      }
      
      data_lock.unlock();
      return -1;
    } else {
      pd_remote_proxy_port = check_me;
    }

  } else if (!strcmp(entryname, IOP_HTTP_PATH)) {
    pd_default_path = CORBA::string_dup(data);

  } else if (!strcmp(entryname, IOP_HTTP_MIME_TYPE)) {
    pd_default_mime_type = CORBA::string_dup(data);

  } else if (!strcmp(entryname, IOP_HTTP_IP_MAPPING_FILENAME)) {
    if (!ipf.load_file(data)) {
      return -1;
    }
  } else if (!strcmp(entryname, IOP_HTTP_SSL_POLICY)) {
    sscanf(data,"%d", &check_me);

    if ((check_me >= SSL_LAST_VALUE) || (check_me<0)) {
      if (omniORB::traceLevel > 0) {
	omniORB::log << " [IOP_HTTP]: Bad IOP_HTTP_SSL_POLICY value in config file!.\n";
	omniORB::log.flush();
      }
      

      data_lock.unlock();
      return -1;
    }
    
    pd_SSL_requiered = check_me;
  } else {
    /* No match has been found ! */
    data_lock.unlock();
    return 0;
  }
  
  data_lock.unlock();
  return 1;
}

void
iop_http::read_orb_write_http(int listenfd, HttpConnectionInterfaceI *http, 
			      struct iop_http_handshake &handshake, 
			      const char *mime_type,
			      const char *path) 
{
  /* Buffers used for data transmission. */
  char                      iop_buffer[2048];
  CORBA::Boolean persistent_connection = true;
  
  while(persistent_connection) {
      CORBA::ULong size_to_be_read;
      int ret;
      /* 
	 We read the GIOP header. The giop message_size field
	 is necessary because in order to send chuncks, some packets
	 are sended in separates packets. We are going to reassemble 
	 all the packets in one HTTP packet.

	 So, the full size of the HTTP packet is
	 GIOP_Header + Hanshake + message_size
      */
      GIOP::MessageHeader hdr;
      ret = ::recv(listenfd, (char*)&hdr, sizeof(GIOP::MessageHeader), MSG_WAITALL);
      if (ret != sizeof(GIOP::MessageHeader)) {
	/* We cannot know if the socket is still opened or not, or if
	   this is a "real" error. In the case catch a bad descriptor,
	   that mean to say the socket has been closed by the ORB. */
	
	if ((GETERRNO() == EBADF) || (GETERRNO() == ENOTCONN)){
	  return;
	}
      
	/* If it was not a EBADF or ENOTCONN, we have a real error. Throw an exception. */
	// This error happens when the orb closed its Strand.
	throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
      }
      
      size_to_be_read = hdr.message_size + sizeof(GIOP::MessageHeader);

      memcpy(iop_buffer, &handshake, sizeof(struct iop_http_handshake));
      memcpy(iop_buffer+sizeof(struct iop_http_handshake), &hdr, sizeof(GIOP::MessageHeader));

      // Start the HTTP connection.
      if ((http->SetHeaderContentType(mime_type) != HttpConnectionInterfaceI::HTTP_RETURN_OK) ||
	  (http->SetHeaderContentLength(size_to_be_read + sizeof(struct iop_http_handshake)) != HttpConnectionInterfaceI::HTTP_RETURN_OK)) {
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
      }


      // We keep this value static in order to reduce the amount of tests.
      if (pd_remote_proxy_auth != NULL) {
	if ((http->SetHeader("Proxy-authorization",pd_remote_proxy_auth) != HttpConnectionInterfaceI::HTTP_RETURN_OK) ||
	    (http->SetHeader("Proxy-Connection","Keep-Alive") != HttpConnectionInterfaceI::HTTP_RETURN_OK) ||
	    (http->SetHeader("Connection","Keep-Alive") != HttpConnectionInterfaceI::HTTP_RETURN_OK)) {
	  throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
	}
	
      }

      HttpConnectionInterfaceI::HTTPReturnValueT http_ret;
      http_ret = http->SendRequestHeader(path, HttpConnectionInterfaceI::HTTP_METHOD_POST);
      
      // The proxy ask us to provide an authentification.
      if (http_ret == HttpConnectionInterfaceI::HTTP_RETURN_PROXY_AUTHENTICATION_REQUIRED) {  
	
	if (omniORB::traceLevel > 0) {
	  omniORB::log << " [IOP_HTTP]: Proxy Authentification problem. Check your login and password are corrects.\n";
	  omniORB::log.flush();
	}
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);      
      } else 
	if (http_ret != HttpConnectionInterfaceI::HTTP_RETURN_OK) { 	  // Error.
	  if (omniORB::traceLevel >= 5) {
	    omniORB::log << " [IOP_HTTP]: Impossible to send HttpHeader!\n";
	    omniORB::log << " [IOP_HTTP]: HTTP Return Code "<<http_ret <<"\n";
	    omniORB::log.flush();
	  }
	  
	  throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);      
	}

      if (http->BeginBodyOutput() != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
      }
      
      // Send the Handshake and GIOP init data
      unsigned int write_ret;
      
      if (http->WriteBodyOutput(iop_buffer, sizeof(struct iop_http_handshake) + sizeof(GIOP::MessageHeader), write_ret) != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	if (omniORB::traceLevel >= 5) {
	  omniORB::log << " [IOP_HTTP]: Error while writting.!\n";
	  omniORB::log.flush();
	}
	
	throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);   
      }
      size_t amount_read = sizeof(GIOP::MessageHeader);   

      /* 5- Read the real data */
      while (amount_read < size_to_be_read) {
	if ((ret = ::recv(listenfd, iop_buffer, sizeof(iop_buffer),0)) < 1) {
	  
	  if (omniORB::traceLevel >= 5) {
	    omniORB::log << " [IOP_HTTP]: bad receive operation on internal socket!\n";
	    omniORB::log << " [IOP_HTTP]: recv() " << strerror(GETERRNO()) << "\n.";
	    omniORB::log.flush();
	  }
	  
	  throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);      
	} /* End if */
	if (http->WriteBodyOutput(iop_buffer, ret,write_ret) != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	  throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);      
	}
	
	amount_read += ret;
      } /* End of while */
      if (http->EndBodyOutput() != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
      }

      if (omniORB::traceLevel >= 5) {
	omniORB::log << "XXX ["<<listenfd <<"] ["<<amount_read <<"]\n";
	omniORB::log.flush();
      }
      if (http->ReceiveResponseHeader() != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
      }
      
      size_to_be_read = http->GetHeaderContentLength();
      
      if (size_to_be_read <= 0) {
	if (omniORB::traceLevel >= 5) {
	    omniORB::log << " [IOP_HTTP]: Reply without any content!\n";
	    omniORB::log.flush();
	  }
      }
      
      /* This specific header is set by some proxies which do not
	 support persistent connections. */
      char *header_connection_field = http->GetHeader("Connection");
      if (header_connection_field != NULL) {
	if (!strcmp(header_connection_field, "close")) {
	  persistent_connection = false;
	}
	
	delete [] header_connection_field;
	header_connection_field = NULL;
      }

      if (size_to_be_read <=0) {
	continue;
      }
      
      if (http->BeginBodyInput() != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
      }
      
      amount_read = 0;

      unsigned int retry_loop = 0;
      while (amount_read != size_to_be_read) {
	char str[1024];
	unsigned int ret;
	
	HttpConnectionInterfaceI::HTTPReturnValueT http_return_value;
	
	// Read only what is necessary. We could block.
	if (size_to_be_read - amount_read <= sizeof(str)) {
	  http_return_value = http->ReadBodyInput(str, size_to_be_read - amount_read, ret);
	} else {
	  http_return_value = http->ReadBodyInput(str, sizeof(str), ret);
	}
	
	if ((ret == -1) || (http_return_value != HttpConnectionInterfaceI::HTTP_RETURN_OK) ) {
	    if (omniORB::traceLevel >= 5) {
	      omniORB::log << " [IOP_HTTP]: Cannot read HTTP data. \n";
	      omniORB::log << " [IOP_HTTP]: ReadBodyInput() " << strerror(GETERRNO()) << "\n.";
	      omniORB::log.flush();
	    }
	    throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
	  }
	
	if (ret>0) {
	  if (::send(listenfd, str, ret,0)<1) {
	    /* No way to send data back to the ORB engine. */
	    if (omniORB::traceLevel >= 5) {
	      omniORB::log << " [IOP_HTTP]: Cannot send data back to the ORB. \n";
	      omniORB::log << " [IOP_HTTP]: send() " << strerror(GETERRNO()) << "\n.";
	      omniORB::log.flush();
	    }
	    
	    throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
	  }
	    
	  amount_read+=ret;
	}

	// If we receive nothing, we give up after some time.
	if (!ret) {
	  retry_loop++;
	} else {
	  retry_loop = 0;
	}
	
	if (retry_loop > MAX_LOOP_RETRY) {
	  if (omniORB::traceLevel >= 5) {
	    omniORB::log << " [IOP_HTTP]: Got a timeout while trying to read the BodyInput. \n";
	    omniORB::log.flush();
	  }
	  throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
	}
      }
      if (http->EndBodyInput() != HttpConnectionInterfaceI::HTTP_RETURN_OK) {
	throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
      }
      
  } /* End of while */
}

int 
iop_http::thread_iop_http_proxy(void *incoming_fd) {

  int                       listenfd;
  struct iop_http_handshake handshake;

  /* Parameters for http connection. */
  char                     *host_to_send = NULL;
  char                     *mime_type    = NULL;
  char                     *path         = NULL;
  unsigned int              port_to_send = 0;
  ip_file_mapping_t         ipm          = NULL;
  CORBA::Boolean            is_ssl       = 0;

  if (incoming_fd == NULL) {
    return -1;
  }

  /* 1- Read the handshake data */
  listenfd = (int) incoming_fd;    
  /* Read the information header sent by the ORB. */
  if ((::recv(listenfd,  (char*)&handshake.pd_magic, sizeof(handshake.pd_magic),0) == RC_SOCKET_ERROR) ||
      (::recv(listenfd,  (char*)&handshake.pd_version_proto, sizeof(handshake.pd_version_proto),0) == RC_SOCKET_ERROR) ||
      (::recv(listenfd,  (char*)&handshake.pd_hostname, sizeof(CORBA::Char)*128,0) == RC_SOCKET_ERROR) ||
      (::recv(listenfd,  (char*)&handshake.pd_port, sizeof(handshake.pd_port),0) == RC_SOCKET_ERROR)) {
      
    if (omniORB::traceLevel >= 5) {
      omniORB::log << " [IOP_HTTP]: Cannot receive handshake from ORB!\n";
      omniORB::log << " [IOP_HTTP]: recv() " << strerror(GETERRNO()) << "\n.";
      omniORB::log.flush();
    }
    throw CORBA::COMM_FAILURE(GETERRNO(),CORBA::COMPLETED_NO);
  }

  /* 
     2- Check the magic number that must be in the structure. This is a
     very simple way to be sure that the guy who is talking to us
     knows really what he is doing. It prevents "unknown talked"
     too. 
  */
  if (ntohl(handshake.pd_magic) != IOP_HTTP_MAGIC_HANDSHAKE) {
    if (omniORB::traceLevel >= 5) {
      omniORB::log << " [IOP_HTTP]: Bad magic number during handshake!\n";
      omniORB::log.flush();
    }
    
    throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
  } 
  if (omniORB::traceLevel >= 5) {
    omniORB::log << "Infos Handshake: ["<<ntohl(handshake.pd_magic) <<"] ["<<ntohs(handshake.pd_version_proto) << "] [" <<handshake.pd_hostname << "] [" <<ntohs(handshake.pd_port)<<"] \n";
    omniORB::log.flush();
  }

  /* 3- Can we find a mapping ? */
  if ((ipm = ipf.ip_match((const char*)handshake.pd_hostname)) != NULL) {
    /* Get the informations. */
    host_to_send = (char *)ipf.get_host(ipm);
    mime_type    = (char *)ipf.get_mimetype(ipm);
    path         = (char *)ipf.get_pathname(ipm);

    /* The port to send data if based on the SSL policy. */
    if (ipf.get_ssl_policy(ipm) == SSL_REQUIRED) {
      port_to_send = ipf.get_port_https(ipm);      
      is_ssl=1;
    } else { 
      port_to_send = ipf.get_port_http(ipm);
    }
  } else {
    /* No mapping: let's use default values. */
   
    host_to_send = pd_default_hostname;

    if (pd_SSL_requiered == SSL_REQUIRED) {
      port_to_send = pd_default_port_https;
      is_ssl=1;
    } else {
      port_to_send = pd_default_port;
    }
    
    mime_type    = pd_default_mime_type;
    path         = pd_default_path;
  }
  /*
    HTTP Proxy - only for HTTP protocol - not for SSL currently.
  */
  char real_path[2048];
  
  if ((!is_ssl) && (pd_remote_proxy_hostname != NULL)) {
    if (is_ssl) {
      sprintf(real_path, "https://%s:%d",
	      host_to_send,
	      port_to_send);
    } else {
      sprintf(real_path, "http://%s:%d",
	      host_to_send,
	      port_to_send);
    }
    if (path[0] != '/') {
      strcat(real_path,"/");
    }
    strcat(real_path, path);

    if (omniORB::traceLevel >=  5) {
      omniORB::log << " [IOP_HTTP]: HTTP requests forwarded via proxy ("<<pd_remote_proxy_hostname << " " << pd_remote_proxy_port << ")\n";
      omniORB::log.flush();
    }
    host_to_send = pd_remote_proxy_hostname;
    port_to_send = pd_remote_proxy_port;
  } else {
    strcpy(real_path, path);
  }
  if (omniORB::traceLevel >=  5) {
    omniORB::log <<" [IOP_HTTP]: Real path = "<< real_path << "\n";  
    omniORB::log.flush();
  }
  /* 
     Setup the HTTP connection. The idea is to try to keep the connection 
     open as long as possible in order to reduce the overload due
     to the open/close network operation. BTW, it will save a lot of 
     process for SSL, because the handshaking is really slower.
  */
  if (omniORB::traceLevel >= 5) {
    omniORB::log << " [IOP_HTTP]: Create connection "<< host_to_send << "Port "<<port_to_send<<"\n";
    omniORB::log.flush();
  }

  HttpConnectionInterfaceI *http = NULL;
  if (is_ssl) {
#ifndef HAS_SSL      
      omniORB::log << " [IOP_HTTP]: SSL not implemented!\n";
      omniORB::log.flush();
      throw CORBA::COMM_FAILURE(0,CORBA::COMPLETED_NO);
#else 
      // Note: we do not provide the cipher list here because we make the
      // assomption that the toolkit will choose the strongest one during
      // the negocation. 
      http = GetHttpConnectionInterfaceManager()->CreateHttpConnection(host_to_send, port_to_send,
								       true, 
								       pd_SSL_root_certificates);
#endif /* HAS_SSL */
  } else {    
    http = GetHttpConnectionInterfaceManager()->CreateHttpConnection(host_to_send, port_to_send);
  }
  
  if (http == NULL) {
    if (omniORB::traceLevel > 0) {
      omniORB::log << " [IOP_HTTP]: Cannot create a HTTP connection object.\n";
      omniORB::log << " [IOP_HTTP]: This feature may not be available in your installation.\n.";
      omniORB::log.flush();
    } 
    
    return -1;
  }
  
  /* Initialize the ProxyAuthorization stuff. - HTTP only !*/
  if ((!is_ssl) && (pd_remote_proxy_auth == NULL) && 
      (pd_remote_proxy_login != NULL) && (pd_remote_proxy_password != NULL)) {
      
      char auth_tmp[2048];
      sprintf(auth_tmp, "%s:%s", pd_remote_proxy_login, pd_remote_proxy_password);

      // Encode64... and add the Basic key word - dump it. 
      char *b64 = http->EncodeBase64(auth_tmp);
      sprintf(auth_tmp, "Basic %s", b64);
      pd_remote_proxy_auth = CORBA::string_dup(auth_tmp);
      delete []b64;
      
      if (omniORB::traceLevel >= 5) {
	  omniORB::log << " [IOP_HTTP]: Proxy Auth fixed "<< pd_remote_proxy_auth << "\n";
	  omniORB::log.flush();
      }
  }

  try {
    // Read everything we get on the incoming socket and send it as 
    // HTTP packets. 
    read_orb_write_http(listenfd, http, handshake, mime_type, real_path);
  } catch (...) {
    // If an exception has been thrown, delete the http object
    // and rethrow the exception.
      delete http;
    
    throw;
  }
      delete http;

  return 0;
}

CORBA::Boolean
iop_http::need_to_relay(const char *host_to_send) 
{
    if (host_to_send==NULL) {
	return 0;
    }
  
    if (IopHttpManager().ipf.is_mapping_feature()) {

	if (IopHttpManager().ipf.ip_match(host_to_send) != NULL) {
	    return 1;
	} 

    } else {
	/* If no mapping table exists, we check if we have to relay */
	if (IopHttpManager().have_to_relay()) {
	    return 1;
	}
    }
    return 0;
}

/* Static */

/* The thread creating from OmniORB does not allow to specify a method
   as entry point. This is why we use this static function, which is
   going to call the appropriated method. */
void 
iop_http::iop_http_proxy_s(void *fd) {

  if (fd != NULL) {
    try {
	IopHttpManager().thread_iop_http_proxy(fd);
    } catch(...) { }

    CLOSESOCKET((int)fd);
  }

  omni_thread::exit(NULL);
}

void 
iop_http::listen_bind_socket_s(void *fd) {

    if (fd != NULL) {
	IopHttpManager().thread_listen_bind_socket(fd);

	CLOSESOCKET((int)fd);
    }
    omni_thread::exit(NULL);
}

