// -*- Mode: C++; -*-
//                            Package   : omniEvents
//  omniEvents.cc             Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniOEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//    Event Services Channel Factory implementation. The factory registers
//    itself with the naming service. Clients wishing to create event
//    channels can either use the factory by resolving its name with the
//    naming service or create in-process channels.
//	

/*
  $Log: omniEvents.cc,v $
  Revision 1.5  2000/08/30 04:37:03  naderp
  Port to omniORB 3.0.1.

  Revision 1.4  2000/03/06 13:19:09  naderp
  Initialising EventChannelFactory with port number.

  Revision 1.3  2000/03/06 04:12:56  naderp
  No longer pre-generating factory key.
  Removed internal dependency between factory and naming service.

  Revision 1.2  2000/03/02 04:21:36  naderp
  Printing factory IOR.

  Revision 1.1  1999/11/02 02:29:40  naderp
  Added <signal.h>.

  Revision 1.0  1999/11/01 17:06:59  naderp
  omniEvents 2.0
  Added logfile initialisation and creation of default factory.

Revision 0.5  99/08/27  17:04:15  17:04:15  naderp (Paul Nader)
Changed filename to omniEvents.cc

Revision 0.4  99/08/27  11:48:49  11:48:49  naderp (Paul Nader)
Partitioned EventChannelFactory_i from CosEvent_i.

Revision 0.3  99/04/23  16:03:51  16:03:51  naderp (Paul Nader)
gcc port.
Ignore unix SIGPIPE if connections are severed.

Revision 0.2  99/04/23  09:33:11  09:33:11  naderp (Paul Nader)
Windows Port.

Revision 0.1  98/11/25  14:07:33  14:07:33  naderp (Paul Nader)
Initial Revision

*/

#include <args.h>
#include <naming.h>
#include <CosEvent_i.h>
#include <omniEventsLog.h>
#include <EventChannelFactory_i.h>
#ifdef __WIN32__
#include <getopt.h>
#else
#include <unistd.h>
#include <signal.h>
#endif

extern int optind;
void usage();

int
main(int argc, char **argv)
{
  //
  // Process Options
  int c;
  int port = 0;
  int first = 0;
  char *logDir = NULL;
  char *factoryName = (char *) "EventChannelFactory";
  char *factoryKind = (char *) "EventChannelFactory";

  while ((c = getopt(argc,argv,"O:s:l:N:K:h")) != EOF)
  {
     switch (c)
     {
        case 's': first = 1;
                  port = atoi(optarg);
                  if (port <= 0)
                  {
                     cerr << "\nError: port must be a positive integer" << endl;
                     usage();
                  }
                  removeArgs(argc, argv, 1, 2);
                  optind = 1;
                  break;

        case 'l': logDir = optarg;
                  removeArgs(argc, argv, 1, 2);
                  optind = 1;
                  break;

        case 'N': factoryName = optarg;
                  removeArgs(argc, argv, 1, 2);
                  optind = 1;
                  break;

        case 'K': factoryKind = optarg;
                  removeArgs(argc, argv, 1, 2);
                  optind = 1;
                  break;

        case 'O': break;

        case 'h':
        default : usage();
                  break;
     }
  }

  //
  // Create log file instance.
  omniEventsLog logfile(port, logDir);

  //
  // Add a fake command line option to tell the BOA which port to use.
  port = logfile.getPort();
  insertArgs(argc, argv, 1, 2);
  argv[1] = strdup("-BOAiiop_port");
  argv[2] = new char[32 + 1];
  sprintf(argv[2], "%d", port);

  //
  // Initialise orb and boa.
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB3");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB3_BOA");

  //
  // Create a default factory.
  EventChannelFactory_i *factory = NULL;
  if (first)
  {
     //
     // Get Name Service root context.
     CosNaming::NamingContext_ptr rootContext;
     rootContext = getRootNamingContext(orb);

     //
     // Create Event Channel Factory.
     factory = new EventChannelFactory_i(port);

     //
     // Create Name
     CosNaming::Name *name = new CosNaming::Name;
     name->length(1);
     (*name)[0].id = CORBA::string_dup (factoryName);
     (*name)[0].kind = CORBA::string_dup (factoryKind);

     //
     // Register Event Channel Factory with the Naming Service.
     bindName2Object(rootContext, *name, factory->_this());
  }

  //
  // If omniEvents is restarting then the omniEventsLog object
  // will take care of creating the factory and any subordinate
  // event channels, proxies, etc under it.
  logfile.init(orb, boa, factory);

  //
  // Print the factory IOR.
  if (! CORBA::is_nil (factory->_this()))
  {
     char *iorstr;
     iorstr = orb->object_to_string(factory->_this());
     cout << "Channel Factory is " << iorstr << "\n";
     delete iorstr;
  }

  // Ignore broken pipes
#ifndef __WIN32__
  signal(SIGPIPE, SIG_IGN);
#endif

  //
  // Start the server implementation, blocking.
  boa->impl_is_ready();

  return 0;
}

void
usage()
{
   cerr << "\nusage: omniEvents [-s port] [-l logdir] [-N name] [-K kind] [-h]\n" << endl;
   cerr << "         -s server port" << endl;
   cerr << "         -l log and data directory" << endl;
   cerr << "         -N name specify factory name [\"EventChannelFactory\"]" << endl;
   cerr << "         -K kind specify factory kind [\"EventChannelFactory\"]" << endl;
   cerr << "         -h     display usage" << endl;
   cerr << "\nUse -s option to start omniEvents for the first time." << endl;
   cerr << "Use -l option to specify the directory where the log/data files are kept.\n";
   cerr << "You can also set the environment variable " << LOGDIR_ENV_VAR << endl;
   cerr << "to specify the directory where the log/data files are kept.\n" << endl;
   exit(-1);
}
