%option never-interactive

%{
/*

//                            Package   : omniEvents
// oep.ll                     Created   : 27/9/99
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    
//    omniEvents persistency lexical scanner.
//

*/

/*
  $Log: oep.ll,v $
  Revision 1.3  2000/09/05 00:59:44  naderp
  Added MaxQueueLength QOS.

  Revision 1.2  2000/03/06 13:09:20  naderp
  Removed common.

  Revision 1.1  2000/03/02 01:14:59  naderp
  Added nil IOR.

  Revision 1.0  1999/11/01 16:29:20  naderp
  Initial revision

*/

#include <omniEvents.h>

#if defined(__VMS)
#include "y_tab.h"
#else
#include "y.tab.h"
#endif

static unsigned char oep_escape_reader(char *);
static double oep_atof(char *);
static long oep_atoi(char *, long);

%}

%%

channelFactory	return CHANNELFACTORY_LABEL;
eventChannel	return EVENTCHANNEL_LABEL;
consumerAdmin	return CONSUMERADMIN_LABEL;
supplierAdmin	return SUPPLIERADMIN_LABEL;
pxyPushSupplier	return PROXYPUSHSUPP_LABEL;
pxyPushConsumer	return PROXYPUSHCONS_LABEL;
pxyPullSupplier	return PROXYPULLSUPP_LABEL;
pxyPullConsumer	return PROXYPULLCONS_LABEL;

PORT		return PORT_LABEL;
IOR		return IOR_LABEL;
KEY		return KEY_LABEL;
PULLRETRYPERIOD	return PULLRETRYPERIOD_LABEL;
MAXQUEUELENGTH	return MAXQUEUELENGTH_LABEL;
MAXEVENTSPERCONSUMER	return MAXEVENTSPERCONSUMER_LABEL;

True		return TRUETOK;
False		return FALSETOK;

-?[1-9][0-9]*	{
		  yylval.ival = oep_atoi(yytext, 10);
		  return INTEGER_LITERAL;
	        }

-?0[xX][a-fA-F0-9]+ {
		  yylval.ival = oep_atoi(yytext, 16);
		  return INTEGER_LITERAL;
	        }

-?0[0-7]*	{
		  yylval.ival = oep_atoi(yytext, 8);
		  return INTEGER_LITERAL;
	      	}

"\""([^\"\\]|\\.)*"\""  {
                  // strip leading and trailing quotes
                  yytext++;
		  char *z = (char *) malloc(strlen(yytext));
		  strncpy(z, yytext, strlen(yytext) - 1);
		  z[strlen(yytext) - 1] = NULL;
		  yylval.sval = z;
		  return STRING_LITERAL;
                }

[0-9a-f]*	{
		  omniORB::objectKey *k = new omniORB::objectKey();
		  int l = strlen(yytext) / 2;
		  omniORB::seqOctets os(l);
		  os.length(l);
		  char* p = yytext;
		  for (int i = 0; i < l; i++)
		  {
		    int n;
		    sscanf(p,"%02x",&n);
		    os[i] = n;
		    p += 2;
		  }
		  *k = omniORB::octetSequenceToKey(os);
		  yylval.kval = k;
		  return KEY_VALUE;
		}

IOR:[0-9a-f]*|nil {
		  char *z = (char *) malloc(strlen(yytext) + 1);
		  strcpy(z, yytext);
		  yylval.sval = z;
		  return IOR_VALUE;
		}

\/\/.*\n	{
		  /* ignore comments */
  		  ((OEP_GlobalData *) oep_global)->incLineNo();
		}

"/*"		{
		  for(;;) {
		    char c = yyinput();
		    if (c == '*') {
		      char next = yyinput();
		      if (next == '/')
			break;
		      else
			/* yyunput(c); */
			unput(c);
	              if (c == '\n') 
		         ((OEP_GlobalData *) oep_global)->incLineNo();
		    }
	          }
		}

[ \t]*		;
\n              {
		  ((OEP_GlobalData *) oep_global)->incLineNo();
                }
.		return yytext[0];

%%
        /* subroutines */

/*
 * oep_atoi - Convert a string of digits into an integer according to base b
 */
static long
oep_atoi(char *s, long b)
{
        long    r = 0;
        long    negative = 0;

        if (*s == '-') {
          negative = 1;
          s++;
        }
        if (b == 8 && *s == '0')
          s++;
        else if (b == 16 && *s == '0' && (*(s + 1) == 'x' || *(s + 1) == 'X'))
          s += 2;

        for (; *s; s++)
          if (*s <= '9' && *s >= '0')
            r = (r * b) + (*s - '0');
          else if (b > 10 && *s <= 'f' && *s >= 'a')
            r = (r * b) + (*s - 'a' + 10);
          else if (b > 10 && *s <= 'F' && *s >= 'A')
            r = (r * b) + (*s - 'A' + 10);
          else
            break;

        if (negative)
          r *= -1;

        return r;
}

/*
 * Convert a string to a float; atof doesn't seem to work, always.
 */
static double
oep_atof(char *s)
{
        char    *h = s;
        double  d = 0.0;
        double  f = 0.0;
        double  e, k;
        long    neg = 0, negexp = 0;

        if (*s == '-') {
          neg = 1;
          s++;
        }
        while (*s >= '0' && *s <= '9') {
                d = (d * 10) + *s - '0';
                s++;
        }
        if (*s == '.') {
                s++;
                e = 10;
                while (*s >= '0' && *s <= '9') {
                        d += (*s - '0') / (e * 1.0);
                        e *= 10;
                        s++;
                }
        }
        if (*s == 'e' || *s == 'E') {

                s++;
                if (*s == '-') {
                        negexp = 1;
                        s++;
                } else if (*s == '+')
                        s++;
                e = 0;
                while (*s >= '0' && *s <= '9') {
                        e = (e * 10) + *s - '0';
                        s++;
                }
                if (e > 0) {
                        for (k = 1; e > 0; k *= 10, e--);
                        if (negexp)
                                d /= k;
                        else
                                d *= k;
                }
        }

        if (neg) d *= -1.0;

        return d;
}

/*
 * Convert (some) escaped characters into their ascii values
 */
static unsigned char
oep_escape_reader(
    char *str
)
{
    if (str[0] != '\\') {
        return str[0];
    }

    switch (str[1]) {
      case 'n':
        return '\n';
      case 't':
        return '\t';
      case 'v':
        return '\v';
      case 'b':
        return '\b';
      case 'r':
        return '\r';
      case 'f':
        return '\f';
      case 'a':
        return '\a';
      case '\\':
        return '\\';
      case '\?':
        return '?';
      case '\'':
        return '\'';
      case '"':
        return '"';
      case 'x':
        {
            // hex value
            int i;
            for (i = 2; str[i] != '\0' && isxdigit(str[i]); i++) {
                continue;
            }
            char save = str[i];
            str[i] = '\0';
            char out = (char)oep_atoi(&str[2], 16);
            str[i] = save;
            return out;
        }
        break;
      default:
        // check for octal value
        if (str[1] >= '0' && str[1] <= '7') {
            int i;
            for (i = 1; str[i] >= '0' && str[i] <= '7'; i++) {
                continue;
            }
            char save = str[i];
            str[i] = '\0';
            char out = (char)oep_atoi(&str[1], 8);
            str[i] = save;
            return out;
        } else {
          return str[1] - 'a';
        }
        break;
    }
}

