// -*- Mode: C++; -*-
//                            Package   : omniEvents
//   pushcons.cc              Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//    Push Model consumer implementation
//	

/*
  $Log: pushcons.cc,v $
  Revision 0.13  2000/08/30 04:39:48  naderp
  Port to omniORB 3.0.1.

  Revision 0.12  2000/03/16 05:37:27  naderp
  Added stdlib.h for getopt.

  Revision 0.11  2000/03/06 13:27:02  naderp
  Using util getRootNamingContext function.
  Using stub headers.
  Fixed error messages.

  Revision 0.10  2000/03/02 03:20:24  naderp
  Added retry resiliency for handling COMM_FAUILURE exceptions.

  Revision 0.9  1999/11/02 13:39:15  naderp
  Added <signal.h>

  Revision 0.8  1999/11/02 07:57:04  naderp
  Updated usage.

Revision 0.7  99/11/01  18:10:29  18:10:29  naderp (Paul Nader)
Added ahndling of COMM_FAILURE exception for connect_push_consumer.

Revision 0.6  99/11/01  16:11:03  16:11:03  naderp (Paul Nader)
omniEvents 2.0 Release.

Revision 0.5  99/10/27  19:46:01  19:46:01  naderp (Paul Nader)
Ignoring Unix SIGPIPE signal.
Catching COMM_FAILURE exception for obtain_push_supplier.
Continuing if it fails to obtain Proxy Supplier.
Try/Catch block for disconnect_push_supplier.

Revision 0.4  99/04/23  16:05:46  16:05:46  naderp (Paul Nader)
gcc port.

Revision 0.3  99/04/23  09:34:03  09:34:03  naderp (Paul Nader)
Windows Port.

Revision 0.2  99/04/21  18:06:26  18:06:26  naderp (Paul Nader)
*** empty log message ***

Revision 0.1.1.1  98/11/27  16:59:37  16:59:37  naderp (Paul Nader)
Added -s option to sleep after disconnecting.

Revision 0.1  98/11/25  14:08:21  14:08:21  naderp (Paul Nader)
Initial Revision

*/

//
//
#include <CosEventComm.hh>
#include <CosEventChannelAdmin.hh>
#ifdef __WIN32__
#include <getopt.h>
#include <iostream>
using namespace std;
#else
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include <signal.h>
#endif

static omni_mutex mutex;
static omni_condition connect_cond(&mutex);
static void usage();
extern CosNaming::NamingContext_ptr getRootNamingContext(CORBA::ORB_ptr orb);

class Consumer_i : virtual public CosEventComm::_sk_PushConsumer {
public:
  Consumer_i (long disconnect = 0) : _disconnect(disconnect) {}

  void push (const CORBA::Any& data);
  void disconnect_push_consumer ();

private:
  long _disconnect;
};

void Consumer_i::push (const CORBA::Any& data) {
  CORBA::ULong l;
  static int i = 0;

  i++;
  data >>= l;
  cout << "Push Consumer: push () called. Data : "
       << l
       << endl;

  // Exersize Disconnect
  if (i == _disconnect) {
     i = 0;
     // NOTE : The proxy_supplier object is disposed at the server
     //        during the disconnect_push_supplier call. Do NOT
     //        use the proxy_supplier reference after disconnecting.

     // Signal main thread to disconnect and re-connect.
#ifndef __WIN32__
     mutex.unlock(); // Why does NT break if the mutex is unlocked?
#endif
     connect_cond.signal();
  }
}

void Consumer_i::disconnect_push_consumer () {
  cout << "Push Consumer: disconnected." << endl;
}

int
main(int argc, char **argv)
{
  //
  // Start orb and boa.
  CORBA::ORB_ptr orb = CORBA::ORB_init(argc,argv,"omniORB3");
  CORBA::BOA_ptr boa = orb->BOA_init(argc,argv,"omniORB3_BOA");

  // Process Options
  int c;
  int discnum = 0;
  int sleepInterval = 0;
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";

  while ((c = getopt(argc,argv,"hd:s:n:k:")) != EOF)
  {
     switch (c)
     {
        case 'd': discnum = atoi(optarg);
                  break;

        case 's': sleepInterval = atoi(optarg);
                  break;

        case 'n': channelName = optarg;
                  break;

        case 'k': channelKind = optarg;
                  break;

        case 'h':
        default : usage();
                  exit(-1);
                  break;
     }
  }

  // Ignore broken pipes
#ifndef __WIN32__
  signal(SIGPIPE, SIG_IGN);
#endif

  Consumer_i* consumer = new Consumer_i (discnum);
  consumer->_obj_is_ready(boa);

  // Let the BOA know we are ready.
  boa->impl_is_ready(0,1);

  //
  // Obtain and narrow reference to Name Service.
  CosNaming::NamingContext_ptr rootContext;
  rootContext = getRootNamingContext(orb);

  //
  // Obtain Event Channel Object
  CosNaming::Name name;
  name.length (1);
  name[0].id = CORBA::string_dup (channelName);
  name[0].kind = CORBA::string_dup (channelKind);

  cout << "Looking for EventChannel" << endl;
  CosEventChannelAdmin::EventChannel_var channel;
  try {
    CORBA::Object_var obj = rootContext->resolve(name);
    channel = CosEventChannelAdmin::EventChannel::_narrow(obj);
    if (CORBA::is_nil(channel))
    {
       cerr << "Failed to narrow Event Channel reference !" << endl;
       exit(1);
    }
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the "
          << "naming service !" << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception !" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot find event channel ! [\""
          << channelName << "\", \"" << channelKind << "\"]"
          << endl;
     exit (1);
  }

  //
  // Get Consumer admin interface - retrying on Comms Failure.
  CosEventChannelAdmin::ConsumerAdmin_var consumer_admin;
  while (1)
  {
     try {
        consumer_admin = channel->for_consumers ();
        if (CORBA::is_nil (consumer_admin))
        {
           cerr << "Event Channel returned consumer_admin nil reference !"
                << endl;
           exit (1);
        }
        break;
     }
     catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught COMM_FAILURE exception "
             << "obtaining Consumer Admin !. Retrying..."
             << endl;
        continue;
     }
     catch (...) {
        cerr << "Unexpected System Exception."
             << "Failed to obtain ConsumerAdmin !" << endl;
        exit (1);
     }
  }
  cerr << "Obtained ConsumerAdmin." << endl;

  while (1) {
     //
     // Get proxy supplier - retrying on Comms Failure.
     CosEventChannelAdmin::ProxyPushSupplier_var proxy_supplier;
     while (1)
     {
        try {
           proxy_supplier = consumer_admin->obtain_push_supplier ();
           if (CORBA::is_nil (proxy_supplier))
           {
              cerr << "Consumer Admin returned proxy_supplier nil reference !"
                   << endl;
              exit (1);
           }
	   break;
        }
        catch (CORBA::COMM_FAILURE& ex) {
           cerr << "Caught COMM_FAILURE Exception "
                << "obtaining Push Supplier !. Retrying..."
                << endl;
           continue;
        }
        catch (...) {
           cerr << "Unexpected System Exception."
	        << "Failed to obtain Proxy Supplier !"
                << endl;
           exit (1);
        }
     }
     cerr << "Obtained ProxyPushSupplier." << endl;
   
     //
     // Connect Push Consumer - retrying on Comms Failure.
     CosEventComm::PushConsumer::_duplicate(consumer->_this());
     while (1)
     {
        try {
           proxy_supplier->connect_push_consumer(consumer->_this());
	   break;
        }
        catch (CORBA::BAD_PARAM& ex) {
           cerr << "Caught BAD_PARAM Exception connecting Push Consumer !"
                << endl;
           exit (1);
        }
        catch (CosEventChannelAdmin::AlreadyConnected& ex) {
           cerr << "Push Consumer already connected !" 
                << endl;
           break;
        }
        catch (CORBA::COMM_FAILURE& ex) {
           cerr << "Caught COMM_FAILURE exception "
	        << "connecting push consumer !. Retrying..."
                << endl;
           continue;
        }
        catch (...) {
           cerr << "Unexpected System Exception. "
	        << "Failed to connect Push Consumer !"
                << endl;
           exit (1);
        }
     }
     cerr << "Connected Push Consumer." << endl;

     // Wait for indication to disconnect before re-connecting.
     mutex.lock();
     connect_cond.wait();

     // Disconnect - retrying on Comms Failure.
     while (1)
     {
        try {
           proxy_supplier->disconnect_push_supplier();
           break;
        }
        catch (CORBA::COMM_FAILURE& ex) {
           cerr << "Caught COMM_FAILURE Exception "
	        << "disconnecting Push Consumer !. Retrying..."
                << endl;
           continue;
        }
        catch (...) {
           cerr << "Unexpected System Exception. "
                << "Failed to disconnect Push Consumer!."
                << endl;
           exit (1);
        }
     }
     cerr << "Disconnected Push Consumer." << endl;
   
     // Yawn
     cerr << "Sleeping " << sleepInterval << " Seconds." << endl;
     omni_thread::sleep(sleepInterval);
  }

  // Not Reached
  // Destory Channel
  channel->destroy();
  cerr << "Destroyed channel !" << endl;

  return 0;
}

static void
usage()
{
   cerr << "\nusage: pushcons [-d n [-s n]] [-n name] [-k kind] [-h]\n" << endl;
   cerr << "         -d n   disconnect after n pushes" << endl;
   cerr << "         -s n   sleep n seconds after disconnecting" << endl;
   cerr << "         -n name specify channel name [\"EventChannel\"]" << endl;
   cerr << "         -k kind specify channel kind [\"EventChannel\"]" << endl;
   cerr << "         -h     display usage" << endl;
   cerr << endl;
}
