// -*- Mode: C++; -*-
//                            Package   : omniEvents
//  omniEventsLog.cc          Created   : 1/10/99
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniOEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//	

/*
  $Log: omniEventsLog.cc,v $
  Revision 1.4  2000/03/06 13:19:58  naderp
  Moved port from global to factory persistency data.

  Revision 1.3  2000/03/06 04:15:09  naderp
  Removed internal dependency between factory and Naming Service.

  Revision 1.2  2000/03/02 04:20:09  naderp
  Initialising factory refernce in init().

  Revision 1.1  2000/03/02 02:00:25  naderp
  Re-open active logfile during re-start.

  Revision 1.0  1999/11/01 17:04:08  naderp
  Initial revision

*/

#include <omniEventsLog.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <time.h>
#include <fcntl.h>

#include <EventChannelFactory_i.h>

extern int yyparse();
extern int yydebug;
extern FILE *yyin;

OEP_GlobalData *oep_global;
CORBA::ORB_ptr omniEventsLog::orb = NULL;
CORBA::BOA_ptr omniEventsLog::boa = NULL;
omniEventsLog *omniEventsLog::theLog = NULL;

// Minimum idle period before we take a checkpoint (15 mins)
#define DEFAULT_IDLE_TIME_BTW_CHKPT  (15*60)

#if defined(__VMS) && __CRTL_VER < 70000000
#include <omniVMS/unlink.hxx>
#include <omniVms/utsname.hxx>
#endif

#ifdef __WIN32__

#include <io.h>
#include <winbase.h>

#define stat(x,y) _stat(x,y)
#define unlink(x) _unlink(x)

#else

#include <unistd.h>
#include <sys/utsname.h>

#endif

#if defined(__nextstep__)
#include <libc.h>
#include <sys/param.h>
#endif

#if defined(__nextstep__)
#include <sys/param.h>
#endif

#ifndef O_SYNC
#ifdef  O_FSYNC              // FreeBSD 3.2 does not have O_SYNC???
#define O_SYNC O_FSYNC
#endif
#endif

#ifdef _NO_STRDUP

// we have no strdup
static char *
strdup (char* str)
{
  char *newstr;

  newstr = (char *) malloc (strlen (str) + 1);
  if (newstr)
    strcpy (newstr, str);
  return newstr;
}
#endif  // _NO_STRDUP

#if !defined(__SUNPRO_CC) || __SUNPRO_CC < 0x500
#if !defined(__VMS) && !defined(_MSC_VER)
#define USE_ATTACH_FD
#endif
#endif


//------------------------------------------------------------------------
//           timestamp Implementation
//------------------------------------------------------------------------
// This class can be used to generate timestamps.  The t() method normally
// returns a timestamp string, but if the same timestamp (to the nearest
// second) would be returned as last time then an empty string is returned
// instead.
//

class timestamp {
  char str[29];
public:
  timestamp(void) {
    str[0] = '\n';
    str[1] = str[28] = '\0';
  }
  char* t(void) {
    time_t t = time(NULL);
    char *p = ctime(&t);
    if (strncmp(p, &str[1], 24) == 0) {
      return &str[28];
    }
    strncpy(&str[1], p, 24);
    str[25] = ':';
    str[26] = '\n';
    str[27] = '\n';
    return str;
  }
};

timestamp ts;

#define DB(l,x) ((omniORB::traceLevel >= l) && (cerr << x << endl))

//------------------------------------------------------------------------
//           omniEvents Log Implementation
//------------------------------------------------------------------------
omniEventsLog::omniEventsLog(int p, char* l) :
  port(p),
  logdir(l),
  factory(NULL),
  checkpointNeeded(1)
{

  omniEventsLog::theLog = this;

#ifdef __WIN32__
  struct _stat sb;
#else
  struct stat sb;
#endif

  if (!logdir && (logdir = getenv(LOGDIR_ENV_VAR)) == NULL)
    logdir = strdup(DEFAULT_LOGDIR);

#if !defined(__WIN32__) && !defined(__VMS)
  if (logdir[0] != '/') {
    cerr << ts.t() << "Error: " << LOGDIR_ENV_VAR << " (" << logdir
         << ") is not an absolute path name." << endl;
    exit(1);
  }

  if (logdir[strlen(logdir)-1] == '/') {
    logdir[strlen(logdir)-1] = '\0';            // strip trailing '/'
  }

#ifndef _USE_GETHOSTNAME
  struct utsname un;
  if (uname(&un) < 0) {
    cerr << ts.t() << "Error: cannot get the name of this host." << endl;

    exit(1);
  }

  char* logname = new char[strlen(logdir) + strlen("/omnievents-")
                           + strlen(un.nodename) + 1];
  sprintf(logname, "%s/omnievents-%s", logdir, un.nodename);
#else

  // Apparently on some AIX versions, MAXHOSTNAMELEN is too small (32) to
  // reflect the true size a hostname can be. Check and fix the value.
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#elif   MAXHOSTNAMELEN < 64
#undef  MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#endif

  char hostname[MAXHOSTNAMELEN+1];

  if (gethostname(hostname, MAXHOSTNAMELEN) < 0) {
    cerr << ts.t() << "Error: cannot get the name of this host." << endl;

    exit(1);
  }
  char* logname = new char[strlen(logdir) + strlen("/omnievents-")
                           + strlen(hostname) + 1];
  sprintf(logname, "%s/omnievents-%s", logdir, hostname);
#endif // _USE_GETHOSTNAME
#elif defined(__WIN32__)

  // Get host name:

  DWORD machineName_len = MAX_COMPUTERNAME_LENGTH+1;
  char* machineName = new char[machineName_len];
  if (!GetComputerName((LPTSTR) machineName, &machineName_len)) {
    cerr << ts.t() << "Error: cannot get the name of this host." << endl;

    exit(1);
  }

  char* logname = new char[strlen(logdir) + strlen("\\omnievents-")
                           + strlen(machineName) + 1];
  sprintf(logname, "%s\\omnievents-%s", logdir, machineName);

  delete[] machineName;
#else
  char last(
    logdir[strlen(logdir)-1]
  );
  if (last != ':' && last != ']') {
    cerr << ts.t() << "Error: " << LOGDIR_ENV_VAR << " (" << logdir
         << ") is not a directory name." << endl;
    exit(1);
  }

  struct utsname un;
  if (uname(&un) < 0) {
    cerr << ts.t() << "Error: cannot get the name of this host." << endl;

    exit(1);
  }

  char* logname = new char[strlen(logdir) + strlen("/omnievents-")
                           + strlen(un.nodename) + 1];
  sprintf(logname, "%somnievents-%s", logdir, un.nodename);
#endif

#ifndef __VMS
  active = new char[strlen(logname)+strlen(".log")+1];
  sprintf(active,"%s.log",logname);
  backup = new char[strlen(logname)+strlen(".bak")+1];
  sprintf(backup,"%s.bak",logname);
  checkpt = new char[strlen(logname)+strlen(".ckp")+1];
  sprintf(checkpt,"%s.ckp",logname);
#else
  // specify latest version:
  active = new char[strlen(logname)+strlen(".log;")+1];
  sprintf(active,"%s.log;",logname);
  backup = new char[strlen(logname)+strlen(".bak;")+1];
  sprintf(backup,"%s.bak;",logname);
  checkpt = new char[strlen(logname)+strlen(".ckp;")+1];
  sprintf(checkpt,"%s.ckp;",logname);
#endif

  delete [] logname;

  if (port != 0)
  {

    //
    // Starting for the first time - make sure log file doesn't exist, and
    // for safety, that there is no backup file either.

    firstTime = 1;

    if (stat(active,&sb) == 0)
    {
      cerr << ts.t() << "Error: log file '" << active
           << "' exists.  Can't use -s option." << endl;
      exit(1);
    }
    if (stat(backup,&sb) == 0)
    {
      cerr << ts.t() << "Error: backup file '" << backup
           << "' exists.  Can't use -s option." << endl;
      exit(1);
    }
  }
  else
  {

    //
    // Restart - parse log file.

    firstTime = 0;

    FILE *file = fopen(active, "r");

    if (!file)
    {
      cerr << ts.t() << "Error: cannot open log file '" << active << "'."
           << endl;

      if (stat(backup,&sb) == 0) 
      {
        cerr << "Info: backup file '" << backup << "' exists." << endl
             << "This must be removed if you want to use the -start option."
             << endl;
      }
      exit(1);
    }

    yyin = file;
    oep_global = new OEP_GlobalData();
    ::oep_global = this->oep_global;
    int result = yyparse();
    if (result == 1)
    {
      cerr << ts.t() << "Error: parsing log file '" << active << "'."
           << endl;
      exit(1);
    }
    fclose(file);
    cerr << ts.t() << "Read log file successfully." << endl;

    //
    // Set the server port number
    OEP_cfps *fdata = oep_global->getFactory();
    port = fdata->getPort();
  }
}

void
omniEventsLog::init(CORBA::ORB_ptr o,CORBA::BOA_ptr b,EventChannelFactory_i *&f)
{
  omniEventsLog::orb = o;
  omniEventsLog::boa = b;
  factory = f;

  if (firstTime)
  {
    //
    // starting for the first time - create an initial log file.
    cerr << ts.t() << "Starting omniEvents on port "
         << port << endl;

    try {

#if defined(USE_ATTACH_FD)
#ifdef __WIN32__
      int fd = _open(active, O_WRONLY | O_CREAT | O_TRUNC, _S_IWRITE);
#else
      int fd = open(active, O_WRONLY | O_CREAT | O_TRUNC | O_SYNC, 0666);
#endif /* __WIN32__ */

      if (fd < 0) throw IOError();
      logfile.attach(fd);
#else
#ifdef __WIN32__
      logfile.open(active,ios::out|ios::trunc);
#else
      logfile.open(active,ios::out|ios::trunc,0666);
#endif /* __WIN32__ */
      if (!logfile) throw IOError();

#endif /* USE_ATTACH_FD */

       _lock.lock();
       logfile << *this;
       _lock.unlock();

    } catch (IOError& ex) {
      cerr << ts.t() << "Error: cannot create initial log file '" << active
           << "': " << endl;
      perror("");
      cerr << "\nYou can set the environment variable " << LOGDIR_ENV_VAR
           << " to specify the\ndirectory where the log files are kept.\n"
           << endl;
      logfile.close();
      unlink(active);
      exit(1);
    }

    checkpointNeeded = 1;
    cerr << ts.t() << "Wrote initial log file.\n" << endl;
  }
  else
  {

    try {

#if defined(USE_ATTACH_FD)
#ifdef __WIN32__
      int fd = _open(active, O_WRONLY|O_APPEND, _S_IWRITE);
#else
      int fd = open(active, O_WRONLY|O_APPEND, 0666);
#endif /* __WIN32__ */

      if (fd < 0) throw IOError();
      logfile.attach(fd);
#else
#ifdef __WIN32__
      logfile.open(active,ios::out|ios::app);
#else
      logfile.open(active,ios::out|ios::app, 0666);
#endif /* __WIN32__ */
      if (!logfile) throw IOError();

#endif /* USE_ATTACH_FD */

    } catch (IOError& ex) {
      cerr << ts.t() << "Error: cannot open log file '" << active
           << "': " << endl;
      perror("");
      cerr << "\nYou can set the environment variable " << LOGDIR_ENV_VAR
           << " to specify the\ndirectory where the log files are kept.\n"
           << endl;
      logfile.close();
      unlink(active);
      exit(1);
    }

    //
    // Get the persisted factory data
    OEP_cfps *fdata = oep_global->getFactory();

    //
    // Create the default factory
    unsigned int port = fdata->getPort();
    const omniORB::objectKey *key = fdata->getKey();
    list<OEP_ecps *> &channels = fdata->getChannels();
    factory = new EventChannelFactory_i(port, *key, channels);
    f = factory;

    delete fdata;
  }

  //
  // Create a worker thread to checkpoint the log file at regular intervals.
  recorder = new omniEventsLogWorker(this,
                                     &omniEventsLog::checkpoint,
                                     omni_thread::PRIORITY_NORMAL);
  return;
}

void
omniEventsLog::persist()
{
  omniEventsLog *log = omniEventsLog::theLog;
  if (log != NULL)
  {
    omni_mutex_lock l(log->_lock);
    (log->logfile) << *log << flush;
    log->checkpointNeeded = 1;
  }
}

ostream& 
operator<<(ostream &os, const omniEventsLog &log)
{
  os << *(log.factory) << endl;
  return os;
}

int
omniEventsLog::getPort()
{
  return port;
}

void
omniEventsLog::checkpoint(void)
{

  int idle_time_btw_chkpt;
  char *itbc = getenv("OMNIEVENTS_ITBC");
  if (itbc == NULL || sscanf(itbc,"%d",&idle_time_btw_chkpt) != 1)
  {
    idle_time_btw_chkpt = DEFAULT_IDLE_TIME_BTW_CHKPT;
  }

  omni_mutex mutex;
  omni_condition cond(&mutex);

  mutex.lock();
  while (1) {

    if (!checkpointNeeded) {
      break;
    }
  
    cerr << ts.t() << "Checkpointing Phase 1: Prepare." << endl;
  
    ofstream ckpf;
    int fd = -1;
  
    try {
  
#if defined(USE_ATTACH_FD)
#ifdef __WIN32__
      fd = _open(checkpt, O_WRONLY | O_CREAT | O_TRUNC, _S_IWRITE);
#else
      fd = open(checkpt, O_WRONLY | O_CREAT | O_TRUNC | O_SYNC, 0666);
#endif
  
      if (fd < 0) {
        cerr << ts.t() << "Error: cannot open checkpoint file '"
             << checkpt << "' for writing." << endl;
        throw IOError();
      }
  
      ckpf.attach(fd);
#else
#ifdef __WIN32__
      ckpf.open(checkpt,ios::out|ios::trunc);
#else
      ckpf.open(checkpt,ios::out|ios::trunc,0666);
#endif /* __WIN32__ */
      if (!ckpf) {
        cerr << ts.t() << "Error: cannot open checkpoint file '"
             << checkpt << "' for writing." << endl;
        throw IOError();
      }
#endif
  
      ckpf << *this;
  
      ckpf.close();
      if (!ckpf)
        throw IOError();
  
  // a bug in sparcworks C++ means that the fd doesn't get closed.
#if defined(__sunos__) && defined(__SUNPRO_CC) && __SUNPRO_CC < 0x500
      if (close(fd) < 0)
        throw IOError();
#endif
  
    } catch (IOError& ex) {
      cerr << ts.t() << flush;
      perror("I/O error writing checkpoint file");
      cerr << "Abandoning checkpoint" << endl;
      ckpf.close();
  // a bug in sparcworks C++ means that the fd doesn't get closed.
#if defined(__sunos__) && defined(__SUNPRO_CC) && __SUNPRO_CC < 0x500
      close(fd);
#endif
      unlink(checkpt);
      return;
    }
  
    //
    // Now commit the checkpoint to become the active log.
    //
  
    cerr << ts.t() << "Checkpointing Phase 2: Commit." << endl;
    _lock.lock();

  // a bug in sparcworks C++ means that the fd doesn't get closed.
#if defined(__sunos__) && defined(__SUNPRO_CC) && __SUNPRO_CC < 0x500
    close(logfile.rdbuf()->fd());
#endif
  
    logfile.close();
  
    unlink(backup);
  
#if defined(__WIN32__)
    if (rename(active, backup) != 0) {
#elif defined(__VMS)
    if (rename(active, backup) < 0) {
#else
    if (link(active,backup) < 0) {
#endif
      // Failure here leaves old active and checkpoint file.
      cerr << ts.t() << "Error: failed to link backup file '" << backup
           << "' to old log file '" << active << "'." << endl;
      exit(1);
    }
  
#if !defined( __VMS) && !defined(__WIN32__)
    if (unlink(active) < 0) {
      // Failure here leaves active and backup pointing to the same (old) file.
      cerr << ts.t() << "Error: failed to unlink old log file '" << active
           << "'." << endl;
      perror("");
      exit(1);
    }
#endif
  
#if defined(__WIN32__)
    if (rename(checkpt,active) != 0) {
#elif defined(__VMS)
    if (rename(checkpt,active) < 0) {
#else
    if (link(checkpt,active) < 0) {
#endif
      // Failure here leaves no active but backup points to the old file.
      cerr << ts.t() << "Error: failed to link log file '" << active
           << "' to checkpoint file '" << checkpt << "'." << endl;
      exit(1);
    }
  
// #ifndef __VMS
#if !defined( __VMS) && !defined(__WIN32__)
    if (unlink(checkpt) < 0) {
      // Failure here leaves active and checkpoint pointing to the same file.
      cerr << ts.t() << "Error: failed to unlink checkpoint file '" << checkpt
           << "'." << endl;
      exit(1);
    }
#endif
  
#if defined(USE_ATTACH_FD)
#ifdef __WIN32__
    fd = _open(active, O_WRONLY | O_APPEND);
#else
    fd = open(active, O_WRONLY | O_APPEND | O_SYNC);
#endif
  
    if (fd < 0) {
      cerr << ts.t() << "Error: cannot open new log file '" << active
           << "' for writing." << endl;
      exit(1);
    }
    logfile.attach(fd);
#else
#ifdef __WIN32__
    logfile.open(active,ios_base::out|ios_base::app);
#else
    logfile.open(active,ios::out|ios::app,0666);
#endif /* __WIN32__ */
    if (!logfile.is_open()) {
      cerr << ts.t() << "Error: cannot open log file '" << active
           << "' for writing." << endl;
      perror("");
      exit(1);
    }
#endif
  
    cerr << ts.t() << "Checkpointing completed." << endl;
  
    _lock.unlock();
    checkpointNeeded = 0;

    unsigned long s, n;
    omni_thread::get_time(&s, &n, idle_time_btw_chkpt);
    cond.timedwait(s,n);
  }
  mutex.unlock();
}

//------------------------------------------------------------------------
//           OmniEvents Log Worker Implementation
//------------------------------------------------------------------------
omniEventsLogWorker::omniEventsLogWorker(omniEventsLog *object,
                                       Method method,
                                       priority_t priority) :
   omni_thread(NULL, priority)
{

   DB(10, "omniEventsLogWorker : Constructor Start");

   _method = method;
   _object = object;

   start_undetached();

   DB(10, "omniEventsLogWorker : Constructor End");
}

void*
omniEventsLogWorker::run_undetached (void *) {

   DB(10, "omniEventsLogWorker : run_undetached Start");

   (_object->*_method)();

   DB(10, "omniEventsLogWorker : run_undetached End");

   return(0);
}

omniEventsLogWorker::~omniEventsLogWorker () {

   DB(10, "omniEventsLogWorker : Destructor Start");

   DB(10, "omniEventsLogWorker : Destructor End");
}



