// -*- Mode: C++; -*-
//                            Package   : omniEvents
// CosEvent_i.cc              Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//      Implementation of the COSS Event Services
//	

/*
  $Log: CosEvent_i.cc,v $
  Revision 1.3  2000/03/02 01:43:16  naderp
  Changed Proxy containment from channel to admin objects.
  Added Support for multiple admin objects per channel.
  Mulitple bug fixes. See CHANGES_201 file.

  Revision 1.2  1999/11/02 15:16:54  naderp
  Changed maxEventsPerConsumer type to unsigned.

Revision 1.1  99/11/01  18:50:40  18:50:40  naderp (Paul Nader)
Harmonized Proxy output.

Revision 1.0  99/11/01  17:01:28  17:01:28  naderp (Paul Nader)
omniEvents R2.0.
Implemented Persistency.

Revision 0.7  99/08/27  11:33:00  11:33:00  naderp (Paul Nader)
Partitioned EventChannelFactory_i from CosEvent_i.

Revision 0.6  99/08/04  14:52:30  14:52:30  naderp (Paul Nader)
Automatically deregsitering proxy push supplier if push consumer is
uncontactable when pushing event. 

Revision 0.5  99/04/29  15:50:47  15:50:47  naderp (Paul Nader)
Added comparator for set container for architectures that don't
support default template arguments.
Typedef'd set containers.

Revision 0.4  99/04/23  15:57:18  15:57:18  naderp (Paul Nader)
gcc port.

Revision 0.3  99/04/23  09:32:42  09:32:42  naderp (Paul Nader)
Windows Port.

Revision 0.2  99/03/02  19:31:05  19:31:05  naderp (Paul Nader)


Revision 0.1.1.4  99/03/02  19:09:49  19:09:49  naderp (Paul Nader)
Replaced container size() by empty().
Inserted explicit comparator for set container required byAIX 4.2.1/xlC 3.1.4.7/STLport.
Added support for agent mode operation (pullsupp + pushcons).

Revision 0.1.1.3  99/01/29  14:51:29  14:51:29  naderp (Paul Nader)
Replaced tests for empty containers by calls to _events.empty()

Revision 0.1.1.2  98/12/01  15:31:14  15:31:14  naderp (Paul Nader)
Added Agent Mode code.

Revision 0.1.1.1  98/11/25  14:30:51  14:30:51  naderp (Paul Nader)
Removed throw parenthesis for egcs on Linux.
Added scope EventChannelAdmin to EventChannelFactory::_nil().

Revision 0.1  98/11/25  14:05:20  14:05:20  naderp (Paul Nader)
Initial Revision

*/

#include <omniEventsLog.h>
#include <naming.h>
#include <defaults.h>
#include <CosEvent_i.h>

#ifdef _MSC_VER
#include <iostream>
#include <fstream>
#else
#include <iostream.h>
#include <fstream.h>
#endif

#define DB(l,x) ((omniORB::traceLevel >= l) && (cerr << x << endl))

//------------------------------------------------------------------------
//           Proxy Push Consumer Interface Implementation
//------------------------------------------------------------------------
ProxyPushConsumer_i::ProxyPushConsumer_i (SupplierAdmin_i *admin)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Constructor Start");

    key = _key();
    this->admin = admin;
    _proxyState = notConnected;
    _supplier = CosEventComm::PushSupplier::_nil();
    this->_obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPushConsumer_i : Activated ProxyPushConsumer");

    DB(10, "ProxyPushConsumer_i : Constructor End");
}

#if defined(__WIN32__) && defined(_MSC_VER)

// Work-around MSVC++ nested class bug
typedef CosEventChannelAdmin::_sk_ProxyPushConsumer CosEventChannelAdmin__sk_ProxyPushConsumer;

ProxyPushConsumer_i::ProxyPushConsumer_i (SupplierAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
  CosEventChannelAdmin__sk_ProxyPushConsumer(k),
  key(k)

#else

ProxyPushConsumer_i::ProxyPushConsumer_i (SupplierAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
  CosEventChannelAdmin::_sk_ProxyPushConsumer(k),
  key(k)

#endif

{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Constructor Start");

    this->admin = admin;

    if (ior == NULL)
    {
       _supplier = CosEventComm::PushSupplier::_nil();
       _proxyState = notConnected;
    }
    else
    {
       if (strcmp(ior,"nil") == 0)
       {
          _supplier = CosEventComm::PushSupplier::_nil();
       }
       else
       {
          CORBA::Object_ptr obj = omniEventsLog::orb->string_to_object(ior);
          _supplier = CosEventComm::PushSupplier::_narrow(obj);
       }
       _proxyState = connected;
    }

    this->_obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPushConsumer_i : Activated ProxyPushConsumer");

    DB(10, "ProxyPushConsumer_i : Constructor End");
}

ProxyPushConsumer_i::~ProxyPushConsumer_i()
{
    DB(10, "ProxyPushConsumer_i : Destructor Start");

    DB(10, "ProxyPushConsumer_i : Destructor End");
}

void
ProxyPushConsumer_i::connect_push_supplier (
  CosEventComm::PushSupplier_ptr push_supplier)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Connecting PushSupplier");

    if (!CORBA::is_nil(_supplier))
    {
       throw CosEventChannelAdmin::AlreadyConnected ();
    }

    if (!CORBA::is_nil(push_supplier))
    {
       _supplier = CosEventComm::PushSupplier::_duplicate (push_supplier);
    }
    
    _proxyState = connected;
    admin->channel->incProxies(EventChannel_i::ProxyPushConsumer);

    omniEventsLog::persist();

    DB(10, "ProxyPushConsumer_i : PushSupplier Connected");
}

void
ProxyPushConsumer_i::push (const CORBA::Any& data)
{
    if (_proxyState != ProxyPushConsumer_i::connected)
        throw CosEventComm::Disconnected ();

    event_t event;
    event.data = data;
    event.stamp.get_time();

    admin->addEvent (event);
    omni_thread::yield();
}

void
ProxyPushConsumer_i::disconnect_push_consumer ()
{

    admin->_drg_pxy_push_consumer(this);
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Disconecting ProxyPushConsumer");

    admin->channel->decProxies(EventChannel_i::ProxyPushConsumer);
    _proxyState = disconnected;

    if (! CORBA::is_nil(_supplier))
    {
       CORBA::release(_supplier);
    }

    this->_dispose();
    omniEventsLog::persist();
    DB(10, "ProxyPushConsumer_i : ProxyPushConsumer Disconnected");
}

ostream&
operator<<(ostream &os, const ProxyPushConsumer_i &ppc)
{
   if (ppc._proxyState != ProxyPushConsumer_i::disconnected)
   {
      os << "\t\t\tpxyPushConsumer\n"
         << "\t\t\t{\n"
         << "\t\t\t\tKEY\t" << ppc.key << "\n";

      if (ppc._proxyState == ProxyPushConsumer_i::connected)
      {
         if (! CORBA::is_nil (ppc._supplier))
         {
            char *iorstr;
            iorstr =  omniEventsLog::orb->object_to_string(ppc._supplier);
            os << "\t\t\t\tIOR\t" << iorstr << "\n";
            delete iorstr;
         }
	 else
	 {
            os << "\t\t\t\tIOR\tnil\n";
	 }
      }

      os << "\t\t\t}\n";
   }
   return os;
}

//------------------------------------------------------------------------
//           Proxy Push Supplier Interface Implementation
//------------------------------------------------------------------------
ProxyPushSupplier_i::ProxyPushSupplier_i (ConsumerAdmin_i *admin) :
   _nonEmpty(&_headLock),
   _maxEvents(MAX_EVENTS_PER_CONSUMER)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Constructor Start");

    key = _key();
    this->admin = admin;
    _proxyState = notConnected;
    _consumer = CosEventComm::PushConsumer::_nil();
    _dispatcher = new ProxyPushSupplierWorker(this,
                                   &ProxyPushSupplier_i::_dispatch,
                                   omni_thread::PRIORITY_NORMAL);
    _obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPushSupplier_i : Activated ProxyPushSupplier");

    DB(10, "ProxyPushSupplier_i : Constructor End");
}

#if defined(__WIN32__) && defined(_MSC_VER)

// Work-around MSVC++ nested class bug
typedef CosEventChannelAdmin::_sk_ProxyPushSupplier CosEventChannelAdmin__sk_ProxyPushSupplier;

ProxyPushSupplier_i::ProxyPushSupplier_i (ConsumerAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
  CosEventChannelAdmin__sk_ProxyPushSupplier(k),
  _nonEmpty(&_headLock),
  key(k)

#else

ProxyPushSupplier_i::ProxyPushSupplier_i (ConsumerAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
CosEventChannelAdmin::_sk_ProxyPushSupplier(k),
  _nonEmpty(&_headLock),
  key(k)

#endif
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Constructor Start");

    this->admin = admin;

    if (ior == NULL)
    {
       _consumer = CosEventComm::PushConsumer::_nil();
       _proxyState = notConnected;
    }
    else
    {
       CORBA::Object_ptr obj = omniEventsLog::orb->string_to_object(ior);
       _consumer = CosEventComm::PushConsumer::_narrow(obj);
       _proxyState = connected;
    }

    _dispatcher = new ProxyPushSupplierWorker(this,
                                   &ProxyPushSupplier_i::_dispatch,
                                   omni_thread::PRIORITY_NORMAL);

    _obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPushSupplier_i : Activated ProxyPushSupplier");

    DB(10, "ProxyPushSupplier_i : Constructor End");
}

ProxyPushSupplier_i::~ProxyPushSupplier_i()
{
    DB(10, "ProxyPushSupplier_i : Destructor Start");

    DB(10, "ProxyPushSupplier_i : Destructor End");
}

void
ProxyPushSupplier_i::connect_push_consumer (
    CosEventComm::PushConsumer_ptr push_consumer)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Connecting PushConsumer");

    if (!CORBA::is_nil(_consumer))
       throw CosEventChannelAdmin::AlreadyConnected();

    if (CORBA::is_nil (push_consumer)) {
       throw CORBA::BAD_PARAM(0,CORBA::COMPLETED_NO);
    }

    _consumer = CosEventComm::PushConsumer::_duplicate (push_consumer);
    _proxyState = connected;
    admin->channel->incProxies(EventChannel_i::ProxyPushSupplier);

    omniEventsLog::persist();

    DB(10, "ProxyPushSupplier_i : PushConsumer Connected");
}

void
ProxyPushSupplier_i::disconnect_push_supplier ()
{
    admin->_drg_pxy_push_supplier(this);
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Disconecting ProxyPushSupplier");

    admin->channel->decProxies(EventChannel_i::ProxyPushSupplier);
    _proxyState = disconnected;
    _nonEmpty.signal();
    _dispatcher->join(0);
    _dispatcher = 0;

    if (! CORBA::is_nil(_consumer))
    {
       CORBA::release(_consumer);
    }

    this->_dispose();
    omniEventsLog::persist();
    DB(10, "ProxyPushSupplier_i : ProxyPushSupplier Disconnected");
}

void
ProxyPushSupplier_i::addEvent (const event_t &event)
{
    DB(10, "ProxyPushSupplier_i : addEvent Start");

    omni_mutex_lock l(_headLock);

    if ((_proxyState == connected) && (_stamp <= event.stamp))
    {
       _events.push_front(event);
       if ((_maxEvents > 0) && (_events.size() == _maxEvents))
       {
          _events.pop_back();
       }
       _nonEmpty.signal();
    }

    DB(10, "ProxyPushSupplier_i : addEvent Start");
}

void
ProxyPushSupplier_i::_dispatch ()
{
   DB(10, "ProxyPushSupplier_i : _dispatch Start");

   event_t event;

   while (_proxyState != disconnected) {

      _headLock.lock();
      while (_events.empty() || (CORBA::is_nil(_consumer))) {
            if (_proxyState == disconnected)
               break;
            _nonEmpty.wait();
      }

      if (_proxyState == disconnected) {
          _headLock.unlock();
          break;
      }

      DB(20,"ProxyPushSupplier_i : Queue Length : " << _events.size());

      while (! _events.empty()) {

         // Get next event.
         event = _events.back();
         _events.pop_back();
         _headLock.unlock();
      
         try {
            if (! CORBA::is_nil(_consumer))
            {
               // Push event data
               _consumer->push(event.data);
            }
         }
         catch (...)
         {
//           DB(20, "ProxyPushSupplier_i : Exception notifying PushConsumer!");
//           DB(20, "ProxyPushSupplier_i : Disconnectig PushConsumer.");
//           // Cant disconnect directly because disconnect_push_supplier
//           // signals this thread and disposes of the object. Create a
//           // new thread to do the disconnection instead.
//           omni_thread *disconnect_thread = new ProxyPushSupplierWorker(this,
//                              &ProxyPushSupplier_i::disconnect_push_supplier,
//                              omni_thread::PRIORITY_HIGH);
//           disconnect_thread->join(0);
         }
         _headLock.lock();
      }

      _headLock.unlock();
      omni_thread::yield();
   }

   DB(10, "ProxyPushSupplier_i : _dispatch End");
}

void
ProxyPushSupplier_i::setMaxEvents(unsigned long m)
{
  _maxEvents = m;
}

ostream&
operator<<(ostream &os, const ProxyPushSupplier_i &pps)
{
   if (pps._proxyState != ProxyPushSupplier_i::disconnected)
   {
      os << "\t\t\tpxyPushSupplier\n"
         << "\t\t\t{\n"
         << "\t\t\t\tKEY\t" << pps.key << "\n";

      if (! CORBA::is_nil (pps._consumer))
      {
         char *iorstr;
         iorstr =  omniEventsLog::orb->object_to_string(pps._consumer);
         os << "\t\t\t\tIOR\t" << iorstr << "\n";
         delete iorstr;
      }
 
      os << "\t\t\t}\n";
   }
   return os;
}

//------------------------------------------------------------------------
//           Proxy Push Supplier Worker Implementation
//------------------------------------------------------------------------
ProxyPushSupplierWorker::ProxyPushSupplierWorker(ProxyPushSupplier_i *object,
                                       Method method,
                                       priority_t priority) :
   omni_thread(NULL, priority)
{

   DB(10, "ProxyPushSupplierWorker : Constructor Start");

   _method = method;
   _object = object;

   start_undetached();

   DB(10, "ProxyPushSupplierWorker : Constructor End");
}

void*
ProxyPushSupplierWorker::run_undetached (void *) {

   DB(10, "ProxyPushSupplierWorker : run_undetached Start");

   (_object->*_method)();

   DB(10, "ProxyPushSupplierWorker : run_undetached End");

   return(0);
}

ProxyPushSupplierWorker::~ProxyPushSupplierWorker () {

   DB(10, "ProxyPushSupplierWorker : Destructor Start");

   DB(10, "ProxyPushSupplierWorker : Destructor End");
}


//------------------------------------------------------------------------
//           Proxy Pull Supplier Interface Implementation
//------------------------------------------------------------------------
ProxyPullSupplier_i::ProxyPullSupplier_i (ConsumerAdmin_i *admin) :
    _nonEmpty(&_headLock),
   _maxEvents(MAX_EVENTS_PER_CONSUMER)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Constructor Start");

    key = _key();
    this->admin = admin;
    _proxyState = notConnected;
    _consumer = CosEventComm::PullConsumer::_nil();
    _obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPullSupplier_i : Activated ProxyPullSupplier");

    DB(10, "ProxyPullSupplier_i : Constructor End");
}

#if defined(__WIN32__) && defined(_MSC_VER)

// Work-around MSVC++ nested class bug
typedef CosEventChannelAdmin::_sk_ProxyPullSupplier CosEventChannelAdmin__sk_ProxyPullSupplier;

ProxyPullSupplier_i::ProxyPullSupplier_i (ConsumerAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
  CosEventChannelAdmin__sk_ProxyPullSupplier(k),
  _nonEmpty(&_headLock),
  key(k)

#else

ProxyPullSupplier_i::ProxyPullSupplier_i (ConsumerAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
CosEventChannelAdmin::_sk_ProxyPullSupplier(k),
  _nonEmpty(&_headLock),
  key(k)

#endif

{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Constructor Start");

    this->admin = admin;

    if (ior == NULL)
    {
       _consumer = CosEventComm::PullConsumer::_nil();
       _proxyState = notConnected;
    }
    else
    {
       if (strcmp(ior,"nil") == 0)
       {
          _consumer = CosEventComm::PullConsumer::_nil();
       }
       else
       {
          CORBA::Object_ptr obj = omniEventsLog::orb->string_to_object(ior);
          _consumer = CosEventComm::PullConsumer::_narrow(obj);
       }
       _proxyState = connected;
    }

    _obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPullSupplier_i : Activated ProxyPullSupplier");

    DB(10, "ProxyPullSupplier_i : Constructor End");
}

ProxyPullSupplier_i::~ProxyPullSupplier_i()
{
    DB(10, "ProxyPullSupplier_i : Destructor Start");

    DB(10, "ProxyPullSupplier_i : Destructor End");
}


void
ProxyPullSupplier_i::connect_pull_consumer (
    CosEventComm::PullConsumer_ptr pull_consumer)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Connecting PullConsumer");

    if (!CORBA::is_nil (_consumer))
    {
       throw CosEventChannelAdmin::AlreadyConnected();
    }

    if (!CORBA::is_nil (pull_consumer))
    {
       _consumer = CosEventComm::PullConsumer::_duplicate (pull_consumer);
    }
    _proxyState = connected;
    admin->channel->incProxies(EventChannel_i::ProxyPullSupplier);

    omniEventsLog::persist();

    DB(10, "ProxyPullSupplier_i : PullConsumer Connected");
}

void
ProxyPullSupplier_i::disconnect_pull_supplier ()
{
    admin->_drg_pxy_pull_supplier(this);
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Disconecting ProxyPullSupplier");

    admin->channel->decProxies(EventChannel_i::ProxyPullSupplier);
    _proxyState = disconnected;

    if (! CORBA::is_nil(_consumer))
    {
       CORBA::release(_consumer);
    }

    this->_dispose();
    omniEventsLog::persist();

    DB(10, "ProxyPullSupplier : ProxyPullSupplier Disconnected");
}

CORBA::Any*
ProxyPullSupplier_i::pull ()
{
    // Check if consumer is disconnected
    if (_proxyState != connected)
    {
       throw CosEventComm::Disconnected();
    }

    // Check for queued events. If empty, request an event and wait. 
    _headLock.lock();
    if (_events.empty())
    {
       admin->reqEvent();
       _nonEmpty.wait();
    }

    DB(10, "ProxyPullSupplier_i : Pull Queue Size : " << _events.size());

    event_t event;
    event = _events.back();
    _events.pop_back();
    _headLock.unlock();

    return new CORBA::Any(event.data);
}

CORBA::Any*
ProxyPullSupplier_i::try_pull (CORBA::Boolean &has_event)
{
    omni_mutex_lock l(_lock);

    // Check if consumer is disconnected
    if (_proxyState != connected)
    {
        throw CosEventComm::Disconnected();
    }

    event_t event;
    _headLock.lock();

    DB(10, "ProxyPullSupplier_i : Pull Queue Size : " << _events.size());
    
    if (! _events.empty())
    {
       event = _events.back();
       _events.pop_back();
       has_event = true;
       _headLock.unlock();
       omni_thread::yield();
       return new CORBA::Any(event.data);
    }
    else
    {
       has_event = false;
       _headLock.unlock();
       omni_thread::yield();
       return new CORBA::Any();
    }
}

void
ProxyPullSupplier_i::addEvent (const event_t &event)
{
    DB(10, "ProxyPullSupplier_i : addEvent Start");

    if ((_proxyState == connected) && (_stamp <= event.stamp))
    {
       _headLock.lock();
       _events.push_front(event);
       if ((_maxEvents > 0) && (_events.size() == _maxEvents))
       {
          _events.pop_back();
       }
       _headLock.unlock();
       _nonEmpty.signal();
    }

    DB(10, "ProxyPullSupplier_i : addEvent End");
}

void
ProxyPullSupplier_i::setMaxEvents(unsigned long m)
{
  _maxEvents = m;
}


ostream&
operator<<(ostream &os, const ProxyPullSupplier_i &pps)
{
   if (pps._proxyState != ProxyPullSupplier_i::disconnected)
   {
      os << "\t\t\tpxyPullSupplier\n"
         << "\t\t\t{\n"
         << "\t\t\t\tKEY\t" << pps.key << "\n";

      if (pps._proxyState == ProxyPullSupplier_i::connected)
      {
         if (! CORBA::is_nil (pps._consumer))
         {
            char *iorstr;
            iorstr =  omniEventsLog::orb->object_to_string(pps._consumer);
            os << "\t\t\t\tIOR\t" << iorstr << "\n";
            delete iorstr;
         }
	 else
	 {
            os << "\t\t\t\tIOR\tnil\n";
	 }
      }

      os << "\t\t\t}\n";
   }
   return os;
}

//------------------------------------------------------------------------
//           Proxy Pull Consumer Interface Implementation
//------------------------------------------------------------------------
ProxyPullConsumer_i::ProxyPullConsumer_i (SupplierAdmin_i *admin)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Constructor Start");

    key = _key();
    this->admin = admin;
    _proxyState = notConnected;
    _supplier = CosEventComm::PullSupplier::_nil();
    this->_obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPullConsumer_i : Activated ProxyPullConsumer");

    DB(10, "ProxyPullConsumer_i : Constructor End");
}

#if defined(__WIN32__) && defined(_MSC_VER)

// Work-around MSVC++ 4.2 nested class bug
typedef CosEventChannelAdmin::_sk_ProxyPullConsumer CosEventChannelAdmin__sk_ProxyPullConsumer;

ProxyPullConsumer_i::ProxyPullConsumer_i (SupplierAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
  CosEventChannelAdmin__sk_ProxyPullConsumer(k),
  key(k)

#else

ProxyPullConsumer_i::ProxyPullConsumer_i (SupplierAdmin_i *admin,
                                          const omniORB::objectKey &k,
                                          const char *ior) :
CosEventChannelAdmin::_sk_ProxyPullConsumer(k),
  key(k)

#endif

{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Constructor Start");

    this->admin = admin;

    if (ior == NULL)
    {
       _supplier = CosEventComm::PullSupplier::_nil();
       _proxyState = notConnected;
    }
    else
    {
       CORBA::Object_ptr obj = omniEventsLog::orb->string_to_object(ior);
       _supplier = CosEventComm::PullSupplier::_narrow(obj);
       _proxyState = connected;
    }

    this->_obj_is_ready(CORBA::BOA::getBOA());
    DB(10, "ProxyPullConsumer_i : Activated ProxyPullConsumer");

    DB(10, "ProxyPullConsumer_i : Constructor End");
}

ProxyPullConsumer_i::~ProxyPullConsumer_i()
{
    DB(10, "ProxyPullConsumer_i : Destructor Start");

    DB(10, "ProxyPullConsumer_i : Destructor End");
}

void
ProxyPullConsumer_i::connect_pull_supplier (
    CosEventComm::PullSupplier_ptr pull_supplier)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Connecting PullSupplier");

    if (CORBA::is_nil (pull_supplier))
    {
        throw CORBA::BAD_PARAM(0,CORBA::COMPLETED_NO);
    }

    if (!CORBA::is_nil (_supplier))
    {
        throw CosEventChannelAdmin::AlreadyConnected();
    }

    _supplier = CosEventComm::PullSupplier::_duplicate (pull_supplier);
    _proxyState = connected;
    admin->channel->incProxies(EventChannel_i::ProxyPullConsumer);

    omniEventsLog::persist();

    DB(10, "ProxyPullConsumer_i : PullSupplier Connected");
}

event_t *
ProxyPullConsumer_i::getEvent(CORBA::Boolean &has_event)
{
    CORBA::Any *data;
    data = _supplier->try_pull(has_event);

    event_t *event = new event_t;
    event->data = *data;
    (event->stamp).get_time();

    delete data;
    return (event);
}

void
ProxyPullConsumer_i::disconnect_pull_consumer ()
{
    admin->_drg_pxy_pull_consumer(this);
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Disconecting ProxyPullConsumer");

    admin->channel->decProxies(EventChannel_i::ProxyPullConsumer);
    _proxyState = disconnected;

    if (! CORBA::is_nil(_supplier))
    {
       CORBA::release(_supplier);
    }

    this->_dispose();
    omniEventsLog::persist();
    DB(10, "ProxyPullConsumer_i : ProxyPullConsumer Disconnected");
}

ostream&
operator<<(ostream &os, const ProxyPullConsumer_i &ppc)
{
   if (ppc._proxyState != ProxyPullConsumer_i::disconnected)
   {
      os << "\t\t\tpxyPullConsumer\n"
         << "\t\t\t{\n"
         << "\t\t\t\tKEY\t" << ppc.key << "\n";

      if (! CORBA::is_nil (ppc._supplier))
      {
         char *iorstr;
         iorstr =  omniEventsLog::orb->object_to_string(ppc._supplier);
         os << "\t\t\t\tIOR\t" << iorstr << "\n";
         delete iorstr;
      }

      os << "\t\t\t}\n";
   }
   return os;
}

//------------------------------------------------------------------------
//           Consumer Admin Interface Implementation
//------------------------------------------------------------------------
ConsumerAdmin_i::ConsumerAdmin_i (EventChannel_i *ec)
{
    DB(10, "ConsumerAdmin_i : Constructor Start");

    key = _key();
    channel = ec;
    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ConsumerAdmin_i : Constructor End");
}

ConsumerAdmin_i::ConsumerAdmin_i(EventChannel_i *ec, const omniORB::objectKey &k, const OEP_caps & caps) :
CosEventChannelAdmin::_sk_ConsumerAdmin(k),
  key(k)
{
    DB(10, "ConsumerAdmin_i : Constructor Start");

    channel = ec;

    //
    // Build Proxies
    list<OEP_prxy *> pdat;
    list<OEP_prxy *>::iterator iter;

    //
    // Build Push Supplier proxies
    pdat = caps.getProxyPushSuppliers();
    for (iter=pdat.begin(); iter != pdat.end(); iter++)
    {
       ProxyPushSupplier_i *p;
       const omniORB::objectKey *key = (*iter)->getKey();
       const char *ior = (*iter)->getIor();
       p = new ProxyPushSupplier_i(this, *key, ior);
       _reg_pxy_push_supplier(p);
       if (ior != NULL)
       {
          channel->incProxies(EventChannel_i::ProxyPushSupplier);
       }
    }

    //
    // Build Pull Supplier proxies
    pdat = caps.getProxyPullSuppliers();
    for (iter=pdat.begin(); iter != pdat.end(); iter++)
    {
       ProxyPullSupplier_i *p;
       const omniORB::objectKey *key = (*iter)->getKey();
       const char *ior = (*iter)->getIor();
       p = new ProxyPullSupplier_i(this, *key, ior);
       _reg_pxy_pull_supplier(p);
       if (ior != NULL)
       {
          channel->incProxies(EventChannel_i::ProxyPullSupplier);
       }
    }

    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ConsumerAdmin_i : Constructor End");
}

ConsumerAdmin_i::~ConsumerAdmin_i ()
{
    DB(10, "ConsumerAdmin_i : Destructor Start");

    DB(10, "ConsumerAdmin_i : Destructor End");
}


CosEventChannelAdmin::ProxyPushSupplier_ptr 
ConsumerAdmin_i::obtain_push_supplier ()
{
    omni_mutex_lock l(lock);

    DB(10, "ConsumerAdmin_i : obtain_push_supplier Start");

    // Create Proxy Push Supplier
    ProxyPushSupplier_i *pps;
    CosEventChannelAdmin::ProxyPushSupplier_ptr  proxy;
    pps = new ProxyPushSupplier_i (this);
    unsigned long maxEvents = channel->getMaxEventsPerConsumer();
    pps->setMaxEvents(maxEvents);
    _reg_pxy_push_supplier(pps);

    proxy = pps;
    CosEventChannelAdmin::ProxyPushSupplier::_duplicate(proxy);

    omniEventsLog::persist();

    DB(10, "ConsumerAdmin_i : obtain_push_supplier End");
    return proxy;
}

CosEventChannelAdmin::ProxyPullSupplier_ptr 
ConsumerAdmin_i::obtain_pull_supplier ()
{
    omni_mutex_lock l(lock);

    DB(10, "ConsumerAdmin_i : obtain_pull_supplier Start");

    // Create Proxy Pull Supplier
    ProxyPullSupplier_i *pps;
    CosEventChannelAdmin::ProxyPullSupplier_ptr proxy;
    pps = new ProxyPullSupplier_i (this);
    unsigned long maxEvents = channel->getMaxEventsPerConsumer();
    pps->setMaxEvents(maxEvents);
    _reg_pxy_pull_supplier(pps);

    proxy = pps;
    CosEventChannelAdmin::ProxyPullSupplier::_duplicate(proxy);

    omniEventsLog::persist();

    DB(10, "ConsumerAdmin_i : obtain_pull_supplier End");
    return proxy;
}

void
ConsumerAdmin_i::_reg_pxy_push_supplier (ProxyPushSupplier_i *pxy_push_supp)
{
    omni_mutex_lock l(proxyPushSuppliersLock);

    DB(10, "ConsumerAdmin_i : Registering ProxyPushSupplier Start");

    _pxy_push_supp.insert(pxy_push_supp);

    DB(10, "ConsumerAdmin_i : Registering ProxyPushSupplier End");
}

void
ConsumerAdmin_i::_reg_pxy_pull_supplier (ProxyPullSupplier_i *pxy_pull_supp)
{
    omni_mutex_lock l(proxyPullSuppliersLock);

    DB(10, "ConsumerAdmin_i : Registering ProxyPullSupplier Start");

    _pxy_pull_supp.insert(pxy_pull_supp);

    DB(10, "ConsumerAdmin_i : Registering ProxyPullSupplier End");
}

void
ConsumerAdmin_i::_drg_pxy_push_supplier (ProxyPushSupplier_i *pxy_push_supp)
{
    omni_mutex_lock l(proxyPushSuppliersLock);

    DB(10, "ConsumerAdmin_i : De-Registering ProxyPushSupplier Start");

    _pxy_push_supp.erase(pxy_push_supp);

    DB(10, "ConsumerAdmin_i : De-Registering ProxyPushSupplier End");
}

void
ConsumerAdmin_i::_drg_pxy_pull_supplier (ProxyPullSupplier_i *pxy_pull_supp)
{
    omni_mutex_lock l(proxyPullSuppliersLock);

    DB(10, "ConsumerAdmin_i : De-Registering ProxyPullSupplier Start");

    _pxy_pull_supp.erase(pxy_pull_supp);

    DB(10, "ConsumerAdmin_i : De-Registering ProxyPullSupplier End");
}

void
ConsumerAdmin_i::addEvent (const event_t &event)
{
    omni_mutex_lock l(lock);

    DB(10, "ConsumerAdmin_i : addEvent Start");

    //
    // Push Model Consumers
    //
    ProxyPushSupplierSet::iterator i;
    proxyPushSuppliersLock.lock();
    for (i = _pxy_push_supp.begin(); i != _pxy_push_supp.end(); i++)
    {
        try {
           if (! CORBA::is_nil(*i))
           {
              (*i)->addEvent(event);
           }
        }
        catch (...)
        {
           DB(20, "Exception notifying ProxyPushSupplier !");
        }
    }
    proxyPushSuppliersLock.unlock();

    //
    // Pull Model Consumers
    //
    ProxyPullSupplierSet::iterator j;
    proxyPullSuppliersLock.lock();
    for (j = _pxy_pull_supp.begin(); j != _pxy_pull_supp.end(); j++)
    {
        try {
           if (! CORBA::is_nil(*j))
           {
              (*j)->addEvent(event);
           }
        }
        catch (...)
        {
           DB(20, "Exception notifying ProxyPullSupplier !");
        }
    }
    proxyPullSuppliersLock.unlock();

    DB(10, "ConsumerAdmin_i : addEvent End");
}

void
ConsumerAdmin_i::reqEvent (void)
{
    DB(10, "ConsumerAdmin_i : reqEvent Start");

    channel->reqEvent();

    DB(10, "ConsumerAdmin_i : reqEvent End");
}

void
ConsumerAdmin_i::destroy ()
{
    DB(10, "ConsumerAdmin_i : Destroy Start");
    DB(10, "ConsumerAdmin_i : Destroy End");
}


ostream&
operator<<(ostream &os, ConsumerAdmin_i &ca)
{
    os << "\t\tconsumerAdmin\n"
       << "\t\t{\n"
       << "\t\t\tKEY\t" << ca.key << "\n";

    {
       ProxyPushSupplierSet::iterator i;
       omni_mutex_lock l(ca.proxyPushSuppliersLock);
       ProxyPushSupplierSet pps = ca._pxy_push_supp;
       for (i = pps.begin(); i != pps.end(); i++)
       {
          os << **i;
       }
    }

    {
       ProxyPullSupplierSet::iterator k;
       omni_mutex_lock l(ca.proxyPullSuppliersLock);
       ProxyPullSupplierSet ppls = ca._pxy_pull_supp;
       for (k = ppls.begin(); k != ppls.end(); k++)
       {
          os << **k;
       }
    }

    os << "\t\t}\n";

    return os;
}

//------------------------------------------------------------------------
//           Supplier Admin Interface Implementation
//------------------------------------------------------------------------
SupplierAdmin_i::SupplierAdmin_i (EventChannel_i *ec)
{
    DB(10, "SupplierAdmin_i : Constructor Start");

    key = _key();
    channel = ec;
    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "SupplierAdmin_i : Constructor End");
}

SupplierAdmin_i::SupplierAdmin_i(EventChannel_i *ec, const omniORB::objectKey &k, const OEP_saps &saps ) :
CosEventChannelAdmin::_sk_SupplierAdmin(k),
  key(k)
{
    DB(10, "SupplierAdmin_i : Constructor Start");

    channel = ec;

    //
    // Build Proxies
    list<OEP_prxy *> pdat;
    list<OEP_prxy *>::iterator iter;

    //
    // Build Push Consumer proxies
    pdat = saps.getProxyPushConsumers();
    for (iter=pdat.begin(); iter != pdat.end(); iter++)
    {
       ProxyPushConsumer_i *p;
       const omniORB::objectKey *key = (*iter)->getKey();
       const char *ior = (*iter)->getIor();
       p = new ProxyPushConsumer_i(this, *key, ior);
       _reg_pxy_push_consumer(p);
       if (ior != NULL)
       {
          channel->incProxies(EventChannel_i::ProxyPushConsumer);
       }
    }

    //
    // Build Pull Consumer proxies
    pdat = saps.getProxyPullConsumers();
    for (iter=pdat.begin(); iter != pdat.end(); iter++)
    {
       ProxyPullConsumer_i *p;
       const omniORB::objectKey *key = (*iter)->getKey();
       const char *ior = (*iter)->getIor();
       p = new ProxyPullConsumer_i(this, *key, ior);
       _reg_pxy_pull_consumer(p);
       if (ior != NULL)
       {
          channel->incProxies(EventChannel_i::ProxyPullConsumer);
       }
    }

    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "SupplierAdmin_i : Constructor End");
}

SupplierAdmin_i::~SupplierAdmin_i ()
{
    DB(10, "SupplierAdmin_i : Destructor Start");

    DB(10, "SupplierAdmin_i : Destructor End");
}

CosEventChannelAdmin::ProxyPushConsumer_ptr 
SupplierAdmin_i::obtain_push_consumer ()
{
    omni_mutex_lock l(lock);

    DB(10, "SupplierAdmin_i : obtain_push_consumer Start");

    // Create a new proxy
    ProxyPushConsumer_i *ppc;
    CosEventChannelAdmin::ProxyPushConsumer_ptr proxy;
    ppc = new ProxyPushConsumer_i (this);
    _reg_pxy_push_consumer (ppc);
    proxy = ppc;
    CosEventChannelAdmin::ProxyPushConsumer::_duplicate(proxy);

    omniEventsLog::persist();

    DB(10, "SupplierAdmin_i : obtain_push_consumer End");
    return proxy;
}

CosEventChannelAdmin::ProxyPullConsumer_ptr 
SupplierAdmin_i::obtain_pull_consumer ()
{
    omni_mutex_lock l(lock);

    DB(10, "SupplierAdmin_i : obtain_pull_consumer Start");

    // Create a new proxy
    ProxyPullConsumer_i *ppc;
    CosEventChannelAdmin::ProxyPullConsumer_ptr proxy;
    ppc = new ProxyPullConsumer_i (this);
    _reg_pxy_pull_consumer (ppc);
    proxy = ppc;
    CosEventChannelAdmin::ProxyPullConsumer::_duplicate(proxy);

    omniEventsLog::persist();

    DB(10, "SupplierAdmin_i : obtain_pull_consumer End");
    return proxy;
}
void
SupplierAdmin_i::addEvent (const event_t &event)
{
    DB(10, "SupplierAdmin_i : addEvent Start");
    
    channel->addEvent(event);

    DB(10, "SupplierAdmin_i : addEvent End");
}

event_t *
SupplierAdmin_i::getEvent(CORBA::Boolean &has_event)
{
    omni_mutex_lock l(proxyPullConsumersLock);

    DB(10, "SupplierAdmin_i : getEvent Start");

    event_t *event = NULL;
    ProxyPullConsumerSet::iterator i;
    for (i = _pxy_pull_cons.begin(); i != _pxy_pull_cons.end(); i++)
    {
        try {
           if (! CORBA::is_nil(*i))
           {
              event = (*i)->getEvent(has_event);
              if (has_event)
              {
	         break;
              }
           }
        }
        catch (...)
        {
           DB(20, "Exception pulling SupplierAdmin !");
        }
    }

    DB(10, "SupplierAdmin_i : getEvent End");

    return event;
}

void
SupplierAdmin_i::_reg_pxy_push_consumer ( ProxyPushConsumer_i *pxy_push_cons)
{
    omni_mutex_lock l(proxyPushConsumersLock);

    DB(10, "SupplierAdmin_i : Registering ProxyPushConsumer Start");

    _pxy_push_cons.insert(pxy_push_cons);

    DB(10, "SupplierAdmin_i : Registering ProxyPushConsumer End");
}

void
SupplierAdmin_i::_reg_pxy_pull_consumer (ProxyPullConsumer_i *pxy_pull_cons)
{
    omni_mutex_lock l(proxyPullConsumersLock);

    DB(10, "SupplierAdmin_i : Registering ProxyPullConsumer Start");

    _pxy_pull_cons.insert(pxy_pull_cons);

    DB(10, "SupplierAdmin_i : Registering ProxyPullConsumer End");
}

void
SupplierAdmin_i::_drg_pxy_push_consumer (ProxyPushConsumer_i *pxy_push_cons)
{
    omni_mutex_lock l(proxyPushConsumersLock);

    DB(10, "SupplierAdmin_i : De-Registering ProxyPushConsumer Start");

    _pxy_push_cons.erase(pxy_push_cons);

    DB(10, "SupplierAdmin_i : De-Registering ProxyPushConsumer End");
}

void
SupplierAdmin_i::_drg_pxy_pull_consumer (ProxyPullConsumer_i *pxy_pull_cons)
{
    omni_mutex_lock l(proxyPullConsumersLock);

    DB(10, "SupplierAdmin_i : De-Registering ProxyPullConsumer Start");

    _pxy_pull_cons.erase(pxy_pull_cons);

    DB(10, "SupplierAdmin_i : De-Registering ProxyPullConsumer End");
}

void
SupplierAdmin_i::destroy ()
{
    DB(10, "SupplierAdmin_i : Destroy Start");
    DB(10, "SupplierAdmin_i : Destroy End");
}

ostream&
operator<<(ostream &os, SupplierAdmin_i &sa)
{
    os << "\t\tsupplierAdmin\n"
       << "\t\t{\n"
       << "\t\t\tKEY\t" << sa.key << "\n"; 

    {
       ProxyPushConsumerSet::iterator j;
       omni_mutex_lock l(sa.proxyPushConsumersLock);
       ProxyPushConsumerSet ppc = sa._pxy_push_cons;
       for (j = ppc.begin(); j != ppc.end(); j++)
       {
          os << **j;
       }
    }

    {
       ProxyPullConsumerSet::iterator k;
       omni_mutex_lock l(sa.proxyPullConsumersLock);
       ProxyPullConsumerSet pplc = sa._pxy_pull_cons;
       for (k = pplc.begin(); k != pplc.end(); k++)
       {
          os << **k;
       }
    }

    os << "\t\t}\n";

    return os;
}

//------------------------------------------------------------------------
//           Event Channel Interface Implementation
//------------------------------------------------------------------------

EventChannel_i::EventChannel_i () :
   _maxEventsPerConsumer(MAX_EVENTS_PER_CONSUMER),
   _pullRetryPeriod(PULL_RETRY_PERIOD),
   dispatchThread(NULL),
   agentThread(NULL),
   pullThread(NULL),
   agentSemaphore(0),
   pullSemaphore(0),
   pushSuppliers(0),
   pushConsumers(0),
   pullSuppliers(0),
   pullConsumers(0)
{
    DB(10, "EventChannel_i : Constructor Start");

    key = _key();
    this->_obj_is_ready(CORBA::BOA::getBOA());
    _nonEmpty = new omni_condition(&_headLock);

    // Create dispatcher thread.
    dispatchThread = new EventChannelWorker(*this,
                                         &EventChannel_i::_dispatch,
                                         omni_thread::PRIORITY_NORMAL);
    // Create agent worker thread.
    agentThread = new EventChannelWorker(*this,
                                         &EventChannel_i::_agent,
                                         omni_thread::PRIORITY_NORMAL);
    // Create pull worker thread.
    pullThread = new EventChannelWorker(*this,
                                         &EventChannel_i::_pull,
                                         omni_thread::PRIORITY_NORMAL);

    DB(10, "EventChannel_i : Constructor End");
}

EventChannel_i::EventChannel_i(const omniObjectKey &k, const OEP_ecps &ecps) :
CosEventChannelAdmin::_sk_EventChannel(k),
   _maxEventsPerConsumer(MAX_EVENTS_PER_CONSUMER),
   _pullRetryPeriod(PULL_RETRY_PERIOD),
   dispatchThread(NULL),
   agentThread(NULL),
   pullThread(NULL),
   agentSemaphore(0),
   pullSemaphore(0),
   pushSuppliers(0),
   pushConsumers(0),
   pullSuppliers(0),
   pullConsumers(0),
   key(k)
{
    DB(10, "EventChannel_i : Constructor Start");

    //
    // Builds an EventChannel_i from parsed logfile data
    this->_obj_is_ready(CORBA::BOA::getBOA());

    _nonEmpty = new omni_condition(&_headLock);

    //
    // Build Supplier Admin Objects
    list<OEP_saps *> sad;
    list<OEP_saps *>::iterator sai;

    sad = ecps.getSupplierAdmins();
    for (sai=sad.begin(); sai != sad.end(); sai++)
    {
       SupplierAdmin_i *s;
       const omniORB::objectKey *key = (*sai)->getKey();
       s = new SupplierAdmin_i(this, *key, **sai);
       _reg_supplier_admin(s);
    }

    //
    // Build Consumer Admin Objects
    list<OEP_caps *> cad;
    list<OEP_caps *>::iterator cai;

    cad = ecps.getConsumerAdmins();
    for (cai=cad.begin(); cai != cad.end(); cai++)
    {
       ConsumerAdmin_i *c;
       const omniORB::objectKey *key = (*cai)->getKey();
       c = new ConsumerAdmin_i(this, *key, **cai);
       _reg_consumer_admin(c);
    }

    // Create dispatcher thread.
    dispatchThread = new EventChannelWorker(*this,
                                         &EventChannel_i::_dispatch,
                                         omni_thread::PRIORITY_NORMAL);
    // Create agent worker thread.
    agentThread = new EventChannelWorker(*this,
                                         &EventChannel_i::_agent,
                                         omni_thread::PRIORITY_NORMAL);
    // Create pull worker thread.
    pullThread = new EventChannelWorker(*this,
                                         &EventChannel_i::_pull,
                                         omni_thread::PRIORITY_NORMAL);

    DB(10, "EventChannel_i : Constructor End");
}

EventChannel_i::~EventChannel_i () 
{
    DB(10, "EventChannel_i : Destructor Start");

    DB(10, "EventChannel_i : Destructor End");
}

void
EventChannel_i::setPullRetryPeriod(long p)
{
    _pullRetryPeriod = p;
}

long
EventChannel_i::getPullRetryPeriod(void) const
{
    return _pullRetryPeriod;
}

void
EventChannel_i::setMaxEventsPerConsumer(unsigned long m)
{
    _maxEventsPerConsumer = m;
}

unsigned long
EventChannel_i::getMaxEventsPerConsumer(void) const
{
    return _maxEventsPerConsumer;
}

void
EventChannel_i::enable()
{
    DB(10, "EventChannel_i : enable Start");

    omni_mutex_lock l(_lock);

    if ((dispatchThread != NULL) &&
        (dispatchThread->state() == omni_thread::STATE_NEW))
    {
       dispatchThread->start();
    }

    if ((agentThread != NULL) &&
        (agentThread->state() == omni_thread::STATE_NEW))
    {
       agentThread->start();
    }

    if ((pullThread != NULL) &&
        (pullThread->state() == omni_thread::STATE_NEW))
    {
       pullThread->start();
    }

    DB(10, "EventChannel_i : enable End");
} 

CosEventChannelAdmin::ConsumerAdmin_ptr 
EventChannel_i::for_consumers ()
{
    ConsumerAdmin_i *consumer_admin;
    consumer_admin = new ConsumerAdmin_i(this);
    _reg_consumer_admin(consumer_admin);

    omniEventsLog::persist();

    return ConsumerAdmin_i::_duplicate(consumer_admin);
}

CosEventChannelAdmin::SupplierAdmin_ptr 
EventChannel_i::for_suppliers ()
{
    SupplierAdmin_i *supplier_admin;
    supplier_admin = new SupplierAdmin_i(this);
    _reg_supplier_admin(supplier_admin);

    omniEventsLog::persist();

    return SupplierAdmin_i::_duplicate(supplier_admin);
}

void
EventChannel_i::destroy ()
{
    DB(10, "EventChannel : Destroy Start");

    SupplierAdminSet::iterator si;
    SupplierAdminSet sas = this->supplierAdmins;
    for (si = sas.begin(); si != sas.end(); si++)
    {
       (*si)->destroy();
    }

    ConsumerAdminSet::iterator ci;
    ConsumerAdminSet cas = this->consumerAdmins;
    for (ci = cas.begin(); ci != cas.end(); ci++)
    {
       (*ci)->destroy();
    }
    delete _nonEmpty;
    DB(10, "EventChannel : Destroy End");
    this->_dispose();
}

void
EventChannel_i::_reg_supplier_admin ( SupplierAdmin_i *supplierAdmin)
{
    omni_mutex_lock l(supplierAdminsLock);

    DB(10, "EventChannel_i : Registering SupplierAdmin Start");

    supplierAdmins.insert(supplierAdmin);

    DB(10, "EventChannel_i : Registering SupplierAdmin End");
}

void
EventChannel_i::_reg_consumer_admin ( ConsumerAdmin_i *consumerAdmin)
{
    omni_mutex_lock l(consumerAdminsLock);

    DB(10, "EventChannel_i : Registering ConsumerAdmin Start");

    consumerAdmins.insert(consumerAdmin);

    DB(10, "EventChannel_i : Registering ConsumerAdmin End");
}

void
EventChannel_i::_drg_supplier_admin ( SupplierAdmin_i *supplierAdmin)
{
    omni_mutex_lock l(supplierAdminsLock);

    DB(10, "EventChannel_i : De-Registering SupplierAdmin Start");

    supplierAdmins.erase(supplierAdmin);

    DB(10, "EventChannel_i : De-Registering SupplierAdmin End");
}

void
EventChannel_i::_drg_consumer_admin ( ConsumerAdmin_i *consumerAdmin)
{
    omni_mutex_lock l(consumerAdminsLock);

    DB(10, "EventChannel_i : De-Registering ConsumerAdmin Start");

    consumerAdmins.erase(consumerAdmin);

    DB(10, "EventChannel_i : De-Registering ConsumerAdmin End");
}
void
EventChannel_i::addEvent (const event_t &event)
{
    _headLock.lock();
    _events.push_front(event);
    _headLock.unlock();
    _nonEmpty->signal();
}

void
EventChannel_i::reqEvent (void)
{
    DB(10, "EventChannel_i : reqEvent Start");

    omni_mutex_lock l(_headLock);
    if (_events.empty())
    {
       pullSemaphore.post();
    }

    DB(10, "EventChannel_i : reqEvent End");
}

void
EventChannel_i::incProxies(const proxy_t &proxy)
{
    DB(10, "EventChannel_i : incProxies Start");

    // Increment connected client counters.
    clientCounterLock.lock();
    switch (proxy)
    {
       case ProxyPushSupplier:
               pushConsumers++;
	       break;
       case ProxyPushConsumer:
               pushSuppliers++;
	       break;
       case ProxyPullSupplier:
               pullConsumers++;
	       break;
       case ProxyPullConsumer:
               pullSuppliers++;
	       break;
       default:
	       break;
    }
    clientCounterLock.unlock();

    // Post agent semaphore if channel is in
    // agent mode and agent thread is blocking.
    if (agentMode() && !agentSemaphore.trywait())
    {
       agentSemaphore.post();
    }

    DB(10, "EventChannel_i : incProxies End");
}

void
EventChannel_i::decProxies(const proxy_t &proxy)
{
    DB(10, "EventChannel_i : decProxies Start");

    // Decrement connected client counters.
    clientCounterLock.lock();
    switch (proxy)
    {
       case ProxyPushSupplier:
               pushConsumers--;
	       break;
       case ProxyPushConsumer:
               pushSuppliers--;
	       break;
       case ProxyPullSupplier:
               pullConsumers--;
	       break;
       case ProxyPullConsumer:
               pullSuppliers--;
	       break;
       default:
	       break;
    }
    clientCounterLock.unlock();

    // Post agent semaphore if channel is in
    // agent mode and agent thread is blocking.
    if (agentMode() && !agentSemaphore.trywait())
    {
       agentSemaphore.post();
    }

    DB(10, "EventChannel_i : decProxies End");
}

void
EventChannel_i::_dispatch (void)
{
    DB(10, "EventChannel_i : _dispatch Start");

    event_t event;

    while (1) {

       _headLock.lock();
       while (_events.empty())
       {
          _nonEmpty->wait();
       }

       DB(20, "EventChannel_i : Queue Length = " << _events.size());

       while (! _events.empty())
       {
         // Get the next event
         event = _events.back();
         _events.pop_back();
         _headLock.unlock();

         //
         // Consumer Admins
         //
         _lock.lock();
         ConsumerAdminSet::iterator i;
         for (i = consumerAdmins.begin(); i != consumerAdmins.end(); i++)
         {
	     try{
                if (! CORBA::is_nil(*i))
                {
                   (*i)->addEvent(event);
                }
             }
             catch (...)
             {
                DB(20, "Exception notifying ConsumerAdmin !");
             }
         }
         _lock.unlock();
         _headLock.lock();
       }

       _headLock.unlock();
       omni_thread::yield();
    }

    DB(10, "EventChannel_i : _dispatch End");
}

int EventChannel_i::agentMode(void)
{
    omni_mutex_lock l(clientCounterLock);
    //
    // Return true if there are only pull suppliers
    // and push consumers clients connected.
    return(pushConsumers && pullSuppliers && !pushSuppliers && !pullConsumers);
}

void EventChannel_i::_agent (void)
{
   DB(10, "EventChannel_i : _agent Start");

   SupplierAdminSet::iterator j;

   while (1)
   {
      // Block until the channel is in agent mode.
      agentSemaphore.wait();

      // Iterate through Pull Model Suppliers requesting events.
// HERE
      j = supplierAdmins.begin();
      while (1)
      {
         if (!agentMode())
         {
            break;
         }
         if ((pullSuppliers > 0) && (!supplierAdmins.empty()) &&
	     (j != supplierAdmins.end()))
         {
            DB(20, "EventChannel_i : _agent Pulling events");
            try {
               if (! CORBA::is_nil(*j))
               {
                  event_t *event;
                  CORBA::Boolean has_event = false;
                  event = (*j)->getEvent(has_event);
                  if (has_event)
                  {
                     this->addEvent(*event);
                     delete event;
                     omni_thread::yield();
                  }
               }
            }
            catch (...)
            {
               DB(20, "Exception pulling Supplier Admin!");
            }
         }

         if ((pullSuppliers == 0) || (supplierAdmins.empty()))
         {
             DB(20, "EventChannel_i : _agent Sleeping Pull Retry Period = " \
                     << _pullRetryPeriod);
             omni_thread::sleep(_pullRetryPeriod);
             j = supplierAdmins.begin();
         }
         else
         {
	    if (j == supplierAdmins.end())
	    {
               j = supplierAdmins.begin();
	    }
	    else
	    {
               j++;
	    }
         }
      }
   }

   DB(10, "EventChannel_i : _agent End");
}

void EventChannel_i::_pull (void)
{
   DB(10, "EventChannel_i : _pull Start");

   SupplierAdminSet::iterator j;

   while (1)
   {
      // Block until a pull consumer requests an event.
      pullSemaphore.wait();

      // Iterate through Pull Model Suppliers requesting events.
      j = supplierAdmins.begin();
      while (1)
      {
         if ((pullSuppliers > 0) && (!supplierAdmins.empty()) &&
	     (j != supplierAdmins.end()))
         {
            DB(20, "EventChannel_i : _pull Pulling events");
            try {
               if (! CORBA::is_nil(*j))
               {
                  event_t *event;
                  CORBA::Boolean has_event = false;
                  event = (*j)->getEvent(has_event);
                  if (has_event)
                  {
                     this->addEvent(*event);
                     delete event;
                     omni_thread::yield();
		     break;
                  }
               }
            }
            catch (...)
            {
               DB(20, "Exception pulling Supplier Admin!");
            }
         }

         if ((pullSuppliers == 0) || (supplierAdmins.empty()))
         {
             DB(20, "EventChannel_i: _pull Sleeping Pull Retry Period = " \
                    << _pullRetryPeriod);
             omni_thread::sleep(_pullRetryPeriod);
             j = supplierAdmins.begin();
         }
         else
         {
	    if (j == supplierAdmins.end())
	    {
               j = supplierAdmins.begin();
	    }
	    else
	    {
               j++;
	    }
         }
      }
   }

   DB(10, "EventChannel_i : _pull End");
}

ostream&
operator<<(ostream &os, EventChannel_i &ec)
{
  os << "\teventChannel\n"
     << "\t{\n"
     << "\t\tKEY\t" << ec.key << "\n"
     << "\t\tMAXEVENTSPERCONSUMER\t" << ec._maxEventsPerConsumer << "\n"
     << "\t\tPULLRETRYPERIOD\t" << ec._pullRetryPeriod << "\n";

  {
     SupplierAdminSet::iterator si;
     omni_mutex_lock l(ec.supplierAdminsLock);
     SupplierAdminSet sas = ec.supplierAdmins;
     for (si = sas.begin(); si != sas.end(); si++)
     {
        os << **si;
     }
  }

  {
     ConsumerAdminSet::iterator ci;
     omni_mutex_lock l(ec.consumerAdminsLock);
     ConsumerAdminSet cas = ec.consumerAdmins;
     for (ci = cas.begin(); ci != cas.end(); ci++)
     {
        os << **ci;
     }
  }

  os << "\t}\n";

  return os;
}

//------------------------------------------------------------------------
//           Event Channel Worker Implementation
//------------------------------------------------------------------------
EventChannelWorker::EventChannelWorker(EventChannel_i &object,
                                       Method method,
                                       priority_t priority) :
   omni_thread(NULL, priority)
{

   DB(10, "EventChannelWorker : Constructor Start");

   _method = method;
   _object = &object;

   DB(10, "EventChannelWorker : Constructor End");
}

void
EventChannelWorker::run (void *) {

    DB(10, "EventChannelWorker : run Start");

    (_object->*_method)();

    DB(10, "EventChannelWorker : run End");
}
