// -*- Mode: C++; -*-
//                            Package   : omniEvents
// CosEvent_i.cc              Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//      Implementation of the COSS Event Services Event Channel Factory
//	

/*
  $Log:	EventChannelFactory_i.cc,v $
Revision 1.1  99/11/01  18:51:38  18:51:38  naderp (Paul Nader)
Using defaults in factory.

Revision 1.0  99/11/01  17:03:36  17:03:36  naderp (Paul Nader)
omniEvents 2.0.
Implementade persistency.

Revision 0.1  99/08/27  11:47:48  11:47:48  naderp (Paul Nader)
Partitioned EventChannelFactory_i from CosEvent_i.

*/

#include <defaults.h>
#include <omniEventsLog.h>
#include <naming.h>
#include "CosEvent_i.h"
#include "EventChannelFactory_i.h"

#ifdef _MSC_VER
#include <iostream>
#include <fstream>
#else
#include <iostream.h>
#include <fstream.h>
#endif

#define DB(l,x) ((omniORB::traceLevel >= l) && (cerr << x << endl))

//------------------------------------------------------------------------
//           Event Channel Factory Interface Implementation
//------------------------------------------------------------------------
EventChannelFactory_i::EventChannelFactory_i()
{
  // empty
}

EventChannelFactory_i::EventChannelFactory_i(const omniObjectKey &k,
                                             const CosNaming::Name &n) :
  EventChannelAdmin::_sk_EventChannelFactory(k),
  name(n),
  key(k)
{
  //
  // Builds an EventChannelFactory_i from the parsed logfile data

  //
  // Get Name Service root context.
  CosNaming::NamingContext_ptr rootContext;
  rootContext = getRootNamingContext(omniEventsLog::orb);

  //
  // Register the object with the Naming Service.
  bindName2Object(rootContext, name, this);

  //
  // Enable implementation.
  this->_obj_is_ready(CORBA::BOA::getBOA());
}

EventChannelFactory_i::EventChannelFactory_i(const omniObjectKey &k,
                                             const CosNaming::Name &n,
                                             list<OEP_ecps *> &channels) :
  EventChannelAdmin::_sk_EventChannelFactory(k),
  name(n),
  key(k)
{
  //
  // Builds an EventChannelFactory_i from the parsed logfile data

  //
  // Get Name Service root context.
  CosNaming::NamingContext_ptr rootContext;
  rootContext = getRootNamingContext(omniEventsLog::orb);

  //
  // Register the object with the Naming Service.
  bindName2Object(rootContext, name, this);

  //
  // Create event channels
  list<OEP_ecps *>::iterator iter;
  for (iter=channels.begin(); iter != channels.end(); iter++)
  {
     long pullRetryPeriod = (*iter)->getPullRetryPeriod();
     long maxEventsPerConsumer = (*iter)->getMaxEventsPerConsumer();
     const omniORB::objectKey *key = (*iter)->getKey();
     EventChannel_i *channel = new EventChannel_i(*key, **iter);
     channel->setPullRetryPeriod(pullRetryPeriod);
     channel->setMaxEventsPerConsumer(maxEventsPerConsumer);
     channel->enable();
     eventChannels.insert(channel);
  }

  //
  // Enable implementation.
  this->_obj_is_ready(CORBA::BOA::getBOA());
}


CORBA::Boolean
EventChannelFactory_i::supports(const CosLifeCycle::Key &k)
{
   //
   // returns true if the key passed has the following contents :
   //
   // id   : "EventChannel"
   // kind : "object interface"

   if ((k.length() == 1) &&
       (strcmp(k[0].id, "EventChannel") == 0) &&
       (strcmp(k[0].kind, "object interface") == 0))
     return (true);
   else
     return (false);
}

CORBA::Object_ptr
EventChannelFactory_i::create_object (const CosLifeCycle::Key &k,
                                      const CosLifeCycle::Criteria &criteria)
{
    DB(10, "EventChannelFactory : Creating Channel Start");

    // Check the key
    if (! this->supports(k))
       throw CosLifeCycle::NoFactory(k);

    // Process criteria
    CORBA::ULong pullRetryPeriod = PULL_RETRY_PERIOD;
    CORBA::ULong maxEventsPerConsumer = MAX_EVENTS_PER_CONSUMER;
    for (size_t i=0; i < criteria.length(); i++)
    {
       if (strcmp(criteria[i].name, "PullRetryPeriod") == 0)
       {
          if (! (criteria[i].value >>= pullRetryPeriod))
             throw CosLifeCycle::InvalidCriteria(criteria);

          if (pullRetryPeriod < 0)
             throw CosLifeCycle::CannotMeetCriteria(criteria);
       }
       if (strcmp(criteria[i].name, "MaxEventsPerConsumer") == 0)
       {
          if (! (criteria[i].value >>= maxEventsPerConsumer))
             throw CosLifeCycle::InvalidCriteria(criteria);

          if (maxEventsPerConsumer < 0)
             throw CosLifeCycle::CannotMeetCriteria(criteria);
       }
    }

    try
    {
       EventChannel_i *channel = new EventChannel_i();
       channel->setPullRetryPeriod(pullRetryPeriod);
       channel->setMaxEventsPerConsumer(maxEventsPerConsumer);
       channel->enable();
       eventChannels.insert(channel);
       EventChannel_i::_duplicate (channel);

       omniEventsLog::persist();

       DB(10, "EventChannelFactory : Creating Channel End");
       return channel;
    }
    catch (...)
    {
       return (CORBA::Object::_nil());
    }
    return (CORBA::Object::_nil());
}

ostream&
operator<<(ostream &os, const EventChannelFactory_i &f)
{
  os << "channelFactory\n{"
     << "\n\tKEY\t" << f.key
     << "\n\tNAME\t" << f.name;

  EventChannelSet::iterator i;
  EventChannelSet ecs = f.eventChannels;
  for (i = ecs.begin(); i != ecs.end(); i++)
  {
     os << "\n" << **i;
  }

  os << "\n}\n";

  return os; 
}
