// -*- Mode: C++; -*-
//                            Package   : omniEvents
//   pullsupp.cc              Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//    Pull Model supplier implementation.
//	

/*
  $Log:	pullsupp.cc,v $
Revision 0.7  99/11/02  13:39:13  13:39:13  naderp (Paul Nader)
Added <signal.h>

  Revision 0.6  1999/11/02 07:57:22  naderp
  Updated usage.

Revision 0.5  99/11/01  16:10:12  16:10:12  naderp (Paul Nader)
omniEvents 2.0 Release.
Ignoring SIGPIPE for UNIX platforms.

Revision 0.4  99/04/23  16:05:44  16:05:44  naderp (Paul Nader)
gcc port.

Revision 0.3  99/04/23  09:34:01  09:34:01  naderp (Paul Nader)
Windows Port.

Revision 0.2  99/04/21  18:06:25  18:06:25  naderp (Paul Nader)
*** empty log message ***

Revision 0.1.1.1  98/11/27  16:59:33  16:59:33  naderp (Paul Nader)
Added -s option to sleep after disconnecting.

Revision 0.1  98/11/25  14:08:07  14:08:07  naderp (Paul Nader)
Initial Revision

*/

//
//
#include <CosEvent_i.h>
#ifdef __WIN32__
#include <getopt.h>
#include <iostream>
#else
#include <unistd.h>
#include <iostream.h>
#include <signal.h>
#endif

static omni_mutex mutex;
static omni_condition connect_cond(&mutex);
static void usage();

class Supplier_i : virtual public CosEventComm::_sk_PullSupplier {
public:
  Supplier_i (long disconnect = 0) : i(0), _disconnect(disconnect), l(0) {};
  CORBA::Any *pull();
  CORBA::Any *try_pull(CORBA::Boolean &has_event);
  void disconnect_pull_supplier ();

private:
  long i;
  long _disconnect;
  CORBA::ULong l;
};

void
Supplier_i::disconnect_pull_supplier () {
  cout << "EventSupplier: disconnected." << endl;
}

CORBA::Any *
Supplier_i::pull() {
   cout << "Supplier_i: pull() called. Data : ";
   CORBA::Any *any = new CORBA::Any();
   *any <<= l++;
   cout << l-1 << endl;

   // Exersize Disconnect
   if ((_disconnect > 0) && (i == _disconnect)) {
      i = 0;
      // Signal and wait for main thread to re-connect.
      mutex.unlock();
      connect_cond.signal();
   }
   i++;
   return (any);
}

CORBA::Any *
Supplier_i::try_pull(CORBA::Boolean &has_event) {
   cout << "Supplier_i: try_pull() called. Data : ";
   CORBA::Any *any = new CORBA::Any();
   *any <<= l++;
   cout << l-1 << endl;
   has_event = 1;

   // Exersize Disconnect
   if ((_disconnect > 0) && (i == _disconnect)) {
      i = 0;
      // Signal and wait for main thread to re-connect.
      mutex.unlock();
      connect_cond.signal();
   }
   i++;
   return (any);
}


int
main (int argc, char** argv)
{
  CORBA::ORB_var orb = CORBA::ORB_init (argc, argv, "omniORB2");
  CORBA::BOA_var boa = orb->BOA_init (argc, argv, "omniORB2_BOA");

  // Process Options
  int c;
  int discnum = 0;
  int sleepInterval = 0;
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";
  while ((c = getopt(argc,argv,"d:s:n:k:h")) != EOF)
  {
     switch (c)
     {
        case 'd': discnum = atoi(optarg);
                  break;

        case 's': sleepInterval = atoi(optarg);
                  break;

        case 'n': channelName = optarg;
                  break;

        case 'k': channelKind = optarg;
                  break;

        case 'h':
        default : usage();
                  exit(-1);
                  break;
     }
  }

  // Ignore broken pipes
#ifndef __WIN32__
  signal(SIGPIPE, SIG_IGN);
#endif

  Supplier_i* supplier = new Supplier_i (discnum);
  supplier->_obj_is_ready(boa);

  // Let the BOA know we are ready, blocking forever...
  boa->impl_is_ready(0,1);

  //
  // Obtain and narrow reference to Name Service.
  CosNaming::NamingContext_var rootContext;
  try {

     // Get NameService reference
     CORBA::Object_var initServ;
     initServ = orb->resolve_initial_references ("NameService");
     if (CORBA::is_nil(initServ))
     {
        cerr << "Failed to obtain NameService reference !" << endl;
        exit(1);
     }

     // Narrow NameService reference
     rootContext = CosNaming::NamingContext::_narrow(initServ);
     if (CORBA::is_nil(rootContext))
     {
        cerr << "Failed to narrow naming context !" << endl;
        exit(1);
     }
  }
  catch(CORBA::ORB::InvalidName& ex) {
     cerr << "Naming Service is invalid [does not exist] !" << endl;
     exit(1);
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE !"
          << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception !" << endl;
     throw;
  }
  catch (...) {
     cerr << "Caught a system exception while resolving the naming service !"
          << endl;
     exit(1);
  }
  

  // Obtain Event Channel Object.
  CosNaming::Name name;
  name.length (1);
  name[0].id = CORBA::string_dup (channelName);
  name[0].kind = CORBA::string_dup (channelKind);

  cout << "Looking for EventChannel" << endl;
  CosEventChannelAdmin::EventChannel_var event_channel;
  try {
    CORBA::Object_var obj = rootContext->resolve (name);
    event_channel = CosEventChannelAdmin::EventChannel::_narrow (obj);
    if (CORBA::is_nil (event_channel))
    {
       cerr << "Failed to narrow Event Channel !" << endl;
       exit(1);
    }
  }
  catch (CORBA::ORB::InvalidName& ex) {
     cerr << "Service required is invalid [does not exist]. !" << endl;
     exit(1);
  }
  catch (CORBA::COMM_FAILURE& ex) {
     cerr << "Caught system exception COMM_FAILURE, unable to contact the naming service !"
          << endl;
     exit(1);
  }
  catch (omniORB::fatalException& ex) {
     cerr << "Caught Fatal Exception !" << endl;
     throw;
  }
  catch (...) {
     cerr << "Cannot find event channel ! [\""
          << channelName << "\", \"" << channelKind << "\"]"
          << endl;
    exit(1);
  }

  //
  // Get Supplier admin interface.
  CosEventChannelAdmin::SupplierAdmin_var supplier_admin;
  try {
     supplier_admin = event_channel->for_suppliers ();
     if (CORBA::is_nil(supplier_admin))
     {
        cerr << "Supplier Admin returned supplier_admin nil reference!"
             << endl;
        exit(1);
     }
  }
  catch (...) {
     cerr << "Failed to obtain Supplier Admin !" << endl;
     exit(1);
  }
  cerr << "Obtained SupplierAdmin." << endl;

  CosEventChannelAdmin::ProxyPullConsumer_var proxy_consumer;
  while (1)
  {
     //
     // Get proxy consumer interface.
     try {
        proxy_consumer = supplier_admin->obtain_pull_consumer ();
        if (CORBA::is_nil(proxy_consumer))
        {
           cerr << "Failed to find Proxy Consumer !" << endl;
           exit(1);
        }
     }
     catch (...) {
        cerr << "Failed to obtain Proxy Consumer !" << endl;
        exit(1);
     }
     cerr << "Obtained ProxyPullConsumer." << endl;
   
     // Connect Pull Supplier
     try {
        proxy_consumer->connect_pull_supplier(CosEventComm::PullSupplier::_duplicate(supplier));
     }
     catch (CORBA::BAD_PARAM& ex) {
        cerr << "Caught BAD_PARAM Exception while connecting Pull Supplier !"
             << endl;
        exit (1);
     }
     catch (CosEventChannelAdmin::AlreadyConnected& ex) {
        cerr << "Pull Supplier already connected !"
             << endl;
     }
     catch (...) {
        cerr << "Failed to connect Pull Supplier !" << endl;
        exit (1);
     }
     cerr << "Connected Pull Supplier." << endl;

     // Wait for indication to disconnect before re-connecting.
     mutex.lock();
     connect_cond.wait();

     // Disconnect.
     proxy_consumer->disconnect_pull_consumer();
     cerr << "Disconnected from ProxyPullConsumer." << endl;

     // Yawn.
     cerr << "Sleeping " << sleepInterval << " Seconds." << endl;
     omni_thread::sleep(sleepInterval);
  }

  // Not Reached
  return (0);
}

static void
usage()
{
   cerr << "\nusage: pullsupp [-d n [-s n]] [-n name] [-k kind] [-h]\n" << endl;
   cerr << "         -d n   disconnect after n pulls" << endl;
   cerr << "         -s n   sleep n Seconds afetr disconnecting" << endl;
   cerr << "         -n name specify channel name [\"EventChannel\"]" << endl;
   cerr << "         -k kind specify channel kind [\"EventChannel\"]" << endl;
   cerr << "         -h     display usage" << endl;
   cerr << endl;
}
