// -*- Mode: C++; -*-
//                            Package   : omniEvents
// CosEvent_i.cc              Created   : 1/4/98
//                            Author    : Paul Nader (pwn)
//
//    Copyright (C) 1998 Paul Nader.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This application is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
//
// Description:
//      Implementation of the COSS Event Services
//	

/*
  $Log:	CosEvent_i.cc,v $
Revision 0.5  99/04/29  15:50:47  15:50:47  naderp (Paul Nader)
Added comparator for set container for architectures that don't
support default template arguments.
Typedef'd set containers.

Revision 0.4  99/04/23  15:57:18  15:57:18  naderp (Paul Nader)
gcc port.

Revision 0.3  99/04/23  09:32:42  09:32:42  naderp (Paul Nader)
Windows Port.

Revision 0.2  99/03/02  19:31:05  19:31:05  naderp (Paul Nader)


Revision 0.1.1.4  99/03/02  19:09:49  19:09:49  naderp (Paul Nader)
Replaced container size() by empty().
Inserted explicit comparator for set container required byAIX 4.2.1/xlC 3.1.4.7/STLport.
Added support for agent mode operation (pullsupp + pushcons).

Revision 0.1.1.3  99/01/29  14:51:29  14:51:29  naderp (Paul Nader)
Replaced tests for empty containers by calls to _events.empty()

Revision 0.1.1.2  98/12/01  15:31:14  15:31:14  naderp (Paul Nader)
Added Agent Mode code.

Revision 0.1.1.1  98/11/25  14:30:51  14:30:51  naderp (Paul Nader)
Removed throw parenthesis for egcs on Linux.
Added scope EventChannelAdmin to EventChannelFactory::_nil().

Revision 0.1  98/11/25  14:05:20  14:05:20  naderp (Paul Nader)
Initial Revision

*/

#include "CosEvent_i.h"

#ifdef _MSC_VER
#include <iostream>
#include <fstream>
#else
#include <iostream.h>
#include <fstream.h>
#endif

#define DB(l,x) ((omniORB::traceLevel >= l) && (cerr << x << endl))

//------------------------------------------------------------------------
//           Proxy Push Consumer Interface Implementation
//------------------------------------------------------------------------
ProxyPushConsumer_i::ProxyPushConsumer_i (EventChannel_i *impl)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Constructor Start");

    _channel = impl;
    _proxyState = notConnected;
    _supplier = CosEventComm::PushSupplier::_nil();
    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ProxyPushConsumer_i : Activated ProxyPushConsumer");

    _channel->_reg_pxy_push_consumer (this);

    DB(10, "ProxyPushConsumer_i : Constructor End");
}

void ProxyPushConsumer_i::connect_push_supplier 
(CosEventComm::PushSupplier_ptr push_supplier)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Connecting PushSupplier");

    if (CORBA::is_nil(push_supplier))
        throw CORBA::BAD_PARAM(0,CORBA::COMPLETED_NO);

    if (!CORBA::is_nil(_supplier))
        throw CosEventChannelAdmin::AlreadyConnected ();

    _proxyState = connected;
    _supplier = CosEventComm::PushSupplier::_duplicate (push_supplier);

    DB(10, "ProxyPushConsumer_i : PushSupplier Connected");
}

void ProxyPushConsumer_i::push (const CORBA::Any& data)
{
    if (_proxyState != ProxyPushConsumer_i::connected)
        throw CosEventComm::Disconnected ();

    _channel->addEvent (data);
    omni_thread::yield();
}

void ProxyPushConsumer_i::disconnect_push_consumer ()
{
    _channel->_drg_pxy_push_consumer(this);

    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushConsumer_i : Disconecting ProxyPushConsumer");

    _proxyState = disconnected;

    if (! CORBA::is_nil(_supplier)) {
       CORBA::release(_supplier);
    }

    this->_dispose();
    
    DB(10, "ProxyPushConsumer_i : ProxyPushConsumer Disconnected");
}

//------------------------------------------------------------------------
//           Proxy Push Supplier Interface Implementation
//------------------------------------------------------------------------
ProxyPushSupplier_i::ProxyPushSupplier_i (EventChannel_i *impl)
    : _nonEmpty(&_headLock)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Constructor Start");

    _channel = impl;
    _proxyState = notConnected;
    _consumer = CosEventComm::PushConsumer::_nil();
    _dispatcher = new ProxyPushSupplierWorker(this,
                                   &ProxyPushSupplier_i::_dispatch,
                                   omni_thread::PRIORITY_NORMAL);
    _obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ProxyPushSupplier_i : Activated ProxyPushSupplier");

    _channel->_reg_pxy_push_supplier (this);

    DB(10, "ProxyPushSupplier_i : Constructor End");
}

void ProxyPushSupplier_i::connect_push_consumer (
    CosEventComm::PushConsumer_ptr push_consumer)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Connecting PushConsumer");

    if (!CORBA::is_nil (_consumer))
       throw CosEventChannelAdmin::AlreadyConnected();

    if (CORBA::is_nil (push_consumer)) {
       throw CORBA::BAD_PARAM(0,CORBA::COMPLETED_NO);
    }

    _proxyState = connected;
    _consumer = CosEventComm::PushConsumer::_duplicate (push_consumer);

    DB(10, "ProxyPushSupplier_i : PushConsumer Connected");
}

void ProxyPushSupplier_i::disconnect_push_supplier ()
{
    _channel->_drg_pxy_push_supplier(this);

    omni_mutex_lock l(_lock);

    DB(10, "ProxyPushSupplier_i : Disconecting ProxyPushSupplier");

    _proxyState = disconnected;
    _nonEmpty.signal();
    _dispatcher->join(0);
    _dispatcher = 0;

    if (! CORBA::is_nil(_consumer)) {
       CORBA::release(_consumer);
    }
    this->_dispose();

    DB(10, "ProxyPushSupplier_i : ProxyPushSupplier Disconnected");
}

ProxyPushSupplier_i::~ProxyPushSupplier_i() {

    DB(10, "ProxyPushSupplier_i : Destructor Start");

    DB(10, "ProxyPushSupplier_i : Destructor End");
}

void
ProxyPushSupplier_i::addEvent (const event_t &event)
{
    omni_mutex_lock l(_headLock);

    if ((_proxyState == connected) && (_stamp <= event.stamp)) {
       _events.push_front(event);
       _nonEmpty.signal();
    }
}

void
ProxyPushSupplier_i::_dispatch ()
{
   DB(10, "ProxyPushSupplier_i : _dispatch Start");

   event_t event;

   while (_proxyState != disconnected) {

      _headLock.lock();
      while (_events.empty() || (CORBA::is_nil(_consumer))) {
            if (_proxyState == disconnected)
               break;
            _nonEmpty.wait();
      }

      if (_proxyState == disconnected) {
          _headLock.unlock();
          break;
      }

      DB(20,"ProxyPushSupplier_i : Queue Length : " << _events.size());

      while (! _events.empty()) {

         // Get next event.
         event = _events.back();
         _events.pop_back();
         _headLock.unlock();
      
         try {
            if (! CORBA::is_nil(_consumer))
            {
               // Push event data
               _consumer->push(event.data);
            }
         }
         catch (...)
         {
           DB(20, "ProxyPushSupplier_i : Exception notifying PushConsumer!");
         }
         _headLock.lock();
      }

      _headLock.unlock();
      omni_thread::yield();
   }

   DB(10, "ProxyPushSupplier_i : _dispatch End");
}

//------------------------------------------------------------------------
//           Proxy Push Supplier Worker Implementation
//------------------------------------------------------------------------
ProxyPushSupplierWorker::ProxyPushSupplierWorker(ProxyPushSupplier_i *object,
                                       Method method,
                                       priority_t priority) :
   omni_thread(NULL, priority)
{

   DB(10, "ProxyPushSupplierWorker : Constructor Start");

   _method = method;
   _object = object;

   start_undetached();

   DB(10, "ProxyPushSupplierWorker : Constructor End");
}

void*
ProxyPushSupplierWorker::run_undetached (void *) {

   DB(10, "ProxyPushSupplierWorker : run_undetached Start");

   (_object->*_method)();

   DB(10, "ProxyPushSupplierWorker : run_undetached End");

   return(0);
}

ProxyPushSupplierWorker::~ProxyPushSupplierWorker () {

   DB(10, "ProxyPushSupplierWorker : Destructor Start");

   DB(10, "ProxyPushSupplierWorker : Destructor End");
}


//------------------------------------------------------------------------
//           Proxy Pull Supplier Interface Implementation
//------------------------------------------------------------------------
ProxyPullSupplier_i::ProxyPullSupplier_i (EventChannel_i *impl)
    : _nonEmpty(&_headLock)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Constructor Start");

    _channel = impl;
    _proxyState = notConnected;
    _consumer = CosEventComm::PullConsumer::_nil();

    _obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ProxyPullSupplier_i : Activated ProxyPullSupplier");

    _channel->_reg_pxy_pull_supplier (this);

    DB(10, "ProxyPullSupplier_i : Constructor End");
}

void
ProxyPullSupplier_i::connect_pull_consumer (
    CosEventComm::PullConsumer_ptr pull_consumer)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Connecting PullConsumer");

    if (!CORBA::is_nil (_consumer))
       throw CosEventChannelAdmin::AlreadyConnected();

    if (CORBA::is_nil (pull_consumer)) {
       throw CORBA::BAD_PARAM(0,CORBA::COMPLETED_NO);
    }

    _proxyState = connected;
    _consumer = CosEventComm::PullConsumer::_duplicate (pull_consumer);

    DB(10, "ProxyPullSupplier_i : PullConsumer Connected");
}

void
ProxyPullSupplier_i::disconnect_pull_supplier ()
{
    _channel->_drg_pxy_pull_supplier(this);

    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullSupplier_i : Disconecting ProxyPullSupplier");

    _proxyState = disconnected;

    if (! CORBA::is_nil(_consumer)) {
       CORBA::release(_consumer);
    }
    this->_dispose();

    DB(10, "ProxyPullConsumer_i : ProxyPullSupplier Disconnected");
}

CORBA::Any*
ProxyPullSupplier_i::pull ()
{
    if (_proxyState != connected)
        throw CosEventComm::Disconnected();

    // Check Proxy Pull Supplier for queued events. If none found,
    // block until either a (Pull or Push) Supplier supplies data.
    // Indicate the channel to pull events from Pull Suppliers

    event_t event;
    _headLock.lock();
    _channel->reqEvent();

    while (_events.empty() || (CORBA::is_nil(_consumer))) {
         if (_proxyState == disconnected)
            break;
         _nonEmpty.wait();
    }

    // Return if consumer has disconnected
    if (_proxyState == disconnected) {
        _headLock.unlock();
        throw CosEventComm::Disconnected();
    }

    DB(10, "ProxyPullSupplier_i : Pull Queue Size : " << _events.size());

    if (! _events.empty())
    {
       event = _events.back();
       _events.pop_back();
       _headLock.unlock();
       return new CORBA::Any(event.data);
    }
    else
    {
       _headLock.unlock();
       return new CORBA::Any();
    }
}

CORBA::Any*
ProxyPullSupplier_i::try_pull (CORBA::Boolean &has_event)
{
    event_t event;
    _headLock.lock();

    // Return if consumer has disconnected
    if (_proxyState != connected) {
        _headLock.unlock();
        throw CosEventComm::Disconnected();
    }

    DB(10, "ProxyPullSupplier_i : Pull Queue Size : " << _events.size());
    if (! _events.empty()) {
       event = _events.back();
       _events.pop_back();
       has_event = 1;
       _headLock.unlock();
       omni_thread::yield();
       return new CORBA::Any(event.data);
    }
    else
    {
       has_event = 0;
       _headLock.unlock();
       omni_thread::yield();
       return new CORBA::Any();
    }
}

void
ProxyPullSupplier_i::addEvent (const event_t &event)
{
    omni_mutex_lock l(_headLock);

    if ((_proxyState == connected) && (_stamp <= event.stamp)) {
       _events.push_front(event);
       _nonEmpty.signal();
    }
}

//------------------------------------------------------------------------
//           Proxy Pull Consumer Interface Implementation
//------------------------------------------------------------------------
ProxyPullConsumer_i::ProxyPullConsumer_i (EventChannel_i *impl)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Constructor Start");

    _channel = impl;
    _proxyState = notConnected;
    _supplier = CosEventComm::PullSupplier::_nil();
    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ProxyPullConsumer_i : Activated ProxyPullConsumer");

    _channel->_reg_pxy_pull_consumer (this);

    DB(10, "ProxyPullConsumer_i : Constructor End");
}

void
ProxyPullConsumer_i::connect_pull_supplier (
    CosEventComm::PullSupplier_ptr pull_supplier)
{
    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Connecting PullSupplier");

    if (CORBA::is_nil (pull_supplier))
        throw CORBA::BAD_PARAM(0,CORBA::COMPLETED_NO);

    if (!CORBA::is_nil (_supplier))
        throw CosEventChannelAdmin::AlreadyConnected();

    _proxyState = connected;
    _supplier = CosEventComm::PullSupplier::_duplicate (pull_supplier);

    DB(10, "ProxyPullConsumer_i : PullSupplier Connected");
}

CORBA::Any *
ProxyPullConsumer_i::getEvent(CORBA::Boolean &has_event)
{
    omni_mutex_lock l(_lock);

    CORBA::Any *data;
    data = _supplier->try_pull(has_event);
    return (data);
}

void
ProxyPullConsumer_i::disconnect_pull_consumer ()
{
    _channel->_drg_pxy_pull_consumer(this);

    omni_mutex_lock l(_lock);

    DB(10, "ProxyPullConsumer_i : Disconecting ProxyPullConsumer");

    _proxyState = disconnected;

    if (! CORBA::is_nil(_supplier)) {
       CORBA::release(_supplier);
    }
    this->_dispose();

    DB(10, "ProxyPullConsumer_i : ProxyPullConsumer Disconnected");
}

//------------------------------------------------------------------------
//           Consumer Admin Interface Implementation
//------------------------------------------------------------------------
ConsumerAdmin_i::ConsumerAdmin_i (EventChannel_i *channel)
{
    DB(10, "ConsumerAdmin_i : Constructor Start");

    event_channel = channel;
    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "ConsumerAdmin_i : Constructor End");
}

ConsumerAdmin_i::~ConsumerAdmin_i ()
{
    DB(10, "ConsumerAdmin_i : Destructor Start");

    DB(10, "ConsumerAdmin_i : Destructor End");
}


CosEventChannelAdmin::ProxyPushSupplier_ptr 
ConsumerAdmin_i::obtain_push_supplier ()
{
    DB(10, "ConsumerAdmin_i : obtain_push_supplier Start");

    // Create Proxy Push Supplier
    CosEventChannelAdmin::ProxyPushSupplier_ptr  ptr;
    ptr = new ProxyPushSupplier_i (event_channel);
    CosEventChannelAdmin::ProxyPushSupplier::_duplicate(ptr);

    DB(10, "ConsumerAdmin_i : obtain_push_supplier End");
    return (ptr);
}

CosEventChannelAdmin::ProxyPullSupplier_ptr 
ConsumerAdmin_i::obtain_pull_supplier ()
{
    DB(10, "ConsumerAdmin_i : obtain_pull_supplier Start");

    // Create Proxy Pull Supplier
    CosEventChannelAdmin::ProxyPullSupplier_ptr ptr;
    ptr = new ProxyPullSupplier_i (event_channel);
    CosEventChannelAdmin::ProxyPullSupplier::_duplicate(ptr);

    DB(10, "ConsumerAdmin_i : obtain_pull_supplier End");
    return ptr;
}

//------------------------------------------------------------------------
//           Supplier Admin Interface Implementation
//------------------------------------------------------------------------
SupplierAdmin_i::SupplierAdmin_i (EventChannel_i *channel)
{
    DB(10, "SupplierAdmin_i : Constructor Start");

    event_channel = channel;
    this->_obj_is_ready(CORBA::BOA::getBOA());

    DB(10, "SupplierAdmin_i : Constructor End");
}

SupplierAdmin_i::~SupplierAdmin_i ()
{
    DB(10, "SupplierAdmin_i : Destructor Start");

    DB(10, "SupplierAdmin_i : Destructor End");
}

CosEventChannelAdmin::ProxyPushConsumer_ptr 
SupplierAdmin_i::obtain_push_consumer ()
{
    DB(10, "ConsumerAdmin_i : obtain_push_consumer Start");

    // Create a new proxy
    CosEventChannelAdmin::ProxyPushConsumer_ptr ptr;
    ptr = new ProxyPushConsumer_i (event_channel);
    CosEventChannelAdmin::ProxyPushConsumer::_duplicate(ptr);

    DB(10, "ConsumerAdmin_i : obtain_push_consumer End");
    return ptr;
}

CosEventChannelAdmin::ProxyPullConsumer_ptr 
SupplierAdmin_i::obtain_pull_consumer ()
{
    DB(10, "ConsumerAdmin_i : obtain_pull_consumer Start");

    // Create a new proxy
    CosEventChannelAdmin::ProxyPullConsumer_ptr ptr;
    ptr = new ProxyPullConsumer_i (event_channel);
    CosEventChannelAdmin::ProxyPullConsumer::_duplicate(ptr);

    DB(10, "ConsumerAdmin_i : obtain_pull_consumer End");
    return (ptr);
}

//------------------------------------------------------------------------
//           Event Channel Interface Implementation
//------------------------------------------------------------------------

EventChannel_i::EventChannel_i (unsigned long pullRetryPeriod) :
_pullRetryPeriod(pullRetryPeriod)
{
    DB(10, "EventChannel_i : Constructor Start");

    _empty = true;
    this->_obj_is_ready(CORBA::BOA::getBOA());

    supplier_admin = new SupplierAdmin_i(this);
    consumer_admin = new ConsumerAdmin_i(this);

    _nonEmpty = new omni_condition(&_headLock);
    _pullEvent = new omni_condition(&_pullLock);
    _collectEvent = new omni_condition(&_collectLock);

    _dispatcher = new EventChannelWorker(*this,
                                         &EventChannel_i::_dispatch,
                                         omni_thread::PRIORITY_NORMAL);
    _dispatcher->start();

    _pullWorker = new EventChannelWorker(*this,
                                     &EventChannel_i::_pull,
                                     omni_thread::PRIORITY_NORMAL);
    _pullWorker->start();

    _agent= new EventChannelWorker(*this,
                                   &EventChannel_i::_collect,
                                   omni_thread::PRIORITY_NORMAL);
    _agent->start();


    DB(10, "EventChannel_i : Constructor End");
}

EventChannel_i::~EventChannel_i () 
{
    DB(10, "EventChannel_i : Destructor Start");

    DB(10, "EventChannel_i : Destructor End");
}

CORBA::Boolean
EventChannel_i::_save_object ()
{
    return 1; 
}

CosEventChannelAdmin::ConsumerAdmin_ptr 
EventChannel_i::for_consumers ()
{
    return ConsumerAdmin_i::_duplicate(consumer_admin);
}

CosEventChannelAdmin::SupplierAdmin_ptr 
EventChannel_i::for_suppliers ()
{
    return SupplierAdmin_i::_duplicate(supplier_admin);
}

void EventChannel_i::destroy ()
{
    DB(10, "EventChannel : Destroy Start");

    disconnect ();
    supplier_admin->_dispose();
    consumer_admin->_dispose();
    this->_dispose();

    delete _nonEmpty;
    delete _pullEvent;

    DB(10, "EventChannel : Destroy End");
}

void
EventChannel_i::_reg_pxy_push_consumer ( ProxyPushConsumer_i *pxy_push_cons)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : Registering ProxyPushConsumer Start");

    _pxy_push_cons.insert(pxy_push_cons);

    DB(10, "EventChannel_i : Registering ProxyPushConsumer End");
}

void
EventChannel_i::_reg_pxy_push_supplier (ProxyPushSupplier_i *pxy_push_supp)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : Registering ProxyPushSupplier Start");

    _pxy_push_supp.insert(pxy_push_supp);

    _collectEvent->signal();

    DB(10, "EventChannel_i : Registering ProxyPushSupplier End");
}

void
EventChannel_i::_reg_pxy_pull_consumer (ProxyPullConsumer_i *pxy_pull_cons)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : Registering ProxyPullConsumer Start");

    _pxy_pull_cons.insert(pxy_pull_cons);

    _collectEvent->signal();

    DB(10, "EventChannel_i : Registering ProxyPullConsumer End");
}

void
EventChannel_i::_reg_pxy_pull_supplier (ProxyPullSupplier_i *pxy_pull_supp)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : Registering ProxyPullSupplier Start");

    _pxy_pull_supp.insert(pxy_pull_supp);

    DB(10, "EventChannel_i : Registering ProxyPullSupplier End");
}

void
EventChannel_i::_drg_pxy_push_consumer (ProxyPushConsumer_i *pxy_push_cons)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : De-Registering ProxyPushConsumer Start");

    _pxy_push_cons.erase(pxy_push_cons);

    DB(10, "EventChannel_i : De-Registering ProxyPushConsumer End");
}


void
EventChannel_i::_drg_pxy_push_supplier (ProxyPushSupplier_i *pxy_push_supp)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : De-Registering ProxyPushSupplier Start");

    _pxy_push_supp.erase(pxy_push_supp);

    DB(10, "EventChannel_i : De-Registering ProxyPushSupplier End");
}

void
EventChannel_i::_drg_pxy_pull_consumer (ProxyPullConsumer_i *pxy_pull_cons)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : De-Registering ProxyPullConsumer Start");

    _pxy_pull_cons.erase(pxy_pull_cons);

    DB(10, "EventChannel_i : De-Registering ProxyPullConsumer End");
}

void
EventChannel_i::_drg_pxy_pull_supplier (ProxyPullSupplier_i *pxy_pull_supp)
{
    omni_mutex_lock l(_lock);

    DB(10, "EventChannel_i : De-Registering ProxyPullSupplier Start");

    _pxy_pull_supp.erase(pxy_pull_supp);

    DB(10, "EventChannel_i : De-Registering ProxyPullSupplier End");
}

void EventChannel_i::addEvent (const CORBA::Any &data)
{
    _headLock.lock();

    event_t event;
    event.data = data;
    event.stamp.get_time();

    _events.push_front(event);
    _empty = false;
    _nonEmpty->signal();

    _headLock.unlock();
    omni_thread::yield();
}

void EventChannel_i::reqEvent (void)
{
    _headLock.lock();
    if (_events.empty())
    {
       // record channel status
       _empty = true;

       // signal pull worker to start work
       _pullEvent->signal();
    }
    _headLock.unlock();
}

void EventChannel_i::_dispatch (void)
{
   DB(10, "EventChannel_i : _dispatch Start");

   event_t event;

   while (1) {

      _headLock.lock();
      while (_events.empty())
         _nonEmpty->wait();

      DB(20, "EventChannel_i : Queue Length = " << _events.size());

      while (! _events.empty()) {

        // Get the next event
        event = _events.back();
        _events.pop_back();
        _headLock.unlock();

        _lock.lock();

        //
        // Push Model Consumers
        //
        ProxyPushSupplierSet::iterator i;
        for (i = _pxy_push_supp.begin(); i != _pxy_push_supp.end(); i++)
        {
            try {
               if (! CORBA::is_nil(*i))
               {
                  (*i)->addEvent(event);
               }
            }
            catch (...)
            {
               DB(20, "Exception notifying ProxyPushSupplier !");
            }
        }

        //
        // Pull Model Consumers
        //
        ProxyPullSupplierSet::iterator j;
        for (j = _pxy_pull_supp.begin(); j != _pxy_pull_supp.end(); j++)
        {
            try {
               if (! CORBA::is_nil(*j))
               {
                  (*j)->addEvent(event);
               }
            }
            catch (...)
            {
               DB(20, "Exception notifying ProxyPullSupplier !");
            }
        }

        _lock.unlock();
        _headLock.lock();
      }

      _headLock.unlock();
      omni_thread::yield();
   }

   DB(10, "EventChannel_i : _dispatch End");
}

int EventChannel_i::agentMode(void)
{
    //
    // Test whether there are only pull suppliers
    // and push consumers connected.
    //
    if (! _pxy_pull_supp.size() &&
        ! _pxy_push_cons.size() &&
          _pxy_push_supp.size() &&
          _pxy_pull_cons.size())
       return (true);
    else
       return (false);
}

void EventChannel_i::_collect (void)
{
   DB(10, "EventChannel_i : _collect Start");

   event_t event;
   ProxyPullConsumerSet::iterator j;

   while (1) {
      //
      // Block until requested.
      //
      _collectLock.lock();
      _collectEvent->wait();
      _collectLock.unlock();

      j = _pxy_pull_cons.begin();

      while (1) {
         //
         // Proceed if channel is in agent mode (i.e there are
         // pull suppliers and push consumers only).
         //
         if (! this->agentMode())
            break;

         //
         // Request each pull supplier.
         //
         if ((!_pxy_pull_cons.empty()) && (j != _pxy_pull_cons.end()))
         {
            DB(20, "EventChannel_i : Pulling events (Agent Mode)");
            try {
               if (! CORBA::is_nil(*j))
               {
                  CORBA::Any *data;
                  CORBA::Boolean has_event;
                  data = (*j)->getEvent(has_event);
                  if (has_event)
                  {
                     this->addEvent(*data);
                     delete data;
                     omni_thread::yield();
                  }
               }
            }
            catch (...)
            {
               DB(20, "Exception pulling ProxyPullConsumer !");
            }
         }

         if ((_pxy_pull_cons.empty()) || (j == _pxy_pull_cons.end()))
         {
             j = _pxy_pull_cons.begin();
             omni_thread::sleep(_pullRetryPeriod);
         }
         else
         {
            j++;
         }
      }
   }

   DB(10, "EventChannel_i : _collect End");
}

void EventChannel_i::_pull (void)
{
   DB(10, "EventChannel_i : _pull Start");

   event_t event;
   ProxyPullConsumerSet::iterator j;

   while (1) {

      //
      // Block until a pull consumer requests an event.
      //
      _pullLock.lock();
      _pullEvent->wait();
      _pullLock.unlock();

      //
      // Pull Model Suppliers
      //
      j = _pxy_pull_cons.begin();
      while (1)
      {
         //
         // Proceed if channel is empty
         //
         _headLock.lock();
         if (! _empty)
         {
            _headLock.unlock();
            break;
         }
         _headLock.unlock();

         if ((!_pxy_pull_cons.empty()) && (j != _pxy_pull_cons.end()))
         {
            DB(20, "EventChannel_i : Pulling events");
            try {
               if (! CORBA::is_nil(*j))
               {
                  CORBA::Any *data;
                  CORBA::Boolean has_event;
                  data = (*j)->getEvent(has_event);
                  if (has_event)
                  {
                     this->addEvent(*data);
                     delete data;
                     omni_thread::yield();
                     break;
                  }
               }
            }
            catch (...)
            {
               DB(20, "Exception pulling ProxyPullConsumer !");
            }
         }
         if ((_pxy_pull_cons.empty()) || (j == _pxy_pull_cons.end()))
         {
             j = _pxy_pull_cons.begin();
             omni_thread::sleep(_pullRetryPeriod);
         }
         else
         {
            j++;
         }
      }
   }

   DB(10, "EventChannel_i : _pull End");
}
void
EventChannel_i::disconnect ()
{
}

//------------------------------------------------------------------------
//           Event Channel Worker Implementation
//------------------------------------------------------------------------
EventChannelWorker::EventChannelWorker(EventChannel_i &object,
                                       Method method,
                                       priority_t priority) :
   omni_thread(NULL, priority)
{

   DB(10, "EventChannelWorker : Constructor Start");

   _method = method;
   _object = &object;

   DB(10, "EventChannelWorker : Constructor End");
}

void
EventChannelWorker::run (void *) {

    DB(10, "EventChannelWorker : run Start");

    (_object->*_method)();

    DB(10, "EventChannelWorker : run End");
}

//------------------------------------------------------------------------
//           Event Channel Factory Interface Implementation
//------------------------------------------------------------------------
EventChannelFactory_i::EventChannelFactory_i ()
{
  // empty
}

CORBA::Boolean
EventChannelFactory_i::supports (const CosLifeCycle::Key &k)
{
   // returns true if the key passed has the following contents :
   //
   // id   : "EventChannel"
   // kind : "object interface"

   if ((k.length() == 1) &&
       (strcmp(k[0].id, "EventChannel") == 0) &&
       (strcmp(k[0].kind, "object interface") == 0))
     return (true);
   else
     return (false);
}

CORBA::Object_ptr
EventChannelFactory_i::create_object (const CosLifeCycle::Key &k,
                                      const CosLifeCycle::Criteria &criteria)
{
    DB(10, "EventChannelFactory : Creating Channel Start");

    // Check the key
    if (! this->supports(k))
       throw CosLifeCycle::NoFactory(k);

    // Process criteria
    CORBA::ULong pullRetryPeriod = 1;
    for (size_t i=0; i < criteria.length(); i++)
    {
       if (strcmp(criteria[i].name, "pullRetryPeriod") == 0)
       {
          if (! (criteria[i].value >>= pullRetryPeriod))
             throw CosLifeCycle::InvalidCriteria(criteria);

          if (pullRetryPeriod < 1)
             throw CosLifeCycle::CannotMeetCriteria(criteria);
       }
    }

    try
    {
       EventChannel_i *channel = new EventChannel_i(pullRetryPeriod);
       EventChannel_i::_duplicate (channel);

       DB(10, "EventChannelFactory : Creating Channel End");
       return channel;
    }
    catch (...)
    {
       return (CORBA::Object::_nil());
    }
    return (CORBA::Object::_nil());
}
