// -*- Mode: C++; -*-
//                              File      : ChannelAdmin_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of ConsumerAdmin_i and SupplierAdmin_i
//
 
/*
$Log: ChannelAdmin_i.cc,v $
Revision 1.63  2000/12/02 16:40:54  alcfp
fix for HP aCC

Revision 1.62  2000/12/01 23:01:09  alcfp
fixed 2 bugs, 1 in admin-level subscription_change, 1 in changepool _next_available method

Revision 1.61  2000/11/17 21:35:54  alcfp
fixed some cases where _fa_helper should have been used

Revision 1.60  2000/11/16 05:32:13  alcfp
fixes for aCC

Revision 1.59  2000/11/15 21:17:29  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.58  2000/11/05 04:48:09  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.57  2000/10/30 05:41:46  alcfp
renamed CosNotify.h to CosNotifyShorthands.h and placed CosNotifyShorthands.h in omniNotify/include rather than src/services/include

Revision 1.56  2000/10/30 04:39:25  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.55  2000/08/22 18:23:52  alcfp
added description to each file

Revision 1.54  2000/08/16 20:19:20  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>
#include "corba_wrappers.h"
#include "CosNotifyShorthands.h"
#include "RDIDebug.h"
#include "RDITypeMap.h"
#include "RDIOplocksMacros.h"
#include "CosEventChannelAdmin_i.h"
#include "CosNotifyChannelAdmin_i.h"

// ------------------------------------------------------------- //
// ConsumerAdmin_i implementation                                //
// ------------------------------------------------------------- //

ConsumerAdmin_i::ConsumerAdmin_i(EventChannel_i* channel,
				 CosNA_InterFilterGroupOperator op,
				 const CosNA_AdminID& serial) :
  RDINotifySubscribe(),
  _oplockptr(0), _disposed(0),
  _fa_helper(), _channel(channel), _qosprop(0),
  _serial(serial), _and_or_oper(op), _rqstypes(), _prx_serial(1),
  _num_proxies(0), _cosevent_push(), _cosevent_pull(),
  _prx_any_push(RDI_ULongHash, RDI_ULongRank),
  _prx_any_pull(RDI_ULongHash, RDI_ULongRank),
  _prx_struc_push(RDI_ULongHash, RDI_ULongRank),
  _prx_struc_pull(RDI_ULongHash, RDI_ULongRank),
  _prx_batch_push(RDI_ULongHash, RDI_ULongRank),
  _prx_batch_pull(RDI_ULongHash, RDI_ULongRank)
{
  RDI_OPLOCK_INIT;
  _rqstypes.length(0);
  _prio_filter = CosNF_MappingFilter::_nil();
  _life_filter = CosNF_MappingFilter::_nil();
  _time_born   = RDI_TimeValue::timeofday();
  _prxy_exit   = _time_born;
  WRAPPED_REGISTER_IMPL(this);
}

CosNA_AdminID
ConsumerAdmin_i::MyID( WRAPPED_IMPLARG_VOID ) 
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_AdminID res = _serial;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_EventChannel_ptr
ConsumerAdmin_i::MyChannel( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_EventChannel_ptr res = WRAPPED_IMPL2OREF(CosNA_EventChannel, _channel);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_InterFilterGroupOperator
ConsumerAdmin_i::MyOperator( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_InterFilterGroupOperator res = _and_or_oper;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNF_MappingFilter_ptr
ConsumerAdmin_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_MappingFilter_ptr res = CosNF_MappingFilter::_nil();
  if ( !CORBA::is_nil(_prio_filter) ) {
    res = CosNF_MappingFilter::_duplicate(_prio_filter);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}
 
void
ConsumerAdmin_i::priority_filter(CosNF_MappingFilter_ptr prio_filter   
				 WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  if ( CORBA::is_nil(prio_filter) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  _prio_filter = CosNF_MappingFilter::_duplicate(prio_filter);
  RDI_OPLOCK_RELEASE;
}

CosNF_MappingFilter_ptr
ConsumerAdmin_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{ 
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_MappingFilter_ptr res = CosNF_MappingFilter::_nil();
  if ( ! CORBA::is_nil(_life_filter) ) {
    res = CosNF_MappingFilter::_duplicate(_life_filter); 
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

void
ConsumerAdmin_i::lifetime_filter(CosNF_MappingFilter_ptr life_filter   
				 WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  if ( CORBA::is_nil(life_filter) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  _life_filter = CosNF_MappingFilter::_duplicate(life_filter);
  RDI_OPLOCK_RELEASE;
}

CosEventChannelAdmin::ProxyPushSupplier_ptr 
ConsumerAdmin_i::obtain_push_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosEventChannelAdmin::ProxyPushSupplier_ptr res = 
    CosEventChannelAdmin::ProxyPushSupplier::_nil();
  EventProxyPushSupplier_i* prx=0;
  RDI_DUMP("CosEvent ProxyPushSupplier creation requested");
  if ( ! _channel->incr_consumers() ) {
    // return null res
  } else if ( ! (prx = new EventProxyPushSupplier_i(this, _channel)) ) {
    _channel->decr_consumers();
    // return null res
  } else if ( _cosevent_push.insert_tail(prx) != 0 ) {
    _channel->decr_consumers();
    prx->disconnect_client_and_dispose(0);
    // return null res
  } else { // return valid res
    if ( _channel->_push_consumer ) {        // Register Push Proxy
    _channel->_push_consumer->insert_proxy(prx);
    }
    _num_proxies += 1;
    RDI_DUMP("CosEvent ProxyPushSupplier creation completed");
    res = WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPushSupplier, prx);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

CosEventChannelAdmin::ProxyPullSupplier_ptr
ConsumerAdmin_i::obtain_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosEventChannelAdmin::ProxyPullSupplier_ptr res = 
    CosEventChannelAdmin::ProxyPullSupplier::_nil();
  EventProxyPullSupplier_i* prx=0;
  RDI_DUMP("CosEvent ProxyPullSupplier creation requested");
  if ( ! _channel->incr_consumers() ) {
    // return null res
  } else if ( ! (prx = new EventProxyPullSupplier_i(this, _channel)) ) {
    _channel->decr_consumers();
    // return null res
  } else if ( _cosevent_pull.insert_tail(prx) != 0 ) {
    _channel->decr_consumers();
    prx->disconnect_client_and_dispose(0);
    // return null res
  } else { // return valid res
    _num_proxies += 1;
    RDI_DUMP("CosEvent ProxyPullSupplier creation completed");
    res = WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPullSupplier, prx);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

// called externally by consumers
void
ConsumerAdmin_i::subscription_change(const CosN_EventTypeSeq& added,
				     const CosN_EventTypeSeq& deled   
				     WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  // A consumer can invoke this operation as long as filters have
  // not been placed at this admin
  if ( _fa_helper.has_filters() ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  }

  CosN_EventTypeSeq added_copy = added;
  CosN_EventTypeSeq deled_copy = deled;
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  nadded.length(0);
  ndeled.length(0);
  CORBA::ULong   ix = 0;
  if ( ! RDI_EventType::valid_sequence(added_copy, ix) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(added[ix]);
  }
  if ( ! RDI_EventType::valid_sequence(deled_copy, ix) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(deled[ix]);
  }
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
    RDI_EventType::compute_diff(_rqstypes, added_copy, deled_copy, nadded, ndeled);
    if ( nadded.length() || ndeled.length() ) {
      (void) _channel->update_mapping(nadded, ndeled, this, 0);
      RDI_EventType::update(_rqstypes, added_copy, deled_copy);
    }
  } else {
    ndeled.length(0);
    _rqstypes = added_copy; 
    RDI_EventType::update(_rqstypes, added_copy, deled_copy);
    (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
  RDI_OPLOCK_RELEASE;
}

// called internally due to filter activity
void
ConsumerAdmin_i::propagate_subscription_change(const CosN_EventTypeSeq& added,
					       const CosN_EventTypeSeq& deled,
					       Filter_i*                flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void
ConsumerAdmin_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

CosNA_ProxyIDSeq*
ConsumerAdmin_i::pull_suppliers( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i * >  bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  if ( ! seq ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_MAYBE);
  }
  num = _prx_any_pull.length() +
    _prx_struc_pull.length() + _prx_batch_pull.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_pull.cursor(); ac.is_valid(); ++ac, ++num )
    (*seq)[num] = ac.key();
  for ( sc = _prx_struc_pull.cursor(); sc.is_valid(); ++sc, ++num )
    (*seq)[num] = sc.key();
  for ( bc = _prx_batch_pull.cursor(); bc.is_valid(); ++bc, ++num )
    (*seq)[num] = bc.key();
  RDI_OPLOCK_RELEASE;
  return seq;
}

CosNA_ProxyIDSeq*
ConsumerAdmin_i::push_suppliers( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i * >  bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  if ( ! seq ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_MAYBE);
  }
  num = _prx_any_push.length() +
    _prx_struc_push.length() + _prx_batch_push.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_push.cursor(); ac.is_valid(); ++ac, ++num )
    (*seq)[num] = ac.key();
  for ( sc = _prx_struc_push.cursor(); sc.is_valid(); ++sc, ++num )
    (*seq)[num] = sc.key();        
  for ( bc = _prx_batch_push.cursor(); bc.is_valid(); ++bc, ++num )
    (*seq)[num] = bc.key();
  RDI_OPLOCK_RELEASE;
  return seq;
}

CosNA_ProxySupplier_ptr 
ConsumerAdmin_i::get_proxy_supplier(CosNA_ProxyID proxy_id WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  ProxyPushSupplier_i*           apush=0;
  ProxyPullSupplier_i*           apull=0;
  StructuredProxyPushSupplier_i* spush=0;
  StructuredProxyPullSupplier_i* spull=0;
  SequenceProxyPushSupplier_i*   bpush=0;
  SequenceProxyPullSupplier_i*   bpull=0;

  CosNA_ProxySupplier_ptr res = CosNA_ProxySupplier::_nil();

  if ( proxy_id <= _prx_serial ) {
    if ( _prx_any_push.lookup(proxy_id, apush) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, apush);
    } else if ( _prx_any_pull.lookup(proxy_id, apull) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, apull);
    } else if ( _prx_struc_push.lookup(proxy_id, spush) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, spush);
    } else if ( _prx_struc_pull.lookup(proxy_id, spull) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, spull);
    } else if ( _prx_batch_push.lookup(proxy_id, bpush) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, bpush);
    } else if ( _prx_batch_pull.lookup(proxy_id, bpull) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, bpull);
    }
  }
  RDI_OPLOCK_RELEASE;
  if (CORBA::is_nil(res)) {
    throw CosNA_ProxyNotFound();
  }
  return res;
}

CosNA_ProxySupplier_ptr 
ConsumerAdmin_i::obtain_notification_pull_supplier(CosNA_ClientType ctype, 
						   CosNA_ProxyID&   proxy_id
						   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  // Check with  the channel if we are about to exceed the maximum
  // number of consumers supported by the channel
  RDI_DUMP("Creation of a new ProxySupplier -- type " << ctype);
  if ( ! _channel->incr_consumers() ) {
    CosNA_AdminLimit limit;
    RDI_DUMP("Consumer limit has been reached creation failed");
    limit.name    = (const char *) "MaxConsumers";
    limit.value <<= _channel->max_consumers();
    RDI_OPLOCK_RELEASE;
    throw CosNA_AdminLimitExceeded(limit);
  }

  CosNA_ProxySupplier_ptr res = CosNA_ProxySupplier::_nil();
  if ( ctype == CosNA_ANY_EVENT ) {
    ProxyPullSupplier_i* prx = 0;
    if ( ! (prx = new ProxyPullSupplier_i(this, _channel,_prx_serial)) ) {
      _channel->decr_consumers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_any_pull.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_consumers(); 
      } else {
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
      }
    }
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
    StructuredProxyPullSupplier_i* prx = 
      new StructuredProxyPullSupplier_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_consumers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_struc_pull.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_consumers();
      } else {
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
      }
    }
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
    SequenceProxyPullSupplier_i* prx =
      new SequenceProxyPullSupplier_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_consumers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_batch_pull.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_consumers();
      } else {
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
      }
    }
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_ProxySupplier_ptr 
ConsumerAdmin_i::obtain_notification_push_supplier(CosNA_ClientType ctype, 
						   CosNA_ProxyID&   proxy_id
						   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  // Check with  the channel if we are about to exceed the maximum
  // number of consumers supported by the channel
  RDI_DUMP("Creation of a new ProxySupplier -- type " << ctype);
  if ( ! _channel->incr_consumers() ) {
    CosNA_AdminLimit limit;
    RDI_DUMP("Consumer limit has been reached creation failed");
    limit.name    = (const char *) "MaxConsumers";
    limit.value <<= _channel->max_consumers();
    RDI_OPLOCK_RELEASE;
    throw CosNA_AdminLimitExceeded(limit);
  }

  CosNA_ProxySupplier_ptr res = CosNA_ProxySupplier::_nil();
  if ( ctype == CosNA_ANY_EVENT ) {
    ProxyPushSupplier_i* prx = 0;
    if ( ! (prx = new ProxyPushSupplier_i(this, _channel, _prx_serial)) ) {
      _channel->decr_consumers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_any_push.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_consumers(); 
      } else {
	if ( _channel->_push_consumer ) {    // Register Push Proxy
	  _channel->_push_consumer->insert_proxy(prx);
	}
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
      }
    }
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
    StructuredProxyPushSupplier_i* prx = 
      new StructuredProxyPushSupplier_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      RDI_DUMP("\tfailed to create proxy");
      _channel->decr_consumers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_struc_push.insert(proxy_id, prx) != 0 ) {
	RDI_DUMP("\tfailed to register proxy with id " << proxy_id);
	prx->disconnect_client_and_dispose(0);
	_channel->decr_consumers();
      } else {
	if ( _channel->_push_consumer ) {    // Register Push Proxy
	  _channel->_push_consumer->insert_proxy(prx);
	}
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
      }
    }
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
    SequenceProxyPushSupplier_i* prx = 
      new SequenceProxyPushSupplier_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_consumers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_batch_push.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_consumers();
      } else { 
	if ( _channel->_push_consumer ) {    // Register Push Proxy
	  _channel->_push_consumer->insert_proxy(prx);
	}
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
      }
    }
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

void
ConsumerAdmin_i::destroy( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _serial == 0 ) {
    RDI_DUMP("Default ConsumerAdmin cannot be destroyed!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _disconnect_clients_and_dispose(1); // 1 means update channel
}

CosN_QoSProperties*
ConsumerAdmin_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_NotifQoS*   qosp = qos_properties();
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  CosN_QoSProperties* res = qosp->get_qos(RDI_C_ADMIN);
  RDI_OPLOCK_RELEASE;
  return res;
}

void
ConsumerAdmin_i::set_qos(const CosN_QoSProperties& r_qos  WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS* p_qos = _channel->qos_properties();
  RDI_NotifQoS* a_qos = (_qosprop ? _qosprop : p_qos);
  
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  if ( ! _qosprop ) {
    _qosprop = new RDI_NotifQoS();
    RDI_AssertAllocThrowNo(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos); 
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::validate_qos
(const CosN_QoSProperties& r_qos,
 WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
 WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS*   p_qos = _channel->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_AssertAllocThrowNo(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,*rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  RDI_OPLOCK_RELEASE;
}

CosNF_FilterID
ConsumerAdmin_i::add_filter(CosNF_Filter_ptr fltr WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
    _rqstypes.length(0);
  }
  CosNF_FilterID res = _fa_helper.add_filter_i(fltr, (RDINotifySubscribe_ptr) this);
  RDI_OPLOCK_RELEASE;
  return res;
}

void
ConsumerAdmin_i::remove_filter(CosNF_FilterID fltrID WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_filter(fltrID);
  RDI_OPLOCK_RELEASE;
}

CosNF_Filter_ptr
ConsumerAdmin_i::get_filter(CosNF_FilterID fltrID WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_Filter_ptr res = _fa_helper.get_filter(fltrID);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNF_FilterIDSeq*
ConsumerAdmin_i::get_all_filters( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_FilterIDSeq* res = _fa_helper.get_all_filters();
  RDI_OPLOCK_RELEASE;
  return res;
}

void
ConsumerAdmin_i::remove_all_filters( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_all_filters();
  RDI_OPLOCK_RELEASE;
}


// -------------------- ** Local-only routines ** ----------------------------

void
ConsumerAdmin_i::disconnect_clients_and_dispose()
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_clients_and_dispose();
}

// *** requires OPLOCK already held
//  If update_channel is true, the admin unregisters itself
void
ConsumerAdmin_i::_disconnect_clients_and_dispose(CORBA::Boolean update_channel)
{
  if (_disposed) {
    RDI_DUMP("ConsumerAdmin_i::_disconnect_clients_and_dispose() called more than once");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _disposed = 1; // acts as guard: the following is executed by only one thread 
  if (update_channel) {
    // Unregister from the channel -- must be done first
    // to avoid deadlock when channel is being destroyed
    _channel->unregister(this);
  }

  RDI_DUMP("Destruction of ConsumerAdmin " << (void*)this << " [" << _serial << "]");

  unsigned int i;
  RDI_ListCursor<EventProxyPushSupplier_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullSupplier_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullcur;

  cpushcur = _cosevent_push.cursor();
  for (i=0; i < _cosevent_push.length(); ++i, ++cpushcur)
    (*cpushcur)->disconnect_client_and_dispose();
  cpullcur = _cosevent_pull.cursor();
  for (i=0; i < _cosevent_pull.length(); ++i, ++cpullcur)
    (*cpullcur)->disconnect_client_and_dispose();
  for ( apushcur = _prx_any_push.cursor(); apushcur.is_valid(); ++apushcur )
    apushcur.val()->disconnect_client_and_dispose();
  for ( apullcur = _prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur )
    apullcur.val()->disconnect_client_and_dispose();
  for ( spushcur = _prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur )
    spushcur.val()->disconnect_client_and_dispose();
  for ( spullcur = _prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur )
    spullcur.val()->disconnect_client_and_dispose();
  for ( bpushcur = _prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur )
    bpushcur.val()->disconnect_client_and_dispose();
  for ( bpullcur = _prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur )
    bpullcur.val()->disconnect_client_and_dispose();

  _cosevent_push.drain();  _cosevent_pull.drain();
  _prx_any_push.clear();   _prx_any_pull.clear();
  _prx_struc_push.clear(); _prx_struc_pull.clear();
  _prx_batch_push.clear(); _prx_batch_pull.clear();

  if ( ! CORBA::is_nil(_prio_filter) ) CORBA::release(_prio_filter);
  if ( ! CORBA::is_nil(_life_filter) ) CORBA::release(_life_filter);
  if ( _rqstypes.length() == 0 ) {
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  if ( _qosprop )      delete _qosprop;

  RDI_DUMP("ConsumerAdmin @ " << (void*)this << " [" << _serial << "] destroyed");
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(EventProxyPushSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_ListCursor<EventProxyPushSupplier_i *> lc;
  unsigned int    indx = 0;

  for ( lc = _cosevent_push.cursor(); indx < _cosevent_push.length(); indx++ ) {
    EventProxyPushSupplier_i* proxy = *lc;
    if ( proxy == prx ) {
      _cosevent_push.remove(lc);
      _num_proxies -= 1;
      _prxy_exit    = RDI_TimeValue::timeofday();
      _channel->decr_consumers();
      if ( _channel->_push_consumer ) {       // Deregister Proxy
	_channel->_push_consumer->remove_proxy(prx);
      }
      break;
    }
    ++lc;
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(EventProxyPullSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_ListCursor<EventProxyPullSupplier_i *> lc;
  unsigned int    indx = 0;

  for ( lc = _cosevent_pull.cursor(); indx < _cosevent_pull.length(); indx++ ) {
    EventProxyPullSupplier_i* proxy = *lc;
    if (  proxy == prx ) {
      _cosevent_pull.remove(lc);
      _num_proxies -= 1;
      _prxy_exit    = RDI_TimeValue::timeofday();
      _channel->decr_consumers();
      break;
    }
    ++lc;
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(ProxyPushSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_any_push.exists(prx->_proxy_id()) ) {
    _prx_any_push.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_consumers();
    if ( _channel->_push_consumer ) {        // Deregister Proxy
      _channel->_push_consumer->remove_proxy(prx);
    }
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(ProxyPullSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_any_pull.exists(prx->_proxy_id()) ) {
    _prx_any_pull.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_consumers();
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(StructuredProxyPushSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_struc_push.exists(prx->_proxy_id()) ) {
    _prx_struc_push.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_consumers();
    if ( _channel->_push_consumer ) {	// Deregister Proxy
      _channel->_push_consumer->remove_proxy(prx);
    }
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(StructuredProxyPullSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_struc_pull.exists(prx->_proxy_id()) ) {
    _prx_struc_pull.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_consumers();
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(SequenceProxyPushSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_batch_push.exists(prx->_proxy_id()) ) {
    _prx_batch_push.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_consumers();
    if ( _channel->_push_consumer ) {     // Deregister Proxy
      _channel->_push_consumer->remove_proxy(prx);
    }
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
ConsumerAdmin_i::remove_proxy(SequenceProxyPullSupplier_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_batch_pull.exists(prx->_proxy_id()) ) {
    _prx_batch_pull.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_consumers();
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

// OPLOCK is acquired/released by caller [EventChannel_i::admin_dispatch()]
void
ConsumerAdmin_i::dispatch_event(RDI_StructuredEvent* event)
{
  RDI_ListCursor<EventProxyPushSupplier_i *> psc;
  RDI_ListCursor<EventProxyPullSupplier_i *> plc;
  unsigned int i=0;
  for (i=0,psc=_cosevent_push.cursor(); i<_cosevent_push.length(); ++i,++psc)
    (*psc)->add_event(event);
  for (i=0,plc=_cosevent_pull.cursor(); i<_cosevent_pull.length(); ++i,++plc)
    (*plc)->add_event(event);
}

// OPLOCK is acquired/released by caller
//  [EventChannel_i::admin_dispatch() or EventChannel_i::proxy_dispatch()]
void
ConsumerAdmin_i::dispatch_event(RDI_StructuredEvent* event,
				RDI_FilterState_t    astat, 
				RDI_TypeMap*         tmap)
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullcur;
  RDI_TypeMap::FNode_t* fnode=0;
  CORBA::Boolean match_found=0;
  RDI_TypeMap::FList_t flist;
 
  // When consumers use 'subscription_change()'  to register 
  // interest in specific event types, we have 'has_filters()' return true
  // so that we execute the filter evaluation logic. However, there are no
  // filters registered and, hence, we need the '!fnode->_fltr' below...

  if ( _num_proxies == 0 )
    return;

  const char* dname = event->get_domain_name();
  const char* tname = event->get_type_name();

  // Evaluate Filters for consumers using ProxyPushSupplier_i

  for (apushcur = _prx_any_push.cursor(); apushcur.is_valid(); ++apushcur ) {
    if ( astat == OrMatch ) {
      apushcur.val()->add_event(event);
    } else if ( ! apushcur.val()->has_filters() ) {
      if ( (astat == NoFilters) || (astat == AndMatch) ) {
	apushcur.val()->add_event(event);
      }
    } else {
      tmap->lookup_begin(dname, tname, apushcur.val(), flist);
      if ( ! flist._star_star && ! flist._domn_star &&
	   ! flist._star_type && ! flist._domn_type ) {
	if ( astat == OrMatch ) {
	  apushcur.val()->add_event(event);
	}
      } else {
	match_found = 0;
	for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	  if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	    match_found = 1;
	    break;
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}

	if ( match_found ) {
	  apushcur.val()->add_event(event);
	}
      }
      tmap->lookup_end();
    }
  }

  // Evaluate Filters for consumers using ProxyPullSupplier_i

  for (apullcur = _prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur ) {
    if ( astat == OrMatch ) {
      apullcur.val()->add_event(event);
    } else if ( ! apullcur.val()->has_filters() ) {
      if ( (astat == NoFilters) || (astat == AndMatch) ) {
	apullcur.val()->add_event(event);
      }
    } else {
      tmap->lookup_begin(dname, tname, apullcur.val(), flist);
      if ( ! flist._star_star && ! flist._domn_star &&
	   ! flist._star_type && ! flist._domn_type ) {
	if ( astat == OrMatch ) {
	  apullcur.val()->add_event(event);
	}
      } else {
	match_found = 0;
	for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	  if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	    match_found = 1;
	    break;
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}

	if ( match_found ) {
	  apullcur.val()->add_event(event);
	}
      }
      tmap->lookup_end();
    }
  }

  // Evaluate Filters for consumers using StructuredProxyPushSupplier_i

  for ( spushcur=_prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur ) {
    if ( astat == OrMatch ) {
      spushcur.val()->add_event(event);
    } else if ( ! spushcur.val()->has_filters() ) {
      if ( (astat == NoFilters) || (astat == AndMatch) ) {
	spushcur.val()->add_event(event);
      }
    } else {
      tmap->lookup_begin(dname, tname, spushcur.val(), flist);
      if ( ! flist._star_star && ! flist._domn_star &&
	   ! flist._star_type && ! flist._domn_type ) {
	if ( astat == OrMatch ) {
	  spushcur.val()->add_event(event);
	}
      } else {
	match_found = 0;
	for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	  if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	    match_found = 1;
	    break;
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}

	if ( match_found ) {
	  spushcur.val()->add_event(event);
	}
      }
      tmap->lookup_end();
    }
  }

  // Evaluate Filters for consumers using StructuredProxyPullSupplier_i

  for ( spullcur=_prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur ) {
    if ( astat == OrMatch ) {
      spullcur.val()->add_event(event);
    } else if ( ! spullcur.val()->has_filters() ) {
      if ( (astat == NoFilters) || (astat == AndMatch) ) {
	spullcur.val()->add_event(event);
      }
    } else {
      tmap->lookup_begin(dname, tname, spullcur.val(), flist);
      if ( ! flist._star_star && ! flist._domn_star &&
	   ! flist._star_type && ! flist._domn_type ) {
	if ( astat == OrMatch ) {
	  spullcur.val()->add_event(event);
	}
      } else {
	match_found = 0;
	for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	  if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	    match_found = 1;
	    break;
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}

	if ( match_found ) {
	  spullcur.val()->add_event(event);
	}
      }
      tmap->lookup_end();
    }
  }

  // Evaluate Filters for consumers using SequenceProxyPushSupplier_i

  for ( bpushcur=_prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur ) {
    if ( astat == OrMatch ) {
      bpushcur.val()->add_event(event);
    } else if ( ! bpushcur.val()->has_filters() ) {
      if ( (astat == NoFilters) || (astat == AndMatch) ) {
	bpushcur.val()->add_event(event);
      }
    } else {
      tmap->lookup_begin(dname, tname, bpushcur.val(), flist);
      if ( ! flist._star_star && ! flist._domn_star &&
	   ! flist._star_type && ! flist._domn_type ) {
	if ( astat == OrMatch ) {
	  bpushcur.val()->add_event(event);
	}
      } else {
	match_found = 0;
	for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	  if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	    match_found = 1;
	    break;
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}

	if ( match_found ) {
	  bpushcur.val()->add_event(event);
	}
      }
      tmap->lookup_end();
    }
  }

  // Evaluate Filters for consumers using SequenceProxyPullSupplier_i

  for ( bpullcur=_prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur ) {
    if ( astat == OrMatch ) {
      bpullcur.val()->add_event(event);
    } else if ( ! bpullcur.val()->has_filters() ) {
      if ( (astat == NoFilters) || (astat == AndMatch) ) {
	bpullcur.val()->add_event(event);
      }
    } else {
      tmap->lookup_begin(dname, tname, bpullcur.val(), flist);
      if ( ! flist._star_star && ! flist._domn_star &&
	   ! flist._star_type && ! flist._domn_type ) {
	if ( astat == OrMatch ) {
	  bpullcur.val()->add_event(event);
	}
      } else {
	match_found = 0;
	for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	  if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	    match_found = 1;
	    break;
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}
	if ( ! match_found ) {
	  for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
	    if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
	      match_found = 1;
	      break;
	    }
	  }
	}

	if ( match_found ) {
	  bpullcur.val()->add_event(event);
	}
      }
      tmap->lookup_end();
    }
  }
}

RDI_NotifQoS*
ConsumerAdmin_i::qos_properties()
{ return _qosprop ? _qosprop : _channel->qos_properties(); }

ostream& operator << (ostream& out, ConsumerAdmin_i& adm)
{
  if (!adm.lock()) return out;

  RDI_ListCursor<EventProxyPushSupplier_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullSupplier_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullcur;
  unsigned int cnt = 0;

  out << &adm <<" ID "  << setw(3) << adm._serial << " Born " << 
    adm._time_born << " Last Proxy "  << adm._prxy_exit  << endl;
  if ( adm._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < adm._rqstypes.length(); ix++) {
      out << endl << "\t" << (const char*)adm._rqstypes[ix].domain_name; 
      out << "::" << (const char*)adm._rqstypes[ix].type_name;
    }
  }
  if ( adm._qosprop )
    out << adm._qosprop << endl;
  else
    out << "\tSame as QoSAdmin settings of Event Channel" << endl;
  cpushcur = adm._cosevent_push.cursor();
  for (cnt=0; cnt < adm._cosevent_push.length(); ++cnt, ++cpushcur)
    out << setw(4) << cnt << ": " << *(*cpushcur) << endl;
  cpullcur = adm._cosevent_pull.cursor();
  for (cnt=0; cnt < adm._cosevent_pull.length(); ++cnt, ++cpullcur)
    out << setw(4) << cnt << ": " << *(*cpullcur) << endl;
  cnt = 0;
  for (apushcur = adm._prx_any_push.cursor(); apushcur.is_valid(); ++apushcur)
    out << setw(4) << ++cnt << ": " << *(apushcur.val()) << endl;
  cnt = 0;
  for (apullcur = adm._prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur)
    out << setw(4) << ++cnt << ": " << *(apullcur.val()) << endl;
  cnt = 0;
  for (spushcur = adm._prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur)
    out << setw(4) << ++cnt << ": " << *(spushcur.val()) << endl;
  cnt = 0;
  for (spullcur = adm._prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur)
    out << setw(4) << ++cnt << ": " << *(spullcur.val()) << endl;
  cnt = 0;
  for (bpushcur = adm._prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur)
    out << setw(4) << ++cnt << ": " << *(bpushcur.val()) << endl;
  cnt = 0;
  for (bpullcur = adm._prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur)
    out << setw(4) << ++cnt << ": " << *(bpullcur.val()) << endl;

  adm.unlock();
  return out;
}

// ------------------------------------------------------------- //
// SupplierAdmin_i implementation                                //
// ------------------------------------------------------------- //

SupplierAdmin_i::SupplierAdmin_i(EventChannel_i* channel,
				 CosNA_InterFilterGroupOperator op,
				 const CosNA_AdminID& serial) :
  _oplockptr(0), _disposed(0),
  _fa_helper(), _channel(channel), _qosprop(0), 
  _serial(serial), _and_or_oper(op), _prx_serial(1), 
  _num_proxies(0), 
  _evtypes(RDI_EventType::hash, RDI_EventType::rank),
  _cosevent_push(), _cosevent_pull(),
  _prx_any_push(RDI_ULongHash, RDI_ULongRank),
  _prx_any_pull(RDI_ULongHash, RDI_ULongRank),
  _prx_struc_push(RDI_ULongHash, RDI_ULongRank),
  _prx_struc_pull(RDI_ULongHash, RDI_ULongRank),
  _prx_batch_push(RDI_ULongHash, RDI_ULongRank),
  _prx_batch_pull(RDI_ULongHash, RDI_ULongRank)
{
  RDI_OPLOCK_INIT;
  _time_born = RDI_TimeValue::timeofday();
  _prxy_exit = _time_born;
  WRAPPED_REGISTER_IMPL(this);
}

CosNA_AdminID
SupplierAdmin_i::MyID( WRAPPED_IMPLARG_VOID ) 
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_AdminID res = _serial;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_EventChannel_ptr
SupplierAdmin_i::MyChannel( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_EventChannel_ptr res = WRAPPED_IMPL2OREF(CosNA_EventChannel, _channel);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_InterFilterGroupOperator
SupplierAdmin_i::MyOperator( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_InterFilterGroupOperator res = _and_or_oper;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosEventChannelAdmin::ProxyPushConsumer_ptr 
SupplierAdmin_i::obtain_push_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosEventChannelAdmin::ProxyPushConsumer_ptr res = CosEventChannelAdmin::ProxyPushConsumer::_nil();
  EventProxyPushConsumer_i* prx=0;
  RDI_DUMP("CosEvent ProxyPushConsumer creation requested");
  if ( ! _channel->incr_suppliers() ) {
    // return null res
  } else if ( ! (prx = new EventProxyPushConsumer_i(this, _channel)) ) {
    _channel->decr_suppliers();
    // return null res
  } else if ( _cosevent_push.insert_tail(prx) != 0 ) {
    _channel->decr_suppliers();
    prx->disconnect_client_and_dispose(0);
    // return null res
  } else { // return valid res
    _num_proxies += 1;
    res = WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPushConsumer, prx);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

CosEventChannelAdmin::ProxyPullConsumer_ptr
SupplierAdmin_i::obtain_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosEventChannelAdmin::ProxyPullConsumer_ptr res = 
    CosEventChannelAdmin::ProxyPullConsumer::_nil();
  EventProxyPullConsumer_i* prx=0;
  RDI_DUMP("CosEvent ProxyPullConsumer creation requested");
  if ( ! _channel->incr_suppliers() ) {
    // return null res
  } else if ( ! (prx = new EventProxyPullConsumer_i(this, _channel)) ) {
    _channel->decr_suppliers();
    // return null res
  } else if ( _cosevent_pull.insert_tail(prx) != 0 ) {
    _channel->decr_suppliers();
    prx->disconnect_client_and_dispose(0);
    // return null res
  } else { // return valid res
    if ( _channel->_pull_supplier ) {      // Register Pull Proxy
      _channel->_pull_supplier->insert_proxy(prx);
    }
    _num_proxies += 1;
    res = WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPullConsumer, prx);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

// Called externally -- not clear that suppliers should be doing this
void
SupplierAdmin_i::offer_change(const CosN_EventTypeSeq& added,
			      const CosN_EventTypeSeq& deled   WRAPPED_IMPLARG ) {
  // this code assumes it is OK for suppliers to do this
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  CosN_EventTypeSeq added_copy = added;
  CosN_EventTypeSeq deled_copy = deled;

  // First, validate the entries in the provided lists to make sure
  // that correct values are provided for these event types .......

  CORBA::ULong vl=0;
  if ( ! RDI_EventType::valid_sequence(added_copy, vl) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(added[vl]);
  }
  if ( ! RDI_EventType::valid_sequence(deled_copy, vl) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(deled[vl]);
  }

  // The same call a proxy uses
  _propagate_offer_change(added_copy, deled_copy);
}

CosNA_ProxyIDSeq*
SupplierAdmin_i::pull_consumers( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_HashCursor<CosNA_ProxyID, ProxyPullConsumer_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  RDI_AssertAllocThrowNo(seq, "Memory allocation failed -- ProxyIDSeq");
  num = _prx_any_pull.length() +
    _prx_struc_pull.length() + _prx_batch_pull.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_pull.cursor(); ac.is_valid(); ++ac, ++num )
    (*seq)[num] = ac.key();
  for ( sc = _prx_struc_pull.cursor(); sc.is_valid(); ++sc, ++num )
    (*seq)[num] = sc.key();
  for ( bc = _prx_batch_pull.cursor(); bc.is_valid(); ++bc, ++num )
    (*seq)[num] = bc.key();
  RDI_OPLOCK_RELEASE;
  return seq;
}

CosNA_ProxyIDSeq*
SupplierAdmin_i::push_consumers( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_HashCursor<CosNA_ProxyID, ProxyPushConsumer_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>  bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  RDI_AssertAllocThrowNo(seq, "Memory allocation failed -- ProxyIDSeq");
  num = _prx_any_push.length() +
    _prx_struc_push.length() + _prx_batch_push.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_push.cursor(); ac.is_valid(); ++ac, ++num )
    (*seq)[num] = ac.key();
  for ( sc = _prx_struc_push.cursor(); sc.is_valid(); ++sc, ++num )
    (*seq)[num] = sc.key();        
  for ( bc = _prx_batch_push.cursor(); bc.is_valid(); ++bc, ++num )
    (*seq)[num] = bc.key();
  RDI_OPLOCK_RELEASE;
  return seq;
}

CosNA_ProxyConsumer_ptr
SupplierAdmin_i::get_proxy_consumer(CosNA_ProxyID proxy_id   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  ProxyPushConsumer_i*           apush=0;
  ProxyPullConsumer_i*           apull=0;
  StructuredProxyPushConsumer_i* spush=0;
  SequenceProxyPushConsumer_i*   bpush=0;
  StructuredProxyPullConsumer_i* spull=0;
  SequenceProxyPullConsumer_i*   bpull=0;

  CosNA_ProxyConsumer_ptr res = CosNA_ProxyConsumer::_nil();
  if ( proxy_id <= _prx_serial ) {
    if ( _prx_any_push.lookup(proxy_id, apush) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, apush);
    } else if ( _prx_any_pull.lookup(proxy_id, apull) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, apull);
    } else if ( _prx_struc_push.lookup(proxy_id, spush) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, spush);
    } else if ( _prx_struc_pull.lookup(proxy_id, spull) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, spull);
    } else if ( _prx_batch_push.lookup(proxy_id, bpush) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, bpush);
    } else if ( _prx_batch_pull.lookup(proxy_id, bpull) ) {
      res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, bpull);
    }
  }
  RDI_OPLOCK_RELEASE;
  if (CORBA::is_nil(res)) {
    throw CosNA_ProxyNotFound();
  }
  return res;
}

CosNA_ProxyConsumer_ptr
SupplierAdmin_i::obtain_notification_pull_consumer
(CosNA_ClientType ctype, CosNA_ProxyID& proxy_id   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  RDI_DUMP("Creation of a new ProxyConsumer -- type " << ctype);

  // Check with  the channel if we are about to exceed the maximum
  // number of suppliers supported by the channel
  if ( ! _channel->incr_suppliers() ) {
    CosNA_AdminLimit limit;
    RDI_DUMP("Supplier limit has been reached creation failed");
    limit.name    = (const char *) "MaxSuppliers";
    limit.value <<= _channel->max_suppliers();
    RDI_OPLOCK_RELEASE;
    throw CosNA_AdminLimitExceeded(limit);
  }

  CosNA_ProxyConsumer_ptr res = CosNA_ProxyConsumer::_nil();
  if ( ctype == CosNA_ANY_EVENT ) {
    ProxyPullConsumer_i* prx = 0;
    if ( ! (prx = new ProxyPullConsumer_i(this, _channel, _prx_serial)) ) {
      _channel->decr_suppliers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_any_pull.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_suppliers(); 
      } else {
	if ( _channel->_pull_supplier ) {   // Register Pull Proxy
	  _channel->_pull_supplier->insert_proxy(prx);
	}
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
      }
    }
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
    StructuredProxyPullConsumer_i* prx = 
      new StructuredProxyPullConsumer_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_suppliers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_struc_pull.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_suppliers();
      } else {
	if ( _channel->_pull_supplier ) {   // Register Pull Proxy
	  _channel->_pull_supplier->insert_proxy(prx);
	}
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
      }
    }
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
    SequenceProxyPullConsumer_i* prx =
      new SequenceProxyPullConsumer_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_suppliers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_batch_pull.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_suppliers();
      }  else {
	if ( _channel->_pull_supplier ) {     // Register Pull Proxy
	  _channel->_pull_supplier->insert_proxy(prx);
	}
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
      }
    }
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_ProxyConsumer_ptr
SupplierAdmin_i::obtain_notification_push_consumer
(CosNA_ClientType ctype, CosNA_ProxyID& proxy_id   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  RDI_DUMP("Creation of a new ProxyConsumer -- type " << ctype);

  // Check with  the channel if we are about to exceed the maximum
  // number of suppliers supported by the channel
  if ( ! _channel->incr_suppliers() ) {
    CosNA_AdminLimit limit;
    RDI_DUMP("Supplier limit has been reached creation failed");
    limit.name    = (const char *) "MaxSuppliers";
    limit.value <<= _channel->max_suppliers();
    RDI_OPLOCK_RELEASE;
    throw CosNA_AdminLimitExceeded(limit);
  }

  CosNA_ProxyConsumer_ptr res = CosNA_ProxyConsumer::_nil();
  if ( ctype == CosNA_ANY_EVENT ) {
    ProxyPushConsumer_i* prx = 0;
    if ( ! (prx = new ProxyPushConsumer_i(this, _channel, _prx_serial)) ) {
      _channel->decr_suppliers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_any_push.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_suppliers(); 
      } else {
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
      }
    }
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
    StructuredProxyPushConsumer_i* prx =
      new StructuredProxyPushConsumer_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_suppliers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_struc_push.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_suppliers();
      } else {
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
      }
    }
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
    SequenceProxyPushConsumer_i* prx = 
      new SequenceProxyPushConsumer_i(this,_channel,_prx_serial);
    if ( ! prx ) {
      _channel->decr_suppliers();
    } else {
      proxy_id = _prx_serial++;
      if ( _prx_batch_push.insert(proxy_id, prx) != 0 ) {
	prx->disconnect_client_and_dispose(0);
	_channel->decr_suppliers();
      } else {
	_num_proxies += 1;
	res = WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
      }
    }
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

void
SupplierAdmin_i::destroy( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _serial == 0 ) {
    RDI_DUMP("Default SupplierAdmin is not allowed to be destroyed!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _disconnect_clients_and_dispose(1); // 1 means update channel
}

CosN_QoSProperties*
SupplierAdmin_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_NotifQoS*   qosp = qos_properties();
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  CosN_QoSProperties* res = qosp->get_qos(RDI_C_ADMIN);
  RDI_OPLOCK_RELEASE;
  return res;
}

void
SupplierAdmin_i::set_qos(const CosN_QoSProperties& r_qos   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS*   p_qos = _channel->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);
  
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  if ( ! _qosprop ) {
    _qosprop = new RDI_NotifQoS();
    RDI_AssertAllocThrowNo(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos); 
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::validate_qos
(const CosN_QoSProperties& r_qos,
 WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq 
 WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS*   p_qos = _channel->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_AssertAllocThrowNo(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,*rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  RDI_OPLOCK_RELEASE;
}

CosNF_FilterID
SupplierAdmin_i::add_filter(CosNF_Filter_ptr fltr WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_FilterID res = _fa_helper.add_filter_i(fltr);
  RDI_OPLOCK_RELEASE;
  return res;
}

void
SupplierAdmin_i::remove_filter(CosNF_FilterID fltrID WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_filter(fltrID);
  RDI_OPLOCK_RELEASE;
}

CosNF_Filter_ptr
SupplierAdmin_i::get_filter(CosNF_FilterID fltrID WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_Filter_ptr res = _fa_helper.get_filter(fltrID);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNF_FilterIDSeq*
SupplierAdmin_i::get_all_filters( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_FilterIDSeq* res = _fa_helper.get_all_filters();
  RDI_OPLOCK_RELEASE;
  return res;
}

void
SupplierAdmin_i::remove_all_filters( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_disposed) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_all_filters();
  RDI_OPLOCK_RELEASE;
}

// -------------------- ** Local-only routines ** ----------------------------

void
SupplierAdmin_i::propagate_offer_change(const CosN_EventTypeSeq& added,
					const CosN_EventTypeSeq& deled)
{
  RDI_OPLOCK_ACQUIRE(return);
  _propagate_offer_change(added, deled);
}

void
SupplierAdmin_i::remove_proxy(EventProxyPushConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_ListCursor<EventProxyPushConsumer_i *> lc;
  unsigned int    indx = 0;
 
  for ( lc = _cosevent_push.cursor(); indx < _cosevent_push.length(); indx++ ) {
    EventProxyPushConsumer_i* proxy = *lc;
    if ( proxy == prx ) {
      _cosevent_push.remove(lc);
      _num_proxies -= 1;
      _prxy_exit    = RDI_TimeValue::timeofday();
      _channel->decr_suppliers();
      break;
    }
    ++lc;
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(EventProxyPullConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_ListCursor<EventProxyPullConsumer_i *> lc;
  unsigned int    indx = 0;
 
  for ( lc = _cosevent_pull.cursor(); indx < _cosevent_pull.length(); indx++ ) {
    EventProxyPullConsumer_i* proxy = *lc;
    if ( proxy == prx ) {
      _cosevent_pull.remove(lc);
      _num_proxies -= 1;
      _prxy_exit    = RDI_TimeValue::timeofday();
      _channel->decr_suppliers();
      if ( _channel->_pull_supplier ) {    // Deregister Pull Proxy
	_channel->_pull_supplier->remove_proxy(prx);
      }
      break;
    }
    ++lc;
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(ProxyPushConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_any_push.exists(prx->_proxy_id()) ) {
    _prx_any_push.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_suppliers();
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(ProxyPullConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_any_pull.exists(prx->_proxy_id()) ) {
    _prx_any_pull.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_suppliers();
    if ( _channel->_pull_supplier ) {    // Deregister Pull Proxy
      _channel->_pull_supplier->remove_proxy(prx);
    }
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(StructuredProxyPushConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_struc_push.exists(prx->_proxy_id()) ) {
    _prx_struc_push.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_suppliers();
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(StructuredProxyPullConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_struc_pull.exists(prx->_proxy_id()) ) {
    _prx_struc_pull.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_suppliers();
    if ( _channel->_pull_supplier ) {      // Deregister Pull Proxy
      _channel->_pull_supplier->remove_proxy(prx);
    }
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(SequenceProxyPushConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_batch_push.exists(prx->_proxy_id()) ) {
    _prx_batch_push.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_suppliers();
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

void
SupplierAdmin_i::remove_proxy(SequenceProxyPullConsumer_i* prx)
{
  RDI_OPLOCK_ACQUIRE(return);
  RDI_DUMP("Remove proxy [" << prx->_proxy_id() << "] from admin " << (void*)this);
  if ( _prx_batch_pull.exists(prx->_proxy_id()) ) {
    _prx_batch_pull.remove(prx->_proxy_id());
    _num_proxies -= 1;
    _prxy_exit    = RDI_TimeValue::timeofday();
    _channel->decr_suppliers();
    if ( _channel->_pull_supplier ) {      // Deregister Pull Proxy
      _channel->_pull_supplier->remove_proxy(prx);
    }
  } else {
    RDI_DUMP("Invalid proxy [" << prx->_proxy_id() << "] provided");
  }
  RDI_OPLOCK_RELEASE;
}

CORBA::Boolean
SupplierAdmin_i::match_event(const CORBA::Any& event)
{
  RDI_OPLOCK_ACQUIRE(return 0);
  CORBA::Boolean matched = 1;
  if ( _fa_helper.has_filters() ) {
    CosNF_FilterIDSeq* fltrseq = _fa_helper.get_all_filters();
    CosNF_Filter_ptr   fltr = CosNF_Filter::_nil();
    matched = 0;
    for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
      fltr = _fa_helper.get_filter((*fltrseq)[ix]);
      if ( fltr->match(event) ) {
	matched = 1;
	break;
      }
    }
    delete fltrseq;
  }
  RDI_OPLOCK_RELEASE;
  return matched;
}

CORBA::Boolean
SupplierAdmin_i::match_event(const CosN_StructuredEvent* event, RDI_StructuredEvent* sevnt)
{
  RDI_OPLOCK_ACQUIRE(return 0);
  CORBA::Boolean matched = 1;
  if ( _fa_helper.has_filters() ) {
    CosNF_FilterIDSeq* fltrseq = _fa_helper.get_all_filters();
    CosNF_Filter_ptr filter = CosNF_Filter::_nil();
    Filter_i* rdfltr = 0;
    matched = 0;
    for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
      filter = _fa_helper.get_filter((*fltrseq)[ix]);
      rdfltr = Filter_i::Filter2Filter_i(filter);
      if ( (rdfltr && rdfltr->rdi_match(sevnt, _channel)) ||
	   (!rdfltr && filter->match_structured(*event)) ) {
	matched = 1;
	break;
      }
    }
    delete fltrseq;
  }
  RDI_OPLOCK_RELEASE;
  return matched;
}

RDI_NotifQoS*
SupplierAdmin_i::qos_properties()
{
  RDI_OPLOCK_ACQUIRE(return 0);
  RDI_NotifQoS* res = _qosprop ? _qosprop : _channel->qos_properties();
  RDI_OPLOCK_RELEASE;
  return res;
}

ostream& operator << (ostream& out, SupplierAdmin_i& adm)
{
  RDI_ListCursor<EventProxyPushConsumer_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullConsumer_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushConsumer_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullConsumer_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bpullcur;
  unsigned int cnt = 0;

  if (!adm.lock()) return out;
  out << &adm <<" ID "  << setw(3) << adm._serial << " Born " << 
    adm._time_born << " Last Proxy "  << adm._prxy_exit  << endl;
  if ( adm._qosprop )
    out << adm._qosprop << endl;
  else  
    out << "\tSame as QoSAdmin settings of Event Channel" << endl;
  cpushcur = adm._cosevent_push.cursor();
  for (cnt=0; cnt < adm._cosevent_push.length(); ++cnt, ++cpushcur)
    out << setw(4) << cnt << ": " << *(*cpushcur) << endl;
  cpullcur = adm._cosevent_pull.cursor();
  for (cnt=0; cnt < adm._cosevent_pull.length(); ++cnt, ++cpullcur)
    out << setw(4) << cnt << ": " << *(*cpullcur) << endl;
  cnt = 0;
  for (apushcur = adm._prx_any_push.cursor(); apushcur.is_valid(); ++apushcur)
    out << setw(4) << ++cnt << ": " << *(apushcur.val()) << endl;
  cnt = 0;
  for (apullcur = adm._prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur)
    out << setw(4) << ++cnt << ": " << *(apullcur.val()) << endl;
  cnt = 0;
  for (spushcur = adm._prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur)
    out << setw(4) << ++cnt << ": " << *(spushcur.val()) << endl;
  cnt = 0;
  for (spullcur = adm._prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur)
    out << setw(4) << ++cnt << ": " << *(spullcur.val()) << endl;
  cnt = 0;
  for (bpushcur = adm._prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur)
    out << setw(4) << ++cnt << ": " << *(bpushcur.val()) << endl;
  cnt = 0;
  for (bpullcur = adm._prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur)
    out << setw(4) << ++cnt << ": " << *(bpullcur.val()) << endl;

  adm.unlock();
  return out;
}

// ---------------------- ** Private routines ** ------------------------------

void
SupplierAdmin_i::disconnect_clients_and_dispose()
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_clients_and_dispose();
}

// ** requires OPLOCK already held
//  If update_channel is true, the admin unregisters itself and also
//  does a propagate_offer_change on the channel if there are
//  event types registered with the admin.
void
SupplierAdmin_i::_disconnect_clients_and_dispose(CORBA::Boolean update_channel)
{
  if (_disposed) {
    RDI_DUMP("SupplierAdmin_i::_disconnect_clients_and_dispose() called more than once");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _disposed = 1; // acts as guard: the following is executed by only one thread 

  if (update_channel) {
    // If there are any event types registered with this administrative
    // object, update the channel's aggregate information about them.
    // NOTE: we release the lock before we inkoke channel operations so
    //      that we avoid potential deadlocks when the channel is being
    //      destroyed at the same time
    CosN_EventTypeSeq added;
    CosN_EventTypeSeq deled;
    added.length(0);
    if ( _evtypes.length() ) {
      RDI_HashCursor<CosN_EventType, CORBA::ULong> curs;
      CORBA::ULong ix=0;
      deled.length( _evtypes.length() );
      for ( ix=0, curs = _evtypes.cursor(); curs.is_valid(); ++ix, ++curs )
	deled[ix] = curs.key();
    } else {
      deled.length(0);
    }
    _channel->unregister(this);
    EventChannel_i* chan = _channel;
    RDI_OPLOCK_RELEASE; // debump after invoking channel ops
    chan->propagate_offer_change(added, deled);
    RDI_OPLOCK_REACQUIRE(return);
  }
  RDI_DUMP("Begin Destruction of SupplierAdmin @ " << (void*)this << " [" << _serial << "]");
  unsigned int i;
  RDI_ListCursor<EventProxyPushConsumer_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullConsumer_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushConsumer_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullConsumer_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bpullcur;

  cpushcur = _cosevent_push.cursor();
  for (i=0; i < _cosevent_push.length(); ++i, ++cpushcur)
    (*cpushcur)->disconnect_client_and_dispose();
  cpullcur = _cosevent_pull.cursor();
  for (i=0; i < _cosevent_pull.length(); ++i, ++cpullcur)
    (*cpullcur)->disconnect_client_and_dispose();
  for ( apushcur = _prx_any_push.cursor(); apushcur.is_valid(); ++apushcur )
    apushcur.val()->disconnect_client_and_dispose();
  for ( apullcur = _prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur )
    apullcur.val()->disconnect_client_and_dispose();
  for ( spushcur = _prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur )
    spushcur.val()->disconnect_client_and_dispose();
  for ( spullcur = _prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur )
    spullcur.val()->disconnect_client_and_dispose();
  for ( bpushcur = _prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur )
    bpushcur.val()->disconnect_client_and_dispose();
  for ( bpullcur = _prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur )
    bpullcur.val()->disconnect_client_and_dispose();

  _cosevent_push.drain();  _cosevent_pull.drain();
  _prx_any_push.clear();   _prx_any_pull.clear();
  _prx_struc_push.clear(); _prx_struc_pull.clear();
  _prx_batch_push.clear(); _prx_batch_pull.clear();

  if ( _fa_helper.has_filters() ) _fa_helper.remove_all_filters();
  if ( _qosprop )      delete _qosprop;

  RDI_DUMP("SupplierAdmin @ " << (void*)this << " [" << _serial << "] destroyed");
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ** Requires OPLOCK is held
// ** Effect: Update the local hash table and if new entries are created or
//      deleted, update the channel as well, after releasing OPLOCK
//      to avoid deadlocks due to reverse order of lock acquisitions.
void
SupplierAdmin_i::_propagate_offer_change(const CosN_EventTypeSeq& added,
					 const CosN_EventTypeSeq& deled)
{
  CORBA::ULong ix=0, sz=0, vl=0;
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  new_added.length(0);
  old_deled.length(0);

  for (sz=0, ix = 0; ix < added.length(); ix++) {
    if ( _evtypes.lookup(added[ix], vl) ) {
      vl += 1; _evtypes.replace(added[ix], vl);
    } else {
      vl  = 1; _evtypes.insert(added[ix], vl);
      new_added.length(sz + 1);
      new_added[sz++] = added[ix];
    }
  }
  for (sz=0, ix = 0; ix < deled.length(); ix++) {
    if ( _evtypes.lookup(deled[ix], vl) ) {
      if ( vl == 1 ) {
	_evtypes.remove(deled[ix]);
	old_deled.length(sz + 1);
	old_deled[sz++] = deled[ix];
      } else {
	vl -= 1; _evtypes.replace(deled[ix], vl);
      }
    } else {
      RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
    }
  }
  EventChannel_i* chan = _channel;
  RDI_OPLOCK_RELEASE;
  if ( new_added.length() || old_deled.length() )
    chan->propagate_offer_change(new_added, old_deled);
}

