// -*- Mode: C++; -*-
//                              File      : RDIFileSys.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of RDI_FileSys [Monotoring file system changes]
//
 
/*
$Log: RDIFileSys.cc,v $
Revision 1.12  2000/11/15 21:17:30  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.11  2000/11/05 04:48:11  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.10  2000/10/30 05:42:37  alcfp
renamed CosNotify.h to CosNotifyShorthands.h and placed CosNotifyShorthands.h in omniNotify/include rather than src/services/include

Revision 1.9  2000/10/04 15:15:06  alcfp
more small updates to get rid of compiler warnings

Revision 1.8  2000/10/04 02:40:05  alcfp
small fixes to avoid some compiler warnings

Revision 1.7  2000/08/22 18:23:56  alcfp
added description to each file

Revision 1.6  2000/08/16 20:19:51  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/param.h>
#include "CosNotifyShorthands.h"
#include "CosNotifyChannelAdmin_i.h"
#include "RDIUtil.h"
#include "RDIFileSys.h"

#ifndef MAXPATHLEN
#  ifdef PATH_MAX
#    define MAXPATHLEN PATH_MAX
#  else 
#    define MAXPATHLEN 1024
#  endif
#endif

////////////////////////////////////////////////////

static const char* rdi_lfile_hdr =
		"*** omniNotify Notification Service - 2000 - Version 1.1.BETA ***";

////////////////////////////////////////////////////

class RDI_FileSysWorker : public omni_thread {
public:
  typedef void (RDI_FileSys::*FileSysMethod) (void);

  RDI_FileSysWorker(RDI_FileSys* object, FileSysMethod method) :
			omni_thread(), _object(object), _method(method)
			{ start_undetached(); }
  void* run_undetached(void *)	{ (_object->*_method)(); return 0; }
private:
  RDI_FileSys*  _object;
  FileSysMethod _method;
  RDI_FileSysWorker() {;}
};

////////////////////////////////////////////////////

RDI_FileSys::RDI_FileSys(const char* lfile, EventChannel_i* echan) :
		_mylock(),  _isover(&_mylock), _thread(0), 
		_finish(0), _totaln(0), _evchan(echan), _lgfile(0)
{
  for (unsigned int indx=0; indx < HASH_SIZE; indx++)
     _htable[ indx ] = 0;
  if ( lfile ) {
     // This log file may have been used in the past -- establish the
     // previous state, and then continue to use the file for logging 
     establish_state(lfile);
  }
  _thread = new RDI_FileSysWorker(this, & RDI_FileSys::monitor_loop);
}

////////////////////////////////////////////////////

RDI_FileSys::~RDI_FileSys()
{
  // Set the '_finish' flag to true and notify the monitoring
  // thread to terminate. Wait till the thread exits and then
  // clean up all allocated structures

  _finish = 1;
  _isover.signal();
  _thread->join(0);

  for (unsigned int hidx=0; hidx < HASH_SIZE; hidx++) {
     while ( _htable[hidx] ) {
	node_t* node  = _htable[hidx];
	_htable[hidx] = node->_next;
	delete node;
     }
     _htable[hidx] = 0;
  }
  if ( _lgfile )
     delete [] _lgfile;
  _lgfile = 0;
}

////////////////////////////////////////////////////

void RDI_FileSys::add_file(const char* name)
{
  if ( _finish ) {
     return;
  } else if ( ! name || (strlen(name) == 0) ) {
     throw RDI_FileSys::InvalidName();
  } else { 
     _mylock.lock();
     if ( add_entry(name) != 0 ) {
	_mylock.unlock();
     	throw RDI_FileSys::SystemError("stat", errno);
     } else {
	append_log("ADD_FILE", name);
	_isover.signal();
	_mylock.unlock();
     }
  }
}

////////////////////////////////////////////////////

void RDI_FileSys::add_dire(const char* name)
{
  if ( _finish ) {
     return;
  } else if ( ! name || (strlen(name) == 0) ) {
     throw RDI_FileSys::InvalidName();
  } else { 
     _mylock.lock();
     if ( add_dir_entry(name) != 0 ) {
	_mylock.unlock();
	throw RDI_FileSys::SystemError("opendir", errno);
     } else {
	append_log("ADD_DIRE", name);
	_isover.signal();
	_mylock.unlock();
     }
  }
}

////////////////////////////////////////////////////

void RDI_FileSys::remove_file(const char* name)
{
  if ( _finish || ! name || (strlen(name) == 0) ) {
     return;
  } else {
    _mylock.lock();
    if ( rem_entry(name) == 0 ) 
    	append_log("DEL_FILE", name);
    _mylock.unlock();
  }
}

////////////////////////////////////////////////////

void RDI_FileSys::remove_dire(const char* name)
{
  if ( _finish || ! name || (strlen(name) == 0) ) {
     return;
  } else {
    _mylock.lock();
    if ( rem_dir_entry(name) == 0 )
    	append_log("DEL_DIRE", name);
    _mylock.unlock();
  }
}

////////////////////////////////////////////////////

ostream& operator << (ostream& out, const RDI_FileSys& f)
{
  RDI_FileSys::node_t* fn;
  for (unsigned int ix=0; ix < RDI_FileSys::HASH_SIZE; ix++) {
     for (fn = f._htable[ix]; fn; fn = fn->_next) {
	out << fn->_name << " refc " << fn->_refc << " dirc " << fn->_dirc << 
	       " dirf " << fn->_dirf << " usid " << fn->_usid << endl;
     }
  }
  return out;
}

////////////////////////////////////////////////////

static const char* FileCreate = "FileCreate";
static const char* FileDelete = "FileDelete";
static const char* FileUpdate = "FileUpdate";

static const uid_t RDI_Invalid_UID = (uid_t)-9999;

void RDI_FileSys::monitor_loop()
{
  CosN_StructuredEvent se;
  unsigned int indx = 0;
  node_t *node = 0, *frst = 0, *prev = 0;
  unsigned long strt, curr, temp;
  struct stat sbuf;

  se.header.fixed_header.event_type.domain_name = (const char *) "FileSystem";
  se.header.variable_header.length(0);
  se.filterable_data.length(7);
  se.filterable_data[0].name = (const char *) "name";
  se.filterable_data[1].name = (const char *) "mode";
  se.filterable_data[2].name = (const char *) "uid";
  se.filterable_data[3].name = (const char *) "size";
  se.filterable_data[4].name = (const char *) "atime";
  se.filterable_data[5].name = (const char *) "mtime";
  se.filterable_data[6].name = (const char *) "ctime";

  omni_thread::get_time(&strt, &temp);

  while ( ! _finish ) {
     _mylock.lock();
     while ( (_totaln == 0) && ! _finish ) 
	_isover.wait();
     if ( _finish ) {
	_mylock.unlock();
	return;
     }

     while ( (indx != HASH_SIZE) && ! _htable[ indx ] )
	indx++;

     // If we have already examined all entries in the file table, check
     // if the time it took us is less than 5 secs, which is our period.
     // If this is the case, sleep for the remainder of the period prior
     // to starting over again from the beginning of the file table

     if ( indx == HASH_SIZE ) {
  	omni_thread::get_time(&curr, &temp);
	if ( (curr - strt) < 5 ) {
	   omni_thread::get_time(&curr, &temp, 5-(curr-strt));
	   _isover.timedwait(curr, 0);
	   if ( _finish ) {
	      _mylock.unlock();
	      return;
	   }
	}
	_mylock.unlock();
  	omni_thread::get_time(&strt, &temp);
	indx = 0;
	continue;
     }

     // We will be processing '_htable[indx]'. To avoid holding the lock
     // while doing so , we increment the entry reference counter by one
     // and we remember the very first entry at this point; **** all new
     // entries are inserted at '_htable[indx]' and, thus, we don't have
     // to address pointer inconsistencies ****

     frst = _htable[ indx ];
     for ( node = frst; node; node = node->_next )
	node->_refc += 1;
     _mylock.unlock();

     // The main processing loop -- check if the specific file has
     // been modified in any way

     for ( node = frst; node; node = node->_next ) {
	if ( (node->_refc == 1) && (node->_dirf == 0) ) {
           // The entry will be deleted since neither a directory nor an
	   // explict file monitoring specification does use it anymore.
	   continue;
	}
	if ( stat(node->_name, &sbuf) != 0 ) {
	   if ( (errno == ENOENT) && (node->_usid != RDI_Invalid_UID) ) {
	      se.header.fixed_header.event_type.type_name = FileDelete;
	      se.filterable_data[0].value <<= (const char *) node->_name;
              se.filterable_data[1].value <<= (CORBA::ULong) node->_mode;
              se.filterable_data[2].value <<= (CORBA::Long)  node->_usid;
              se.filterable_data[3].value <<= (CORBA::ULong) node->_size;
              se.filterable_data[4].value <<= (CORBA::ULong) node->_atms;
              se.filterable_data[5].value <<= (CORBA::ULong) node->_mtms;
              se.filterable_data[6].value <<= (CORBA::ULong) node->_ctms;
	      if ( _evchan ) 
		 _evchan->new_structured_event(se);
	      node->_usid = RDI_Invalid_UID;
	   }
	   continue;
	}

	// File exists.  If 'node->_usid == RDI_Invalid_UID', raise a 'FileCreate'
  	// event. Otherwise, examine if the file has been modified since
   	// the last time we checked its status

  	if ( node->_usid == RDI_Invalid_UID ) {
	   node->_atms = sbuf.st_atime; node->_mtms = sbuf.st_mtime;
           node->_ctms = sbuf.st_ctime; node->_size = sbuf.st_size;
           node->_mode = sbuf.st_mode;  node->_usid = sbuf.st_uid;

           se.header.fixed_header.event_type.type_name = FileCreate;
           se.filterable_data[0].value <<= (const char *) node->_name;
           se.filterable_data[1].value <<= (CORBA::ULong) node->_mode;
           se.filterable_data[2].value <<= (CORBA::Long)  node->_usid;
           se.filterable_data[3].value <<= (CORBA::ULong) node->_size;
           se.filterable_data[4].value <<= (CORBA::ULong) node->_atms;
           se.filterable_data[5].value <<= (CORBA::ULong) node->_mtms;
           se.filterable_data[6].value <<= (CORBA::ULong) node->_ctms;
           if ( _evchan )
              _evchan->new_structured_event(se);
	} else if (sbuf.st_mode !=node->_mode || sbuf.st_uid  !=node->_usid ||
		   sbuf.st_atime!=node->_atms || sbuf.st_mtime!=node->_mtms ||
		   sbuf.st_ctime!=node->_ctms || sbuf.st_size !=node->_size ) {
	   node->_atms = sbuf.st_atime; node->_mtms = sbuf.st_mtime;
	   node->_ctms = sbuf.st_ctime; node->_size = sbuf.st_size;
	   node->_mode = sbuf.st_mode;  node->_usid = sbuf.st_uid;

	   if ( (node->_refc > 1) || (node->_dirf && ! node->_dirc) ) {
	      se.header.fixed_header.event_type.type_name = FileUpdate;
              se.filterable_data[0].value <<= (const char *) node->_name;
              se.filterable_data[1].value <<= (CORBA::ULong) node->_mode;
              se.filterable_data[2].value <<= (CORBA::Long)  node->_usid;
              se.filterable_data[3].value <<= (CORBA::ULong) node->_size;
              se.filterable_data[4].value <<= (CORBA::ULong) node->_atms;
              se.filterable_data[5].value <<= (CORBA::ULong) node->_mtms;
              se.filterable_data[6].value <<= (CORBA::ULong) node->_ctms;
              if ( _evchan )
                 _evchan->new_structured_event(se);
           } 
	   if ( node->_dirc && S_ISDIR(sbuf.st_mode) ) {
	      update_dir_entry(node->_name);
	   }
	}
     }

     // Decrement the reference counters of all entries examined in
     // the above loop before processing the next batch. Here, some
     // entries may be deleted 

     _mylock.lock();
     prev = 0;
     for ( node = _htable[ indx ]; node != frst; node = node->_next )
	prev = node;
     node = frst;
     while ( node ) {
	node->_refc -= 1;
	if ( (node->_refc == 0) && (node->_dirc == 0) && 
	     ((node->_dirf == 0 || node->_usid == RDI_Invalid_UID)) ) {
	   if ( ! prev ) {
	      _htable[ indx ] = node->_next;
	      delete node;
	      node = _htable[ indx ];
	   } else { 
	      prev->_next = node->_next;
	      delete node;
	      node = prev->_next;
	   }
	   _totaln -= 1;
	} else {
	   prev = node;
	   node = node->_next;
	}
     }
     _mylock.unlock();
     indx += 1;
  }
}

////////////////////////////////////////////////////

RDI_FileSys::node_t* RDI_FileSys::lookup(const char* fname)
{
  unsigned int hidx = RDI_StrHash(fname) % HASH_SIZE;
  node_t* node = _htable[ hidx ];

  while ( node && (strcmp(node->_name, fname) != 0) )
	node = node->_next;
  return node;
}

////////////////////////////////////////////////////

int RDI_FileSys::add_entry(const char*   fname,
			   CORBA::Boolean dFile, 
			   CORBA::Boolean isDir,
			   CORBA::Boolean isNew)
{
  unsigned int hidx = RDI_StrHash(fname) % HASH_SIZE;
  node_t* node = _htable[ hidx ];
  struct stat sbuf;

  while ( node && (strcmp(node->_name, fname) != 0) )
     node = node->_next;
  if ( node ) {
     node->_dirc = isDir ? (node->_dirc + 1) : node->_dirc;
     node->_dirf = dFile ? 1 : node->_dirf;
     node->_refc = dFile ? node->_refc : (node->_refc + 1);
  } else {
     node = new node_t(fname);
     node->_dirc = isDir ? 1 : 0;
     node->_dirf = dFile ? 1 : 0;
     node->_refc = dFile ? 0 : 1;
     if ( dFile && isNew ) {
	node->_usid = RDI_Invalid_UID;	// See notes on 'update_dir_entry'
     } else if ( stat(fname, &sbuf) == 0 ) {
	node->_mode = sbuf.st_mode;
	node->_usid = sbuf.st_uid;
	node->_size = sbuf.st_size;
	node->_atms = sbuf.st_atime;
	node->_mtms = sbuf.st_mtime;
        node->_ctms = sbuf.st_ctime;
     } else if ( errno == ENOENT ) {
	node->_usid = RDI_Invalid_UID; // File may be created in future!!
     } else {			// We do not have access to file!!
	delete node;
	return -1;
     }
     node->_next = _htable[ hidx ];
     _htable[ hidx ] = node;
     _totaln += 1;
  }
  return 0;
}

////////////////////////////////////////////////////

int RDI_FileSys::rem_entry(const char*   fname, 
		           CORBA::Boolean dFile,
			   CORBA::Boolean isDir)
{
  unsigned int hidx = RDI_StrHash(fname) % HASH_SIZE;
  node_t* node = _htable[ hidx ];
  node_t* prev = 0;

  while ( node && (strcmp(node->_name, fname) != 0) ) {
     prev = node;
     node = node->_next;
  }
  if ( node ) {
     if ( !dFile && !isDir && !node->_refc ) {
	cerr << "No registered file entry for [" << fname << "]" << endl;
	return - 1;
     }
     node->_refc = dFile ? node->_refc : (node->_refc - 1);
     node->_dirc = isDir ? (node->_dirc - 1) : node->_dirc;
     node->_dirf = dFile ? 0 : node->_dirf;
     if ( (node->_refc == 0) && (node->_dirc == 0) && (node->_dirf == 0) ) {
        if ( prev ) {
	   prev->_next = node->_next;
	} else {
	   _htable[ hidx ] = node->_next;
 	}
        delete node;
        _totaln -= 1;
     }
     return 0;
  } else {
     return -1;
  }
}

////////////////////////////////////////////////////

int RDI_FileSys::add_dir_entry(const char* name)
{
  unsigned int hidx = RDI_StrHash(name) % HASH_SIZE;
  node_t* node = _htable[ hidx ];
  struct dirent* dentry = 0;
  DIR* dirp = 0;
  char fname[MAXPATHLEN];

  while ( node && (strcmp(node->_name, name) != 0) )
     node = node->_next;
  if ( node && node->_dirc ) {
     node->_dirc += 1;
  } else {
     if ( (dirp = opendir(name)) != (DIR *) 0 ) {
        while ( (dentry = readdir(dirp)) != (struct dirent *) 0 ) {
	   if ( !strcmp(dentry->d_name, ".") || !strcmp(dentry->d_name, "..") )
	      continue;
	   (void) sprintf(fname, "%s/%s", name, dentry->d_name);
	   (void) add_entry(fname, 1, 0, 0);
        }
        (void) closedir(dirp);
        (void) add_entry(name, 1, 1, 0);
     } else  {
	cerr << "Cannot access directory [" << name << "]" << endl;
        return -1;
     }
  }
  return 0;
}

////////////////////////////////////////////////////

int RDI_FileSys::rem_dir_entry(const char* name)
{
  unsigned int hidx = RDI_StrHash(name) % HASH_SIZE;
  node_t* node = _htable[ hidx ];
  struct dirent* dentry = 0;
  DIR* dirp = 0;
  char fname[MAXPATHLEN];

  while ( node && (strcmp(node->_name, name) != 0) )
     node = node->_next;
  if ( ! node ) {
     return -1;
  } else if ( node->_dirc == 0 ) {
     cerr << "No registered directory entry for [" << name << "]" << endl;
     return -1;
  } else if ( node->_dirc == 1 ) {
     if ( (dirp = opendir(name)) != (DIR *) 0 ) {
        while ( (dentry = readdir(dirp)) != (struct dirent *) 0 ) {
	   if ( !strcmp(dentry->d_name, ".") || !strcmp(dentry->d_name, "..") ) 
	      continue;        
           (void) sprintf(fname, "%s/%s", name, dentry->d_name);
	   (void) rem_entry(fname, 1, 0);
        }
        (void) closedir(dirp);
        (void) rem_entry(name, 1, 1);
     } else {
        cerr << "Failed to open monitored directory [" << name << "]" << endl;
        return -1;
     }
  } else {
     node->_dirc -= 1;
  }
  return 0;
}

////////////////////////////////////////////////////

void RDI_FileSys::update_dir_entry(const char* name)
{
  struct dirent* dentry = 0;
  DIR* dirp = 0;
  char fname[MAXPATHLEN];

  // We should add to the list of monitored files only those files
  // that did not exist the last time we visited the directory. In
  // addition, for these new entries, we will pretend that they do
  // not exist so that 'FileCreate' events are generated;

  if ( (dirp = opendir(name)) == (DIR *) 0 ) {
     cerr << "Failed to open monitored directory [" << name << "]" << endl;
     return;
  }
  while ( (dentry = readdir(dirp)) != (struct dirent *) 0 ) {
     if ( ! strcmp(dentry->d_name, ".") || ! strcmp(dentry->d_name, "..") )
	continue;
     (void) sprintf(fname, "%s/%s", name, dentry->d_name);
     (void) add_entry(fname, 1, 0, 1);
  }
  (void) closedir(dirp);
}

////////////////////////////////////////////////////

void RDI_FileSys::establish_state(const char* lfile)
{
  char *oper = 0, *name = 0;
  unsigned int cntr = 0;
  FILE* filep = 0;
  char fline[1096];

  if ( access(lfile, F_OK) == 0 ) {
     if ( (filep = fopen(lfile, "r")) == (FILE *) 0 ) {
	cerr << "Unable to open provided log file [" << lfile << "]" << endl;
	cerr << "No logging is going to be used at this point ....." << endl;
	return;
     }
     if ( (name = fgets(fline, 1096, filep)) != (char *) 0 ) {
	// Get rid of extra new line character at the end of string
	if ( fline[ strlen(fline) - 1 ] == '\n' )
	   fline[ strlen(fline) - 1 ] = '\0';
     }
     if ( ! name || strcmp(fline, rdi_lfile_hdr) ) {
	cerr << "File [" << lfile << "] is **NOT** a READY log file" << endl;
	cerr << "No logging is going to be used at this point ....." << endl;
	(void) fclose(filep);
	return;
     }

     while ( fgets(fline, 1024, filep) ) {
	cntr += 1;
	// We should have a timestamp (25 chars), operation (8 chars), 
 	// name (>= 1 char), and two spaces minimum in the above line
	if ( strlen(fline) < 36 ) {
	   cerr << "[" << lfile << ":" << cntr << "] invalid log entry '" <<
	           fline << "' ... skipping entry ..." << endl;
	   continue;
	}
	// Get rid of extra new line character at the end of string
	if ( fline[ strlen(fline) - 1 ] == '\n' )
	   fline[ strlen(fline) - 1 ] = '\0';
	oper = & fline[26];
	name = & fline[35];
	if ( strncmp(oper, "ADD_FILE", 8) == 0 ) {
	   (void) add_entry(name);
	} else if ( strncmp(oper, "DEL_FILE", 8) == 0 ) {
	   rem_entry(name);
	} else if ( strncmp(oper, "ADD_DIRE", 8) == 0 ) {
	   (void) add_dir_entry(name);
	} else if ( strncmp(oper, "DEL_DIRE", 8) == 0 ) {
	   rem_dir_entry(name);
	} else {
	   cerr << "[" << lfile << ":" << cntr << "] invalid log entry '" <<
	           fline << "' ... skipping entry ..." << endl;
	}
     }

     (void) fclose(filep);
     _lgfile = new char[strlen(lfile) + 1];
     strcpy(_lgfile, lfile);

     // If we had too many entries into the log file, we compress the log
     // file by eliminating entries not needed anymore; this is done only
     // when the log entries are more than double the registered entries.

     if ( (cntr > 200) && (cntr > 2*_totaln) )
	cleanup_log();

  } else {
     if ( (filep = fopen(lfile, "w")) == (FILE *) 0 ) {
     	cerr << "Cannot create provided log file ["  << lfile << "]" << endl;
	cerr << "No logging is going to be used at this point ....." << endl;
     } else {
	// Write READY log header as the first line in the file
	(void) fprintf(filep, "%s\n", rdi_lfile_hdr);
	(void) fflush(filep);
	(void) fclose(filep);

  	_lgfile = new char[strlen(lfile) + 1];
  	strcpy(_lgfile, lfile);
     }
  }
  return;
}

////////////////////////////////////////////////////

void RDI_FileSys::append_log(const char* action, const char* name)
{
  if ( _lgfile ) {
     FILE*  filep = 0;
     time_t currt = time(0);
     char*  ptime = ctime(&currt);

     ptime[24] = ':';
     if ( (filep = fopen(_lgfile, "a+")) != (FILE *) 0 ) {
	(void) fprintf(filep, "%s %s %s\n", ptime, action, name);
	(void) fflush(filep);
	(void) fclose(filep);
     } else {
	cerr << "Failed to open log file [" << _lgfile << "]" << endl;
	cerr << "Continue with no logging from this point on" << endl;
	delete [] _lgfile;
	_lgfile = 0;
     } 
  }
}

////////////////////////////////////////////////////

void RDI_FileSys::cleanup_log()
{
  FILE*   filep = 0;
  char    nname[1024];
  char*   ptime = 0;
  node_t* node  = 0;
  time_t  currt = time(0);
  unsigned int ix, jx;

  // Create the new log file in the same directory as the old log
  // file. If the creation fails, we will continue to use the old
  // log file -- should we raise an error instead?

  sprintf(nname, "%s.new", _lgfile);
  if ( (filep = fopen(nname, "a+")) == (FILE *) 0 )
     return;

  // First,  write the READY log header as the first line in this
  // file and, then, insert an entry for every directory and file 
  // that is being monitored *** the timestamp of all the entries
  // is set to the current time ***

  (void) fprintf(filep, "%s\n", rdi_lfile_hdr);
  ptime = ctime(&currt);
  ptime[24] = ':';

  for (ix=0; ix < HASH_SIZE; ix++) {
     for (node = _htable[ix]; node; node = node->_next) {
       	for (jx=0; jx < node->_refc; jx++) {
  	   (void) fprintf(filep, "%s %s %s\n", ptime, "ADD_FILE", node->_name);
   	}
	for (jx=0; jx < node->_dirc; jx++) {
	   (void) fprintf(filep, "%s %s %s\n", ptime, "ADD_DIRE", node->_name);
	}
     }
  }
  (void) fflush(filep);
  (void) fclose(filep); 

  // Next, unlink the old log file and rename the new one to have
  // the same name as the old one

  (void) unlink(_lgfile);
  (void) rename(nname, _lgfile);
}

