// -*- Mode: C++; -*-
//                              File      : RDIEventQueue.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of RDI_EventQueue [Channel's event queue]
//
 
/*
$Log: RDIEventQueue.cc,v $
Revision 1.36  2000/12/14 22:58:59  alcfp
minor fix to mutex release

Revision 1.35  2000/12/14 18:35:38  alcfp
fixes in channel shutdown sequence

Revision 1.34  2000/11/15 21:17:30  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.33  2000/10/01 19:19:52  alcfp
missing semi fixed

Revision 1.32  2000/10/01 13:39:53  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.31.2.1  2000/09/30 15:39:53  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.31  2000/08/22 18:23:56  alcfp
added description to each file

Revision 1.30  2000/08/16 20:19:49  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>
#include "RDI.h"
#include "RDIDebug.h"
#include "RDIEventQueue.h"
#include "RDIThreadPriority.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//              Support class used for garbage collection                //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

class EventQueueWorker : public omni_thread {
public:
  typedef void (RDI_EventQueue::*QueueMethod)(void);

  EventQueueWorker(RDI_EventQueue* queue, QueueMethod method,
                   priority_t prio=RDI_EVENT_QUEUE_PRIORITY) :
    omni_thread(NULL, prio), _equeue(queue), _method(method) {;}
  void run(void *)      { (_equeue->*_method)(); }
private:
  RDI_EventQueue* _equeue;
  QueueMethod     _method;
  EventQueueWorker()  {;}
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

RDI_EventQueue::RDI_EventQueue(unsigned int qsize, 
			       unsigned int concr, CORBA::Short dsqos) :
  _oplock(), _qempty(&_oplock), _gclock(), _qclean(&_gclock),
  _period(300), _gcdone(1), _finish(0), _worker(0), _evhead(0),
  _evtail(0), _disqos(dsqos), _length(0), _maxnum(qsize), 
  _announ(0), _nmdrop(0), _numreg(concr), _numblk(0)
{
  _worker = new EventQueueWorker(this, & RDI_EventQueue::garbage_collect);
  _worker->start();
}

RDI_EventQueue::~RDI_EventQueue()
{
  _oplock.lock();
  RDI_DUMP("   - event queue is shutting down");
  if (!_finish) {
    // Set '_finish' flag to true and signal the garbage collection and
    // any other blocked threads. Wait for _numblk zero and gcdone
    _finish = 1;
    _qclean.signal();
    _qempty.broadcast();
    while (_numblk != 0 || ! _gcdone) {
      RDI_DUMP("~RDI_EventQueue waiting on " << _numblk << " blocked threads [gcdone = " << (int)_gcdone << "]");
      _qclean.signal();
      _qempty.broadcast();
      _oplock.unlock();
      omni_thread::yield();
      _oplock.lock();
    }
    RDI_DUMP("~RDI_EventQueue: _numblk is zero and gc thread done");
  }
  while ( _evhead ) {	// Delete all enqueued events
    RDI_StructuredEvent* tmpevnt = _evhead;
    _evhead = tmpevnt->_next;
    delete tmpevnt;
  }
  // Need to release the mutex to avoid EBUSY errors during 
  // destruction of the mutex object
  _oplock.unlock();
  _length = _numreg = 0;
  _evhead = _evtail = 0;
}

// Set '_finish' flag to true and signal the garbage collection and
// any other blocked threads.  If wait_on_threads is true, wait until
// there are no blocked threads and gcdone before returning.
void RDI_EventQueue::set_finished(CORBA::Boolean wait_on_threads) {
  _oplock.lock();
  RDI_DUMP("RDI_EventQueue::set_finished called");
  if (_finish) {
    RDI_DUMP("RDI_EventQueue::set_finished called more than once");
    _oplock.unlock();
    return;
  }
  _finish = 1;
  _qclean.signal();
  _qempty.broadcast();
  RDI_DUMP("RDI_EventQueue::set_finished: _numblk = " << _numblk << " [gcdone = " << (int)_gcdone << "]");
  if (wait_on_threads) {
    while (_numblk != 0 || ! _gcdone) {
      RDI_DUMP("RDI_EventQueue::set_finished waiting on " << _numblk << " blocked threads [gcdone = " << (int)_gcdone << "]");
      _qclean.signal();
      _qempty.broadcast();
      _oplock.unlock();
      omni_thread::yield();
      _oplock.lock();
    }
    RDI_DUMP("RDI_EventQueue::set_finished: _numblk is zero and _gcdone is true");
  }
  _oplock.unlock();
}

int RDI_EventQueue::insert(RDI_StructuredEvent* event)
{
  _oplock.lock();
  _announ += 1;

  if ( _finish ) {		// Are we shutting down???
    _oplock.unlock();
    return -1;
  }
  if ( _maxnum && (_length >= _maxnum) && discard_events() ) {
    _oplock.unlock();
    return -1;
  }

  // The reference counter of the event MUST by 1 at this point
  RDI_Assert((event->ref_counter()==1), "Invalid event refCnt");

  // Increment event reference counter by 'concurrency' level.
  // Not very flexible since we cannot update the concurrency
  // level during run time
  event->incr_ref_counter(_numreg, RDI_LOCK_EVENT);

  _length     += 1;
  event->_next = 0;
  if ( _evtail ) { _evtail->_next = event; _evtail = event; } 
  else           { _evtail = event;        _evhead = event; }

  if ( _numblk != 0 )	// Signal threads waiting for new event
    _qempty.broadcast();
  if ( run_garbage_collect() && _gcdone ) // Garbage collection
    _qclean.signal();

  _oplock.unlock();
  return 0;
}

// Return the next event to be processed, given that the previously
// processed event is 'eprev'. The assumption made in the following
// method is that  accessing a pointer or a four-byte integer is an
// atomic operation and, hence,  we do not need to acquire the lock
// on the queue unless we have already processed all events .......

RDI_StructuredEvent* 
RDI_EventQueue::next_event(RDI_StructuredEvent* eprev, CORBA::Boolean block)
{
  RDI_StructuredEvent* event = 0;

  if ( _finish )  {	// We are shutting down
    return 0;
  }

  if ( ! eprev ) {
    // First time in the method; we assume that only future
    // events are of interest and, thus, we use '_evtail'
    if ( (event = _evtail) ) {
      if ( event->get_state() == RDI_StructuredEvent::NEWBORN )
	event->set_state(RDI_StructuredEvent::PENDING);
      return event;
    } else if ( ! block ) {
      return 0;
    } else {
      // To avoid missing events we use '_evhead'. In
      // addition,  we acuire '_oplock' since we will
      // update the queue state.
      _oplock.lock();
      _numblk += 1;
      while ( ! _finish && ! _evhead )
	_qempty.wait();
      _numblk -= 1;
      if ( _finish ) {	// We are shutting down
	_oplock.unlock();
	return 0;
      }
      event = _evhead;
      _oplock.unlock();
      if ( event->get_state() == RDI_StructuredEvent::NEWBORN )
	event->set_state(RDI_StructuredEvent::PENDING);
      return event;
    }
  } else {
    // We decrement the reference counter of an event which
    // has already being accessed ONLY AFTER a new event is
    // available to 'prevent' any racing condition with the
    // garbage collection worker thread
    if ( eprev->_next ) {
      event = eprev->_next;
      eprev->decr_ref_counter( RDI_LOCK_EVENT );
      if ( event->get_state() == RDI_StructuredEvent::NEWBORN )
	event->set_state(RDI_StructuredEvent::PENDING);
      return event;
    } else if ( ! block ) {
      return 0;
    } else {
      _oplock.lock();
      _numblk += 1;
      while ( ! _finish && ! eprev->_next )
	_qempty.wait();
      _numblk -= 1;
      if ( _finish ) {        // We are shutting down
	_oplock.unlock();
	return 0;
      }
      event = eprev->_next;
      _oplock.unlock();
      eprev->decr_ref_counter( RDI_LOCK_EVENT );
      if ( event->get_state() == RDI_StructuredEvent::NEWBORN )
	event->set_state(RDI_StructuredEvent::PENDING);
      return event;
    }
  }
  return 0;
}

// Based on the discard policy specified,  either no new events are
// inserted into the queue or existing events are dropped. An event
// can be dropped only when it has not been processed yet or it has
// been processed and not needed any longer.  Returns -1 when event
// cannot be inserted into the queue.

int RDI_EventQueue::discard_events()
{
  RDI_StructuredEvent* tmpevnt = 0;

  if ( _disqos == CosN_LifoOrder ) {
    RDI_DUMP("CosN_LifoOrder not implemented yet");
    _nmdrop += 1;
    return -1;
  } else if ( _disqos == CosN_DeadlineOrder ) {
    RDI_DUMP("CosN_DeadlineOrder not implemented yet");
    _nmdrop += 1;
    return -1;
  } else if ( _disqos == CosN_PriorityOrder ) {
    RDI_DUMP("CosN_PriorityOrder not implemented yet");
    _nmdrop += 1;
    return -1;
  } else if ( (_disqos == CosN_AnyOrder) ||
	      (_disqos == CosN_FifoOrder) ) {
    if ( (_evhead->ref_counter() > 1) ||
	 (_evhead->get_state() == RDI_StructuredEvent::NEWBORN) )
      return -1;
    tmpevnt = _evhead;
    _evhead = _evhead->_next;
    delete tmpevnt;
    _nmdrop += 1;
  } else { 
    RDI_DUMP("Invalid CosNotification Discard Policy specified");
    return -1;
  }
  return 0;
}

// Garbage collect processed events that are not needed any longer.
// These events have a reference counter==1, and their state is not
// RDI_StructuredEvent::NEWBORN. Garbage collection starts from the
// head of the queue and stops once an event that does not meet the
// above criteria is encountered 

void RDI_EventQueue::garbage_collect()
{
  RDI_StructuredEvent*  tmpevnt;
  unsigned int cursize, evncntr;
  unsigned long secs, nsecs;

  while ( 1 ) {
    _gclock.lock();
    if ( _finish ) {
      RDI_DUMP("\tGC thread "<<omni_thread::self()->id()<<" exits");
      omni_thread::exit();
      return;
    }
    omni_thread::get_time(&secs, &nsecs, _period);
    _qclean.timedwait(secs, 0);
    if ( _finish ) {
      RDI_DUMP("\tGC thread "<<omni_thread::self()->id()<<" exits");
      _gclock.unlock();
      omni_thread::exit();
      return;
    }
    _gclock.unlock();

    // Remember the current length of the queue and set the
    // garbage collection flag to indicate work in progress

    _oplock.lock();
    cursize = _length;
    _gcdone = 0;		// Garbage collection started
    _oplock.unlock();
    evncntr = 0;
    while ( --cursize && _evhead && (_evhead->ref_counter() == 1) &&
	    (_evhead->get_state() != RDI_StructuredEvent::NEWBORN) ) {
      tmpevnt = _evhead;
      _evhead = _evhead->_next;
      delete tmpevnt;
      if ( ++evncntr % 100 == 0 ) 
	omni_thread::yield();
    }
    _oplock.lock();
    if ( evncntr ) {
      RDI_DUMP("GC: disposed "<<evncntr<<" of "<<_length<<" events");
    }
    _length -= evncntr;
    _gcdone  = 1;		// Garbage collection finished
    _oplock.unlock();
    omni_thread::yield();
  }
}

void RDI_EventQueue::set_max_size(unsigned int new_max_size)
{
  _oplock.lock(); 
  _maxnum = new_max_size; 
  _oplock.unlock();
}

void RDI_EventQueue::set_gc_period(unsigned long nsecs)
{
  _oplock.lock();
  _period = nsecs ? nsecs : 1;
  _qclean.signal();
  _oplock.unlock();
}

void RDI_EventQueue::display_stats(CORBA::Boolean show_events)
{
  _oplock.lock();
  
  cout << "RDI_EventQueue" << endl << "--------------" << endl;
  if ( show_events ) {
    RDI_StructuredEvent* evnode = _evhead;
    while ( evnode ) {
      cout << "\t" << evnode << " - " << evnode->get_domain_name() << "::" <<
	evnode->get_type_name() << " [" << evnode->get_event_name()  <<
	"] " << " refCnt " << evnode->ref_counter();
      switch ( evnode->get_state() ) {
      case RDI_StructuredEvent::INVALID   : cout<<" INVALID"   <<endl; break;
      case RDI_StructuredEvent::NEWBORN   : cout<<" NEWBORN"   <<endl; break;
      case RDI_StructuredEvent::DISPATCHED: cout<<" DISPATCHED"<<endl; break;
      case RDI_StructuredEvent::PENDING   : cout<<" PENDING"   <<endl; break;
      }
      evnode = evnode->_next;
    }
  }
  cout << "Size " << _length << " [Max " << _maxnum << "] #announced " << 
    _announ << " #dropped " << _nmdrop << " period " << _period << endl;
  _oplock.unlock();
}
