// -*- Mode: C++; -*-
//                              File      : ProxyConsumer.cc
//                              Package   : omniNotify-Library
//                              Created on: 18-Oct-2000
//                              Authors   : gruber
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of Consumer proxies
//
 
/*
$Log: ProxyConsumer.cc,v $
Revision 1.8  2000/12/01 23:01:09  alcfp
fixed 2 bugs, 1 in admin-level subscription_change, 1 in changepool _next_available method

Revision 1.7  2000/11/29 21:42:02  alcfp
fixed bug in SequenceProxyPullConsumer_i::_pull_event

Revision 1.6  2000/11/17 21:36:48  alcfp
fixed oversight in matching logic when filters are only located at proxy or admin but not both

Revision 1.5  2000/11/16 05:32:15  alcfp
fixes for aCC

Revision 1.4  2000/11/15 21:17:29  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.3  2000/11/05 19:49:51  alcfp
scope fix

Revision 1.2  2000/11/05 04:48:10  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.1  2000/10/30 04:40:15  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

*/
 
#include <iomanip.h>

#include "RDIDebug.h"
#include "RDITimeDefs.h"
#include "CosNotifyChannelAdmin_i.h"
#include "RDIOplocksMacros.h"

extern const char* RDI_PRX_TYPE(const CosNA_ProxyType& type);

// ------------------------------------------------------------- //
// Support classes needed for implementing  the thread-per-proxy //
// notification approach                                         //
// ------------------------------------------------------------- //

class AnyPullWorker : public omni_thread {
public:
  typedef void (ProxyPullConsumer_i::*Method)(void);
  AnyPullWorker(ProxyPullConsumer_i* p, Method m) :
    omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  ProxyPullConsumer_i* _proxy;
  Method               _method;
  AnyPullWorker()  {;}
};

class StrPullWorker : public omni_thread {
public:
  typedef void (StructuredProxyPullConsumer_i::*Method)(void);
  StrPullWorker(StructuredProxyPullConsumer_i* p, Method m) :
    omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  StructuredProxyPullConsumer_i* _proxy;
  Method                         _method;
  StrPullWorker()  {;}
};

class SeqPullWorker : public omni_thread {
public:
  typedef void (SequenceProxyPullConsumer_i::*Method)(void);
  SeqPullWorker(SequenceProxyPullConsumer_i* p, Method m) :
    omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  SequenceProxyPullConsumer_i* _proxy;
  Method                       _method;
  SeqPullWorker()  {;}
};

// ------------------------------------------------------------- //
// RDIProxyConsumer                                              //
// ------------------------------------------------------------- //

RDIProxyConsumer::RDIProxyConsumer(SupplierAdmin_i*      admin,
				   EventChannel_i*       chann,
				   const RDI_ObjectKind  otype,
				   const CosNA_ProxyType prtype,
				   const CosNA_ProxyID&  prxid) : 
  _oplockptr(0), _fa_helper(), _channel(chann), _myadmin(admin),
  _otype(otype), _prxtype(prtype), _pserial(prxid), _nevents(0),
  _pxstate(RDI_NotConnected), _active(0), 
  _qosprop(0), _sc_off(0),
  _evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  RDI_OPLOCK_INIT;
  _sc_subscriber = CosNC_NotifySubscribe::_nil();
}

// ----------- ** Methods that implement public CORBA interfaces ** ---------------

CosNA_ProxyType RDIProxyConsumer::MyType()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_ProxyType res = _prxtype;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_ProxyID RDIProxyConsumer::MyID()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_ProxyID res = _pserial;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_SupplierAdmin_ptr RDIProxyConsumer::MyAdmin()
{ 
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_SupplierAdmin_ptr res = WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosN_EventTypeSeq*
RDIProxyConsumer::obtain_subscription_types(CosNA_ObtainInfoMode mode)
{
  // to achieve atomicity for CosNA_ALL_NOW_UPDATES_ON we must
  // ask _channel to do this operation so that lock acquisition
  // order will be _chanel (actually _type_map)
  // followed by this + _channel->_schange_pool
  // (via call to _enable_updates or _disable_updates)

  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_OPLOCK_BUMP_RELEASE;
  CosN_EventTypeSeq* res = _channel->pxy_obtain_subscription_types(this, mode);
  RDI_OPLOCK_ACQUIRE_DEBUMP_RELEASE(RDI_THROW_INV_OBJREF);
  return res;
}

void RDIProxyConsumer::validate_event_qos
(const CosN_QoSProperties& r_qos,
 WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_AssertAllocThrowNo(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  RDI_OPLOCK_RELEASE;
}

CosN_QoSProperties* RDIProxyConsumer::get_qos()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  CosN_QoSProperties* res = qosp->get_qos(_otype);
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxyConsumer::set_qos(const CosN_QoSProperties& r_qos)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);
  
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,_otype,eseq,rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  if ( ! _qosprop ) {
    _qosprop = new RDI_NotifQoS();
    RDI_AssertAllocThrowNo(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos); 
  RDI_OPLOCK_RELEASE;
}

void RDIProxyConsumer::validate_qos
(const CosN_QoSProperties& r_qos,
 WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_AssertAllocThrowNo(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,_otype,eseq,*rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  RDI_OPLOCK_RELEASE;
}

CosNF_FilterID RDIProxyConsumer::add_filter(CosNF_Filter_ptr filter)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_FilterID res = _fa_helper.add_filter_i(filter);
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxyConsumer::remove_filter(CosNF_FilterID fltrID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_filter(fltrID);
  RDI_OPLOCK_RELEASE;
}

void RDIProxyConsumer::remove_all_filters()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_all_filters();
  RDI_OPLOCK_RELEASE;
}

CosNF_Filter_ptr RDIProxyConsumer::get_filter(CosNF_FilterID fltrID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_Filter_ptr res = _fa_helper.get_filter(fltrID);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNF_FilterIDSeq* RDIProxyConsumer::get_all_filters()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_FilterIDSeq* res = _fa_helper.get_all_filters();
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxyConsumer::offer_change
(const CosN_EventTypeSeq& added, 
 const CosN_EventTypeSeq& deled)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_EventTypeSeq added_copy = added;
  CosN_EventTypeSeq deled_copy = deled;
  CORBA::ULong ix = 0, sz = 0, vl = 0;
  if ( ! RDI_EventType::valid_sequence(added_copy, ix) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(added[ix]);
  }
  if ( ! RDI_EventType::valid_sequence(deled_copy, ix) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(deled[ix]);
  }
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  new_added.length(0);
  old_deled.length(0);
  for (sz = 0, ix = 0; ix < added_copy.length(); ix++) {
    if ( _evtypes.lookup(added_copy[ix], vl) ) {
      vl += 1; _evtypes.replace(added_copy[ix], vl);
    } else {
      vl  = 1; _evtypes.insert(added_copy[ix], vl);
      new_added.length(sz + 1);
      new_added[sz].domain_name = added_copy[ix].domain_name;
      new_added[sz++].type_name = added_copy[ix].type_name;
    }
  }
  for (sz = 0, ix = 0; ix < deled_copy.length(); ix++) {
    if ( _evtypes.lookup(deled_copy[ix], vl) ) {
      if ( vl == 1 ) {
	_evtypes.remove(deled_copy[ix]);
	old_deled.length(sz + 1);
	old_deled[sz].domain_name = deled_copy[ix].domain_name;
	old_deled[sz++].type_name = deled_copy[ix].type_name;
      } else {
	vl -= 1; _evtypes.replace(deled_copy[ix], vl);
      } 
    } else {
      RDI_DUMP("Invalid " << (const char*)deled_copy[ix].domain_name << "::" << (const char*)deled_copy[ix].type_name);
    }
  }
  SupplierAdmin_i* myadmin = _myadmin;
  RDI_OPLOCK_RELEASE;
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() )
    myadmin->propagate_offer_change(new_added, old_deled);
}

// -------------------- ** Local-only Methods ** -----------------------------------

void RDIProxyConsumer::_enable_updates() {
  RDI_OPLOCK_ACQUIRE(return);
  if ( (_pxstate != RDI_Connected) || CORBA::is_nil(_sc_subscriber) || ! _channel->_schange_pool ) {
    // _sc_off status is meaningless without connected _nc_supplier and a valid _schange_pool
    _sc_off = 0;
    RDI_OPLOCK_RELEASE;
    return;
  }
  if (!_sc_off) {
    // make subscription_change msgs reflect only subsequent updates
    // by removing 'this' before re-inserting it
    _channel->_schange_pool->remove_proxy(this);
  }
  _sc_off = 0;
  _channel->_schange_pool->insert_proxy(this);
  RDI_OPLOCK_RELEASE;
}

void RDIProxyConsumer::_disable_updates() {
  RDI_OPLOCK_ACQUIRE(return);
  if ( (_pxstate != RDI_Connected) || CORBA::is_nil(_sc_subscriber) || ! _channel->_schange_pool ) {
    // _sc_off status is meaningless without connected _nc_supplier and a valid _schange_pool
    _sc_off = 1;
    RDI_OPLOCK_RELEASE;
    return;
  }
  if (!_sc_off) {
    _sc_off = 1;
    _channel->_schange_pool->remove_proxy(this);
  }
  RDI_OPLOCK_RELEASE;
}

// Return true only if there are admin-level filters and at least one matches event
// ** N.B.: oplock should be held during call to this helper routine
CORBA::Boolean RDIProxyConsumer::_match_event_admin_level(const CORBA::Any& event) {
  return (_myadmin->has_filters() && _myadmin->match_event(event));
}

// Return true only if there are proxy-level filters and at least one matches event
// ** N.B.: oplock should be held during call to this helper routine
CORBA::Boolean RDIProxyConsumer::_match_event_proxy_level(const CORBA::Any& event) {
  CORBA::Boolean matched = 0;
  if ( _fa_helper.has_filters() ) {
    CosNF_FilterIDSeq* fseq = _fa_helper.get_all_filters();
    CosNF_Filter_ptr filter;
    for (CORBA::ULong ix = 0; ix < fseq->length(); ix++) {
      filter = _fa_helper.get_filter((*fseq)[ix]);
      if ( filter->match(event) ) {
	matched = 1;
	break;
      }
    }
    delete fseq;
  }
  return matched;
}

// Return true if event should be added to the channel
// ** N.B.: oplock should be held during call to this helper routine  
CORBA::Boolean RDIProxyConsumer::_match_event(const CORBA::Any& event) {
  // no filters at all => matched 
  if ( (! _fa_helper.has_filters()) && (! _myadmin->has_filters()) ) {
    return 1;
  }
  // proxy or admin or both have filters
  if (!_fa_helper.has_filters()) {
    // only admin has filters -- _myadmin->_admin_operator() does not matter
    return _match_event_admin_level(event);
  } else if (!_myadmin->has_filters()) {
    // only proxy has filters -- _myadmin->_admin_operator() does not matter
    return _match_event_proxy_level(event);
  } else {
    // both proxy and admin have filters -- _myadmin->_admin_operator() matters
    if (_myadmin->_admin_operator() == CosNA_AND_OP) {
      return (_match_event_proxy_level(event) &&  _match_event_admin_level(event));
    } else {
      return (_match_event_proxy_level(event) ||  _match_event_admin_level(event));
    }
  }
}

// Return true only if there are admin-level filters and at least one matches event
// ** N.B.: oplock should be held during call to this helper routine
CORBA::Boolean RDIProxyConsumer::_match_event_admin_level
(const CosN_StructuredEvent* event, RDI_StructuredEvent* sevnt) {
  return (_myadmin->has_filters() && _myadmin->match_event(event, sevnt));
}

// Return true only if there are proxy-level filters and at least one matches event
// ** N.B.: oplock should be held during call to this helper routine
CORBA::Boolean RDIProxyConsumer::_match_event_proxy_level
(const CosN_StructuredEvent* event, RDI_StructuredEvent* sevnt) {
  CORBA::Boolean matched = 0;
  if ( _fa_helper.has_filters() ) {
    CosNF_FilterIDSeq* fltrseq = _fa_helper.get_all_filters();
    CosNF_Filter_ptr filter;
    Filter_i* rdfltr = 0;
    for (CORBA::ULong ix = 0; ix < fltrseq->length(); ix++) {
      filter = _fa_helper.get_filter((*fltrseq)[ix]);
      rdfltr = Filter_i::Filter2Filter_i(filter);
      if ( (rdfltr && rdfltr->rdi_match(sevnt, _channel)) ||
	   (!rdfltr && filter->match_structured(*event)) ) {
	matched = 1;
	break;
      }
    }
    delete fltrseq;
  }
  return matched;
}

// Return true if event should be added to the channel
// ** N.B.: oplock should be held during call to this helper routine
CORBA::Boolean RDIProxyConsumer::_match_event
(const CosN_StructuredEvent* event, RDI_StructuredEvent* sevnt) {
  // no filters at all => matched 
  if ( (! _fa_helper.has_filters()) && (! _myadmin->has_filters()) ) {
    return 1;
  }
  // proxy or admin or both have filters
  if (!_fa_helper.has_filters()) {
    // only admin has filters -- _myadmin->_admin_operator() does not matter
    return _match_event_admin_level(event, sevnt);
  } else if (!_myadmin->has_filters()) {
    // only proxy has filters -- _myadmin->_admin_operator() does not matter
    return _match_event_proxy_level(event, sevnt);
  } else {
    // both proxy and admin have filters -- _myadmin->_admin_operator() matters
    if (_myadmin->_admin_operator() == CosNA_AND_OP) {
      return (_match_event_proxy_level(event, sevnt) &&  _match_event_admin_level(event, sevnt));
    } else {
      return (_match_event_proxy_level(event, sevnt) ||  _match_event_admin_level(event, sevnt));
    }
  }
}

void RDIProxyConsumer::disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_client_and_dispose(remove_from_admin);
}

// If proxy has a connected CosNotify-style supplier, _sc_subscriber will be non-null.
// Attempt to send it a subscription_change msg.
//
// If the message cannot be delivered, or if CORBA::NO_IMPLEMENT or another
// exception is returned, return value TRUE
// (indicating the proxy should not be sent any further offer_change msgs), otherwise FALSE.
//
// XXX On a transient failure, the channel could retry one or more times before
// XXX giving up and returning value TRUE.  At the moment, channel gives up immediately.
//
// XXX ** On hard failure, change _pxstate ??? **
//
CORBA::Boolean RDIProxyConsumer::send_subscription_change(const CosN_EventTypeSeq& added,
							  const CosN_EventTypeSeq& deled) {
  RDI_OPLOCK_ACQUIRE_BUMP(return 1); // debump on method exit

  CORBA::Boolean res = 0; // OK
  if (_sc_off) {
    RDI_DUMP("send_subscription_change called on proxy with _sc_off true");
    res = 1;
  } else if ( (_pxstate == RDI_Connected) && ! CORBA::is_nil(_sc_subscriber) ) {
    // do not hold OPLOCK across subscription_change call
    RDI_OPLOCK_RELEASE;
    try {
      _sc_subscriber->subscription_change(added, deled);
    } catch(CORBA::NO_IMPLEMENT& ex) {
      res = 1; // do not try any more of these
    } catch (...) {
      RDI_DUMP("RDIProxyConsumer -- caught exception other than CORBA::NO_IMPLEMENT " <<
	       "when sending a subscription_change; could retry if this is a soft failure " <<
	       "but for now, giving up immediately");
      res = 1;
    }
    RDI_OPLOCK_REACQUIRE(return 1);
  } else {
    RDI_DUMP("RDIProxyConsumer -- sc_subscriber not available => did not send subscription_change");
    res = 1;
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
  return res;
}

// ------------------------------------------------------------- //
// ProxyPullConsumer_i                                           //
// ------------------------------------------------------------- //

ProxyPullConsumer_i::ProxyPullConsumer_i(SupplierAdmin_i* admin,
					 EventChannel_i*  chann,
					 const CosNA_ProxyID&  prxid) : 
  RDIProxyConsumer(admin, chann, RDI_C_AnyPRX, CosNA_PULL_ANY, prxid),
  _worker(0), _thrdone(0), _timeout_s(0), _timeout_n(0)
{
  _supplier = CosEventComm::PullSupplier::_nil();
  _nc_supplier = CosNC_PullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
    _worker = new AnyPullWorker(this, &ProxyPullConsumer_i::_pull_event);
    RDI_AssertAllocThrowNo(_worker, "Memory allocation failed -- omni_thread");
    _worker->start();
    _thrdone = 0;
    RDI_DUMP("Pull thread for proxy " << (void*)this << " -- " << _worker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void ProxyPullConsumer_i::connect_any_pull_supplier
(CosEventComm::PullSupplier_ptr supplier WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  CosEventComm::PullSupplier_var evsupl;
  if ( CORBA::is_nil(supplier) || (_pxstate != RDI_NotConnected) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate  = RDI_Connected;
  _active   = 1;
  _supplier = CosEventComm::PullSupplier::_duplicate(supplier);
  // If supplier is a CosNC_PullSupplier, set _nc_supplier
  _nc_supplier = CosNC_PullSupplier::_narrow(supplier); // implicit duplicate

  if (! CORBA::is_nil(_nc_supplier)) {
    _sc_subscriber = CosNC_NotifySubscribe::_narrow(_nc_supplier); // implicit duplicate
    if (CORBA::is_nil(_sc_subscriber)) {
      RDI_DUMP("** UNEXPECTED: CosNC_PullSupplier could not be narrowed to CosNC_NotifySubscribe");
    }
    // if enabled, register this proxy for subscription_change msgs
    if ( ! _sc_off && _channel->_schange_pool ) {
      _channel->_schange_pool->insert_proxy(this);
    }
  }

  if ( _worker ) {    // Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }
  if ( _qosprop ) {
    delete _qosprop;
  }
  RDI_OPLOCK_RELEASE;
}

void ProxyPullConsumer_i::suspend_connection(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( ! _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyInactive();
  }
  _active = 0;
  RDI_OPLOCK_RELEASE;
}

void ProxyPullConsumer_i::resume_connection(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyActive();
  }
  _active = 1;
  if ( _worker ) {	// Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }
  RDI_OPLOCK_RELEASE;
}

// is_available determines whether a pull should occur.
// in this case the requirement is that either the pull interval is zero
// or an interval timeout has occurred
CORBA::Boolean ProxyPullConsumer_i::is_available(unsigned long* wait_s, unsigned long* wait_n) {
  RDI_OPLOCK_ACQUIRE(return 0);
  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  if ( (_pxstate != RDI_Connected) || !_active ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }

  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
  if ( (pull_interval_s == 0) && (pull_interval_n == 0) ) {
    _timeout_s = 0; _timeout_n = 0;
    RDI_OPLOCK_RELEASE;
    return 1; // pull as fast as possible
  }

  omni_thread::get_time(&time_s, &time_n);

  if ( (_timeout_s == 0) && (_timeout_n == 0) ) {
    // proxy has not established timeout yet : set it now
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  }
  if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
    RDI_OPLOCK_RELEASE;
    return 1; // interval timeout occurred for this proxy
  }
  // update wait_s/_n to reflect the future timeout point of this proxy
  if ( ( ((*wait_s) == 0) && ((*wait_n) == 0)                   ) ||
       ( ((*wait_s) > _timeout_s)                               ) ||
       ( ((*wait_s) == _timeout_s) && ((*wait_n) > _timeout_n)) ) {
    (*wait_s) = _timeout_s;
    (*wait_n) = _timeout_n;
  }
  RDI_OPLOCK_RELEASE;
  return 0;
}

// REQUIREMENT: A thread from push pool only calls this method
// if is_available() is true and it thinks state is connected and active.
// (For safety we verify that these conditions actually hold.)
void ProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit

  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;
  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);

  CORBA::Boolean hasev = 0;
  CORBA::Any*    event = 0;

  invalid = 0;
  if ( (_pxstate != RDI_Connected) || !_active ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }

  // sanity check: if pull interval is positive, we should only be called if a timeout occurred
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&time_s, &time_n);
    if ((_timeout_s == 0) && (_timeout_n == 0)) { // timeout not set; set it now
      omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
    } 
    if ( (time_s < _timeout_s) || ((time_s == _timeout_s) && (time_n < _timeout_n)) ) {
      RDI_DUMP("** INTERNAL ERROR: ProxyPullConsumer_i::pull_event called too soon");
      RDI_OPLOCK_DEBUMP_RELEASE;
      return;
    }
  }

  // update timeout before releasing OPLOCK -- this means we do the update
  // before doing the try_pull (seems OK)
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }

  // Do not hold OPLOCK across try_pull
  RDI_OPLOCK_RELEASE;

  CORBA::Boolean outcall_worked = 0;
  try {
    event = _supplier->try_pull(hasev);
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Supplier object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Exception while pulling event from supplier");
  }

  RDI_OPLOCK_REACQUIRE(return);

  // add new event to channel even if we have been disconnected in the meantime
  if (outcall_worked && hasev && event) {
    _nevents++; 
    if ( _match_event(*event) ) {
      _channel->new_any_event(*event);
    }
  }
  // new_any_event copies event so can always delete it
  if ( event ) {
    delete event; event = 0;
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during try_pull -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      // unregister this proxy -- no more subscription_change msgs
      if ( ! _sc_off && ! CORBA::is_nil(_nc_supplier) && _channel->_schange_pool ) {
	_channel->_schange_pool->remove_proxy(this);
      }
      _pxstate = RDI_Exception;
      invalid  = 1;
    }
  }

  RDI_OPLOCK_DEBUMP_RELEASE;
}

void ProxyPullConsumer_i::_pull_event()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  CORBA::Boolean hasev = 0, update_timeout = 0, do_yield = 0;
  CORBA::Any*    event = 0;
  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  // invariant: oplock is held at top of loop
  while ( 1 ) {
    do_yield = 1;
    while ( 1 ) {
      // must recompute these here because they can change across a wait/timedwait
      RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
      if (pull_interval_s || pull_interval_n) {
	// if timeout not set, or if update_timeout true due to pull, recompute timeout
	if ( update_timeout || ((_timeout_s == 0) && (_timeout_n == 0)) ) {
	  omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
	}
      } else {
	_timeout_s = 0; _timeout_n = 0;
      }
      update_timeout = 0;
      // breaking from inner loop requires a pull interval timeout (if set)
      // and state must be RDI_Connected and _active
      if ((_pxstate != RDI_NotConnected) && (_pxstate != RDI_Connected)) {
	break; // also break on exceptional state
      }
      if ((_pxstate == RDI_Connected) && _active) {
	if ((_timeout_s == 0) && (_timeout_n == 0)) {
	  break; // pulling as fast as possible so pull now
	}
	omni_thread::get_time(&time_s, &time_n);
	if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
	  break; // pull interval has passed so pull now
	}
	do_yield = 0;
	RDI_OPLOCK_TIMEDWAIT(_timeout_s, _timeout_n); // must wait for pull interval
      } else {
	do_yield = 0;
	RDI_OPLOCK_WAIT; // must wait _pxstate or _active to change
      }
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("pull thread");
    }

    // Do not hold OPLOCK across try_pull
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    CORBA::Boolean outcall_worked = 0;
    try {
      event = _supplier->try_pull(hasev);
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Supplier object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Communication Failure");
    } catch (...) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Exception while pulling event from supplier");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("pull thread [**unexpected REACQUIRE failure**]"));

    update_timeout = 1; // force update of timeout at top of loop

    // add new event to channel even if we have been disconnected in the meantime
    if (outcall_worked && hasev && event) {
      _nevents += 1;
      if ( _match_event(*event) ) {
	_channel->new_any_event(*event);
      }
    }
    // new_any_event copies event so we can delete it regardless
    if ( event ) {
      delete event; event = 0;
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during try_pull -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	// unregister this proxy -- no more subscription_change msgs
	if ( ! _sc_off && ! CORBA::is_nil(_nc_supplier) && _channel->_schange_pool ) {
	  _channel->_schange_pool->remove_proxy(this);
	}
	_pxstate = RDI_Exception;
      } 
    }
  }
}

ostream& operator << (ostream& out, const ProxyPullConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) << 
    " ID " << setw(3) << prx._pserial;
  if ( CORBA::is_nil(prx._nc_supplier) ) out << " CosEventComm Consumer";
  out << prx._pxstate;
  out << (prx._active ? "Active " : "Suspended ");
  return out << " #Pull " << prx._nevents;
}

void ProxyPullConsumer_i::disconnect_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void ProxyPullConsumer_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** ProxyPullConsumer_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  if ( ! _sc_off && (_pxstate == RDI_Connected) && ! CORBA::is_nil(_nc_supplier) && _channel->_schange_pool ) {
    _channel->_schange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding try_pull call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("ProxyPullConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _active  = 0;
  _fa_helper.remove_all_filters();
  _supplier = CosEventComm::PullSupplier::_nil();
  _nc_supplier = CosNC_PullSupplier::_nil();
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// StructuredProxyPullConsumer_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPullConsumer_i::StructuredProxyPullConsumer_i(
							     SupplierAdmin_i* admin,	
							     EventChannel_i*  chann,
							     const CosNA_ProxyID&  prxid) : 
  RDIProxyConsumer(admin, chann, RDI_C_StrPRX, CosNA_PULL_STRUCTURED, prxid),
  _worker(0), _thrdone(0), _timeout_s(0), _timeout_n(0)
{
  _supplier = CosNC_StructuredPullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
    _worker = new StrPullWorker(this, &StructuredProxyPullConsumer_i::_pull_event);
    RDI_AssertAllocThrowNo(_worker, "Memory allocation failed -- omni_thread");
    _worker->start();
    _thrdone = 0;
    RDI_DUMP("Pull thread for proxy " << (void*)this << " -- " << _worker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void StructuredProxyPullConsumer_i::connect_structured_pull_supplier
(CosNC_StructuredPullSupplier_ptr supplier WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) || (_pxstate != RDI_NotConnected) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _supplier = CosNC_StructuredPullSupplier::_duplicate(supplier);
  _sc_subscriber = CosNC_NotifySubscribe::_narrow(_supplier); // implicit duplicate
  if (CORBA::is_nil(_sc_subscriber)) {
    RDI_DUMP("** UNEXPECTED: CosNC_StructuredPullSupplier could not be narrowed to CosNC_NotifySubscribe");
  }
  _pxstate  = RDI_Connected;
  _active   = 1;

  // if enabled, register this proxy for subscription_change msgs
  if ( ! _sc_off && _channel->_schange_pool ) {
    _channel->_schange_pool->insert_proxy(this);
  }

  if ( _worker ) {    // Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }
  if ( _qosprop ) {
    delete _qosprop;
  }
  RDI_OPLOCK_RELEASE;
}

void StructuredProxyPullConsumer_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( ! _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyInactive();
  }
  _active = 0;
  RDI_OPLOCK_RELEASE;
}

void StructuredProxyPullConsumer_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyActive();
  }
  _active = 1;
  if ( _worker ) {	// Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }
  RDI_OPLOCK_RELEASE;
}

// is_available determines whether a pull should occur.
// in this case the requirement is that either the pull interval is zero
// or an interval timeout has occurred
CORBA::Boolean StructuredProxyPullConsumer_i::is_available(unsigned long* wait_s, unsigned long* wait_n) {
  RDI_OPLOCK_ACQUIRE(return 0);
  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  if ( (_pxstate != RDI_Connected) || ! _active ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }

  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
  if ( (pull_interval_s == 0) && (pull_interval_n == 0) ) {
    _timeout_s = 0; _timeout_n = 0;
    RDI_OPLOCK_RELEASE;
    return 1; // pull as fast as possible
  }

  omni_thread::get_time(&time_s, &time_n);

  if ( (_timeout_s == 0) && (_timeout_n == 0) ) {
    // proxy has not established timeout yet : set it now
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  }
  if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
    RDI_OPLOCK_RELEASE;
    return 1; // interval timeout occurred for this proxy
  }
  // update wait_s/_n to reflect the future timeout point of this proxy
  if ( ( ((*wait_s) == 0) && ((*wait_n) == 0)                   ) ||
       ( ((*wait_s) > _timeout_s)                               ) ||
       ( ((*wait_s) == _timeout_s) && ((*wait_n) > _timeout_n)) ) {
    (*wait_s) = _timeout_s;
    (*wait_n) = _timeout_n;
  }
  RDI_OPLOCK_RELEASE;
  return 0;
}

// REQUIREMENT: A thread from push pool only calls this method
// if is_available() is true and it thinks state is connected and active.
// (For safety we verify that these conditions actually hold.)
void StructuredProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit

  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;
  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);

  CORBA::Boolean hasev = 0, out_of_space = 0, outcall_worked = 0, matched = 0;
  CosN_StructuredEvent* event = 0;
  RDI_StructuredEvent* sevnt = 0;

  invalid = 0;
  if ( (_pxstate != RDI_Connected) || ! _active ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }

  // sanity check: if pull interval is positive, we should only be called if a timeout occurred
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&time_s, &time_n);
    if ((_timeout_s == 0) && (_timeout_n == 0)) { // timeout not set; set it now
      omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
    } 
    if ( (time_s < _timeout_s) || ((time_s == _timeout_s) && (time_n < _timeout_n)) ) {
      RDI_DUMP("** INTERNAL ERROR: StructuredProxyPullConsumer_i::pull_event called too soon");
      RDI_OPLOCK_DEBUMP_RELEASE;
      return;
    }
  }

  // update timeout before releasing OPLOCK -- this means we do the update
  // before doing the try_pull (seems OK)
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }

  // Do not hold OPLOCK across try_pull
  RDI_OPLOCK_RELEASE;

  outcall_worked = 0;
  try {
    RDI_DUMP("StructuredProxyPullConsumer_i::pull_event doing try_pull_structured_event");
    event = _supplier->try_pull_structured_event(hasev);
    RDI_DUMP("StructuredProxyPullConsumer_i::RETURN FROM pull_event doing try_pull_structured_event");
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Supplier object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Exception while pulling event from supplier");
  }

  RDI_OPLOCK_REACQUIRE(return);

  // add new event to channel even if we have been disconnected in the meantime
  out_of_space = 0;
  if (outcall_worked && hasev && event) {
    _nevents++; 
    sevnt = new RDI_StructuredEvent(*event);
    RDI_AssertAllocThrowNo(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
    if ( (matched = _match_event(event, sevnt)) ) {
      out_of_space = _channel->new_structured_event(sevnt);
    }
    // Do not delete sevnt unless it was not added to channel
    if (sevnt && (!matched || out_of_space)) {
      delete sevnt;
    }
    sevnt = 0;
  }
  // Creation of sevnt copies event, so delete it regardless
  if ( event ) {
    delete event; event = 0;
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during try_pull -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      // unregister this proxy -- no more subscription_change msgs
      if ( ! _sc_off && _channel->_schange_pool ) {
	_channel->_schange_pool->remove_proxy(this);
      }
      _pxstate = RDI_Exception;
      invalid  = 1;
    }
  }

  RDI_OPLOCK_DEBUMP_RELEASE;
}

void StructuredProxyPullConsumer_i::_pull_event()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  CORBA::Boolean hasev = 0, update_timeout = 0, do_yield = 0;
  CORBA::Boolean outcall_worked = 0, out_of_space = 0, matched = 0;
  CosN_StructuredEvent* event = 0;
  RDI_StructuredEvent* sevnt = 0;
  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  // invariant: oplock is held at top of loop
  while ( 1 ) {
    do_yield = 1;
    while ( 1 ) {
      // must recompute these here because they can change across a wait/timedwait
      RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
      if (pull_interval_s || pull_interval_n) {
	// if timeout not set, or if update_timeout true due to pull, recompute timeout
	if ( update_timeout || ((_timeout_s == 0) && (_timeout_n == 0)) ) {
	  omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
	}
      } else {
	_timeout_s = 0; _timeout_n = 0;
      }
      update_timeout = 0;
      // breaking from inner loop requires a pull interval timeout (if set)
      // and state must be RDI_Connected and _active
      if ((_pxstate != RDI_NotConnected) && (_pxstate != RDI_Connected)) {
	break; // also break on exceptional state
      }
      if ((_pxstate == RDI_Connected) && _active) {
	if ((_timeout_s == 0) && (_timeout_n == 0)) {
	  break; // pulling as fast as possible so pull now
	}
	omni_thread::get_time(&time_s, &time_n);
	if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
	  break; // pull interval has passed so pull now
	}
	do_yield = 0;
	RDI_OPLOCK_TIMEDWAIT(_timeout_s, _timeout_n); // must wait for pull interval
      } else {
	do_yield = 0;
	RDI_OPLOCK_WAIT; // must wait _pxstate or _active to change
      }
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("pull thread");
    }
    // Do not hold OPLOCK across try_pull
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    outcall_worked = 0;
    try {
      event = _supplier->try_pull_structured_event(hasev);
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Supplier object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Communication Failure");
    } catch (...) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Exception while pulling event from supplier");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("pull thread [**unexpected REACQUIRE failure**]"));

    update_timeout = 1; // force update of timeout at top of loop

    // add new event to channel even if we have been disconnected in the meantime
    out_of_space = 0;
    if (outcall_worked && hasev && event) {
      _nevents++; 
      sevnt = new RDI_StructuredEvent(*event);
      RDI_AssertAllocThrowNo(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
      if ( (matched = _match_event(event, sevnt)) ) {
	out_of_space = _channel->new_structured_event(sevnt);
      }
      // Do not delete sevnt unless it was not added to channel
      if (sevnt && (!matched || out_of_space)) {
	delete sevnt;
      }
      sevnt = 0;
    }
    // Creation of sevnt copies event, so delete it regardless
    if ( event ) {
      delete event; event = 0;
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during try_pull -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	// unregister this proxy -- no more subscription_change msgs
	if ( ! _sc_off && _channel->_schange_pool ) {
	  _channel->_schange_pool->remove_proxy(this);
	}
	_pxstate = RDI_Exception;
      }
    }
  }
}

ostream& operator << (ostream& out, const StructuredProxyPullConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) << 
    " ID " << setw(3) << prx._pserial;
  out << prx._pxstate;
  out << (prx._active ? "Active " : "Suspended ");
  return out << " #Pull " << prx._nevents;
}

void StructuredProxyPullConsumer_i::disconnect_structured_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void StructuredProxyPullConsumer_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** StructuredProxyPullConsumer_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more subscription_change msgs
  if ( ! _sc_off && (_pxstate == RDI_Connected) && _channel->_schange_pool ) {
    _channel->_schange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding try_pull call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("StructuredProxyPullConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _active  = 0;
  _fa_helper.remove_all_filters();
  _supplier = CosNC_StructuredPullSupplier::_nil();
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// SequenceProxyPullConsumer_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPullConsumer_i::SequenceProxyPullConsumer_i(
							 SupplierAdmin_i* admin,
							 EventChannel_i*  chann,
							 const CosNA_ProxyID&  prxid) : 
  RDIProxyConsumer(admin, chann, RDI_C_SeqPRX, CosNA_PULL_SEQUENCE, prxid),
  _worker(0), _thrdone(0), _timeout_s(0), _timeout_n(0)
{
  _supplier = CosNC_SequencePullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
    _worker = new SeqPullWorker(this, &SequenceProxyPullConsumer_i::_pull_event);
    RDI_AssertAllocThrowNo(_worker, "Memory allocation failed -- omni_thread");
    _worker->start();
    _thrdone = 0;
    RDI_DUMP("Pull thread for proxy " << (void*)this << " -- " << _worker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void SequenceProxyPullConsumer_i::connect_sequence_pull_supplier
(CosNC_SequencePullSupplier_ptr supplier WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) || (_pxstate != RDI_NotConnected) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _supplier = CosNC_SequencePullSupplier::_duplicate(supplier);
  _sc_subscriber = CosNC_NotifySubscribe::_narrow(_supplier); // implicit duplicate
  if (CORBA::is_nil(_sc_subscriber)) {
    RDI_DUMP("** UNEXPECTED: CosNC_SequencePullSupplier could not be narrowed to CosNC_NotifySubscribe");
  }
  _pxstate  = RDI_Connected;
  _active   = 1;

  // if enabled, register this proxy for subscription_change msgs
  if ( ! _sc_off && _channel->_schange_pool ) {
    _channel->_schange_pool->insert_proxy(this);
  }

  if ( _worker ) {	// Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }

  RDI_OPLOCK_RELEASE;
}

void SequenceProxyPullConsumer_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( ! _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyInactive();
  }
  _active = 0;
  RDI_OPLOCK_RELEASE;
}

void SequenceProxyPullConsumer_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyActive();
  }
  _active = 1;
  if ( _worker ) {    // Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }
  RDI_OPLOCK_RELEASE;
}

// is_available determines whether a pull should occur.
// in this case the requirement is that either the pull interval is zero
// or an interval timeout has occurred
CORBA::Boolean SequenceProxyPullConsumer_i::is_available(unsigned long* wait_s, unsigned long* wait_n) {
  RDI_OPLOCK_ACQUIRE(return 0);
  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  if ( (_pxstate != RDI_Connected) || ! _active ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }

  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
  if ( (pull_interval_s == 0) && (pull_interval_n == 0) ) {
    _timeout_s = 0; _timeout_n = 0;
    RDI_OPLOCK_RELEASE;
    return 1; // pull as fast as possible
  }

  omni_thread::get_time(&time_s, &time_n);

  if ( (_timeout_s == 0) && (_timeout_n == 0) ) {
    // proxy has not established timeout yet : set it now
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  }
  if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
    RDI_OPLOCK_RELEASE;
    return 1; // interval timeout occurred for this proxy
  }
  // update wait_s/_n to reflect the future timeout point of this proxy
  if ( ( ((*wait_s) == 0) && ((*wait_n) == 0)                   ) ||
       ( ((*wait_s) > _timeout_s)                               ) ||
       ( ((*wait_s) == _timeout_s) && ((*wait_n) > _timeout_n)) ) {
    (*wait_s) = _timeout_s;
    (*wait_n) = _timeout_n;
  }
  RDI_OPLOCK_RELEASE;
  return 0;
}

// REQUIREMENT: A thread from push pool only calls this method
// if is_available() is true and it thinks state is connected and active.
// (For safety we verify that these conditions actually hold.)
void SequenceProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method return

  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;
  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);

  CORBA::Boolean hasev = 0, out_of_space = 0, outcall_worked = 0, matched = 0;
  CORBA::ULong   mxnum = 5, ex = 0;
  CosN_EventBatch* events = 0;
  RDI_StructuredEvent* sevnt = 0;

  invalid = 0;
  if ( (_pxstate != RDI_Connected) || ! _active ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }

  // sanity check: if pull interval is positive, we should only be called if a timeout occurred
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&time_s, &time_n);
    if ((_timeout_s == 0) && (_timeout_n == 0)) { // timeout not set; set it now
      omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
    } 
    if ( (time_s < _timeout_s) || ((time_s == _timeout_s) && (time_n < _timeout_n)) ) {
      RDI_DUMP("** INTERNAL ERROR: ProxyPullConsumer_i::pull_event called too soon");
      RDI_OPLOCK_DEBUMP_RELEASE;
      return;
    }
  }

  // update timeout before releasing OPLOCK -- this means we do the update
  // before doing the try_pull (seems OK)
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }

  // Do not hold OPLOCK across try_pull

  RDI_OPLOCK_RELEASE;

  outcall_worked = 0;
  hasev = 0;
  try {
    events = _supplier->try_pull_structured_events(mxnum, hasev);
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Supplier object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Exception while pulling event from supplier");
  }

  RDI_OPLOCK_REACQUIRE(return);

  // add new event to channel even if we have been disconnected in the meantime
  out_of_space = 0;
  if (outcall_worked && hasev && events) {
    _nevents += events->length();
    for (ex = 0; (out_of_space == 0) && (ex < events->length()); ex++) {
      sevnt = new RDI_StructuredEvent((*events)[ex]);
      RDI_AssertAllocThrowNo(sevnt,"Memory allocation failure -- RDI_StructuredEvent");
      if ( (matched = _match_event(&(*events)[ex], sevnt)) ) {
	out_of_space = _channel->new_structured_event(sevnt);
      }
      // Do not delete sevnt unless it was not added to channel
      if (sevnt && (!matched || out_of_space)) {
	delete sevnt;
      }
      sevnt = 0;
    }
  }
  // events can all be deleted because each was either copied or not added
  if ( events ) {
    delete events; events = 0;
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during try_pull -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      // unregister this proxy -- no more subscription_change msgs
      if ( ! _sc_off && _channel->_schange_pool ) {
	_channel->_schange_pool->remove_proxy(this);
      }
      _pxstate = RDI_Exception;
      invalid  = 1;
    }
  }

  RDI_OPLOCK_DEBUMP_RELEASE;
}

void SequenceProxyPullConsumer_i::_pull_event()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  CORBA::Boolean hasev = 0, update_timeout = 0, do_yield = 0, matched = 0;
  CORBA::Boolean outcall_worked = 0, out_of_space = 0;
  CORBA::ULong   mxnum = 5, ex = 0;
  CosN_EventBatch* events = 0;
  RDI_StructuredEvent* sevnt = 0;

  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  // invariant: oplock is held at top of loop
  while ( 1 ) {
    do_yield = 1;
    while ( 1 ) {
      // must recompute these here because they can change across a wait/timedwait
      RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
      if (pull_interval_s || pull_interval_n) {
	// if timeout not set, or if update_timeout true due to pull, recompute timeout
	if ( update_timeout || ((_timeout_s == 0) && (_timeout_n == 0)) ) {
	  omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
	}
      } else {
	_timeout_s = 0; _timeout_n = 0;
      }
      update_timeout = 0;
      // breaking from inner loop requires a pull interval timeout (if set)
      // and state must be RDI_Connected and _active
      if ((_pxstate != RDI_NotConnected) && (_pxstate != RDI_Connected)) {
	break; // also break on exceptional state
      }
      if ((_pxstate == RDI_Connected) && _active) {
	if ((_timeout_s == 0) && (_timeout_n == 0)) {
	  break; // pulling as fast as possible so pull now
	}
	omni_thread::get_time(&time_s, &time_n);
	if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
	  break; // pull interval has passed so pull now
	}
	do_yield = 0;
	RDI_OPLOCK_TIMEDWAIT(_timeout_s, _timeout_n); // must wait for pull interval
      } else {
	do_yield = 0;
	RDI_OPLOCK_WAIT; // must wait _pxstate or _active to change
      }
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("pull thread");
    }
    // Do not hold OPLOCK across try_pull
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    outcall_worked = 0;
    hasev = 0;
    try {
      events = _supplier->try_pull_structured_events(mxnum, hasev);
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Supplier object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Communication Failure");
    } catch (...) {
      RDI_DUMP("Pull thread for proxy " << (void*)this << " - Exception while pulling event from supplier");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("pull thread [**unexpected REACQUIRE failure**]"));

    update_timeout = 1; // force update of timeout at top of loop

    // add new event to channel even if we have been disconnected in the meantime
    out_of_space = 0;
    if (outcall_worked && hasev && events) {
      _nevents += events->length();
      for (ex = 0; (out_of_space == 0) && (ex < events->length()); ex++) {
	sevnt = new RDI_StructuredEvent((*events)[ex]);
	RDI_AssertAllocThrowNo(sevnt,"Memory allocation failure -- RDI_StructuredEvent");
	if ( (matched = _match_event(&(*events)[ex], sevnt)) ) {
	  out_of_space = _channel->new_structured_event(sevnt);
	}
	// Do not delete sevnt unless it was not added to channel
	if (sevnt && (!matched || out_of_space)) {
	  delete sevnt;
	}
	sevnt = 0;
      }
    }
    // events can all be deleted because each was either copied or not added
    if ( events ) {
      delete events; events = 0;
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during try_pull -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	// unregister this proxy -- no more subscription_change msgs
	if ( ! _sc_off && _channel->_schange_pool ) {
	  _channel->_schange_pool->remove_proxy(this);
	}
	_pxstate = RDI_Exception;
      }
    }
  }
}

ostream& operator << (ostream& out, const SequenceProxyPullConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
    " ID " << setw(3) << prx._pserial;
  out << prx._pxstate;
  out << (prx._active ? "Active " : "Suspended ");
  return out << " #Pull " << prx._nevents;
}

void SequenceProxyPullConsumer_i::disconnect_sequence_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void SequenceProxyPullConsumer_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** SequenceProxyPullConsumer_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more subscription_change msgs
  if ( ! _sc_off && (_pxstate == RDI_Connected) && _channel->_schange_pool ) {
    _channel->_schange_pool->remove_proxy(this);
  }

  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding try_pull call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("SequenceProxyPullConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _active  = 0;
  _fa_helper.remove_all_filters();
  _supplier = CosNC_SequencePullSupplier::_nil();
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// ProxyPushConsumer_i                                           //
// ------------------------------------------------------------- //

ProxyPushConsumer_i::ProxyPushConsumer_i(SupplierAdmin_i* admin,
					 EventChannel_i*  evchn,
					 const CosNA_ProxyID&  prxid) : 
  RDIProxyConsumer(admin, evchn, RDI_C_AnyPRX, CosNA_PUSH_ANY, prxid)
{
  _supplier = CosEventComm::PushSupplier::_nil();
  _nc_supplier = CosNC_PushSupplier::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

void ProxyPushConsumer_i::push(const CORBA::Any& event WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  _nevents += 1;
  CORBA::Boolean matched = _match_event(event);
  if ( matched ) {
    if (_channel->new_any_event(event) ) { // event not added due to limits
      RDI_OPLOCK_RELEASE;
      throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
    }
  }
  RDI_OPLOCK_RELEASE;
}

void ProxyPushConsumer_i::connect_any_push_supplier
(CosEventComm::PushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( (_pxstate != RDI_NotConnected) || ! CORBA::is_nil(_supplier) ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate = RDI_Connected;
  _active  = 1;
  _supplier = CosEventComm::PushSupplier::_duplicate(supplier);
  // If supplier is a CosNC_PushSupplier, set _nc_supplier
  _nc_supplier = CosNC_PushSupplier::_narrow(supplier); // implicit duplicate

  if (! CORBA::is_nil(_nc_supplier)) {
    _sc_subscriber = CosNC_NotifySubscribe::_narrow(_nc_supplier); // implicit duplicate
    if (CORBA::is_nil(_sc_subscriber)) {
      RDI_DUMP("** UNEXPECTED: CosNC_PushSupplier could not be narrowed to CosNC_NotifySubscribe");
    }
    // if enabled, register this proxy for subscription_change msgs
    if ( ! _sc_off && _channel->_schange_pool ) {
      _channel->_schange_pool->insert_proxy(this);
    }
  }

  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const ProxyPushConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype);
  if ( CORBA::is_nil(prx._nc_supplier ) ) out << " CosEventComm Supplier";
  out << prx._pxstate;
  return out << " #Events " << prx._nevents;
}

void ProxyPushConsumer_i::disconnect_push_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void ProxyPushConsumer_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** ProxyPushConsumer_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more subscription_change msgs
  if ( ! _sc_off && (_pxstate == RDI_Connected) && ! CORBA::is_nil(_nc_supplier) && _channel->_schange_pool ) {
    _channel->_schange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // This proxy does not need to worry about inuse count
  // because it is never bumped above zero.
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("ProxyPushConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _fa_helper.remove_all_filters();
  _supplier = CosEventComm::PushSupplier::_nil();
  _nc_supplier = CosNC_PushSupplier::_nil();
  if ( _qosprop ) 
    delete _qosprop;
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// StructuredProxyPushConsumer_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPushConsumer_i::StructuredProxyPushConsumer_i(
							     SupplierAdmin_i* admin, 	
							     EventChannel_i*  evchn,
							     const CosNA_ProxyID&  prxid) :
  RDIProxyConsumer(admin, evchn, RDI_C_StrPRX, CosNA_PUSH_STRUCTURED, prxid)
{
  _supplier = CosNC_StructuredPushSupplier::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

void StructuredProxyPushConsumer_i::push_structured_event
(const CosN_StructuredEvent& event   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  CORBA::Boolean matched = 0, out_of_space = 0; 
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  _nevents += 1;
  RDI_StructuredEvent* sevnt = new RDI_StructuredEvent(event);
  RDI_AssertAllocThrowNo(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
  if ( (matched = _match_event(&event, sevnt)) ) {
    out_of_space = _channel->new_structured_event(sevnt);
  }
  // Do not delete sevnt unless it was not added to channel
  if (sevnt && (!matched || out_of_space)) {
    delete sevnt;
  }
  sevnt = 0;
  RDI_OPLOCK_RELEASE;
  if ( out_of_space ) {
    throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  }
}

void StructuredProxyPushConsumer_i::connect_structured_push_supplier
(CosNC_StructuredPushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate = RDI_Connected;
  _active  = 1;
  _supplier= CosNC_StructuredPushSupplier::_duplicate(supplier);
  _sc_subscriber = CosNC_NotifySubscribe::_narrow(_supplier); // implicit duplicate
  if (CORBA::is_nil(_sc_subscriber)) {
    RDI_DUMP("** UNEXPECTED: CosNC_StructuredPushSupplier could not be narrowed to CosNC_NotifySubscribe");
  }
  // if enabled, register this proxy for subscription_change msgs
  if ( ! _sc_off && _channel->_schange_pool ) {
    _channel->_schange_pool->insert_proxy(this);
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const StructuredProxyPushConsumer_i& prx)
{
  out << &prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
    " ID " << setw(3) << prx._pserial;
  out << prx._pxstate;
  return out << " #Events " << prx._nevents;
}

void StructuredProxyPushConsumer_i::disconnect_structured_push_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void StructuredProxyPushConsumer_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** StructuredProxyPushConsumer_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more subscription_change msgs
  if ( ! _sc_off && (_pxstate == RDI_Connected) && _channel->_schange_pool ) {
    _channel->_schange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // This proxy does not need to worry about inuse count
  // because it is never bumped above zero.
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("StructuredProxyPushConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _fa_helper.remove_all_filters();
  _supplier = CosNC_StructuredPushSupplier::_nil();
  if ( _qosprop )
    delete _qosprop;
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// SequenceProxyPushConsumer_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPushConsumer_i::SequenceProxyPushConsumer_i(
							 SupplierAdmin_i* admin,
							 EventChannel_i*  evchn,
							 const CosNA_ProxyID&  prxid) :
  RDIProxyConsumer(admin, evchn, RDI_C_SeqPRX, CosNA_PUSH_SEQUENCE, prxid)
{
  _supplier = CosNC_SequencePushSupplier::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

void 
SequenceProxyPushConsumer_i::push_structured_events
(const CosN_EventBatch& events WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  CORBA::ULong ex = 0;
  CORBA::Boolean matched = 0, out_of_space = 0;
  RDI_StructuredEvent* sevnt = 0;
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  _nevents += events.length();
  out_of_space = 0;
  for (ex = 0; (out_of_space == 0) && (ex < events.length()); ex++) {
    sevnt = new RDI_StructuredEvent(events[ex]);
    RDI_AssertAllocThrowNo(sevnt,"Memory allocation failure -- RDI_StructuredEvent");
    if ( (matched = _match_event(&events[ex], sevnt)) ) {
      out_of_space = _channel->new_structured_event(sevnt);
    }
    // Do not delete sevnt unless it was not added to channel
    if (sevnt && (!matched || out_of_space)) {
      delete sevnt;
    }
    sevnt = 0;
  }
  RDI_OPLOCK_RELEASE;
  if ( out_of_space ) {
    throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  }
}

void SequenceProxyPushConsumer_i::connect_sequence_push_supplier
(CosNC_SequencePushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate = RDI_Connected;
  _active  = 1;
  _supplier = CosNC_SequencePushSupplier::_duplicate(supplier);
  _sc_subscriber = CosNC_NotifySubscribe::_narrow(_supplier); // implicit duplicate
  if (CORBA::is_nil(_sc_subscriber)) {
    RDI_DUMP("** UNEXPECTED: CosNC_SequencePushSupplier could not be narrowed to CosNC_NotifySubscribe");
  }
  // if enabled, register this proxy for subscription_change msgs
  if ( ! _sc_off && _channel->_schange_pool ) {
    _channel->_schange_pool->insert_proxy(this);
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const SequenceProxyPushConsumer_i& prx)
{
  out << & prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
    " ID " << setw(3) << prx._pserial;
  out << prx._pxstate;
  return out << " #Events " << prx._nevents;
}

void SequenceProxyPushConsumer_i::disconnect_sequence_push_consumer( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void SequenceProxyPushConsumer_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** SequenceProxyPushConsumer_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more subscription_change msgs
  if ( ! _sc_off && (_pxstate == RDI_Connected) && _channel->_schange_pool ) {
    _channel->_schange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // This proxy does not need to worry about inuse count
  // because it is never bumped above zero.
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("SequenceProxyPushConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _fa_helper.remove_all_filters();
  _supplier = CosNC_SequencePushSupplier::_nil();
  if ( _qosprop )
    delete _qosprop;
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

