// -*- Mode: C++; -*-
//                              File      : RDIDebug.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIDebug.h,v $
Revision 1.16  2000/11/15 21:18:17  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.15  2000/10/30 04:39:15  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.14  2000/08/22 18:23:49  alcfp
added description to each file

Revision 1.13  2000/08/16 20:19:06  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _RDI_DEBUG_H_
#define _RDI_DEBUG_H_

#include <stdio.h>
#include <iostream.h>
#include <strstream.h>
#include "RDIUtil.h"

/** ERROR LOGGING
  * The destination of the logged information depends on the value
  * of the 'medium'. The supported alternatives are listed below:
  *
  *    ether  : no error logging is performed
  *    stderr : errors sent to standard error
  *    syslog : errors sent to syslog facility
  *    fname  : errors sent to the given file
  *
  *    atsvclog : [only applies for RUX -- use the atsvc log for both at svc and ready]
  * 
  * The 'format' argument passed to the  'log' function is used in
  * the same way as the  'format' argument in the system functions
  * fprintf(), printf() and sprintf().
  *
  * The 'level_t' is used in two ways: for tagging each record and
  * for setting the priority. The supported tags/priorities are:
  *
  *    Fatal    : Non-recoverable condition
  *    Critical : Critical conditions
  *    Error    : Error conditions
  *    Warning  : Various warning messages
  *    Notice   : Special handling is required (not an error)
  *    Info     : Informational messages
  *    Debug    : Debbuging information
  *
  * NOTE: When 'log()' is called, logging is done iff '_active' is
  * true and 'level' <= '_llevel'
  */

class RDI_ErrorLog {
public:
  enum level_t  { Fatal, Critical, Error, Warning, Notice, Info, Debug };
  enum medium_t { Ether, Stderr, Syslog, File, AtSvcLog };

  RDI_ErrorLog(const char* indentity,
	       level_t level=Warning, const char* medium="stderr");
  virtual ~RDI_ErrorLog()  	{ close(); if (_elname) delete [] _elname; }

  void    close();
  void    log(level_t level, const char *format, ...);
  void    flush();

  void    set_active(CORBA::Boolean turnon=1)	{ _active=turnon; }
  CORBA::Boolean get_active(void) const	{ return _active; }

  void    set_level(level_t level)	{ _llevel=level;  }
  level_t get_level() const 		{ return _llevel; }

  ostrstream& operator *()		{ return _stream; }
  ostrstream& strstream()		{ return _stream; }

protected:
  CORBA::Boolean  _active;
  medium_t        _medium;
  level_t         _llevel;
  FILE*           _elfile;
  char*           _elname;
  ostrstream      _stream;

  RDI_ErrorLog();
  RDI_ErrorLog(const RDI_ErrorLog&);
};

/** APPLICATION DEBUGGING
  *
  * A specialization of the error log facility used for debugging.
  * Debugging can be enabled/disabled for several file names: list
  * of files separated by white spaces, or commas, is the argument
  * to enable() and disable().
  */

class RDI_Debug : public RDI_ErrorLog {
public:
  RDI_Debug(const char* indentity,
	    const char* list_of_files=0, const char* medium="stderr");
  virtual ~RDI_Debug();

  void enable(const char* list_of_files);
  void enable_all_files(void)			{ _all_files=1;  }
  void disable(const char* list_of_files);
  void disable_all_files(void)			{ _all_files=0; }
  CORBA::Boolean is_enabled(const char* file_name) const;
  // Get up to 'max_entries' of the files that are currently
  // in the list of files for which debugging is enabled.
  // NOTE: You should NOT delete any of the returned names..
  void current_files(char** file_list, unsigned int max_entries) const;

  static void init_default_debug(const char* list_of_files=0, 
			         const char* medium="stderr");
  static void delete_default_debug();
  static RDI_Debug* default_debug()	{ return _def_dbg; }

  static omni_mutex _temp_mutex;

private:
  char**          _file_name;
  unsigned int    _num_files;
  CORBA::Boolean  _all_files;

  static RDI_Debug* _def_dbg;

  RDI_Debug();
  RDI_Debug(const RDI_Debug&);
};

/** DEBUGGING MACROS
  * Various debugging macros  to make our life easier.  All macros
  * use the '_def_dbg' object for debugging. If this object is not
  * initialized, it is initialized to the "stderr".  The format of
  * the 'str' argument in RDI_DUMP() / RDI_FDUMP() follows the C++
  * stream output interface:
  *
  *     RDI_DUMP("Age " << age << " Name " << str);
  *
  * RDI_ENABLE(fname)    : enable debugging output for fname
  * RDI_DISABLE(fname)   : disable debugging output for fname
  * RDI_DUMP(str)        : output 'str' if __FILE__ is enabled
  * RDI_FDUMP(str)       : output 'str' independently of __FILE__
  * RDI_FUNC_ENTER(func) : indicates function 'func' entered
  * RDI_FUNC_RETURN      : most recent entered 'func' is returning
  *
  * RDI_DBG_CLEANUP      : close/dealloc the _def_dbg object
  *
  * All macros but __RDI_ADUMP__ variants turn into "no-ops" when NDEBUG is
  * defined.
  */

#ifndef MacroArg2String
#  define MacroArg2String(s) #s
#endif

#ifdef RDI_LONG_DBG_FILENAMES
#define __RDI_FNAME_SHORTEN(file) file
#else
// shorten /foo/bar/zot/baz.cc  to  zot/baz.cc  (leave one level of subdir context)
inline char* __RDI_FNAME_SHORTEN(char* fname) {
  char* endptr = fname + strlen(fname) - 1;
  while ((endptr > fname) && (*(--endptr) != '/')) {;} // finds last /, if any
  while ((endptr > fname) && (*(--endptr) != '/')) {;} // finds next-to-last /, if any
  if (endptr > fname) return (endptr+1);
  return fname;
}
#endif

#define __RDI_NULL_STMT 	if (0) { }

#define __RDI_ADUMP_2__(doflush, str)  do { RDI_Debug::_temp_mutex.lock(); \
	ostrstream& stream = RDI_Debug::default_debug()->strstream(); \
	stream << str; \
	if (doflush) RDI_Debug::default_debug()->flush(); RDI_Debug::_temp_mutex.unlock(); } while (0)
#define __RDI_ADUMP_4__(doflush, file, line, str)  do { RDI_Debug::_temp_mutex.lock(); \
	ostrstream& stream = RDI_Debug::default_debug()->strstream(); \
	stream << __RDI_FNAME_SHORTEN((char*)file) << ":" << line << " " << str; \
	if (doflush) RDI_Debug::default_debug()->flush(); RDI_Debug::_temp_mutex.unlock(); } while (0)
#define __RDI_ADUMP__(doflush, str) __RDI_ADUMP_4__(doflush, __FILE__, __LINE__, str)

#define __RDI_DBG_WRAP(stmt) do { \
	if ( ! RDI_Debug::default_debug() ) \
		RDI_Debug::init_default_debug(); \
	stmt; } while (0)

#define __RDI_ADUMP_2_WRAPPED(doflush, str) \
	__RDI_DBG_WRAP( __RDI_ADUMP_2__(doflush, str) )

#define __RDI_ADUMP_WRAPPED(doflush, str) \
	__RDI_DBG_WRAP( __RDI_ADUMP__(doflush, str) )

#define RDI_DBG_CLEANUP RDI_Debug::delete_default_debug()

#ifndef NDEBUG
#  define RDI_DBG_ENABLE(fname)	\
	__RDI_DBG_WRAP( RDI_Debug::default_debug()->enable(fname) )
#  define RDI_DBG_DISABLE(fname) \
	__RDI_DBG_WRAP( RDI_Debug::default_debug()->disable(fname) )
#  define RDI_DBG_IS_ENABLED(fname) \
	( RDI_Debug::default_debug() && \
	  RDI_Debug::default_debug()->is_enabled(fname) )
#  define RDI_DBG_ENABLE_ALL \
	__RDI_DBG_WRAP( RDI_Debug::default_debug()->enable_all_files() )
#  define RDI_DBG_DISABLE_ALL \
	__RDI_DBG_WRAP( RDI_Debug::default_debug()->disable_all_files() )
#  define RDI_FDUMP(str) \
	__RDI_DBG_WRAP(__RDI_ADUMP__(1, "[FDUMP] " << str))
#  define RDI_DUMP(str) __RDI_DBG_WRAP( \
	if ( RDI_Debug::default_debug()->is_enabled(__FILE__) == 1 ) \
		__RDI_ADUMP__(1, str) )
#  define RDI_FUNC_ENTER(func) \
	char* __func__ = MacroArg2String(func); \
	RDI_DUMP("Entering funct: " << __func__);
#  define RDI_FUNC_RETURN \
	RDI_DUMP("Returning from: " << __func__);
#else
#  define RDI_DBG_ENABLE(fname) 	__RDI_NULL_STMT
#  define RDI_DBG_DISABLE(fname)	__RDI_NULL_STMT
#  define RDI_DBG_IS_ENABLED(fname)	__RDI_NULL_STMT
#  define RDI_DBG_ENABLE_ALL   		__RDI_NULL_STMT
#  define RDI_DBG_DISABLE_ALL  		__RDI_NULL_STMT
#  define RDI_FDUMP(str)		__RDI_NULL_STMT
#  define RDI_DUMP(str)			__RDI_NULL_STMT
#  define RDI_FUNC_ENTER(func)		__RDI_NULL_STMT
#  define RDI_FUNC_RETURN		__RDI_NULL_STMT
#endif

/** ADDITINAL MACROS
  * RDI_DBG_PARAM(x)
  *	Wrap a parameter that is going to be used only when NDEBUG
  *	is not defined (just a trick to avoid compiler warnings)
  * RDI_MULTI_PARAM(x)
  *	Like the previous macro  but used for parameters which are
  *	required when MULTITHREADED is defined
  * RDI_Fatal(str)
  *	Fatal error:  the message is logged and program is aborted
  * RDI_Assert(x, str)
  *	If 'x' is false,  'str' gets logged and program is aborted
  * RDI_AssertAllocThrowMaybe(x, str)
  *	If 'x' is NULL,  'str' gets logged and
  *       CORBA::NO_MEMORY(0, CORBA::COMPLETED_MAYBE) is thrown
  * RDI_AssertAllocThrowNo(x, str)
  *	If 'x' is NULL,  'str' gets logged and
  *       CORBA::NO_MEMORY(0, CORBA::COMPLETED_NO) is thrown
  */

#ifndef NDEBUG
#  define RDI_DBG_PARAM(x) x
#else
#  define RDI_DBG_PARAM(x)
#endif

#ifdef MULTITHREADED
#  define RDI_MULTI_PARAM(x) x
#else
#  define RDI_MULTI_PARAM(x)
#endif

#define RDI_Fatal(str) \
	__RDI_DBG_WRAP( __RDI_ADUMP__(1, str); \
	RDI_Debug::default_debug()->log(RDI_ErrorLog::Fatal, "fatal error") )

#ifndef NDEBUG
#  define RDI_Assert(x, str) \
	if ( ! (x) ) \
		RDI_Fatal("assert failed " << MacroArg2String(x) << " " << str)
#else
#  define RDI_Assert(x, str) 		__RDI_NULL_STMT
#endif

#define RDI_AssertAllocThrowMaybe(x, str) \
    if ( ! (x) ) { \
       RDI_DUMP(str); \
       throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_MAYBE); }

#define RDI_AssertAllocThrowNo(x, str) \
    if ( ! (x) ) { \
       RDI_DUMP(str); \
       throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_NO); }

#endif
