// -*- Mode: C++; -*-
//                              File      : CosNotifyFilter_i.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    set of C++ definitions for the CosNotifyFilter module
//
 
/*
$Log: CosNotifyFilter_i.h,v $
Revision 1.41  2000/11/15 21:18:17  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.40  2000/10/30 05:41:33  alcfp
renamed CosNotify.h to CosNotifyShorthands.h and placed CosNotifyShorthands.h in omniNotify/include rather than src/services/include

Revision 1.39  2000/10/30 04:39:12  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.38  2000/08/22 18:23:48  alcfp
added description to each file

Revision 1.37  2000/08/16 20:19:03  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _COS_NOTIFY_FILTER_I_H_
#define _COS_NOTIFY_FILTER_I_H_

#include "corba_wrappers.h"

#include "RDIHash.h"
#include "RDIEvent.h"
#include "RDIStaticEvalDefs.h"
#include "RDIDynamicEvalDefs.h"
#include "CosNotifyShorthands.h"

class EventChannel_i;
class Filter_i;


/** RDINotifySubscribe
  *
  * The default 'subscription_change' method does not include filter
  * information and, therefore,  cannot be used in the internal hash
  * tables.  The following class offers a propagate_subscription_change
  * that includes a reference to the filter object whose event types
  * have changed.
  *
  * In addition, when destroy() is called on a given filter, we must
  * inform all objects that reference the filter to cleanup internal
  * structures.
  */

class RDINotifySubscribe {
public:
  virtual void propagate_subscription_change(const CosN_EventTypeSeq& added,
					     const CosN_EventTypeSeq& deled,
					     Filter_i*                filter) = 0;
  virtual void filter_destroy_i(Filter_i* filter) = 0;
};

typedef RDINotifySubscribe* RDINotifySubscribe_ptr;

/** ConstraintImpl
  *
  * For each constraint added to a filter, _constraints[i], there is
  * a corresponding implementation-object _constraint_impls[i], that
  * maintains a node with the parsed filter,  against which matching
  * is performed at runtime. 
  */

class ConstraintImpl {
public:
  ConstraintImpl() : just_types(0), node(0)    {;}
  ~ConstraintImpl() 	{ if (node) delete node; node=0; }

  // Create the internal parse tree and any other state which is
  // required for the given constraint expression. In case of an
  // error or invalid constraint expression, NULL is returned...

  static ConstraintImpl* create(const CosNF_ConstraintExp& constraint);

  ConstraintImpl& operator = (const ConstraintImpl& ci) 
	{ just_types=ci.just_types; node=ci.node; return *this; }

  CORBA::Boolean just_types;
  RDI_PCState*   node;
};

typedef _CORBA_Unbounded_Sequence<ConstraintImpl *> ConstraintImplSeq;

/** Filter_i
  *
  * This class implements the OMG CosNF_Filter interface.
  * A filter has a set of constraints, each having a unique id. Each
  * constraint consists of two parts: an 'event type sequence' and a
  * 'constraint expression'.  The syntax is as follows:
  *
  *  { [{domain, type},{domain, type},...], "constraint expression"}
  *
  * The types in the event type sequence are implicitly OR'd. An '*'
  * in the domain or type name denotes a wildcard,  matching zero or
  * more characters that position.  An empty sequence  is equivalent 
  * to a sequence with the element ["*", "*"].
  *
  * The constraints in a filter are implicitly OR'd. An empty string
  * in constraint expression is considered the same as "true".
  */

class Filter_i : 
	WRAPPED_SKELETON_SUPER(CosNF::, Filter) {
public:
  Filter_i(const char* grammar="EXTENDED_TCL");
  virtual ~Filter_i();

  void  destroy( WRAPPED_DECLARG_VOID );
  char* constraint_grammar( WRAPPED_DECLARG_VOID )
    			{ return CORBA::string_dup(_constraint_grammar); }

  CosNF_ConstraintInfoSeq* add_constraints(
			  const CosNF_ConstraintExpSeq& constraints
			  WRAPPED_DECLARG );
  void modify_constraints(const CosNF_ConstraintIDSeq&   del_list,
			  const CosNF_ConstraintInfoSeq& mod_list
			  WRAPPED_DECLARG );
  CosNF_ConstraintInfoSeq* get_constraints(const CosNF_ConstraintIDSeq& id_list
                                           WRAPPED_DECLARG );
  CosNF_ConstraintInfoSeq* get_all_constraints( WRAPPED_DECLARG_VOID );
  void remove_all_constraints( WRAPPED_DECLARG_VOID );

  CORBA::Boolean match(const CORBA::Any& event WRAPPED_DECLARG );
  CORBA::Boolean match_structured(const CosN_StructuredEvent& event 
                                  WRAPPED_DECLARG );
  CORBA::Boolean match_typed(const CosN_PropertySeq& event WRAPPED_DECLARG );

  CosNF_CallbackID attach_callback(CosNC_NotifySubscribe_ptr callback
                                   WRAPPED_DECLARG );
  void detach_callback(CosNF_CallbackID callback WRAPPED_DECLARG );
  CosNF_CallbackIDSeq* get_callbacks( WRAPPED_DECLARG_VOID );


  // The following methods are not available to Notification clients

  void create_ev_types_from_dom_list(CosN_EventTypeSeq& lst_seq);
  void notify_subscribers_i(const CosN_EventTypeSeq& add_seq, 
			    const CosN_EventTypeSeq& del_seq);

  CORBA::Boolean rdi_match(RDI_StructuredEvent* evnt, EventChannel_i* chan=0);

  CosNF_CallbackID attach_callback_i(RDINotifySubscribe_ptr callback_i);
  void detach_callback_i(CosNF_CallbackID  callbackID);

  CORBA::Boolean  has_callbacks() 	{ return _callbacks.length()   ? 1:0; }
  CORBA::Boolean  has_callbacks_i()	{ return _callbacks_i.length() ? 1:0; }
  unsigned int    getID() const		{ return _fid; }

  // If 'f' is a local filter object with impl type Filter_i,
  // return a valid Filter_i* pointer; otherwise return NULL

  static Filter_i* Filter2Filter_i(CosNF_Filter_ptr f);

  friend ostream& operator << (ostream& out, const Filter_i& fltr);

  static Filter_i* null_filter;
private:
  CORBA::Boolean _exists_constraint(const CosNF_ConstraintID& cstid,
				    CORBA::ULong& position) const;
  void 		 _update_ev_tables(const CosNF_ConstraintExp& cexpr,
			 	    CosN_EventTypeSeq& add_types, 
				    CosN_EventTypeSeq& del_types);
  void           _remove_constraint(const CosNF_ConstraintID& cstid, 
				    CosN_EventTypeSeq& add_types, 
				    CosN_EventTypeSeq& rem_types);
  CORBA::Boolean _event_is_dominated(const CosN_EventType& t1);
  void           _add_ev_type(CosN_EventTypeSeq& tsq, const RDI_EventType& etp);

public:
  struct RDIFilterKeyMapEntry {
    	Filter_i*             iref;
	RDIFilterKeyMapEntry* next;
 	RDIFilterKeyMapEntry(Filter_i* fltr=0) : iref(fltr), next(0) {;}
  };
  typedef RDI_Hash<unsigned long, RDIFilterKeyMapEntry*> RDIFilterKeyMap;
private:
#ifndef NDEBUG
  CORBA::Boolean           _dbgout;
#endif
  omni_mutex               _oplock;
  unsigned int             _fid;
  long                     _idcounter;
  unsigned long            _hashvalue;
  char*                    _constraint_grammar;
  CosNF_ConstraintInfoSeq* _constraints;
  ConstraintImplSeq*       _constraint_impls;
  CosNF_CallbackID         _callback_serial;
  CosNF_CallbackID         _callback_i_serial;
  RDI_Hash<CosNF_CallbackID, CosNC_NotifySubscribe_ptr> _callbacks;
  RDI_Hash<CosNF_CallbackID, RDINotifySubscribe_ptr>    _callbacks_i;
  RDI_Hash<RDI_EventType, void *>       _flt_dom_ev_types; 
  RDI_Hash<RDI_EventType, CORBA::ULong> _flt_all_ev_types;

  static omni_mutex       _classlock;
  static unsigned int     _classctr;
  static RDIFilterKeyMap* _class_keymap;
};


/** MappingFilter_i
  *
  * Implementation of the  CosNF_MappingFilter interface.
  * NOTE: this class is not supported yet
  */

class MappingFilter_i : 
		WRAPPED_SKELETON_SUPER(CosNF::, MappingFilter) 
{
public:
  MappingFilter_i(const char* grammar, const CORBA::Any& def_value);
  virtual ~MappingFilter_i();

  void  destroy( WRAPPED_DECLARG_VOID );
  char* constraint_grammar( WRAPPED_DECLARG_VOID )
    			{ return CORBA::string_dup(_constraint_grammar); }
  CORBA::TypeCode_ptr value_type( WRAPPED_DECLARG_VOID );
  CORBA::Any*         default_value( WRAPPED_DECLARG_VOID );

  CosNF_MappingConstraintInfoSeq* add_mapping_constraints(
				const CosNF_MappingConstraintPairSeq& pair_list
			        WRAPPED_DECLARG );
  void modify_mapping_constraints(
				const CosNF_ConstraintIDSeq& del_list,
				const CosNF_MappingConstraintInfoSeq& mod_list
				WRAPPED_DECLARG );
  CosNF_MappingConstraintInfoSeq* get_mapping_constraints(
				const CosNF_ConstraintIDSeq& id_list
		                WRAPPED_DECLARG ); 
  CosNF_MappingConstraintInfoSeq* get_all_mapping_constraints(
		                WRAPPED_DECLARG_VOID );
  void remove_all_mapping_constraints( WRAPPED_DECLARG_VOID );

  CORBA::Boolean match(const CORBA::Any & event,
		       WRAPPED_OUTARG_TYPE(CORBA::Any) result_to_set
                       WRAPPED_DECLARG );
  CORBA::Boolean match_structured(const CosN_StructuredEvent & event,
		       WRAPPED_OUTARG_TYPE(CORBA::Any) result_to_set
                       WRAPPED_DECLARG );
  CORBA::Boolean match_typed(const CosN_PropertySeq & event,
		       WRAPPED_OUTARG_TYPE(CORBA::Any) result_to_set
                       WRAPPED_DECLARG );
private:
  omni_mutex    _oplock;
  char*         _constraint_grammar;
  CORBA::Any    _def_value;
 CosNF_ConstraintID _constraint_serial;
  CosNF_MappingConstraintInfoSeq* _constraints;
};


/** FilterFactory_i
  *
  * This class implements the  OMG CosNF_FilterFactory 
  * interface.  A filter factory is used to create filter objects
  * for a given constraint language. Each filter factory might be 
  * able to support multiple constraint languages.  However, each
  * one of them MUST support the default constraint language that 
  * is specified in the OMG Notification Service Specification.
  */

class FilterFactory_i : 
		WRAPPED_SKELETON_SUPER(CosNF::, FilterFactory) 
{
public:
  FilterFactory_i(const char* grammar="EXTENDED_TCL",EventChannel_i* channel=0);
  virtual ~FilterFactory_i();

  CosNF_Filter_ptr        create_filter(const char* grammar WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr create_mapping_filter(const char* grammar,
                                                const CORBA::Any& value
                                                WRAPPED_DECLARG );

  // The following methods are not available to Notification clients

  int            add_grammar(const char* grammar);
  void           del_grammar(const char* grammar);
  CORBA::Boolean is_supported(const char* grammar);
private:
  enum { MAXGR = 5 };

  omni_mutex      _oplock;         	// Serialize concurrent operations
  EventChannel_i* _channel;     	// NULL when factory is stand alone
  char*           _clangs[MAXGR];  	// The list of supported grammars
  unsigned int    _nlangs;		// Number of registered grammars
};


/** FAdminHelper
  *
  * An administrative object is responsible for managing filter objects.
  * It implements the methods  from the OMG CosNF_FilterAdmin
  * interface except for add_filter(), which is overriden in all classes
  * that use this helper class.
  */

class FAdminHelper {
public:
  FAdminHelper();
  ~FAdminHelper();

  // Methods from CosNF_FilterAdmin Interface

  void           remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG );
  CosNF_Filter_ptr    get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG );
  CosNF_FilterIDSeq*  get_all_filters( WRAPPED_DECLARG_VOID );
  void           remove_all_filters( WRAPPED_DECLARG_VOID );

  // omniNotify-specific methods

  CosNF_FilterID add_filter_i(CosNF_Filter_ptr fltr, 
			      RDINotifySubscribe_ptr holder);
  CosNF_FilterID add_filter_i(CosNF_Filter_ptr fltr);
  void           rem_filter_i(Filter_i*   fltr);
  CORBA::Boolean has_filters() const { return _filters.length() ? 1 : 0; }
private:
  omni_mutex       _oplock;
  CosNF_FilterID   _serial;
  CosNF_CallbackID _callbackID; 
  RDI_Hash<CosNF_FilterID, CosNF_CallbackID> _rdcbids;
  RDI_Hash<CosNF_FilterID, Filter_i*> _filters;

  void cleanup_filter(CosNF_FilterID fltrID);
};

#endif
