// -*- Mode: C++; -*-
//                              File      : RDILog.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDILog.h,v $
Revision 1.5  2000/11/15 21:18:17  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.4  2000/10/30 04:39:23  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.3  2000/08/22 18:23:50  alcfp
added description to each file

Revision 1.2  2000/08/16 20:19:18  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _RDI_LOG_H_
#define _RDI_LOG_H_

#include <iostream.h>
#include <sys/param.h>
#include <sys/types.h>
#include <time.h>
#ifndef MAXPATHLEN
#  ifdef PATH_MAX
#    define MAXPATHLEN PATH_MAX
#  else 
#    define MAXPATHLEN 1024
#  endif
#endif

#if defined (__OMNIORB3__)
#include <omniORB3/CORBA.h>
#elif defined (__OMNIORB2__)
#include <omniORB2/CORBA.h>
#elif defined (__ORBIX3__)
#include "CORBA.h"
#endif

#include "omnithread.h"

class RDI_LogHdr;
class RDI_LogAnchor;
class RDI_Log;
class RDI_LogIter;


// -------------------- Log Sequence Number -------------------- //
//                                                               //
// Log record LSN values correspond to the physical locations of //
// these records.  Since log files may be reused, records stored //
// at the same offset in a log file must have different LSNs. In //
// addition, when several log files are used, the LSN value of a //
// log record should contain the 'id' of the file containing the //
// record. The '_wrap' LSN member addresses these problems.      //
// ------------------------------------------------------------- //

class RDI_LSN {
public:
  RDI_LSN() : _offs(0), _wrap(0) {;}
  RDI_LSN(const RDI_LSN& l) : _offs(l._offs), _wrap(l._wrap) {;}
  RDI_LSN(off_t offs, unsigned long wrap) : _offs(offs), _wrap(wrap) {;}

  RDI_LSN& operator =  (const RDI_LSN& l)
		{ _offs=l._offs; _wrap=l._wrap; return *this; }
  RDI_LSN& operator += (off_t offs)
		{ _offs+=offs; return *this; }
  int operator == (const RDI_LSN& l) const
		{ return ((_wrap==l._wrap) && (_offs==l._offs)) ? 1 : 0; }
  int operator != (const RDI_LSN& l) const
		{ return ((_wrap!=l._wrap) || (_offs!=l._offs)) ? 1 : 0; }
  int operator >  (const RDI_LSN& l) const
		{ return ((_wrap > l._wrap) || 
		          ((_wrap==l._wrap) && (_offs > l._offs))) ? 1 : 0; }
  int operator >= (const RDI_LSN& l) const
		{ return ((*this == l) || (*this > l)) ? 1 : 0; }
  int operator <  (const RDI_LSN& l) const
		{ return ((_wrap < l._wrap) || 
		  	  ((_wrap == l._wrap) && (_offs < l._offs))) ? 1 : 0; }
  int operator <= (const RDI_LSN& l) const
		{ return ((*this == l) || (*this < l)) ? 1 : 0; }

  void          set_offset(off_t offs)		{ _offs = offs; }
  void          set_wrapno(unsigned long wrap)	{ _wrap = wrap; }

  off_t         get_offset(void) const		{ return _offs; }
  unsigned long get_wrapno(void) const		{ return _wrap; }

  friend ostream& operator << (ostream& out, const RDI_LSN& l)
		{ return out << l._offs << "." << l._wrap; }

  static const RDI_LSN null;
private:
  off_t         _offs;
  unsigned long _wrap;
};


// --------------------- Log Record Header --------------------- //
//                                                               //
// Log records consist of fixed size headers and a variable size //
// bodies. The header is created by the log manager and contains //
// information needed for scanning the log file(s) and detecting //
// data corruption. The '_magic' member is used for implementing //
// a simple corruption detection scheme. For the moment, it only //
// includes a 'magic number'. In the furure, the checksum of the //
// body will be used.                                            //
// ------------------------------------------------------------- //

#define RDI_LREC_MAGIC	0xB38F0E32

enum RDI_LREC_Kind  { RDI_LREC_USRAPP,	/* Application log record */
		      RDI_LREC_LOGSYS, 	/* Log system  log record */
		      RDI_LREC_CHCK_S,	/* Start of  a checkpoint */
		      RDI_LREC_CHCK_F 	/* Finish of a checkpoint */
		    };

class RDI_LRecHdr {
public:
  RDI_LRecHdr(RDI_LREC_Kind  kind=RDI_LREC_USRAPP, 
	      unsigned long  csum=RDI_LREC_MAGIC, 
	      const RDI_LSN& plsn=RDI_LSN::null, size_t size=0);
  RDI_LRecHdr(const RDI_LRecHdr& v) : _kind(v._kind), _csum(v._csum), 
	      _time(v._time), _plsn(v._plsn), _size(v._size) {;}
  RDI_LRecHdr& operator = (const RDI_LRecHdr& v) {
		_kind=v._kind; _csum=v._csum; _time=v._time;
		_plsn=v._plsn; _size=v._size; return *this; }
  friend ostream& operator << (ostream& out, const RDI_LRecHdr& v); 
public:
  RDI_LREC_Kind _kind;	// Kind of this record
  unsigned long _csum;	// Checksum of payload
  time_t        _time;	// Timestamp of record
  RDI_LSN       _plsn;	// Previous record LSN
  size_t        _size;	// Size of the payload
};


// ---------------------- Log File Anchor ---------------------- //
//                                                               //
// The log file anchor maintains important information about the //
// status of the log files and the log records they contain. Log //
// versions are four 8-byte numbers (i.e., a long).  The current //
// version is 1.0.0.0, which corresponds to 16777216L            //
// ------------------------------------------------------------- //

#define RDI_OneKiloByte		1024L
#define RDI_OneMegaByte	     1048576L
#define RDI_OneGigaByte	  1073741824L

#define RDI_CLogVersion	    16777216L
#define RDI_ReadyString	  "omniNotify Notification Log"

class RDI_LogAnchor {
	friend class RDI_Log;
	friend class RDI_LogIter;
public:
  RDI_LogAnchor(CORBA::Boolean reuse=0, size_t size=RDI_OneMegaByte);
  RDI_LogAnchor(const RDI_LogAnchor& a);

  RDI_LogAnchor& operator = (const RDI_LogAnchor& a);

  unsigned long  lversion() const	{ return _lversion; }
  CORBA::Boolean circular() const	{ return _circular; }
  size_t         log_size() const    	{ return _log_size; }
  unsigned long  wrap_cnt() const	{ return _wrap_cnt; }
  const RDI_LSN& frst_lsn() const	{ return _frst_lsn; }
  const RDI_LSN& need_lsn() const	{ return _need_lsn; }
  const RDI_LSN& stbl_lsn() const	{ return _stbl_lsn; }
  const RDI_LSN& ckpt_lsn() const	{ return _ckpt_lsn; }

  friend ostream& operator << (ostream& out, const RDI_LogAnchor& a);
private:
  char          _readystr[64];
  unsigned long _lversion;	// The version number of this log
  CORBA::Boolean _circular;	// If set,  overwrite old records
  unsigned long _wrap_cnt;	// File index when many log files
  size_t        _log_size;	// The maximum size of a log file
  RDI_LSN       _frst_lsn;	// LSN of oldest known log record
  RDI_LSN       _need_lsn;	// LSN of oldest  required record
  RDI_LSN       _stbl_lsn;	// LSN of last on-disk log record
  RDI_LSN       _ckpt_lsn;	// LSN of last *full*  checkpoint

  void clear()	{ _circular=0; _wrap_cnt=0; _log_size=0;
		  _frst_lsn=_need_lsn=_stbl_lsn=_ckpt_lsn=RDI_LSN::null; }
};


// ------------------- Memory Buffer Manager ------------------- //
//                                                               //
// The tail of the log is maintained in main memory.  This class //
// is responsible for managing this memory buffer.               //
// ------------------------------------------------------------- //

class RDI_MemoryLog {
public:
  RDI_MemoryLog(size_t size);
  ~RDI_MemoryLog();

  void append(const void* user_data, size_t data_size, 
	      RDI_LREC_Kind kind, const RDI_LSN& plsn);
  void clear()	{ _offs = 0; }

  const char* data_ptr()   const	{ return _data; }
  size_t      total_space() const	{ return _size; }
  size_t      used_space() const	{ return _offs; }
  size_t      free_space() const	{ return (_size - _offs); }

  friend ostream& operator << (ostream& out, const RDI_MemoryLog& mlog);
private:
  size_t _size;
  size_t _offs;
  char*  _buff;
  char*  _data;
};


// ------------------------ Log Manager ------------------------ //
//                                                               //
// Log records can be stored in memory or on disk. In the latter //
// case, a memory buffer is also used  for temporary storage  of //
// records placed in the log file.  Multiple logs may be open at //
// the same time by different system components.                 //
// ------------------------------------------------------------- //

class RDI_Log {
	friend class RDI_LogIter;
public:
  ~RDI_Log();

  // Does the given path correspond to a valid omniNotify log? If yes,
  // return 1. Otherwise, return 0

  static int valid_log(const char* logpath);

  // Create a new log. If 'logpath' is a directory, a log file is
  // created in the directory. If 'reuse' is 0, a new log file is
  // created when the current one becomes full and 'logpath' is a
  // directory. If 'reuse' is not 0, the same file is overwritten
  // when it becomes full.  If 'format' is not 0, the log file(s)
  // is(are) formatted at creation.  Finally, 'memsize' refers to
  // the size of the memory buffer used to store the log tail

  static RDI_Log* create(const char*    logpath="./_rdi_log_",
			 size_t         logsize=RDI_OneMegaByte,
			 CORBA::Boolean reuse=0,
			 CORBA::Boolean format=0,
			 size_t         memsize=32*RDI_OneKiloByte);

  // Open a log file. If 'logpath' is a directory, the latest log
  // file in the directory is opened. If 'format' is not 0, newly
  // created log file(s) is(are) formatted at creation time.  The
  // value of 'memsize' is used for the memory buffer that stores
  // the log tail

  static RDI_Log* open(const char*    logpath="./_rdi_log_", 
		       CORBA::Boolean format=0,
		       size_t         memsize=32*RDI_OneKiloByte);

  void close(CORBA::Boolean flushlog=1);
  int  flush();

  // Append a log record.  The LSN of the record is returned when
  // the operation succeeds. RDI_LSN::null is returned on failure

  RDI_LSN append(const void* data, size_t size);

  // Typically, checkpoints are application specific.  Therefore,
  // we need to know when checkpoints start and end. In addition,
  // we need to know the LSN of the oldest log record that may be
  // needed by the application; this allows us to reuse log space
  // or delete log files that are not needed anymore. If this LSN
  // is NULL, we assume that all log records are needed

  RDI_LSN begin_checkpoint();
  RDI_LSN finish_checkpoint(const RDI_LSN& olsn=RDI_LSN::null);

  // Some utility functions that might be needed by applications
  // that want to scan the log *** no lock is acquired here ***

  RDI_LSN first_lsn() const             { return _lanchor._frst_lsn; }
  RDI_LSN needed_lsn() const            { return _lanchor._need_lsn; }
  RDI_LSN stable_lsn() const            { return _lanchor._stbl_lsn; }
  RDI_LSN checkpoint_lsn() const        { return _lanchor._ckpt_lsn; }
  RDI_LSN last_lsn() const              { return _lastlsn; }

  friend ostream& operator << (ostream& out, const RDI_Log& l);

private:
  char           _logpath[MAXPATHLEN];	// Log file directory  or log file
  CORBA::Boolean  _rawdisk;		// If != 0, log stored on raw disk
  CORBA::Boolean  _dirpath;		// If != 0, directory path is used
  CORBA::Boolean  _fformat;		// If != 0, log file  is formatted
  CORBA::Boolean  _cactive;		// If != 0, checkpoint in progress
  CORBA::Boolean  _invalid;		// If != 0, log state is not valid
  int            _filedes;		// The file descriptor for the log
  RDI_LogAnchor  _lanchor;		// Anchor for  disk-based  file(s)
  RDI_MemoryLog* _logtail;		// Memory resident tail of the log
  unsigned long  _numrecs;		// #of log records  written so far
  unsigned long  _numchck;		// #of complete checkpoints so far
  RDI_LSN        _memflsn;		// Lsn of first record in _mbuffer
  RDI_LSN        _lastlsn;		// Lsn of last  log record written
  RDI_LSN        _nextlsn;		// Lsn assigned to next log record
  RDI_LSN        _scheckp;		// Lsn of  RDI_LREC_CHCK_S  record
  omni_mutex     _lock;

  RDI_Log();
  RDI_Log(const RDI_Log&);

  int     flush_tail(CORBA::Boolean force_anchor=0);
  int     establish_state();
  int     read_anchor();
  int     write_anchor();
  int     valid_lsn(const RDI_LSN& lsn) const;
  int     allocate_space(size_t nbytes);
  RDI_LSN append(const void* data, size_t size, RDI_LREC_Kind kind);
  void    cleanup();

  static int new_file(const char* name, size_t size=0, 
		      CORBA::Boolean format=0, CORBA::Boolean israw=0);
};


// ----------------------- Log Iterator ------------------------ //
//                                                               //
// Log files are scanned using a log iterator.  Random, forward, //
// and backward scannings are supported.                         //
// ------------------------------------------------------------- //

class RDI_LogIter {
public:
  RDI_LogIter(const RDI_Log& log, CORBA::Boolean forward=1);
  ~RDI_LogIter();

  // Fetch the log record with LSN equal to 'lsn'. The record header 
  // is always fetched. If 'data' is provided, up to 'size' bytes of
  // the record's payload are stored into 'data'

  int fetch(const RDI_LSN& lsn, RDI_LRecHdr& hdr, void* data=0, size_t size=0);

  // Set the scan pointer to the log record with LSN equal to 'lsn'.
  // It returns 0 on success

  int seek(const RDI_LSN& lsn);

  // Change the scan direction of the log iterator

  void change2forward() 		{ _forward = 1;  }
  void change2backward()		{ _forward = 0; }

  // Fetch the next log record, if any, based on the scan direction.
  // If 'data' is provided, up to 'size' bytes are stored in 'data'.
  // The LSN of the record is returned on success.  RDI_LSN::null is
  // returned on failure

  RDI_LSN next(RDI_LRecHdr& hdr, void* data=0, size_t size=0);

private:
  const RDI_Log&  _logmngr;
  int             _filedes;
  unsigned long   _filenum;
  RDI_LSN         _nextlsn;
  CORBA::Boolean  _forward;

  RDI_LogIter();
  RDI_LogIter(const RDI_LogIter&);

  int open_file(unsigned long filenum);
};

#endif
