// -*- Mode: C++; -*-
//                              File      : RDITime.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of RDI_TimeValue [Time-related functionality]
//
 
/*
$Log: RDITime.cc,v $
Revision 1.7  2000/10/30 04:42:02  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.6  2000/08/22 18:23:57  alcfp
added description to each file

Revision 1.5  2000/08/16 20:20:10  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <new.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <iomanip.h>
#include "RDITime.h"

// --------------------------------------------------------------- //
// In multithreaded environments, the APIs of some system-provided //
// time management routines differ.  The following inline function //
// definitions handle this case.                                   //
// --------------------------------------------------------------- //

#ifdef MULTITHREADED
inline struct tm* _RDI_LocalTime(const time_t *clock, struct tm *result)
{ return localtime_r(clock, result); }

inline char* _RDI_CTime(const time_t *clock, char *buf)
{ return ctime_r(clock, buf); }

inline time_t _RDI_MkTime(struct tm *timeptr)
{ return mktime(timeptr); }
#else
inline struct tm* _RDI_LocalTime(const time_t *clock, struct tm *result)
{ struct tm *temp = localtime(clock);
  if ( (result != (struct tm *) 0) && (temp != (struct tm *) 0) ) 
	*result = *temp; 
  return temp;
}

inline char* _RDI_CTime(const time_t *clock, char *buf)
{ char *temp = ctime(clock);
  if ( (temp != (char *) 0) && (buf != (char *) 0) )
	strcpy(buf, temp);
  return temp;
}

inline time_t _RDI_MkTime(struct tm *timeptr)
{ return mktime(timeptr); }
#endif

// -------------------------------------------------------------- //
// Utility functions and consants that are going to be used below //
// -------------------------------------------------------------- //

static const unsigned int DaysInMonth[]= {31,28,31,30,31,30,31,31,30,31,30,31};

static inline char * _RDI_SkipWS(const char *string)
{ 
  char* ptr = (char *) string;
  while ( (ptr!=(char *) 0) && (*ptr!='\0') && ((*ptr==' ') || (*ptr=='\t')) )
	ptr += 1;
  return ((ptr == (char *) 0) || (*ptr == '\0')) ? 0 : ptr;
}

static inline int _RDI_LastDay(int year, unsigned int month)
{
  int daynum = DaysInMonth[ month % 12 ]; 
  if ( (month % 12) == 1 ) { 			// February --- leap year?
	if ( (year % 2000) == 0 ) {
		daynum = 29;
	} else if ( (year % 4) == 0 ) {
		daynum = 29;
		if ( (year % 100) == 0 && (year % 400) != 0 )
			daynum = 28;
	}
  }
  return daynum;
}

// -------------------------------------------------------------- //

// initialize static const class member
const RDI_TimeValue RDI_TimeValue::null = RDI_TimeValue();

RDI_TimeValue RDI_TimeValue::delta(const RDI_TimeValue& lt, 
				   const RDI_TimeValue& rt)
{
  RDI_TimeValue diff;
  if ( lt._tms.tv_sec > rt._tms.tv_sec ) {
	diff._tms.tv_sec  = lt._tms.tv_sec - rt._tms.tv_sec;
	if ( lt._tms.tv_usec > rt._tms.tv_usec )
		diff._tms.tv_usec = lt._tms.tv_usec - rt._tms.tv_usec;
	else {
		diff._tms.tv_sec -= 1;
		diff._tms.tv_usec = lt._tms.tv_usec+1000000-rt._tms.tv_usec;
	}
  } else if ( lt._tms.tv_sec < rt._tms.tv_sec ) {
	diff._tms.tv_sec  = rt._tms.tv_sec - lt._tms.tv_sec;
	if ( lt._tms.tv_usec < rt._tms.tv_usec ) 
		diff._tms.tv_usec = rt._tms.tv_usec - lt._tms.tv_usec;
	else {
		diff._tms.tv_sec -= 1;
		diff._tms.tv_usec = rt._tms.tv_usec+1000000-lt._tms.tv_usec; 
	}
  } else {
	diff._tms.tv_sec  = 0;
	diff._tms.tv_usec = (lt._tms.tv_usec > rt._tms.tv_usec) ? 
					(lt._tms.tv_usec - rt._tms.tv_usec) :
                                        (rt._tms.tv_usec - lt._tms.tv_usec);
  }
  return diff;
}

//////////////////////////////

RDI_TimeValue RDI_TimeValue::make_time(const char* tspec, CORBA::Boolean nextv, 
				       char* ebuff, RDI_TimeValue base)
{
  struct timeval tmval;
  RDI_TimeValue  currt;
  struct tm      tmnow;
  struct tm      tmfut;
  time_t         tsecs;
  int            hours=0, mins=0, secds=0, msecs=0;
  char           tsbuf[256];
  char*          parse = (char *) tsbuf;
  char*          delim = 0;
  CORBA::Boolean atnoon=0, atnow=0, atmidnight=0, lastday=0;
 
  if ( (tspec == (const char *) 0) || (strlen(tspec) > 256) ) {
	if ( ebuff != (char *) 0 )
		sprintf(ebuff, "NULL or too long value for time_spec");
	return null;
  }
  strcpy(tsbuf, tspec);

  // Convert all upper case characters to lower case

  for (unsigned int i=0; i < strlen(tsbuf); i++) {
	if ( isalpha(tsbuf[i]) && isupper(tsbuf[i]) )
		tsbuf[i] = tolower(tsbuf[i]);
  }

  // Construct current time information -- current clock or 'base'

  if ( base == RDI_TimeValue::null )
    	(void) gettimeofday(&tmval, 0);
  else 
	tmval = base;
  (void) _RDI_LocalTime(&tmval.tv_sec, &tmnow);
  currt = tmval;

  tmfut.tm_mday = tmnow.tm_mday;
  tmfut.tm_mon  = tmnow.tm_mon;
  tmfut.tm_year = tmnow.tm_year;
 
  // Skip any white spaces present in the specification

  if ( (parse = _RDI_SkipWS(parse)) == (char *) 0 ) {
	if ( ebuff != (char *) 0 )
		sprintf(ebuff, "White space only specification");
	return null;
  }

  // Parse the time_info part of the time specification

  if ( strncasecmp(parse, "midnight", 8) == 0 ) {
	tmfut.tm_hour =   23;
	tmfut.tm_min  =   59;
	tmfut.tm_sec  =   59;
	msecs         =  999;
	atmidnight    =    1;
	parse += 8;
  } else if ( strncasecmp(parse, "noon", 4) == 0 ) {
	tmfut.tm_hour =   12;
	tmfut.tm_min  =    0;
	tmfut.tm_sec  =    0;
	msecs         =    0;
	atnoon        =    1;
	parse += 4;
  } else if ( strncasecmp(parse, "now", 3) == 0 ) {
	tmfut.tm_hour = tmnow.tm_hour;
	tmfut.tm_min  = tmnow.tm_min;
	tmfut.tm_sec  = tmnow.tm_sec;
	msecs         = 0;
	atnow         = 1;
	parse += 3;
  } else {
	// HH:MM:SS:mm [am|pm] format is expected here

	if ( ! (parse = parse_time(parse, hours, mins, secds, msecs, ebuff)) )
		return null;

	secds += msecs / 1000;
	msecs  = msecs % 1000;
	mins  += secds / 60;
	secds  = secds % 60;
	hours += mins / 60;
	mins   = mins % 60;

	tmfut.tm_sec  = secds;
	tmfut.tm_min  = mins;
	tmfut.tm_hour = hours;
  }

  tmfut.tm_isdst = -1;

  // Skip any white spaces present in the specification and parse
  // the 'date_info' part, if it was provided

  if ( (parse = _RDI_SkipWS(parse)) == (char *) 0 ) {
	tmfut.tm_mon  = tmnow.tm_mon;
	tmfut.tm_mday = tmnow.tm_mday;
	tmfut.tm_year = tmnow.tm_year;
  } else if ( (delim = strchr(parse, '/')) != (char *) 0 ) {  	// mm
	*delim = '\0';
	tmfut.tm_mon = (int) strtoul(parse, 0, 10) - 1;

	parse  = delim + 1;
	if ( (delim = strchr(parse, '/')) == (char *) 0 ) {	// dd
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Invalid mm/dd/yyyy format %s", tspec);
		return null;
	}
	*delim = '\0';
	tmfut.tm_mday = (int) strtoul(parse, 0, 10);

	delim += 1;
	parse  = delim;
	while ( (*delim != '\0') && (*delim != ' ') && (*delim != '\t') )
		delim += 1;
	*delim = '\0';
	tmfut.tm_year = (int) strtoul(parse, 0, 10) - 1900;
  } else {
  	if ( strncasecmp(parse, "today", 5) == 0 ) {
		tmfut.tm_mday = tmnow.tm_mday;
		parse += 5;
  	} else if ( strncasecmp(parse, "tomorrow", 8) == 0 ) {
		tmfut.tm_mday = tmnow.tm_mday + 1;
		parse   += 8;
  	} else if ( strncasecmp(parse, "lastday", 7) == 0 ) {
		lastday = 1;		// Need to compute last day
		parse  += 7;
  	} else {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Invalid 'date_info' format %s", tspec);
		return null;
	}

	while ( (*parse != '\0') && ((*parse == ' ') || (*parse == '\t')) )
        	parse += 1;

	if ( *parse != '\0' ) {		// Check if month is given
		if ( strncasecmp(parse, "jan", 3) == 0 )      tmfut.tm_mon =  0;
        	else if ( strncasecmp(parse, "feb", 3) == 0 ) tmfut.tm_mon =  1;
        	else if ( strncasecmp(parse, "mar", 3) == 0 ) tmfut.tm_mon =  2;
        	else if ( strncasecmp(parse, "apr", 3) == 0 ) tmfut.tm_mon =  3;
        	else if ( strncasecmp(parse, "may", 3) == 0 ) tmfut.tm_mon =  4;
        	else if ( strncasecmp(parse, "jun", 3) == 0 ) tmfut.tm_mon =  5;
        	else if ( strncasecmp(parse, "jul", 3) == 0 ) tmfut.tm_mon =  6;
        	else if ( strncasecmp(parse, "aug", 3) == 0 ) tmfut.tm_mon =  7;
        	else if ( strncasecmp(parse, "sep", 3) == 0 ) tmfut.tm_mon =  8;
        	else if ( strncasecmp(parse, "oct", 3) == 0 ) tmfut.tm_mon =  9;
       	 	else if ( strncasecmp(parse, "nov", 3) == 0 ) tmfut.tm_mon = 10;
        	else if ( strncasecmp(parse, "dec", 3) == 0 ) tmfut.tm_mon = 11;

		// Skip month information first and then any white spaces

		while ((*parse != '\0') && (*parse != ' ') && (*parse != '\t'))
			parse += 1;
		while ( (*parse != '\0') && ((*parse==' ') || (*parse=='\t')) )
                        parse += 1;

		// Check if year is given

		if ( *parse != '\0' ) {
			if ( strncasecmp(parse, "next", 4) == 0 ) {
				tmfut.tm_year = tmnow.tm_year + 1;
			} else {
				tmfut.tm_year = (int) strtoul(parse, 0, 10);
				tmfut.tm_year = tmfut.tm_year - 1900;
			}
		}
	}

	if ( lastday == 1 ) 
		tmfut.tm_mday = _RDI_LastDay(tmfut.tm_year+1900, tmfut.tm_mon);
  }

  // Check if we need to perform any changes to the existing values.

  if ( nextv == 1 ) {
     if ( tmfut.tm_year <= tmnow.tm_year ) {
	tmfut.tm_year = tmnow.tm_year;		// At least current year
	if ( tmfut.tm_mon < tmnow.tm_mon ) {	
	   tmfut.tm_year += 1;			// Same month, next year
	   if ( lastday == 1 )
	      tmfut.tm_mday = _RDI_LastDay(tmfut.tm_year+1900, tmfut.tm_mon);
	} else if ( tmfut.tm_mon == tmnow.tm_mon ) {
	   if ( tmfut.tm_mday <= tmnow.tm_mday ) {
	      tmfut.tm_mday = tmnow.tm_mday;	// At least current day
	      if ( (tmfut.tm_hour < tmnow.tm_hour) ||
	           (tmfut.tm_min  < tmnow.tm_min)  ||
	           (tmfut.tm_sec  < tmnow.tm_sec) ) {
	         if ( atnoon == 1 || atnow == 1 || atmidnight == 1 ) {
		    if ( lastday == 1 ) {
		       tmfut.tm_mon  += 1;
		       tmfut.tm_mday = _RDI_LastDay(tmfut.tm_year+1900, tmfut.tm_mon);
		    } else 
		       tmfut.tm_mday += 1;
		 } else {
		    if ( tmfut.tm_hour < tmnow.tm_hour )
		       tmfut.tm_hour = tmnow.tm_hour;
		    if ( tmfut.tm_min < tmnow.tm_min )
		       tmfut.tm_min = tmnow.tm_min;
		    if ( tmfut.tm_sec < tmnow.tm_sec )
		       tmfut.tm_sec = tmnow.tm_sec;
		 }
	      }
	   }
	}
     }
  }
		
  // At this point we are ready to compute the absolute time.

  if ( (tsecs = _RDI_MkTime(&tmfut)) == (time_t) -1 ) {
	if ( ebuff != (char *) 0 )
		sprintf(ebuff, "Failed to create time for %s", tspec);
	return null;
  }

  currt = RDI_TimeValue(tsecs, (long) msecs*1000);
  return currt;
} 

//////////////////////////////

int RDI_TimeValue::increment(RDI_TimeValue& tval,const char* ispec, char* ebuff)
{
  int   years= 0, months=0, weeks=0, days=0, hours=0, mins=0, secs=0, msecs=0;
  int   curno= 0, negat=0, index=0;
  char* parse= (char *) 0;
  char  tbuff[128];
  struct tm tmval;
  time_t    tsecs;
 
  if ( (parse = _RDI_SkipWS(ispec)) == (char *) 0 ) {
	if ( ebuff != (char *) 0 )
		sprintf(ebuff, "NULL value was provided for incr_spec");
	return -1;
  }

  while ( (parse != (char *) 0) && (*parse != '\0') ) {
	// First, search for a sign, if any is provided  -- default is +

	index = negat = curno = 0;

	if ( *parse == '+' ) {
		negat  = 0;
		parse += 1;
	} else if ( *parse == '-' ) {
		negat  = 1;
		parse += 1;
	}

	// Second, search for a number. If none, report an error  

	if ( ((parse=_RDI_SkipWS(parse)) == (char *) 0) || ! isdigit(*parse) ) {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Number was not found");
		return -1;
	}

	while ( (*parse != '\0') && isdigit(*parse) && (index < 128) )
		tbuff[index++] = *parse++;
	if ( index >= 128 ) {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Too many digits for a number");
		return -1;
	} else if ( *parse == '\0' ) {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Expected time unit after number");
		return -1;
	}

	tbuff[index] = '\0';
	curno = (int) strtoul(tbuff, 0, 10);	

	// Finally, search for the time info string

	if ( ((parse=_RDI_SkipWS(parse)) == (char *) 0) || ! isalpha(*parse) ) {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Expected time unit name in %s", parse);
		return -1;
	}

	index = 0;
	while ( (*parse != '\0') && isalpha(*parse) && (index < 128) ) {
		tbuff[index++] = tolower( *parse );
		parse += 1;
	}
	if ( index >= 128 ) {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Too many characters for a time unit");
		return -1;
	} 
	tbuff[index] = '\0';

	if ( strcasecmp(tbuff, "year") == 0 || 
	     strcasecmp(tbuff, "years") == 0 ) {
		years += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "month") == 0 || 
		    strcasecmp(tbuff, "months") == 0 ) {
		months += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "week") == 0 || 
		    strcasecmp(tbuff, "weeks") == 0 ) {
		weeks += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "day") == 0 || 
		    strcasecmp(tbuff, "days") == 0 ) {
		days += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "hour") == 0 || 
		    strcasecmp(tbuff, "hours") == 0 ) {
		hours += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "min") == 0 || 
		    strcasecmp(tbuff, "minute") == 0 ||
		    strcasecmp(tbuff, "mins") == 0 || 
		    strcasecmp(tbuff, "minutes") == 0 ) {
		mins += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "sec") == 0 || 
		    strcasecmp(tbuff, "second") == 0 ||
		    strcasecmp(tbuff, "secs") == 0 || 
		    strcasecmp(tbuff, "seconds") == 0 ) {
		secs += negat ? -1*curno : curno;
  	} else if ( strcasecmp(tbuff, "msec") == 0 || 
		    strcasecmp(tbuff, "msecs") == 0 ||
		    strcasecmp(tbuff, "msecond") == 0     ||
		    strcasecmp(tbuff, "mseconds") == 0    ||
		    strcasecmp(tbuff, "millisec") == 0    ||
		    strcasecmp(tbuff, "millisecs") == 0   ||
		    strcasecmp(tbuff, "millisecond") == 0 ||
		    strcasecmp(tbuff, "milliseconds") == 0 ) {
		msecs += negat ? -1*curno : curno;
	} else {
		if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Invalid time unit: %s", tbuff);
		return -1;
	}

	parse = _RDI_SkipWS(parse);
  }

  tsecs = tval.secds();

  (void) _RDI_LocalTime(&tsecs, &tmval);
  
  secs  += msecs / 1000;
  msecs  = msecs % 1000;
  mins  += secs / 60;
  secs   = secs % 60;
  hours += mins / 60;
  mins   = mins % 60;

  tmval.tm_sec  += secs;
  tmval.tm_min  += mins;
  tmval.tm_hour += hours + (weeks*7 + days) * 24;
  tmval.tm_mon  += months;
  tmval.tm_year += years;
  tmval.tm_isdst = -1;

  if ( (tsecs = _RDI_MkTime(&tmval)) == (time_t) -1 ) {
	if ( ebuff != (char *) 0 )
		sprintf(ebuff, "Failed to increment absolute time");
	return -1;
  } 

  tval = RDI_TimeValue(tsecs, msecs);
  return 0;
}

//////////////////////////////

RDI_TimeValue RDI_TimeValue::make_period(const char* pspec)
{
  RDI_TimeValue tval;
  int   hours=0, mins=0, secs=0, msecs=0;
  
  if ( parse_time(pspec, hours, mins, secs, msecs, 0) == (char *) 0 )
	return null;
  secs  = secs + msecs/1000 + mins*60 + hours*360;
  msecs = msecs % 1000;
  tval  = RDI_TimeValue(secs, msecs);
  return tval;
}

//////////////////////////////

char * RDI_TimeValue::parse_time(const char* tinfo, int& H, 
			         int& M, int& S, int& m, char* ebuff)
{
  int   index = 0;
  char* parse = 0;
  char  buff[32];

  H = M = S = m = 0;

  if ( ((parse = _RDI_SkipWS(tinfo)) == (char *) 0) || ! isdigit(*parse) ) {
	if ( ebuff != (char *) 0 )
		sprintf(ebuff, "Expected number for HH");
	return 0;
  }

  index = 0;  						// Extract HH
  while ( (*parse != '\0') && isdigit(*parse) )
	buff[index++] = *parse++;
  buff[index] = '\0';
  H = (int) strtoul(buff, 0, 10);

  if ( *parse == ':' ) {				// Extract MM
	index  = 0;
	parse += 1;

	if ( ! isdigit(*parse) ) {
	   	if ( ebuff != (char *) 0 )
			sprintf(ebuff, "Expected number for MM");
		return 0;
	}

	while ( (*parse != '\0') && isdigit(*parse) )
		buff[index++] = *parse++;
	buff[index] = '\0';
	M = (int) strtoul(buff, 0, 10);

	if ( *parse == ':' ) {				// Extract SS
		index  = 0;
		parse += 1;

		if ( ! isdigit(*parse) ) {
			if ( ebuff != (char *) 0 )
				sprintf(ebuff, "Expected number for SS");
			return 0;
		}

		while ( (*parse != '\0') && isdigit(*parse) )
			buff[index++] = *parse++;
		buff[index] = '\0';
		S = (int) strtoul(buff, 0, 10);

		if ( *parse == ':' ) {			// Extract mm
			index  = 0;
			parse += 1;

			if ( ! isdigit(*parse) ) {
				if ( ebuff != (char *) 0 )
					sprintf(ebuff,"Expected number for mm");
				return 0;
			}

			while ( (*parse != '\0') && isdigit(*parse) )
				buff[index++] = *parse++;
			buff[index] = '\0';
			m = (int) strtoul(buff, 0, 10);
		}
	}
  }

  // Check is am or pm is specified and adjust H, if needed 

  while ( (*parse != '\0') && ((*parse == ' ') || (*parse == '\t')) )
	parse += 1;

  if ( *parse == '\0' || ((*parse != 'a') && (*parse != 'p')) )
	return parse;

  parse += 1;
  if ( (*parse == '\0') || (*parse != 'm') ) {	// Shortcut for am | pm
	if ( *(parse - 1) == 'a' ) {
		if ( H >= 12 ) {
			if ( ebuff != (char *) 0 )
				sprintf(ebuff, "am was used with HH >= 12");
			return 0;
		}
	} else {
		if ( H >= 12 ) {
			if ( ebuff != (char *) 0 )
				sprintf(ebuff, "pm was used with HH >= 12");
			return 0;
		} 
		H += 12;
	}
  } else {
	if ( *(parse - 1) == 'a' ) {
		if ( H >= 12 ) {
			if ( ebuff != (char *) 0 )
				sprintf(ebuff, "am was used with HH >= 12");
			return 0;
		}
	} else {
		if ( H >= 12 ) {
			if ( ebuff != (char *) 0 )
				sprintf(ebuff, "pm was used with HH >= 12");
			return 0;
		} 
		H += 12;
	}	
	parse += 1;
  }

  return parse;
}

//////////////////////////////

void RDI_TimeValue::convert2string(char* buf) const
{
  char buffer[32];
  time_t secds = (time_t) _tms.tv_sec;
  unsigned int msecs = _tms.tv_usec / 1000;

  _RDI_CTime(&secds, buffer);

  buffer[7] = '\0';
  buffer[10] = '/';
  buffer[19] = ':';
  buffer[24] = ' '; 
  if (strcasecmp(&buffer[4], "Jan") == 0 )      strncpy(buf, "01/", 3); 
  else if (strcasecmp(&buffer[4], "Feb") == 0 ) strncpy(buf, "02/", 3);
  else if (strcasecmp(&buffer[4], "Mar") == 0 ) strncpy(buf, "03/", 3);
  else if (strcasecmp(&buffer[4], "Apr") == 0 ) strncpy(buf, "04/", 3);
  else if (strcasecmp(&buffer[4], "May") == 0 ) strncpy(buf, "05/", 3);
  else if (strcasecmp(&buffer[4], "Jun") == 0 ) strncpy(buf, "06/", 3);
  else if (strcasecmp(&buffer[4], "Jul") == 0 ) strncpy(buf, "07/", 3);
  else if (strcasecmp(&buffer[4], "Aug") == 0 ) strncpy(buf, "08/", 3);
  else if (strcasecmp(&buffer[4], "Sep") == 0 ) strncpy(buf, "09/", 3);
  else if (strcasecmp(&buffer[4], "Oct") == 0 ) strncpy(buf, "10/", 3);
  else if (strcasecmp(&buffer[4], "Nov") == 0 ) strncpy(buf, "11/", 3);
  else if (strcasecmp(&buffer[4], "Dec") == 0 ) strncpy(buf, "12/", 3);
  char* bptr = buf + 3;
  strncpy(bptr, &buffer[8], 3); // dd slash
  bptr = bptr + 3;
  strncpy(bptr, &buffer[20], 5); // yyyy space
  bptr = bptr + 5;
  strncpy(bptr, &buffer[11], 9); // hh:mm:ss:
  bptr = bptr + 9;
  sprintf(bptr, "%03d", msecs); // xxx
  bptr = bptr + 3;
  *bptr = '\0';
}

//////////////////////////////

ostream& operator << (ostream& out, const RDI_TimeValue& tv)
{
  char buffer[32];
  tv.convert2string(buffer);
  return out << buffer;
}
