// -*- Mode: C++; -*-
//                              File      : RDIConfig.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of RDI_Config
//
 
/*
$Log: RDIConfig.cc,v $
Revision 1.14  2000/11/15 21:17:30  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.13  2000/11/05 04:48:11  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.12  2000/08/22 18:23:55  alcfp
added description to each file

Revision 1.11  2000/08/16 20:19:46  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "RDIConfig.h"

static inline unsigned int MyStringHash(const void* K)
{
  unsigned int   h=0;
  unsigned char* p=0;
  for ( p = (unsigned char *) K; *p; p++ )
    h = 5*h + *p;
  return h;
}

RDI_Config::RDI_Config()
{
  for (unsigned int indx=0; indx < HASH_SIZE; indx++)
    _htbl[indx] = 0;	
}

RDI_Config::~RDI_Config()
{
  for (unsigned int indx=0; indx < HASH_SIZE; indx++) {
    while ( _htbl[indx] ) {
      node_t* tmp = _htbl[indx];
      _htbl[indx] = _htbl[indx]->_next;
      delete tmp;
    }
  }
}

int RDI_Config::parse_arguments(int& argc, char** argv, CORBA::Boolean rm_args)
{
  char *pname=0, *value=0;
  int  indx=0;
  
  while ( ++indx < argc ) {
    if ( argv[indx][0] != '-' || argv[indx][1] != 'D' )
      continue;
    pname = &argv[indx][2];
    value = pname;
    while ( value && (*value != '=') )
      ++value;
    if ( *value != '=' )
      continue;
    *value++ = '\0';
    if ( strcmp(pname, "CONFIGFILE") == 0 ) {
      if ( this->import_settings(value) )
	continue;
    } else if ( this->set_value(pname, value) ) 
      continue;
    if ( rm_args ) {
      for (int ix=indx; ix < argc-1; ix++)
	argv[ix] = argv[ix+1];
      argc -= 1;
      indx -= 1;
    }
  }
  return 0;
}

int RDI_Config::import_settings(const char* fname)
{
  FILE* file = 0;
  char  line[1024];
  char *pname=0, *value=0;
  unsigned int lnum = 0;
  
  if ( ! fname || ! strlen(fname) ) {
    // fprintf(stderr, "NULL configuration file name provided\n");
    return -1;
  }
  if ( (file = fopen(fname, "r")) == (FILE *) 0 ) {
    // fprintf(stderr, "Invalid configuration file name: %s\n", fname);
    return -1;
  }
  while ( fgets(line, sizeof(line)-1, file) != (char *) 0 ) {
    lnum += 1;
    pname = & line[0];
    // skip leading white space for pname
    while ( *pname == ' ' || *pname == '\t' )
      pname++;
    // ignore commented/blank lines
    if ( *pname=='#' || *pname=='\n' || *pname=='\0' )
      continue;
    value = pname;
    while ( *value != ' ' && *value != '\t' && *value != '\0' )
      value++;
    if ( *value == '\0' ) {
      // fprintf(stderr, "Invalid syntax @ %d -- skipping\n", lnum);
      continue;
    }
    *value++ = '\0';
    // skip leading white space for value
    while ( *value == ' ' || *value == '\t' )
      value++;
    // Remove trailing blanks and newlines from value
    while ( value[strlen(value)-1]=='\n' ||
	    value[strlen(value)-1]=='\t' || value[strlen(value)-1]==' ' )
      value[ strlen(value) - 1] = '\0';

    if ( this->set_value(pname, value) ) {
      (void) fclose(file);
      return -1;
    }
  }

  (void) fclose(file);
  return 0;
}

int RDI_Config::export_settings(const char* fname, const char* header) const
{
  FILE*   file = 0;
  node_t* node = 0;

  if ( ! fname || ! strlen(fname) ) {
    // fprintf(stderr, "NULL file name was provided\n");
    return -1;
  }
  if ( (file = fopen(fname, "w")) == (FILE *) 0 ) {
    // fprintf(stderr, "Failed to open configuration file: %s\n", fname);
    return -1;
  }

  fprintf(file, "# ==================================================\n");
  fprintf(file, "#        R E A D Y  Configuration  Parameters       \n");
  fprintf(file, "#                                                   \n");
  fprintf(file, "# You can modify the value of any variable by either\n");
  fprintf(file, "# editing the file and changing the desired value or\n");
  fprintf(file, "# seting the environment variable of the name to the\n");
  fprintf(file, "# desired value before process execution.           \n");
  fprintf(file, "#===================================================\n\n");

  if ( header && strlen(header) ) 
    fprintf(file, "# %s\n", header);

  for (unsigned int indx=0; indx < HASH_SIZE; indx++) {
    for (node = _htbl[indx]; node != (node_t *) 0; node = node->_next)
      fprintf(file, "%-30s  %s\n", node->_name, node->_value);
  }
  (void) fclose(file);
  return 0;
}

int RDI_Config::get_value(const char* pname, char*& value) const
{
  unsigned int indx = MyStringHash(pname) & HASH_MASK;
  node_t*      node = _htbl[indx];

  while ( node && strcmp(node->_name, pname) )
    node = node->_next;
  if ( ! node )
    return -1;
  value = node->_value;
  return 0;
}

int RDI_Config::get_value(const char* pname, int& value) const
{
  char *delim=0, *strvl=0;

  if ( get_value(pname, strvl) )
    return -1;
  value = (int) strtol(strvl, &delim, 0);
  if ( (delim == strvl) || (*delim != '\0') ) {
    // fprintf(stderr, "Expected integer value -- found %s\n", strvl);
    return -1;
  }
  return 0;
}

int RDI_Config::get_value(const char* pname, long& value) const
{
  char *delim=0, *strvl=0;

  if ( get_value(pname, strvl) )
    return -1;
  value = strtol(strvl, &delim, 0);
  if ( (delim == strvl) || (*delim != '\0') ) {
    // fprintf(stderr, "Expected long value -- found %s\n", strvl);
    return -1;
  }
  return 0;
}

int RDI_Config::get_value(const char* pname, double& value) const
{
  char *delim=0, *strvl=0;

  if ( get_value(pname, strvl) )
    return -1;
  value = strtod(strvl, &delim);
  if ( (delim == strvl) || (*delim != '\0') ) {
    // fprintf(stderr, "Expected double value -- found %s\n", strvl);
    return -1;
  }
  return 0;
}

int RDI_Config::getset_value(const char* pname, 
			     char*&  value, const char* dfval)
{ 
  if ( get_value(pname, value) == 0 ) 
    return 0;
  return set_value(pname, dfval) ? -1 : get_value(pname, value);
}

int RDI_Config::getset_value(const char* pname, int& value, int dfval)
{
  char strvl[64];
  if ( get_value(pname, value) == 0 )
    return 0;
  value = dfval;
  sprintf(strvl, "%d", dfval);
  return set_value(pname, strvl);
}

int RDI_Config::getset_value(const char* pname, long& value, long dfval)
{ 
  char strvl[64];
  if ( get_value(pname, value) == 0 )
    return 0;
  value = dfval;
  sprintf(strvl, "%ld", dfval);
  return set_value(pname, strvl);
}

int RDI_Config::getset_value(const char* pname, double& value, double dfval)
{ 
  char strvl[64];
  if ( get_value(pname, value) == 0 )
    return 0;
  value = dfval;
  sprintf(strvl, "%f", dfval);
  return set_value(pname, strvl);
}

int RDI_Config::set_value(const char* pname, const char* value)
{
  unsigned int indx = 0;
  node_t*      node = 0;
  char*        temp = 0;

  if ( ! pname || ! strlen(pname) || ! value || ! strlen(value) ) {
    // if (! pname ) fprintf(stderr, "pname ptr is null\n");
    // if (! value ) fprintf(stderr, "value ptr is null\n");
    // if (!strlen(pname)) fprintf(stderr, "strlen of pname is zero\n");
    // if (!strlen(value)) fprintf(stderr, "strlen of value is zero\n");
    // fprintf(stderr, "One or more NULL parameters were specified\n");
    return -1;
  }

  indx = MyStringHash(pname) & HASH_MASK;
  node = _htbl[indx];

  while ( node && strcmp(node->_name, pname) )
    node = node->_next;
  if ( ! node ) {
    if (!(node=new node_t) || !(node->_name = new char[strlen(pname)+1])) {
      // fprintf(stderr, "Memory allocation failed\n");
      return -1;
    } 
    strcpy(node->_name, pname);
    node->_next = _htbl[indx];
    _htbl[indx] = node;
  }
  if ( ! node->_value || strlen(node->_value) < strlen(value) ) {
    if ( ! (temp = new char [strlen(value) + 1]) ) {
      // fprintf(stderr, "Memory allocation failed\n");
      return -1;
    }
    if ( node->_value )
      delete [] node->_value;
    node->_value = temp;;
  }
  strcpy(node->_value, value);
  return 0;
}

int RDI_Config::env_update(const char* pname)
{
  char*   value = 0;
  node_t* cnode = 0;

  if ( pname ) {
    if ( (value = getenv(pname)) != (char *) 0 ) {
      if ( this->set_value(pname, value) ) {
	// fprintf(stderr, "Failed to set %s value\n", pname);
	return -1;
      }
    }
    return 0;
  }

  // Go over all existing configuration parameters and updated their 
  // values with the corresponding environment values, if specified.

  for (unsigned int indx=0; indx < HASH_SIZE; indx++ ) {
    for (cnode = _htbl[indx]; cnode != (node_t *) 0; cnode = cnode->_next) {
      if ( (value = getenv(cnode->_name)) == (char *) 0 )
	continue;
      if ( strlen(cnode->_value) < strlen(value) ) {
	char* temp = new char [ strlen(value) + 1 ];
	if ( ! temp ) {
				// fprintf(stderr,"Memory allocation failed\n");
	  return -1;
	}
	delete [] cnode->_value;
	cnode->_value = temp;
      }
      strcpy(cnode->_value, value);
    }
  }

  return 0;
}

ostream& operator << (ostream& out, const RDI_Config& cfg)
{
  RDI_Config::node_t* cnode = 0;
  for (unsigned int indx=0; indx < RDI_Config::HASH_SIZE; indx++ ) {
    for ( cnode = cfg._htbl[indx]; cnode; cnode = cnode->_next )
      out << cnode->_name << "\t\t" << cnode->_value << endl;
  }
  return out;
}
