// -*- Mode: C++; -*-
//                              File      : ProxySupplier.cc
//                              Package   : omniNotify-Library
//                              Created on: 18-Oct-2000
//                              Authors   : gruber
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of Supplier proxies
//
 
/*
$Log: ProxySupplier.cc,v $
Revision 1.10  2000/12/14 18:35:20  alcfp
fixes in channel shutdown sequence

Revision 1.9  2000/12/01 23:01:10  alcfp
fixed 2 bugs, 1 in admin-level subscription_change, 1 in changepool _next_available method

Revision 1.8  2000/11/19 19:02:33  alcfp
removed unnecessary debug output

Revision 1.7  2000/11/17 21:38:34  alcfp
small fixes to get rid of some Sun CC 5.0 compiler warnings

Revision 1.6  2000/11/16 05:32:16  alcfp
fixes for aCC

Revision 1.5  2000/11/15 21:17:30  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.4  2000/11/05 04:48:10  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.3  2000/10/30 16:19:29  alcfp
removed unused variable

Revision 1.2  2000/10/30 05:41:54  alcfp
renamed CosNotify.h to CosNotifyShorthands.h and placed CosNotifyShorthands.h in omniNotify/include rather than src/services/include

Revision 1.1  2000/10/30 04:40:24  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

*/
 
#include <iomanip.h>
#include "RDI.h"
#include "RDITimeDefs.h"
#include "RDIDebug.h"
#include "RDIEvent.h"
#include "RDIOplocksMacros.h"

#include "CosNotifyShorthands.h"
#include "CosNotifyChannelAdmin_i.h"

extern const char* RDI_PRX_TYPE(const CosNA_ProxyType& type);

// ------------------------------------------------------------- //
// Support classes needed for implementing  the thread-per-proxy //
// notification approach                                         //
// ------------------------------------------------------------- //

typedef void (ProxyPushSupplier_i::*AnyPushSupplierMethod)(void);

class AnyProxyWorker : public omni_thread {
public:
  AnyProxyWorker(ProxyPushSupplier_i* p, AnyPushSupplierMethod m) 
    : omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  ProxyPushSupplier_i*  _proxy;
  AnyPushSupplierMethod _method;
  AnyProxyWorker()  {;}
};

class AnyProxyBoundWorker : public omni_thread {
public:
  AnyProxyBoundWorker(ProxyPushSupplier_i* p, AnyPushSupplierMethod m) 
    : omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) 
  { start_undetached();   }
  void* run_undetached(void *)	{ (_proxy->*_method)(); return 0; }
private:
  ProxyPushSupplier_i*  _proxy;
  AnyPushSupplierMethod _method;
  AnyProxyBoundWorker()  {;}
};

typedef void (StructuredProxyPushSupplier_i::*StrPushSupplierMethod)(void);

class StrProxyWorker : public omni_thread {
public:
  StrProxyWorker(StructuredProxyPushSupplier_i* p, StrPushSupplierMethod m) 
    : omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  StructuredProxyPushSupplier_i* _proxy;
  StrPushSupplierMethod          _method;
  StrProxyWorker()  {;}
};

class StrProxyBoundWorker : public omni_thread {
public:
  StrProxyBoundWorker(StructuredProxyPushSupplier_i* p, StrPushSupplierMethod m)
    : omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) 
  { start_undetached();   }
  void* run_undetached(void *)  { (_proxy->*_method)(); return 0; }
private:
  StructuredProxyPushSupplier_i* _proxy;
  StrPushSupplierMethod          _method;
  StrProxyBoundWorker()  {;}
};

typedef void (SequenceProxyPushSupplier_i::*SeqPushSupplierMethod)(void);

class SeqProxyWorker : public omni_thread {
public:
  SeqProxyWorker(SequenceProxyPushSupplier_i* p, SeqPushSupplierMethod m) 
    : omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  SequenceProxyPushSupplier_i* _proxy;
  SeqPushSupplierMethod        _method;
  SeqProxyWorker()  {;}
};

class SeqProxyBoundWorker : public omni_thread {
public:
  SeqProxyBoundWorker(SequenceProxyPushSupplier_i* p, SeqPushSupplierMethod m) 
    : omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) 
  { start_undetached();   }
  void* run_undetached(void *)  { (_proxy->*_method)(); return 0; }
private:
  SequenceProxyPushSupplier_i* _proxy;
  SeqPushSupplierMethod        _method;
  SeqProxyBoundWorker()  {;}
};

// ------------------------------------------------------------- //
// RDIProxySupplier                                           //
// ------------------------------------------------------------- //

RDIProxySupplier::RDIProxySupplier(ConsumerAdmin_i*      admin, 
				   EventChannel_i*       chann,
				   const RDI_ObjectKind  otype,
				   const CosNA_ProxyType prtype,
				   const CosNA_ProxyID&  prxid) : 
  RDINotifySubscribe(),
  _oplockptr(0), _fa_helper(), _channel(chann), _myadmin(admin), 
  _otype(otype), _prxtype(prtype), _pserial(prxid), _nevents(0),
  _pxstate(RDI_NotConnected), _active(0),
  _qosprop(0), _oc_off(1), _rqstypes()
{
  RDI_OPLOCK_INIT;
  _oc_subscriber = CosNC_NotifyPublish::_nil();
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
}

// -------- ** Routines that implement public CORBA interfaces ** -------------

CosNA_ProxyType RDIProxySupplier::MyType()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_ProxyType res = _prxtype;
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_ProxyID RDIProxySupplier::MyID()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_ProxyID res = _pserial; 
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNA_ConsumerAdmin_ptr RDIProxySupplier::MyAdmin()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNA_ConsumerAdmin_ptr res = WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNF_MappingFilter_ptr RDIProxySupplier::priority_filter()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_MappingFilter_ptr res = CosNF_MappingFilter::_nil();
  if ( !CORBA::is_nil(_pfilter) ) {
    res =  CosNF_MappingFilter::_duplicate(_pfilter);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxySupplier::priority_filter
(CosNF_MappingFilter_ptr prio_filter)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  if ( CORBA::is_nil(prio_filter) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  _pfilter = CosNF_MappingFilter::_duplicate(prio_filter);
  RDI_OPLOCK_RELEASE;
}

CosNF_MappingFilter_ptr RDIProxySupplier::lifetime_filter()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_MappingFilter_ptr res = CosNF_MappingFilter::_nil();
  if ( !CORBA::is_nil(_lfilter) ) {
    res = CosNF_MappingFilter::_duplicate(_lfilter);
  }
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxySupplier::lifetime_filter
(CosNF_MappingFilter_ptr life_filter)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  if ( CORBA::is_nil(life_filter) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  _lfilter = CosNF_MappingFilter::_duplicate(life_filter); 
  RDI_OPLOCK_RELEASE;
}

CosN_EventTypeSeq* 
RDIProxySupplier::obtain_offered_types(CosNA_ObtainInfoMode mode)
{
  // to achieve atomicity for CosNA_ALL_NOW_UPDATES_ON we must
  // ask _channel to do this operation so that lock acquisition
  // order will be _channel followed by this + _channel->_ochange_pool
  // (via call to _enable_updates or _disable_updates)

  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_OPLOCK_BUMP_RELEASE;
  CosN_EventTypeSeq* res = _channel->pxy_obtain_offered_types(this, mode);
  RDI_OPLOCK_ACQUIRE_DEBUMP_RELEASE(RDI_THROW_INV_OBJREF);
  return res;
}

void RDIProxySupplier::validate_event_qos
(const CosN_QoSProperties& r_qos,
 WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_AssertAllocThrowNo(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  RDI_OPLOCK_RELEASE;
}

CosN_QoSProperties* RDIProxySupplier::get_qos()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  CosN_QoSProperties* res = qosp->get_qos(_otype);
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxySupplier::set_qos(const CosN_QoSProperties& r_qos)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,_otype,eseq,rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  if ( ! _qosprop ) {
    _qosprop = new RDI_NotifQoS();
    RDI_AssertAllocThrowNo(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
  RDI_OPLOCK_RELEASE;
}

void RDIProxySupplier::validate_qos
(const CosN_QoSProperties& r_qos,
 WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosN_PropertyErrorSeq eseq;
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_AssertAllocThrowNo(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,_otype,eseq,*rseq,child,p_qos)) {
    RDI_OPLOCK_RELEASE;
    throw CosN_UnsupportedQoS(eseq);
  }
  RDI_OPLOCK_RELEASE;
}

CosNF_FilterID
RDIProxySupplier::add_filter(CosNF_Filter_ptr filter)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
    _rqstypes.length(0);
  }
  CosNF_FilterID res = _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxySupplier::remove_filter(CosNF_FilterID fltrID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_filter(fltrID);
  RDI_OPLOCK_RELEASE;
}

void RDIProxySupplier::remove_all_filters()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  _fa_helper.remove_all_filters();
  RDI_OPLOCK_RELEASE;
}

CosNF_Filter_ptr RDIProxySupplier::get_filter(CosNF_FilterID fltrID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_Filter_ptr res = _fa_helper.get_filter(fltrID);
  RDI_OPLOCK_RELEASE;
  return res;
}

CosNF_FilterIDSeq* RDIProxySupplier::get_all_filters()
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed
  CosNF_FilterIDSeq* res = _fa_helper.get_all_filters();
  RDI_OPLOCK_RELEASE;
  return res;
}

void RDIProxySupplier::subscription_change
(const CosN_EventTypeSeq& added,
 const CosN_EventTypeSeq& deled)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // in process of being disposed

  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  }

  CosN_EventTypeSeq added_copy = added;
  CosN_EventTypeSeq deled_copy = deled;
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  nadded.length(0);
  ndeled.length(0);
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix = 0;
  if ( ! RDI_EventType::valid_sequence(added_copy, ix, i_am_consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(added[ix]);
  }
  if ( ! RDI_EventType::valid_sequence(deled_copy, ix, i_am_consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CosNC_InvalidEventType(deled[ix]);
  }
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
    RDI_EventType::compute_diff(_rqstypes, added_copy, deled_copy, nadded, ndeled);
    if ( nadded.length() || ndeled.length() ) {
      (void) _channel->update_mapping(nadded, ndeled, this, 0);
      RDI_EventType::update(_rqstypes, added_copy, deled_copy);
    }
  } else {
    ndeled.length(0);
    _rqstypes = added_copy; 
    RDI_EventType::update(_rqstypes, added_copy, deled_copy);
    (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
  RDI_OPLOCK_RELEASE;
}

// ------------------ ** Local-only routines **  --------------------------

void RDIProxySupplier::_enable_updates() {
  RDI_OPLOCK_ACQUIRE(return);
  if ( (_pxstate != RDI_Connected) || CORBA::is_nil(_oc_subscriber) || ! _channel->_ochange_pool ) {
    // _oc_off status is meaningless without connected _nc_consumer and a valid _ochange_pool
    _oc_off = 0;
    RDI_OPLOCK_RELEASE;
    return;
  }
  if (!_oc_off) {
    // make offer_change msgs reflect only subsequent updates
    // by removing 'this' before re-inserting it
    _channel->_ochange_pool->remove_proxy(this);
  }
  _oc_off = 0;
  _channel->_ochange_pool->insert_proxy(this);
  RDI_OPLOCK_RELEASE;
}

void RDIProxySupplier::_disable_updates() {
  RDI_OPLOCK_ACQUIRE(return);
  if ( (_pxstate != RDI_Connected) || CORBA::is_nil(_oc_subscriber) || ! _channel->_ochange_pool ) {
    // _oc_off status is meaningless without connected _nc_consumer and a valid _ochange_pool
    _oc_off = 1;
    RDI_OPLOCK_RELEASE;
    return;
  }
  if (!_oc_off) {
    _oc_off = 1;
    _channel->_ochange_pool->remove_proxy(this);
  }
  RDI_OPLOCK_RELEASE;
}

void RDIProxySupplier::filter_destroy_i(Filter_i* filter)
{
  _fa_helper.rem_filter_i(filter);
}

void RDIProxySupplier::propagate_subscription_change
(const CosN_EventTypeSeq& added,
 const CosN_EventTypeSeq& deled,
 Filter_i*          flter)
{
  (void) _channel->update_mapping(added, deled, this, flter);
}

// --------------------- ** Protected methods ** ------------------------------------- 

void RDIProxySupplier::disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_client_and_dispose(remove_from_admin);
}

// If proxy has a connected CosNotify-style consumer, _oc_subscriber will be non-null.
// Attempt to send it an offer_change msg.
//
// If the message cannot be delivered, or if CORBA::NO_IMPLEMENT or another
// exception is returned, return value TRUE
// (indicating the proxy should not be sent any further offer_change msgs), otherwise FALSE.
//
// XXX On a transient failure, the channel could retry one or more times before
// XXX giving up and returning value TRUE.  At the moment, channel gives up immediately.
//
// XXX ** On hard failure, change _pxstate ??? **
//
CORBA::Boolean RDIProxySupplier::send_offer_change(const CosN_EventTypeSeq& added,
						   const CosN_EventTypeSeq& deled) {
  RDI_OPLOCK_ACQUIRE_BUMP(return 1); // debump on method exit

  CORBA::Boolean res = 0; // OK
  if (_oc_off) {
    RDI_DUMP("send_offer_change called on proxy with _oc_off true");
    res = 1;
  } else if ( (_pxstate == RDI_Connected) && ! CORBA::is_nil(_oc_subscriber) ) {
    // do not hold OPLOCK across offer_change call
    RDI_OPLOCK_RELEASE;
    try {
      _oc_subscriber->offer_change(added, deled);
    } catch(CORBA::NO_IMPLEMENT& ex) {
      res = 1; // do not try any more of these
    } catch (...) {
      RDI_DUMP("RDIProxySupplier -- caught exception other than CORBA::NO_IMPLEMENT " <<
	       "when sending an offer_change; could retry if this is a soft failure " <<
	       "but for now, giving up immediately");
      res = 1;
    }
    RDI_OPLOCK_REACQUIRE(return 1);
  } else {
    RDI_DUMP("RDIProxySupplier -- oc_subscriber not available => did not send offer_change");
    res = 1;
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
  return res;
}

// ------------------------------------------------------------- //
// ProxyPullSupplier_i                                           //
// ------------------------------------------------------------- //

ProxyPullSupplier_i::ProxyPullSupplier_i(ConsumerAdmin_i* admin, 
					 EventChannel_i*  chann,
					 const CosNA_ProxyID&  prxid) : 
  RDIProxySupplier(admin, chann, RDI_S_AnyPRX, CosNA_PULL_ANY, prxid),
  _ntfqueue(32,32)
{
  _consumer = CosEventComm::PullConsumer::_nil();
  _nc_consumer = CosNC_PullConsumer::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

CORBA::Any* ProxyPullSupplier_i::pull( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE_BUMP(RDI_THROW_INV_OBJREF); // debump on method exit
  CORBA::ULong qsize = 0;
  while ( (_pxstate == RDI_Connected) && ((qsize = _ntfqueue.length()) == 0) ) {
    RDI_OPLOCK_WAIT;
  }
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    throw CosEventComm::Disconnected();
  }

  RDI_StructuredEvent* event = _ntfqueue.get_head();
  const CosN_StructuredEvent& cosev = event->get_cos_event();
  _ntfqueue.remove_head();
  _nevents += 1;

  CORBA::Any* res = 0;
  if (  RDI_LOCK_EVENT  ) event->mylock().lock();
  if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
    res = new CORBA::Any(cosev.remainder_of_body);
  } else {
    res = new CORBA::Any;
    (*res) <<= cosev;
  }
  if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
  event->decr_ref_counter( RDI_LOCK_EVENT );

  _channel->incr_num_notifications(--qsize);
  RDI_OPLOCK_DEBUMP_RELEASE;
  return res;
}

CORBA::Any* 
ProxyPullSupplier_i::try_pull(CORBA::Boolean& has_event WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  CORBA::ULong qsize = _ntfqueue.length();
  if ( qsize == 0 ) {
    has_event = 0;
    RDI_OPLOCK_RELEASE;
    return new CORBA::Any();
  }
  RDI_StructuredEvent* event = _ntfqueue.get_head();
  const CosN_StructuredEvent&      cosev = event->get_cos_event();
  _ntfqueue.remove_head();
  _nevents += 1;

  if (  RDI_LOCK_EVENT  ) event->mylock().lock();
  CORBA::Any* res = 0;
  has_event = 1;
  if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
    res = new CORBA::Any(cosev.remainder_of_body);
  } else {
    res = new CORBA::Any;
    (*res) <<= cosev;
  }
  if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
  event->decr_ref_counter( RDI_LOCK_EVENT );

  _channel->incr_num_notifications(--qsize);
  RDI_OPLOCK_RELEASE;
  return res;
}
  
void ProxyPullSupplier_i::connect_any_pull_consumer
(CosEventComm::PullConsumer_ptr consumer WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  // According to the CORBA Notification Service specification, pull
  // consumers can provide a _nil reference [2.6.1].
  if ( ! CORBA::is_nil(consumer) ) {
    _consumer = CosEventComm::PullConsumer::_duplicate(consumer);
    // If consumer is a CosNC_PullConsumer, set _nc_consumer
    _nc_consumer = CosNC_PullConsumer::_narrow(consumer); // implicit duplicate
    if (! CORBA::is_nil(_nc_consumer)) {
      _oc_subscriber = CosNC_NotifyPublish::_narrow(_nc_consumer); // implicit duplicate
      if (CORBA::is_nil(_oc_subscriber)) {
	RDI_DUMP("** UNEXPECTED: CosNC_PullConsumer could not be narrowed to CosNC_NotifyPublish");
      }
      // if enabled, register this proxy for offer_change msgs
      if ( ! _oc_off && _channel->_ochange_pool ) {
	_channel->_ochange_pool->insert_proxy(this);
      }
    }
  }
  _pxstate = RDI_Connected;
  _active  = 1;
  RDI_OPLOCK_RELEASE;
}

void ProxyPullSupplier_i::add_event(RDI_StructuredEvent* entry WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( (_pxstate == RDI_Connected) && entry ) {
    // Check if the 'MaxEventsPerConsumer' threshold has been reached.
    // In this case, we discard events based on 'DiscardPolicy' -- the
    // current implementation assumes FIFO
    RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());
    unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
    if ( msize && (_ntfqueue.length() >= msize) ) {
      RDI_StructuredEvent* evnt = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      evnt->decr_ref_counter( RDI_LOCK_EVENT );
      RDI_DUMP("MaxEventPerConsumer (" << msize << ") reached -- discarded oldest event in consumer's pull queue");
    }
    entry->incr_ref_counter( RDI_LOCK_EVENT );
    _ntfqueue.insert_tail(entry);
    RDI_OPLOCK_SIGNAL;
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const ProxyPullSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype);
  if ( CORBA::is_nil(prx._nc_consumer) ) out << " CosEventComm Consumer";
  out << prx._pxstate;
  out << " QSize "<< prx._ntfqueue.length()<< " #Pull "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < prx._rqstypes.length(); ix++) {
      out << endl << "\t" << prx._rqstypes[ix].domain_name; 
      out << "::" << prx._rqstypes[ix].type_name;
    }
  }
  return out;
}

void ProxyPullSupplier_i::disconnect_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void ProxyPullSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** Internal error -- ProxyPullSupplier_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more offer_change msgs
  if ( ! _oc_off && (_pxstate == RDI_Connected) && ! CORBA::is_nil(_nc_consumer) && _channel->_ochange_pool ) {
    _channel->_ochange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // Wait for inuse to drop to zero.  This ensures that any outstanding blocked pull
  // call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("ProxyPullSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  if ( _rqstypes.length() == 0 ) {
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  _consumer = CosEventComm::PullConsumer::_nil();
  _nc_consumer = CosNC_PullConsumer::_nil();
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter( RDI_LOCK_EVENT );
  }
  _pfilter = CosNF_MappingFilter::_nil();
  _lfilter = CosNF_MappingFilter::_nil();
  if ( _qosprop ) {
    delete _qosprop; _qosprop = 0;
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// StructuredProxyPullSupplier_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPullSupplier_i::StructuredProxyPullSupplier_i(
							     ConsumerAdmin_i* admin, 
							     EventChannel_i*  chann,
							     const CosNA_ProxyID&  prxid) : 
  RDIProxySupplier(admin, chann, RDI_S_StrPRX, CosNA_PULL_STRUCTURED, prxid),
  _ntfqueue(32,32)
{
  _consumer = CosNC_StructuredPullConsumer::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

CosN_StructuredEvent* 
StructuredProxyPullSupplier_i::pull_structured_event( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE_BUMP(RDI_THROW_INV_OBJREF); // debump on method exit
  CosN_StructuredEvent* cosse = 0;
  RDI_StructuredEvent*  entry = 0;

  CORBA::ULong qsize = 0;
  while ( (_pxstate == RDI_Connected) && ((qsize = _ntfqueue.length()) == 0) ) {
    RDI_OPLOCK_WAIT;
  }
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    throw CosEventComm::Disconnected();
  } 

  entry = _ntfqueue.get_head();
  _ntfqueue.remove_head();
  _nevents += 1;
  if (  RDI_LOCK_EVENT  ) entry->mylock().lock();
  cosse = new CosN_StructuredEvent( entry->get_cos_event());
  if (  RDI_LOCK_EVENT  ) entry->mylock().unlock();
  entry->decr_ref_counter( RDI_LOCK_EVENT );
  _channel->incr_num_notifications(--qsize);
  RDI_OPLOCK_DEBUMP_RELEASE;
  return cosse;
}

CosN_StructuredEvent* StructuredProxyPullSupplier_i::try_pull_structured_event
(CORBA::Boolean& has_event   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  CORBA::ULong qsize = _ntfqueue.length();
  if ( qsize == 0 ) {
    has_event = 0;
    RDI_OPLOCK_RELEASE;
    return new CosN_StructuredEvent();
  }
  CosN_StructuredEvent*    cosse = 0; 
  RDI_StructuredEvent* entry=_ntfqueue.get_head();
  _ntfqueue.remove_head();
  _nevents += 1;
  has_event = 1;
  if (  RDI_LOCK_EVENT  ) entry->mylock().lock();
  cosse = new CosN_StructuredEvent( entry->get_cos_event());
  if (  RDI_LOCK_EVENT  ) entry->mylock().unlock();
  entry->decr_ref_counter( RDI_LOCK_EVENT );
  _channel->incr_num_notifications(--qsize);
  RDI_OPLOCK_RELEASE;
  return cosse;
}	

void StructuredProxyPullSupplier_i::connect_structured_pull_consumer
(CosNC_StructuredPullConsumer_ptr consumer WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  // According to the CORBA Notification Service specification, pull
  // consumers can provide a _nil reference [2.6.1] ................
  if ( ! CORBA::is_nil(consumer) ) {
    _consumer = CosNC_StructuredPullConsumer::_duplicate(consumer);
    _oc_subscriber = CosNC_NotifyPublish::_narrow(_consumer); // implicit duplicate
    if (CORBA::is_nil(_oc_subscriber)) {
      RDI_DUMP("** UNEXPECTED: CosNC_StructuredPullConsumer could not be narrowed to CosNC_NotifyPublish");
    }
    // if enabled, register this proxy for offer_change msgs
    if ( ! _oc_off && _channel->_ochange_pool ) {
      _channel->_ochange_pool->insert_proxy(this);
    }
  }
  _pxstate  = RDI_Connected;
  _active  = 1;
  RDI_OPLOCK_RELEASE;
}

void StructuredProxyPullSupplier_i::add_event
(RDI_StructuredEvent* entry   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( (_pxstate == RDI_Connected) && entry ) {
    // Check if the 'MaxEventsPerConsumer' threshold has been reached.
    // In this case, we discard events based on 'DiscardPolicy' -- the
    // current implementation assumes FIFO
    RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());
    unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
    if ( msize && (_ntfqueue.length() >= msize) ) {
      RDI_StructuredEvent* evnt = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      evnt->decr_ref_counter( RDI_LOCK_EVENT );
      RDI_DUMP("MaxEventPerConsumer (" << msize << ") reached -- discarded oldest event in consumer's pull queue");
    }
    entry->incr_ref_counter( RDI_LOCK_EVENT );      
    _ntfqueue.insert_tail(entry);
    RDI_OPLOCK_SIGNAL;
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const StructuredProxyPullSupplier_i& prx)
{
  out << & prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
    " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter ) ) out << " PFilter " << (void*)prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter ) ) out << " LFilter " << (void*)prx._lfilter;
  out << prx._pxstate;
  out << " QSize "<< prx._ntfqueue.length()<< " #Pull "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < prx._rqstypes.length(); ix++) {
      out << endl << "\t" << prx._rqstypes[ix].domain_name; 
      out << "::" << prx._rqstypes[ix].type_name;
    }
  }
  return out;
}

void StructuredProxyPullSupplier_i::disconnect_structured_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void StructuredProxyPullSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** Internal error -- StructuredProxyPullSupplier_i::_disconnect_client_and_dispose " <<
	     "called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more offer_change msgs
  if ( ! _oc_off && (_pxstate == RDI_Connected) && _channel->_ochange_pool ) {
    _channel->_ochange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // Wait for inuse to drop to zero.  This ensures that any outstanding blocked pull
  // call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("StructuredProxyPullSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  if ( _rqstypes.length() == 0 ) { 
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  _consumer = CosNC_StructuredPullConsumer::_nil();
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter( RDI_LOCK_EVENT );
  }
  _pfilter = CosNF_MappingFilter::_nil();
  _lfilter = CosNF_MappingFilter::_nil();
  if ( _qosprop ) {
    delete _qosprop; _qosprop = 0;
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// SequenceProxyPullSupplier_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPullSupplier_i::SequenceProxyPullSupplier_i(
							 ConsumerAdmin_i* admin, 
							 EventChannel_i*  chann,
							 const CosNA_ProxyID&  prxid) : 
  RDIProxySupplier(admin, chann, RDI_S_SeqPRX, CosNA_PULL_SEQUENCE, prxid),
  _ntfqueue(32,32)
{
  unsigned long pacing_interval_s, pacing_interval_n;
  _consumer = CosNC_SequencePullConsumer::_nil();

  // If pacingInterval is non-zero, set initial timeout.
  // Note that the interval typically expires before any events are
  // placed in the notification queue of this proxy, which means
  // the first batch normally contains just a single event.
  // The only way to fix this would be to initialize the pacing
  // interval on the very first event insertion.
  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_TimeT2SecNanosec(qospr->pacingInterval, pacing_interval_s, pacing_interval_n);
  if (pacing_interval_s || pacing_interval_n) { // update timeout
    omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }
  WRAPPED_REGISTER_IMPL(this);
}

CosN_EventBatch* 
SequenceProxyPullSupplier_i::pull_structured_events
(CORBA::Long maxnum   WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE_BUMP(RDI_THROW_INV_OBJREF); // debump on method exit
  if (maxnum < 0)
    maxnum = 0; // we take 0 to mean use the channel's maximumBatchSize
  CosN_EventBatch* notif = 0;
  RDI_StructuredEvent* event = 0;
  RDI_NotifQoS* qospr = 0;
  // pacing interval : both 0 means always wait for full batch size
  unsigned long pacing_interval_s, pacing_interval_n;
  unsigned long time_s, time_n;
  CORBA::Long bsize, qsize, actsize;

  while ( 1 ) {
    // must recompute these here because they can change across a wait/timedwait
    qsize = _ntfqueue.length();
    qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
    RDI_TimeT2SecNanosec(qospr->pacingInterval, pacing_interval_s, pacing_interval_n);
    if (pacing_interval_s || pacing_interval_n) {
      // if timeout not set, recompute timeout
      if ((_timeout_s == 0) && (_timeout_n == 0)) {
	omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
      }
    } else {
      _timeout_s = 0; _timeout_n = 0;
    }
    bsize = qospr->maximumBatchSize;
    if (maxnum && (bsize > maxnum))
      bsize = maxnum;
    // breaking from inner loop requires either a sufficient batch size or a
    // pacing interval timeout (and state must be RDI_Connected and _active)
    if ((_pxstate!=RDI_NotConnected) && (_pxstate!=RDI_Connected)) {
      break; // also break on exceptional state
    }
    if ((_pxstate==RDI_Connected) && _active) {
      if ( qsize >= bsize ) {
	break;
      }
      if ( qsize && (pacing_interval_s || pacing_interval_n) ) {
	omni_thread::get_time(&time_s, &time_n);
	if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
	  // pacing interval has passed so go ahead and send a smaller batch
	  break;
	}
      }
    }
    // must wait
    if ( (_pxstate==RDI_Connected) && _active && qsize &&
	 (pacing_interval_s || pacing_interval_n) ) {
      //      RDI_DUMP("sequence proxy doing timedwait(" << _timeout_s << "," << _timeout_n <<")");
      RDI_OPLOCK_TIMEDWAIT(_timeout_s, _timeout_n);
    } else {
      //      RDI_DUMP("sequence proxy doing normal wait");
      RDI_OPLOCK_WAIT;
    }
  }
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    throw CosEventComm::Disconnected();
  }
  actsize = (qsize < bsize) ? qsize : bsize;
  if ( ! (notif = new CosN_EventBatch(actsize)) ) {
    RDI_DUMP("Memory allocation failed -- CosN_EventBatch");
    RDI_OPLOCK_DEBUMP_RELEASE;
    return 0;
  }
  notif->length(actsize);
  for (CORBA::Long i = 0; i < actsize; i++) {
    event = _ntfqueue.get_head();
    _ntfqueue.remove_head();
    if (  RDI_LOCK_EVENT  ) event->mylock().lock();
    (*notif)[i] = event->get_cos_event();
    if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
    event->decr_ref_counter( RDI_LOCK_EVENT );
    _nevents += 1;
    _channel->incr_num_notifications(--qsize);
  }
  // returning a valid batch => next call to pull_structured_events should use a new timeout:
  if (pacing_interval_s || pacing_interval_n) {
    omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
  return notif;
}
  
CosN_EventBatch* 
SequenceProxyPullSupplier_i::try_pull_structured_events
(CORBA::Long maxnum, CORBA::Boolean& has_event WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  CosN_EventBatch* notif = 0;
  RDI_StructuredEvent* event = 0;

  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }

  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  if (maxnum < 0)
    maxnum = 0; // we take 0 to mean use the channel's maximumBatchSize
  if ( (maxnum == 0) || (maxnum > qospr->maximumBatchSize) )
    maxnum = qospr->maximumBatchSize;
  CORBA::Long qsize = _ntfqueue.length();

  if ( ! (notif = new CosN_EventBatch) ) {
    RDI_DUMP("Memory allocation failed -- CosN_EventBatch");
    RDI_OPLOCK_RELEASE;
    return 0;
  }

  if ( qsize == 0 ) {
    has_event = 0;
    notif->length(0);
  } else {
    CORBA::Long bsize = ((maxnum < qsize) ? maxnum : qsize);
    notif->length(bsize);
    for (CORBA::Long i = 0; i < bsize; i++) {
      event = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      if (  RDI_LOCK_EVENT  ) event->mylock().lock();
      (*notif)[i] = event->get_cos_event();
      if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
      event->decr_ref_counter( RDI_LOCK_EVENT );
      _nevents += 1;
      _channel->incr_num_notifications(--qsize);
    }
    has_event = 1;
  }
  RDI_OPLOCK_RELEASE;
  return notif;
}

void SequenceProxyPullSupplier_i::connect_sequence_pull_consumer
(CosNC_SequencePullConsumer_ptr consumer WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  // According to the CORBA Notification Service specification, pull
  // consumers can provide a _nil reference [2.6.1] ................
  if ( ! CORBA::is_nil(consumer) ) {
    _consumer = CosNC_SequencePullConsumer::_duplicate(consumer);
    _oc_subscriber = CosNC_NotifyPublish::_narrow(_consumer); // implicit duplicate
    if (CORBA::is_nil(_oc_subscriber)) {
      RDI_DUMP("** UNEXPECTED: CosNC_SequencePullConsumer could not be narrowed to CosNC_NotifyPublish");
    }
    // if enabled, register this proxy for offer_change msgs
    if ( ! _oc_off && _channel->_ochange_pool ) {
      _channel->_ochange_pool->insert_proxy(this);
    }
  }
  _pxstate  = RDI_Connected;
  _active  = 1;
  RDI_OPLOCK_RELEASE;
}

void SequenceProxyPullSupplier_i::add_event
(RDI_StructuredEvent* entry   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( (_pxstate == RDI_Connected) && entry ) {
    // Check if the 'MaxEventsPerConsumer' threshold has been reached.
    // In this case, we discard events based on 'DiscardPolicy' -- the
    // current implementation assumes FIFO
    RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());
    unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
    if ( msize && (_ntfqueue.length() >= msize) ) {
      RDI_StructuredEvent* evnt = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      evnt->decr_ref_counter( RDI_LOCK_EVENT );
      RDI_DUMP("MaxEventPerConsumer (" << msize << ") reached -- discarded oldest event in consumer's pull queue");
    }
    entry->incr_ref_counter( RDI_LOCK_EVENT );
    _ntfqueue.insert_tail(entry);
    RDI_OPLOCK_SIGNAL;
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const SequenceProxyPullSupplier_i& prx)
{
  out << & prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
    " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << (void*)prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << (void*)prx._lfilter;
  out << prx._pxstate;
  out << " QSize "<< prx._ntfqueue.length()<< " #Pull "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < prx._rqstypes.length(); ix++) {
      out << endl << "\t" << prx._rqstypes[ix].domain_name; 
      out << "::" << prx._rqstypes[ix].type_name;
    }
  }
  return out;
}

void SequenceProxyPullSupplier_i::disconnect_sequence_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void SequenceProxyPullSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** Internal error -- SequenceProxyPullSupplier_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more offer_change msgs
  if ( ! _oc_off && (_pxstate == RDI_Connected) && _channel->_ochange_pool ) {
    _channel->_ochange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // Wait for inuse to drop to zero.  This ensures that any outstanding blocked pull
  // call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("SequenceProxyPullSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  if ( _rqstypes.length() == 0 ) { 
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  _consumer = CosNC_SequencePullConsumer::_nil();
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter( RDI_LOCK_EVENT );
  }
  _pfilter = CosNF_MappingFilter::_nil();
  _lfilter = CosNF_MappingFilter::_nil();
  if ( _qosprop ) {
    delete _qosprop; _qosprop = 0;
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// ProxyPushSupplier_i                                           //
// ------------------------------------------------------------- //

ProxyPushSupplier_i::ProxyPushSupplier_i(ConsumerAdmin_i* admin, 
					 EventChannel_i*  chann,
					 const CosNA_ProxyID&  prxid) : 
  RDIProxySupplier(admin, chann, RDI_S_AnyPRX, CosNA_PUSH_ANY, prxid),
  _worker(0), _ntfqueue(32, 128)
{
  _consumer = CosEventComm::PushConsumer::_nil();

  // When the number of push threads allocated at the channel level
  // is 0, each proxy uses its own thread for pushing events 
  if ( _channel->push_threads() == 0 ) {
    // _worker = new AnyProxyWorker(this, &ProxyPushSupplier_i::_push_event);
    _worker = new AnyProxyBoundWorker(this, &ProxyPushSupplier_i::_push_event);
    RDI_AssertAllocThrowNo(_worker, "Memory allocation failed -- omni_thread");
    // If we use unbound threads, we should start the thread using
    // _worker->start();
    RDI_DUMP("Push thread for proxy " << (void*)this << " -- " << _worker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}
 
void ProxyPushSupplier_i::connect_any_push_consumer
(CosEventComm::PushConsumer_ptr consumer WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  CosEventComm::PushConsumer_var evcons;
  if ( CORBA::is_nil(consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _active   = 1;
  _pxstate  = RDI_Connected;
  _consumer = CosEventComm::PushConsumer::_duplicate(consumer);
  // If consumer is a CosNC_PushConsumer, set _nc_consumer
  _nc_consumer = CosNC_PushConsumer::_narrow(consumer); // implicit duplicate
  if (! CORBA::is_nil(_nc_consumer)) {
    _oc_subscriber = CosNC_NotifyPublish::_narrow(_nc_consumer); // implicit duplicate
    if (CORBA::is_nil(_oc_subscriber)) {
      RDI_DUMP("** UNEXPECTED: CosNC_PushConsumer could not be narrowed to CosNC_NotifyPublish");
    }
    // if enabled, register this proxy for offer_change msgs
    if ( ! _oc_off && _channel->_ochange_pool ) {
      _channel->_ochange_pool->insert_proxy(this);
    }
  }
    RDI_OPLOCK_RELEASE;
}

void ProxyPushSupplier_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( ! _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyInactive();
  }
  _active = 0;
  RDI_OPLOCK_RELEASE;
}

void ProxyPushSupplier_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyActive();
  }
  _active = 1;
  if ( _worker ) {	// Notify worker thread
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_push_consumer ) {
    _channel->_push_consumer->signal_push_threads();
  }
  RDI_OPLOCK_RELEASE;
}

CORBA::Boolean ProxyPushSupplier_i::has_events(unsigned long* wait_s, unsigned long* wait_t)
{
  RDI_OPLOCK_ACQUIRE(return 0);
  if ( !_oplockptr || (_pxstate != RDI_Connected) || !_active || !_ntfqueue.length() ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }
  RDI_OPLOCK_RELEASE;
  return 1;
}

void ProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit 

  RDI_StructuredEvent* event = 0;
  unsigned int         qsize = 0;

  invalid = 0;
  if ( (_pxstate != RDI_Connected) || !_active || !_ntfqueue.length() ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }
  event = _ntfqueue.get_head();
  qsize = _ntfqueue.length() - 1;
  _ntfqueue.remove_head();
  _nevents += 1;

  // Do not hold OPLOCK across push
  RDI_OPLOCK_RELEASE;

  CORBA::Boolean outcall_worked = 0;
  if ( RDI_LOCK_EVENT ) event->mylock().lock();
  try {
    const CosN_StructuredEvent& sev = event->get_cos_event();
    if ( strcmp(event->get_type_name(), "%ANY") == 0 ) 
      _consumer->push(sev.remainder_of_body);
    else {
      CORBA::Any any_event;
      any_event <<= sev;
      _consumer->push(any_event);
    }
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Consumer object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Exception while pushing event to consumer");
  }

  RDI_OPLOCK_REACQUIRE(return);

  if ( RDI_LOCK_EVENT ) event->mylock().unlock();
  event->decr_ref_counter(RDI_LOCK_EVENT);

  if (outcall_worked) {
    _channel->incr_num_notifications(qsize);
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during push -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      // unregister this proxy -- no more subscription_change msgs
      if ( ! _oc_off && ! CORBA::is_nil(_nc_consumer) && _channel->_ochange_pool ) {
	_channel->_ochange_pool->remove_proxy(this);
      }
      _pxstate = RDI_Exception;
      invalid = 1;
    }
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
}

void ProxyPushSupplier_i::_push_event()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  RDI_StructuredEvent* event = 0;
  unsigned int         qsize = 0;
  CORBA::Boolean       do_yield = 0, outcall_worked = 0;


  while ( 1 ) {
    do_yield = 1;
    while ( (_pxstate==RDI_NotConnected) ||
	    ((_pxstate==RDI_Connected)&&(!_active || !_ntfqueue.length())) ) {
      do_yield = 0;
      RDI_OPLOCK_WAIT;
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("push thread");
    }
    event = _ntfqueue.get_head();
    qsize = _ntfqueue.length() - 1;
    _ntfqueue.remove_head();
    _nevents += 1;

    // Do not hold OPLOCK across push
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    outcall_worked = 0;
    if ( RDI_LOCK_EVENT ) event->mylock().lock();
    try {
      const CosN_StructuredEvent& sev = event->get_cos_event();
      if ( strcmp(event->get_type_name(), "%ANY") == 0 ) 
	_consumer->push(sev.remainder_of_body);
      else {
	CORBA::Any any_event;
	any_event <<= sev;
	_consumer->push(any_event);
      }
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Consumer object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Communication failure while pushing event to consumer");
    } catch (...) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Exception while pushing event to consumer");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("push thread [**unexpected REACQUIRE failure**]"));

    if ( RDI_LOCK_EVENT ) event->mylock().unlock();
    event->decr_ref_counter(RDI_LOCK_EVENT);

    if (outcall_worked) {
      _channel->incr_num_notifications(qsize);
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during push -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	// unregister this proxy -- no more subscription_change msgs
	if ( ! _oc_off && ! CORBA::is_nil(_nc_consumer) && _channel->_ochange_pool ) {
	  _channel->_ochange_pool->remove_proxy(this);
	}
	_pxstate = RDI_Exception;
      }
    }
  }
}

void ProxyPushSupplier_i::add_event(RDI_StructuredEvent* entry WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( (_pxstate == RDI_Connected) && entry ) {
    // Check if the 'MaxEventsPerConsumer' threshold has been reached.
    // In this case, we discard events based on 'DiscardPolicy' -- the
    // current implementation assumes FIFO 
    RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());	
    unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
    if ( msize && (_ntfqueue.length() >= msize) ) {
      RDI_StructuredEvent* evnt = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      evnt->decr_ref_counter(RDI_LOCK_EVENT);
      RDI_DUMP("MaxEventPerConsumer (" << msize << ") reached -- discarded oldest event in consumer's push queue");
    }
    entry->incr_ref_counter(RDI_LOCK_EVENT);
    _ntfqueue.insert_tail(entry);
    if ( _worker ) {
      RDI_OPLOCK_SIGNAL;
    }
    if ( _channel->_push_consumer ) {
      _channel->_push_consumer->signal_push_threads();
    }
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const ProxyPushSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
    " ID " << setw(3) << prx._pserial;
  if ( CORBA::is_nil(prx._nc_consumer) )   out << " CosEventComm Consumer";
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << (void*)prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << (void*)prx._lfilter;
  out << prx._pxstate;
  out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < prx._rqstypes.length(); ix++) {
      out << endl << "\t" << prx._rqstypes[ix].domain_name;
      out << "::" << prx._rqstypes[ix].type_name;
    }
  }
  return out;
}

void ProxyPushSupplier_i::disconnect_push_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void ProxyPushSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** Internal error -- ProxyPushSupplier_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more offer_change msgs
  if ( ! _oc_off && (_pxstate == RDI_Connected) && ! CORBA::is_nil(_nc_consumer) && _channel->_ochange_pool ) {
    _channel->_ochange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding push call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("ProxyPushSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  if ( _rqstypes.length() == 0 ) { 
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  _active  = 0;
  _consumer = CosEventComm::PushConsumer::_nil();
  _nc_consumer = CosNC_PushConsumer::_nil();
  _pfilter = CosNF_MappingFilter::_nil();
  _lfilter = CosNF_MappingFilter::_nil();
  if ( _qosprop ) {
    delete _qosprop; _qosprop = 0;
  }
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter(RDI_LOCK_EVENT);
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// StructuredProxyPushSupplier_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPushSupplier_i::StructuredProxyPushSupplier_i(
							     ConsumerAdmin_i* admin, 
							     EventChannel_i*  chann,
							     const CosNA_ProxyID&  prxid) : 
  RDIProxySupplier(admin, chann, RDI_S_StrPRX, CosNA_PUSH_STRUCTURED, prxid),
  _worker(0), _ntfqueue(128, 128)
{
  _consumer = CosNC_StructuredPushConsumer::_nil();
  // When the number of push threads allocated at the channel level is
  // 0, each proxy uses its own thread to push events to its consumer.
  if (  _channel->push_threads() == 0 ) {
    // _worker = new StrProxyWorker(this,&StructuredProxyPushSupplier_i::_push_event);
    _worker = new StrProxyBoundWorker(this,&StructuredProxyPushSupplier_i::_push_event);
    // If we use unbound threads, we should start the thread using
    // _worker->start();
    RDI_DUMP("PushSupplier Proxy Thread " << _worker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void StructuredProxyPushSupplier_i::connect_structured_push_consumer
(CosNC_StructuredPushConsumer_ptr consumer WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _active  = 1;
  _pxstate = RDI_Connected;
  _consumer= CosNC_StructuredPushConsumer::_duplicate(consumer);
  _oc_subscriber = CosNC_NotifyPublish::_narrow(_consumer); // implicit duplicate
  if (CORBA::is_nil(_oc_subscriber)) {
    RDI_DUMP("** UNEXPECTED: CosNC_StructuredPushConsumer could not be narrowed to CosNC_NotifyPublish");
  }
  // if enabled, register this proxy for offer_change msgs
  if ( ! _oc_off && _channel->_ochange_pool ) {
    _channel->_ochange_pool->insert_proxy(this);
  }
  RDI_OPLOCK_RELEASE;
}

void StructuredProxyPushSupplier_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( ! _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyInactive();
  }
  _active = 0;
  RDI_OPLOCK_RELEASE;
}

void StructuredProxyPushSupplier_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyActive();
  }
  _active = 1;
  if ( _worker ) {
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_push_consumer ) {
    _channel->_push_consumer->signal_push_threads();
  }
  RDI_OPLOCK_RELEASE;
}

CORBA::Boolean StructuredProxyPushSupplier_i::has_events(unsigned long* wait_s, unsigned long* wait_t)
{
  RDI_OPLOCK_ACQUIRE(return 0);
  if ( !_oplockptr || (_pxstate != RDI_Connected) || !_active || !_ntfqueue.length() ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }
  RDI_OPLOCK_RELEASE;
  return 1;
}

void StructuredProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit

  RDI_StructuredEvent* event = 0;;
  unsigned int         qsize = 0;
  CORBA::Boolean outcall_worked = 0;

  invalid = 0;
  if ( (_pxstate != RDI_Connected) || !_active || !_ntfqueue.length() ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }
  event = _ntfqueue.get_head();
  qsize = _ntfqueue.length() - 1;
  _ntfqueue.remove_head();
  _nevents += 1;

  // Do not hold OPLOCK across push
  RDI_OPLOCK_RELEASE;

  outcall_worked = 0;
  if ( RDI_LOCK_EVENT ) event->mylock().lock();
  try {
    _consumer->push_structured_event(event->get_cos_event());
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Consumer object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Exception while pushing event to consumer");
  }

  RDI_OPLOCK_REACQUIRE(return);

  if ( RDI_LOCK_EVENT ) event->mylock().unlock();
  event->decr_ref_counter(RDI_LOCK_EVENT);

  if (outcall_worked) {
    _channel->incr_num_notifications(qsize);
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during push -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      // unregister this proxy -- no more subscription_change msgs
      if ( ! _oc_off && _channel->_ochange_pool ) {
	_channel->_ochange_pool->remove_proxy(this);
      }
      _pxstate = RDI_Exception;
      invalid = 1;
    }
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
}

void StructuredProxyPushSupplier_i::_push_event()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  RDI_StructuredEvent* event = 0;
  unsigned int         qsize = 0;
  CORBA::Boolean       do_yield = 0, outcall_worked = 0;

  while ( 1 ) {
    do_yield = 1;
    while ( (_pxstate==RDI_NotConnected) ||
	    ((_pxstate==RDI_Connected)&&(!_active || !_ntfqueue.length())) ) {
      do_yield = 0;
      RDI_OPLOCK_WAIT;
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("push thread");
    }
    event = _ntfqueue.get_head();
    qsize = _ntfqueue.length() - 1;
    _ntfqueue.remove_head();
    _nevents += 1;

    // Do not hold OPLOCK across push
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    outcall_worked = 0;
    if ( RDI_LOCK_EVENT ) event->mylock().lock();
    try {
      _consumer->push_structured_event(event->get_cos_event());
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Consumer object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Communication failure while pushing event to consumer");
    } catch (...) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Exception while pushing event to consumer");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("push thread [**unexpected REACQUIRE failure**]"));

    if ( RDI_LOCK_EVENT ) event->mylock().unlock();
    event->decr_ref_counter(RDI_LOCK_EVENT);

    if (outcall_worked) {
      _channel->incr_num_notifications(qsize);
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during push -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	// unregister this proxy -- no more subscription_change msgs
	if ( ! _oc_off && _channel->_ochange_pool ) {
	  _channel->_ochange_pool->remove_proxy(this);
	}
	_pxstate = RDI_Exception;
      }
    }
  }
}

void StructuredProxyPushSupplier_i::add_event
(RDI_StructuredEvent* entry   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( (_pxstate == RDI_Connected) && entry ) {
    // Check if the 'MaxEventsPerConsumer' threshold has been reached.
    // In this case, we discard events based on 'DiscardPolicy' -- the
    // current implementation assumes FIFO 
    RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());	
    unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
    if ( msize && (_ntfqueue.length() >= msize) ) {
      RDI_StructuredEvent* evnt = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      evnt->decr_ref_counter(RDI_LOCK_EVENT);
      RDI_DUMP("MaxEventPerConsumer (" << msize << ") reached -- discarded oldest event in consumer's push queue");
    }
    entry->incr_ref_counter(RDI_LOCK_EVENT);
    _ntfqueue.insert_tail(entry);
    if ( _worker ) {
      RDI_OPLOCK_SIGNAL;
    }
    if ( _channel->_push_consumer ) {
      _channel->_push_consumer->signal_push_threads();
    }
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const StructuredProxyPushSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
    " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << (void*)prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << (void*)prx._lfilter;
  out << prx._pxstate;
  out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < prx._rqstypes.length(); ix++) {
      out << endl << "\t" << prx._rqstypes[ix].domain_name; 
      out << "::" << prx._rqstypes[ix].type_name;
    }
  }
  return out;
}

void StructuredProxyPushSupplier_i::disconnect_structured_push_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void StructuredProxyPushSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** Internal error -- StructuredProxyPushSupplier_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more offer_change msgs
  if ( ! _oc_off && (_pxstate == RDI_Connected) && _channel->_ochange_pool ) {
    _channel->_ochange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding push call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("StructuredProxyPushSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  if ( _rqstypes.length() == 0 ) { 
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  _active  = 0;
  _consumer =  CosNC_StructuredPushConsumer::_nil();
  RDI_StructuredEvent* event = 0;
  _pfilter = CosNF_MappingFilter::_nil();
  _lfilter = CosNF_MappingFilter::_nil();
  if ( _qosprop ) {
    delete _qosprop; _qosprop = 0;
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter(RDI_LOCK_EVENT);
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

// ------------------------------------------------------------- //
// SequenceProxyPushSupplier_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPushSupplier_i::SequenceProxyPushSupplier_i(
							 ConsumerAdmin_i* admin, 
							 EventChannel_i*  chann,
							 const CosNA_ProxyID&  prxid) : 
  RDIProxySupplier(admin, chann, RDI_S_SeqPRX, CosNA_PUSH_SEQUENCE, prxid),
  _worker(0), _ntfqueue(128, 128)
{
  unsigned long pacing_interval_s, pacing_interval_n;
  _consumer = CosNC_SequencePushConsumer::_nil();

  // If pacingInterval is non-zero, set initial timeout.
  // Note that the interval typically expires before any events are
  // placed in the notification queue of this proxy, which means
  // the first batch normally contains just a single event.
  // The only way to fix this would be to initialize the pacing
  // interval on the very first event insertion.
  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_TimeT2SecNanosec(qospr->pacingInterval, pacing_interval_s, pacing_interval_n);
  if (pacing_interval_s || pacing_interval_n) { // update timeout
    omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }

  // When the number of push threads allocated at the channel level is
  // 0, each proxy uses its own thread to push events to its consumer.
  if (  _channel->push_threads() == 0 ) {
    // _worker = new SeqProxyWorker(this,&SequenceProxyPushSupplier_i::_push_event);
    _worker = new SeqProxyBoundWorker(this,&SequenceProxyPushSupplier_i::_push_event);
    // If we use unbound threads, we should start the thread using
    // _worker->start();
    RDI_DUMP("PushSupplier Proxy Thread " << _worker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void SequenceProxyPushSupplier_i::connect_sequence_push_consumer
(CosNC_SequencePushConsumer_ptr consumer WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _active   = 1;
  _pxstate  = RDI_Connected;
  _consumer = CosNC_SequencePushConsumer::_duplicate(consumer);
  _oc_subscriber = CosNC_NotifyPublish::_narrow(_consumer); // implicit duplicate
  if (CORBA::is_nil(_oc_subscriber)) {
    RDI_DUMP("** UNEXPECTED: CosNC_SequencePushConsumer could not be narrowed to CosNC_NotifyPublish");
  }
  // if enabled, register this proxy for offer_change msgs
  if ( ! _oc_off && _channel->_ochange_pool ) {
    _channel->_ochange_pool->insert_proxy(this);
  }
  RDI_OPLOCK_RELEASE;
}

void SequenceProxyPushSupplier_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( ! _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyInactive();
  }
  _active = 0;
  RDI_OPLOCK_RELEASE;
}

void SequenceProxyPushSupplier_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_NotConnected();
  }
  if ( _active ) {
    RDI_OPLOCK_RELEASE;
    throw CosNA_ConnectionAlreadyActive();
  }
  _active = 1;
  if ( _worker ) { 
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_push_consumer ) {
    _channel->_push_consumer->signal_push_threads();
  }
  RDI_OPLOCK_RELEASE;
}



// has_events determines whether a push should occur.
// in this case the requirements are that either the max batch size
// has been reached or there is at least 1 event and pacing interval reached
CORBA::Boolean SequenceProxyPushSupplier_i::has_events(unsigned long* wait_s, unsigned long* wait_n) {
  RDI_OPLOCK_ACQUIRE(return 0);

  CORBA::Long qsize = _ntfqueue.length();
  if ( (_pxstate != RDI_Connected) || (!_active) || (qsize == 0) ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }

  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  CORBA::Long bsize = qospr->maximumBatchSize;

  if (qsize >= bsize) {
    RDI_OPLOCK_RELEASE;
    return 1;
  }

  // pacing interval : both 0 means always wait for full batch size
  unsigned long pacing_interval_s, pacing_interval_n;
  unsigned long time_s, time_n;
  RDI_TimeT2SecNanosec(qospr->pacingInterval, pacing_interval_s, pacing_interval_n);
  if ( (pacing_interval_s == 0) && (pacing_interval_n == 0) ) {
    _timeout_s = 0; _timeout_n = 0;
    RDI_OPLOCK_RELEASE;
    return 0; // no pacing interval
  }

  omni_thread::get_time(&time_s, &time_n);

  if ( (_timeout_s == 0) && (_timeout_n == 0) ) {
    // proxy has not established timeout yet : set it now
    omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
  }

  if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
    RDI_OPLOCK_RELEASE;
    return 1; // pacing interval timeout occurred for this proxy
  }

  // update wait_s/_n to reflect the future timeout point of this proxy
  if ( ( ((*wait_s) == 0) && ((*wait_n) == 0)                   ) ||
       ( ((*wait_s) > _timeout_s)                               ) ||
       ( ((*wait_s) == _timeout_s) && ((*wait_n) > _timeout_n)) ) {
    (*wait_s) = _timeout_s;
    (*wait_n) = _timeout_n;
  }
  RDI_OPLOCK_RELEASE;
  return 0;
}

// REQUIREMENT: A thread from push pool only calls this method
// if has_events() is true and it thinks state is connected and active.
// (For safety we verify that these conditions actually hold.)
void SequenceProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit 

  CORBA::Long qsize = _ntfqueue.length();
  if ( (_pxstate != RDI_Connected) || (!_active) || (qsize == 0) ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }

  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  // pacing interval : both 0 means always wait for full batch size
  unsigned long pacing_interval_s, pacing_interval_n;
  unsigned long time_s, time_n;
  RDI_TimeT2SecNanosec(qospr->pacingInterval, pacing_interval_s, pacing_interval_n);
  CORBA::Long   i, bsize = qospr->maximumBatchSize;
  RDI_StructuredEvent** event = 0;
  CORBA::Long actsize;
  CORBA::Boolean outcall_worked = 0;

  invalid = 0;
  if ( (qsize < bsize) && (pacing_interval_s == 0) && (pacing_interval_n == 0) ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }
  // if queue size < bsize, verify that a timeout really occurred
  if ( qsize < bsize) { // one/both of pacing_interval_s/_n must be non-zero
    omni_thread::get_time(&time_s, &time_n);
    if ((_timeout_s == 0) && (_timeout_n == 0)) { // timeout not set; set it now
      omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
    } 
    if ( (time_s < _timeout_s) || ((time_s == _timeout_s) && (time_n < _timeout_n)) ) {
      RDI_DUMP("** INTERNAL ERROR: SequenceProxyPushSupplier_i::push_event called too soon");
      RDI_OPLOCK_DEBUMP_RELEASE;
      return;
    }
  }

  actsize = (qsize < bsize) ? qsize : bsize;
  CosN_EventBatch notif(actsize);
  notif.length(actsize);

  event = new RDI_StructuredEvent * [actsize];
  RDI_AssertAllocThrowNo(event, "Memory allocation failed -- RDI_StructuredEvent");

  for ( i = 0; i < actsize; i++ ) {
    event[i] = _ntfqueue.get_head();
    _ntfqueue.remove_head();
    if ( RDI_LOCK_EVENT ) event[i]->mylock().lock();
    notif[i] = event[i]->get_cos_event();
  }
  _nevents += actsize;
  // update timeout before releasing OPLOCK -- this means we do the update
  // before doing the push (seems OK)
  if (pacing_interval_s || pacing_interval_n) {
    omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }

  // Do not hold OPLOCK across push
  RDI_OPLOCK_RELEASE;

  outcall_worked = 0;
  try {
    _consumer->push_structured_events(notif);
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Consumer object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Exception while pushing event to consumer");
  }

  RDI_OPLOCK_REACQUIRE(return);

  for ( i = 0; i < actsize; i++ ) {
    if ( RDI_LOCK_EVENT ) event[i]->mylock().unlock();
    event[i]->decr_ref_counter(RDI_LOCK_EVENT);
    event[i] = 0;
  }
  delete [] event;
  if (outcall_worked) {
    for ( i = 0; i < actsize; i++ ) {
      _channel->incr_num_notifications(--qsize);
    }
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during push -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      // unregister this proxy -- no more subscription_change msgs
      if ( ! _oc_off && _channel->_ochange_pool ) {
	_channel->_ochange_pool->remove_proxy(this);
      }
      _pxstate = RDI_Exception;
      invalid = 1;
    }
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
}

void SequenceProxyPushSupplier_i::_push_event()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  // pacing interval : both 0 means always wait for full batch size
  unsigned long pacing_interval_s, pacing_interval_n;
  unsigned long time_s, time_n;
  CORBA::Boolean update_timeout = 0, do_yield = 0, outcall_worked = 0;

  // get initial qospr and bsize, setup initial notif and event based on bsize
  RDI_NotifQoS* qospr =  (_qosprop ? _qosprop : _myadmin->qos_properties());
  CORBA::Long  i, bsize = qospr->maximumBatchSize;;

  CosN_EventBatch notif(bsize);
  RDI_StructuredEvent** event = new RDI_StructuredEvent*[bsize];
  RDI_AssertAllocThrowNo(event, "Memory allocation failed -- RDI_StructuredEvent");
  CORBA::Long eventsize = bsize;
  CORBA::Long qsize, actsize;

  while ( 1 ) {
    do_yield = 1;
    while ( 1 ) {
      // must recompute these here because they can change across a wait/timedwait
      qsize = _ntfqueue.length();
      qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
      RDI_TimeT2SecNanosec(qospr->pacingInterval, pacing_interval_s, pacing_interval_n);
      if (pacing_interval_s || pacing_interval_n) {
	// if timeout not set, or if update_timeout true due to push, recompute timeout
	if ( update_timeout || ((_timeout_s == 0) && (_timeout_n == 0)) ) {
	  omni_thread::get_time(&_timeout_s, &_timeout_n, pacing_interval_s, pacing_interval_n);
	}
      } else {
	_timeout_s = 0; _timeout_n = 0;
      }
      update_timeout = 0;
      bsize = qospr->maximumBatchSize;
      if (bsize > eventsize) { // need a larger array
	delete [] event;
	event = new RDI_StructuredEvent*[bsize];
	RDI_AssertAllocThrowNo(event, "Memory allocation failed -- RDI_StructuredEvent");
	eventsize = bsize;
      }
      // breaking from inner loop requires either a sufficient batch size or a
      // pacing interval timeout (and state must be RDI_Connected and _active)
      if ((_pxstate!=RDI_NotConnected) && (_pxstate!=RDI_Connected)) {
	break; // also break on exceptional state
      }
      if ((_pxstate==RDI_Connected) && _active) {
	if ( qsize >= bsize )
	  break;
	if ( qsize && (pacing_interval_s || pacing_interval_n) ) {
	  omni_thread::get_time(&time_s, &time_n);
	  if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
	    // pacing interval has passed so go ahead and send a smaller batch
	    break;
	  }
	}
      }
      // must wait
      if ( qsize && (pacing_interval_s || pacing_interval_n) ) {
	//	RDI_DUMP("sequence proxy doing timedwait(" << _timeout_s << "," << _timeout_n <<")");
	do_yield = 0;
	RDI_OPLOCK_TIMEDWAIT(_timeout_s, _timeout_n);
      } else {
	//	RDI_DUMP("sequence proxy doing normal wait");
	do_yield = 0;
	RDI_OPLOCK_WAIT;
      }
    }
    if ( _pxstate != RDI_Connected ) {
      delete [] event;
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("push thread");
    }
    actsize = (qsize < bsize) ? qsize : bsize;
    notif.length(actsize);
    for ( i = 0; i < actsize; i++ ) {
      event[i] = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      if ( RDI_LOCK_EVENT ) event[i]->mylock().lock();
      notif[i] = event[i]->get_cos_event();
    }
    _nevents += actsize;

    // Do not hold OPLOCK across push
    RDI_OPLOCK_RELEASE;
    
    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    outcall_worked = 0;
    try {
      _consumer->push_structured_events(notif);
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Consumer object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Communication Failure");
    } catch (...) {
      RDI_DUMP("Push thread for proxy " << (void*)this << " - Exception while pushing event to consumer");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("push thread [**unexpected REACQUIRE failure**]"));

    update_timeout = 1; // force update of timeout at top of loop

    for ( i = 0; i < actsize; i++ ) {
      if ( RDI_LOCK_EVENT ) event[i]->mylock().unlock();
      event[i]->decr_ref_counter(RDI_LOCK_EVENT);
      event[i] = 0;
    }

    if (outcall_worked) {
      for ( i = 0; i < actsize; i++ ) {
	_channel->incr_num_notifications(--qsize);
      }
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during push -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	// unregister this proxy -- no more subscription_change msgs
	if ( ! _oc_off && _channel->_ochange_pool ) {
	  _channel->_ochange_pool->remove_proxy(this);
	}
	_pxstate = RDI_Exception;
      }
    }
  }
}

void SequenceProxyPushSupplier_i::add_event
(RDI_StructuredEvent* entry   WRAPPED_IMPLARG )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( (_pxstate == RDI_Connected) && entry ) {
    // Check if the 'MaxEventsPerConsumer' threshold has been reached.
    // In this case, we discard events based on 'DiscardPolicy' -- the
    // current implementation assumes FIFO 
    RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());	
    unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
    if ( msize && (_ntfqueue.length() >= msize) ) {
      RDI_StructuredEvent* evnt = _ntfqueue.get_head();
      _ntfqueue.remove_head();
      evnt->decr_ref_counter(RDI_LOCK_EVENT);
      RDI_DUMP("MaxEventPerConsumer (" << msize << ") reached -- discarded oldest event in consumer's push queue");
    }
    entry->incr_ref_counter(RDI_LOCK_EVENT);
    _ntfqueue.insert_tail(entry);
    if ( _worker ) { 
      RDI_OPLOCK_SIGNAL;
    }
    if ( _channel->_push_consumer ) {
      _channel->_push_consumer->signal_push_threads();
    }
  }
  RDI_OPLOCK_RELEASE;
}

ostream& operator << (ostream& out, const SequenceProxyPushSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
    " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << (void*)prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << (void*)prx._lfilter;
  out << prx._pxstate;
  out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
    for (CORBA::ULong ix = 0; ix < prx._rqstypes.length(); ix++) {
      out << endl << "\t" << prx._rqstypes[ix].domain_name; 
      out << "::" << prx._rqstypes[ix].type_name;
    }
  }
  return out;
}

void SequenceProxyPushSupplier_i::disconnect_sequence_push_supplier( WRAPPED_IMPLARG_VOID )
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if (_pxstate == RDI_Disconnected) { RDI_OPLOCK_RELEASE; RDI_THROW_INV_OBJREF; } // already in process of being disposed
  _disconnect_client_and_dispose(1);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void SequenceProxyPushSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("** Internal error -- SequenceProxyPushSupplier_i::_disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  // unregister this proxy -- no more offer_change msgs
  if ( ! _oc_off && (_pxstate == RDI_Connected) && _channel->_ochange_pool ) {
    _channel->_ochange_pool->remove_proxy(this);
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding push call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("SequenceProxyPushSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  if ( _rqstypes.length() == 0 ) { 
    _fa_helper.remove_all_filters();
  } else {
    CosN_EventTypeSeq added; added.length(0);
    (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }
  _active  = 0;
  _consumer =  CosNC_SequencePushConsumer::_nil();
  _pfilter = CosNF_MappingFilter::_nil();
  _lfilter = CosNF_MappingFilter::_nil();
  if ( _qosprop ) {
    delete _qosprop; _qosprop = 0;
  }
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter(RDI_LOCK_EVENT);
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
}

