// -*- Mode: C++; -*-
//                              File      : CosNotification.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of QoS and Admin Properties
//
 
/*
$Log: CosNotification.cc,v $
Revision 1.12  2000/11/05 04:48:10  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.11  2000/10/31 11:17:05  alcfp
small changes to make defaults follow the spec

Revision 1.10  2000/10/30 04:39:30  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.9  2000/10/04 15:15:06  alcfp
more small updates to get rid of compiler warnings

Revision 1.8  2000/09/27 19:30:11  alcfp
Stopped using tcAliasExpand=1 : dynamic evaluation now uses typecode equivalence test or DynFoo narrow test rather than checking for a specific TCKind

Revision 1.7  2000/08/22 18:23:53  alcfp
added description to each file

Revision 1.6  2000/08/16 20:19:26  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include "RDITimeDefs.h"
#include "RDIDebug.h"
#include "RDIBitmap.h"
#include "CosNotification_i.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//                               QoS Properties                          //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

// XXX The specification says AnyOrder is the default for
// the two ordering properties.  We only support FifoOrder at the moment.
RDI_NotifQoS::RDI_NotifQoS() : 
		eventReliability(CosN_BestEffort),
	       	connectionReliability(CosN_BestEffort),
                priority(CosN_DefaultPriority), 
                timeout(), startTimeSupported(0), stopTimeSupported(0),
                orderPolicy(CosN_FifoOrder),
                discardPolicy(CosN_FifoOrder),
                pacingInterval(), maxEventsPerConsumer(0), 
                maximumBatchSize(8)
{
  timeout.low = 0;
  timeout.high = 0;
  // our choice for default pacing interval = 1 second
  RDI_SecNanosec2TimeT(1,0,pacingInterval);
}

RDI_NotifQoS& RDI_NotifQoS::operator = (const RDI_NotifQoS& qos)
{
  eventReliability      = qos.eventReliability;
  connectionReliability = qos.connectionReliability;
  priority              = qos.priority;
  timeout               = qos.timeout;
  startTimeSupported    = qos.startTimeSupported;
  stopTimeSupported     = qos.stopTimeSupported;
  orderPolicy           = qos.orderPolicy;
  discardPolicy         = qos.discardPolicy;
  pacingInterval        = qos.pacingInterval;
  maxEventsPerConsumer  = qos.maxEventsPerConsumer;
  maximumBatchSize      = qos.maximumBatchSize;
  return *this;
}

CosN_QoSProperties* 
RDI_NotifQoS::get_qos(RDI_ObjectKind otype) const
{
  CosN_QoSProperties* qos = new CosN_QoSProperties();
  RDI_Assert(qos, "Memory allocation failed - CosN_QoSProperties");
  CORBA::ULong idx=0;
  CORBA::Any::from_boolean strt(startTimeSupported);
  CORBA::Any::from_boolean stop(stopTimeSupported);

  if ( otype == RDI_ECHANNEL ) {
     qos->length(RDI_NotifQoS::NUM_PROPERTIES);
     (*qos)[idx].name = CORBA::string_dup(CosN_EventReliability);
     (*qos)[idx++].value <<= eventReliability;
  } else {
     qos->length(RDI_NotifQoS::NUM_PROPERTIES - 1);
  }
  (*qos)[idx].name = CORBA::string_dup(CosN_ConnectionReliability);
  (*qos)[idx++].value <<= connectionReliability;
  (*qos)[idx].name = CORBA::string_dup(CosN_Priority);
  (*qos)[idx++].value <<= priority;
  (*qos)[idx].name = CORBA::string_dup(CosN_Timeout);
  (*qos)[idx++].value <<= timeout;
  (*qos)[idx].name = CORBA::string_dup(CosN_StartTimeSupported);
  (*qos)[idx++].value <<= strt;
  (*qos)[idx].name = CORBA::string_dup(CosN_StopTimeSupported);
  (*qos)[idx++].value <<= stop;
  (*qos)[idx].name = CORBA::string_dup(CosN_OrderPolicy);
  (*qos)[idx++].value <<= orderPolicy;
  (*qos)[idx].name = CORBA::string_dup(CosN_DiscardPolicy);
  (*qos)[idx++].value <<= discardPolicy;
  (*qos)[idx].name = CORBA::string_dup(CosN_PacingInterval);
  (*qos)[idx++].value <<= pacingInterval;
  (*qos)[idx].name = CORBA::string_dup(CosN_MaxEventsPerConsumer);
  (*qos)[idx++].value <<= maxEventsPerConsumer;
  (*qos)[idx].name = CORBA::string_dup(CosN_MaximumBatchSize);
  (*qos)[idx++].value <<= maximumBatchSize;
  return qos;
}

void RDI_NotifQoS::set_qos(const CosN_QoSProperties& r_qos)
{
  TimeBase::TimeT* ull_ptr;
  
  for (CORBA::ULong ix=0; ix < r_qos.length(); ix++) {
    if ( strcmp(r_qos[ix].name, "EventReliability") == 0 ) {
      r_qos[ix].value >>= eventReliability;
    } else if ( strcmp(r_qos[ix].name, "ConnectionReliability") == 0 ) {
      r_qos[ix].value >>= connectionReliability;
    } else if ( strcmp(r_qos[ix].name, "Priority") == 0 ) {
      r_qos[ix].value >>= priority;
    } else if ( strcmp(r_qos[ix].name, "Timeout") == 0 ) {
      // this assumed -DNOLONGLONG used as omniidl flag
      r_qos[ix].value >>= ull_ptr;
      timeout = *ull_ptr;
    } else if ( strcmp(r_qos[ix].name, "StartTimeSupported") == 0 ) {
      CORBA::Any::to_boolean strt(startTimeSupported);
      r_qos[ix].value >>= strt;
    } else if ( strcmp(r_qos[ix].name, "StopTimeSupported") == 0 ) {
      CORBA::Any::to_boolean stop(stopTimeSupported);
      r_qos[ix].value >>= stop;
    } else if ( strcmp(r_qos[ix].name, "OrderPolicy") == 0 ) {
      r_qos[ix].value >>= orderPolicy;
    } else if ( strcmp(r_qos[ix].name, "DiscardPolicy") == 0 ) {
      r_qos[ix].value >>= discardPolicy;
    } else if ( strcmp(r_qos[ix].name, "PacingInterval") == 0 ) {
      r_qos[ix].value >>= ull_ptr;
      pacingInterval = *ull_ptr;
    } else if ( strcmp(r_qos[ix].name, "MaxEventsPerConsumer") == 0 ) {
      r_qos[ix].value >>= maxEventsPerConsumer;
    } else if ( strcmp(r_qos[ix].name, "MaximumBatchSize") == 0 ) {
      r_qos[ix].value >>= maximumBatchSize;
    }
  }
}

ostream& operator << (ostream& out, const RDI_NotifQoS& qos)
{
  unsigned long timeout_s, timeout_n;
  unsigned long pacing_interval_s, pacing_interval_n;
  RDI_TimeT2SecNanosec(qos.timeout, timeout_s, timeout_n);
  RDI_TimeT2SecNanosec(qos.pacingInterval, pacing_interval_s, pacing_interval_n);

  out << "EventReliability     "     << qos.eventReliability <<
    " | ConnectionReliability " << qos.connectionReliability <<
    " | Priority          "     << qos.priority << endl <<
    "Timeout(s,n)     (" << timeout_s << "," << timeout_n << ")" << 
    " | StartTimeSupported    " << (qos.startTimeSupported?1:0) <<
    " | StopTimeSupported "     << (qos.stopTimeSupported?1:0) << endl << 
    "OrderPolicy          "     << qos.orderPolicy << 
    " | DiscardPolicy         " << qos.discardPolicy << 
    " | PacingInterval(s,n) (" << pacing_interval_s << "," << pacing_interval_n << ")" << endl << 
    "MaxEventsPerConsumer "     << qos.maxEventsPerConsumer << 
    " | MaximumBatchSize      " << qos.maximumBatchSize;
  return out;
}

/** RDI_NotifQoS::validate
  *
  * Validate the requested QoS properties, 'r_qos', for a given object.
  * The current QoS properties of this object are 'a_qos', and its kind
  * is 'otype'.  In case the object has a parent, the QoS properties of
  * its parent, 'p_qos', are used to resolve conflicts.  Since some QoS
  * properties cannot be changed once set for objects that manage other
  * objects, 'child' is TRUE when the current object has sub-objects.
  */

CORBA::Boolean RDI_NotifQoS::validate(
			const CosN_QoSProperties& r_qos,
		        const RDI_NotifQoS&  a_qos,
			      RDI_ObjectKind otype,
			      CosN_PropertyErrorSeq& error,
			      CosN_NamedPropertyRangeSeq& range,
			      CORBA::Boolean child, RDI_NotifQoS*  p_qos)
{
  CORBA::Boolean succ=1;
  CORBA::Boolean bval=0;
  CORBA::Short   sval=0;
  CORBA::Long    lval=0;
  CORBA::ULong   ix_e=0;
  RDI_Bitmap     smap(RDI_NotifQoS::NUM_PROPERTIES);

  smap.clear();
  range.length(RDI_NotifQoS::NUM_PROPERTIES);
  error.length(r_qos.length());

  for (CORBA::ULong ix=0; ix < r_qos.length(); ix++) {
     CORBA::TypeCode_var r_tc = r_qos[ix].value.type();

     if ( strcmp(r_qos[ix].name, "EventReliability") == 0 ) {
	smap.set((unsigned int) 0);
	if ( ! CORBA::_tc_short->equivalent(r_tc) ) {
	   error[ix_e].code = CosN_BAD_TYPE;
	   error[ix_e].name = r_qos[ix].name;
	   ix_e += 1; succ  = 0; continue;
	} else if ( (otype != RDI_ECHANNEL) && (otype != RDI_EMESSAGE) ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
	   error[ix_e].name = r_qos[ix].name;
	   ix_e += 1; succ  = 0; continue;
	}
	r_qos[ix].value >>= sval;
	if ( (sval != CosN_BestEffort) &&
	     (sval != CosN_Persistent) ) {
	   error[ix_e].code = CosN_BAD_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= CosN_BestEffort;
	   error[ix_e].available_range.high_val <<= a_qos.eventReliability;
	   ix_e += 1; succ  = 0; continue;
	}
	// If the current object has already created some children objects, 
	// the requested value is considered invalid. The same is true when
	// there are no children and the parent, if any, has a weaker value
	if ( (sval != a_qos.eventReliability) && 
	     (child || (p_qos && (p_qos->eventReliability < sval))) ) {
	   error[ix_e].code = CosN_UNAVAILABLE_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= a_qos.eventReliability;
	   error[ix_e].available_range.high_val <<= a_qos.eventReliability;
	   ix_e += 1; succ  = 0; continue;
	}
     } else if ( strcmp(r_qos[ix].name, "ConnectionReliability") == 0 ) {
	smap.set((unsigned int) 1);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
	   error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
	}
	if ( ! CORBA::_tc_short->equivalent(r_tc) ) {
	   error[ix_e].code = CosN_BAD_TYPE;
	   error[ix_e].name = r_qos[ix].name;
	   ix_e += 1; succ  = 0; continue;
	}
	r_qos[ix].value >>= sval;
	if ( (sval != CosN_BestEffort) &&
             (sval != CosN_Persistent) ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= CosN_BestEffort;
           error[ix_e].available_range.high_val <<= a_qos.connectionReliability;
           ix_e += 1; succ  = 0; continue;
        }
	// If the current object has already created some children objects, 
        // the requested value is considered invalid. The same is true when
        // there are no children and the parent, if any, has a weaker value
        if ( (sval != a_qos.connectionReliability) &&  
             (child || (p_qos && (p_qos->connectionReliability < sval))) ) {
           error[ix_e].code = CosN_UNAVAILABLE_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= a_qos.connectionReliability;
           error[ix_e].available_range.high_val <<= a_qos.connectionReliability;
           ix_e += 1; succ  = 0; continue;
        }
     } else if ( strcmp(r_qos[ix].name, "Priority") == 0 ) {
	smap.set((unsigned int) 2);
	if ( ! CORBA::_tc_short->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	r_qos[ix].value >>= sval;
	if ( (sval < CosN_LowestPriority) ||
	     (sval > CosN_HighestPriority) ) {
	   error[ix_e].code = CosN_BAD_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= 
					CosN_LowestPriority;
	   error[ix_e].available_range.high_val <<= 
					CosN_HighestPriority;
	   ix_e += 1; succ  = 0; continue;
	}
     } else if ( strcmp(r_qos[ix].name, "StartTime") == 0 ) {
	if ( otype != RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! a_qos.startTimeSupported ) {
	   error[ix_e].code = CosN_UNSUPPORTED_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
	}
     } else if ( strcmp(r_qos[ix].name, "StopTime") == 0 ) {
	if ( otype != RDI_EMESSAGE ) {
           error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
        if ( ! a_qos.stopTimeSupported ) {
           error[ix_e].code = CosN_UNSUPPORTED_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
     } else if ( strcmp(r_qos[ix].name, "Timeout") == 0 ) {
	smap.set((unsigned int) 3);
	// ** For the moment, we just IGNORE this property since we **
	// ** do not provide support for this in the current system **
	// error[ix_e].code = CosN_UNSUPPORTED_PROPERTY;
	// error[ix_e].name = r_qos[ix].name;
	// ix_e += 1; succ  = 0; continue;
     } else if ( strcmp(r_qos[ix].name, "StartTimeSupported") == 0 ) {
	CORBA::Any::to_boolean   t_bvl(bval);
	CORBA::Any::from_boolean f_bvl(a_qos.startTimeSupported);
	smap.set((unsigned int) 4);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! CORBA::_tc_boolean->equivalent(r_tc) ) {
	   error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	r_qos[ix].value >>= t_bvl;
	if ( (bval != a_qos.startTimeSupported) && 
	     p_qos && (bval != p_qos->startTimeSupported) ) {
	   error[ix_e].code = CosN_BAD_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= f_bvl;
	   error[ix_e].available_range.high_val <<= f_bvl;
	   ix_e += 1; succ  = 0; continue;
	}
     } else if ( strcmp(r_qos[ix].name, "StopTimeSupported") == 0 ) {
	CORBA::Any::to_boolean   t_bvl(bval);
	CORBA::Any::from_boolean f_bvl(a_qos.stopTimeSupported);
	smap.set((unsigned int) 5);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! CORBA::_tc_boolean->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
        r_qos[ix].value >>= t_bvl;
        if ( (bval != a_qos.stopTimeSupported) && 
             p_qos && (bval != p_qos->stopTimeSupported) ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= f_bvl;
           error[ix_e].available_range.high_val <<= f_bvl;
           ix_e += 1; succ  = 0; continue;
        }
     } else if ( strcmp(r_qos[ix].name, "MaxEventsPerConsumer") == 0 ) {
	smap.set((unsigned int) 6);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! CORBA::_tc_long->equivalent(r_tc) ) {
	   error[ix_e].code = CosN_BAD_TYPE;
	   error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	// The CORBA Notification Specification states that setting
	// this property on ProxyConsumer and SupplierAdmin objects
	// has no meaning.  Consequently, we do not take any action
     } else if ( strcmp(r_qos[ix].name, "OrderPolicy") == 0 ) {
	smap.set((unsigned int) 7);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! CORBA::_tc_short->equivalent(r_tc) ) {
	   error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	r_qos[ix].value >>= sval;
	if ( (sval < CosN_AnyOrder) ||
	     (sval > CosN_DeadlineOrder) ) {
	   error[ix_e].code = CosN_BAD_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= CosN_AnyOrder;
	   error[ix_e].available_range.high_val <<= a_qos.orderPolicy;
	   ix_e += 1; succ  = 0; continue;
	}
	// Here, we assume that a setting of CosN_AnyOrder
	// at the object or the parent implies that any setting can be
	// used.  However, if this is not the case, only the currently
	// set value is supported
	if ( (sval != a_qos.orderPolicy) && 
	     (a_qos.orderPolicy != CosN_AnyOrder) && 
	     p_qos && (p_qos->orderPolicy != CosN_AnyOrder) ) {
	   error[ix_e].code = CosN_UNSUPPORTED_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= CosN_AnyOrder;
	   error[ix_e].available_range.high_val <<= a_qos.orderPolicy;
	   ix_e += 1; succ  = 0; continue;
	}
	// Setting 'OrderPolicy' on ProxyConsumer and SupplierAdmin
	// objects has no meaning  in our implementation.  However,
	// there is no provision for this in the CORBA Notification
	// Specification and, hence, we just do not take any action
     } else if ( strcmp(r_qos[ix].name, "DiscardPolicy") == 0 ) {
	smap.set((unsigned int) 8);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! CORBA::_tc_short->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
        r_qos[ix].value >>= sval;
        if ( (sval < CosN_AnyOrder) ||
             (sval > CosN_LifoOrder) ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= CosN_AnyOrder;
           error[ix_e].available_range.high_val <<= a_qos.discardPolicy;
           ix_e += 1; succ  = 0; continue;
        } 
        // Here, we assume that a setting of CosN_AnyOrder
        // at the object or the parent implies that any setting can be 
        // used.  However, if this is not the case, only the currently
        // set value is supported
        if ( (sval != a_qos.discardPolicy) && 
             (a_qos.orderPolicy != CosN_AnyOrder) && 
             p_qos && (p_qos->discardPolicy != CosN_AnyOrder) ) {
           error[ix_e].code = CosN_UNSUPPORTED_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= CosN_AnyOrder;
           error[ix_e].available_range.high_val <<= a_qos.discardPolicy;
           ix_e += 1; succ  = 0; continue;
        }
        // Setting 'DiscardPolicy' on ProxyConsumer and SupplierAdmin 
        // objects has no meaning in our implementation. However, the
	// CORBA Notification Specification does not address it and, 
	// hence, we just do not take any action
     } else if ( strcmp(r_qos[ix].name, "MaximumBatchSize") == 0 ) {
	smap.set((unsigned int) 9);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	if ( ! CORBA::_tc_long->equivalent(r_tc) ) {
	   error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	r_qos[ix].value >>= lval;
	if ( lval <= 0 ) {
	   error[ix_e].code = CosN_BAD_VALUE;
	   error[ix_e].name = r_qos[ix].name;
	   error[ix_e].available_range.low_val  <<= (CORBA::Long) 1;
	   error[ix_e].available_range.high_val <<= (CORBA::Long) 1000000;
	   ix_e += 1; succ  = 0; continue;
	}
	// Setting 'MaximumBatchSize' on ProxyConsumer and SupplierAdmin
        // objects has no meaning in our implementation. Also, the CORBA
	// Notification Specification states that it only applies to the
	// sequence style proxies -- we do not raise an exception here..
     } else if ( strcmp(r_qos[ix].name, "PacingInterval") == 0 ) {
	smap.set((unsigned int) 10);
	if ( otype == RDI_EMESSAGE ) {
	   error[ix_e].code = CosN_UNAVAILABLE_PROPERTY;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; continue;
        }
	// ** For the moment, we just IGNORE this property since we **
	// ** do not provide support for this in the current system **
	// error[ix_e].code = CosN_UNSUPPORTED_PROPERTY;
        // error[ix_e].name = r_qos[ix].name;
        // ix_e += 1; succ  = 0; continue;
     } else {
	error[ix_e].code = CosN_BAD_PROPERTY;
	error[ix_e].name = r_qos[ix].name;
	ix_e += 1; succ  = 0;
	RDI_DUMP("BAD QoS Property: " << r_qos[ix].name);
     }
  }
  error.length(ix_e);   // Correct the length of this list

  // If no error occured, populate any additional properties that could
  // have been requested and supported

  ix_e = 0;
  if ( succ && (smap.num_set() != RDI_NotifQoS::NUM_PROPERTIES) ) {
     if ( ! smap.is_set((unsigned int) 0) && 
	  ((otype == RDI_ECHANNEL) || (otype == RDI_EMESSAGE)) ) {
	range[ix_e].name = CosN_EventReliability;
	range[ix_e].range.low_val    <<= CosN_BestEffort;
	range[ix_e++].range.high_val <<= a_qos.eventReliability;
     }
     if ( ! smap.is_set((unsigned int) 1) && (otype != RDI_EMESSAGE) ) {
	range[ix_e].name = CosN_ConnectionReliability;
	range[ix_e].range.low_val    <<= CosN_BestEffort;
	range[ix_e++].range.high_val <<= a_qos.connectionReliability;
     }
     if ( ! smap.is_set((unsigned int) 2) ) {
	range[ix_e].name = CosN_Priority;
	range[ix_e].range.low_val    <<= CosN_LowestPriority;
	range[ix_e++].range.high_val <<= CosN_HighestPriority;
     }
     if ( ! smap.is_set((unsigned int) 3) ) {
	range[ix_e].name = CosN_Timeout;
	// range[ix_e].range.low_val    <<= ???
	// range[ix_e++].range.high_val <<= ???
     } 
     if ( ! smap.is_set((unsigned int) 4) && (otype != RDI_EMESSAGE) ) {
	CORBA::Any::from_boolean f_bvl(a_qos.startTimeSupported);
	range[ix_e].name = CosN_StartTimeSupported;
	range[ix_e].range.low_val    <<= f_bvl;
	range[ix_e++].range.high_val <<= f_bvl;
     }
     if ( ! smap.is_set((unsigned int) 5) && (otype != RDI_EMESSAGE) ) {
	CORBA::Any::from_boolean f_bvl(a_qos.stopTimeSupported);
	range[ix_e].name = CosN_StopTimeSupported;
	range[ix_e].range.low_val    <<= f_bvl;
	range[ix_e++].range.high_val <<= f_bvl;
     }
     if ( ! smap.is_set((unsigned int) 6) && 
	  (otype != RDI_C_AnyPRX) && (otype != RDI_C_StrPRX) && 
	  (otype != RDI_C_SeqPRX) && (otype != RDI_EMESSAGE) ) {
        range[ix_e].name = CosN_MaxEventsPerConsumer;
        range[ix_e].range.low_val    <<= (CORBA::Long) 1;
        range[ix_e++].range.high_val <<= (CORBA::Long) 0;
     }
     if ( ! smap.is_set((unsigned int) 7) && (otype != RDI_S_ADMIN) && 
	  (otype != RDI_C_AnyPRX) && (otype != RDI_C_StrPRX) && 
	  (otype != RDI_C_SeqPRX) && (otype != RDI_EMESSAGE) ) {
	range[ix_e].name = CosN_OrderPolicy;
	if ( a_qos.orderPolicy == CosN_AnyOrder ) {
	   range[ix_e].range.low_val    <<= CosN_AnyOrder;
	   range[ix_e++].range.high_val <<= CosN_DeadlineOrder;
	} else {
	   range[ix_e].range.low_val    <<= a_qos.orderPolicy;
	   range[ix_e++].range.high_val <<= a_qos.orderPolicy;
	}
     }
     if ( ! smap.is_set((unsigned int) 8) && (otype != RDI_S_ADMIN) && 
	  (otype != RDI_C_AnyPRX) && (otype != RDI_C_StrPRX) && 
	  (otype != RDI_C_SeqPRX) && (otype != RDI_EMESSAGE) ) {
        range[ix_e].name = CosN_DiscardPolicy;
	if ( a_qos.discardPolicy == CosN_AnyOrder ) {
	   range[ix_e].range.low_val    <<= CosN_AnyOrder;
	   range[ix_e++].range.high_val <<= CosN_LifoOrder;
	} else {
	   range[ix_e].range.low_val    <<= a_qos.discardPolicy;
	   range[ix_e++].range.high_val <<= a_qos.discardPolicy;
	}
     }
     if ( ! smap.is_set((unsigned int) 9) && ((otype == RDI_ECHANNEL) || 
	  (otype == RDI_S_SeqPRX) || (otype == RDI_C_ADMIN)) ) {
	range[ix_e].name = CosN_MaximumBatchSize;
	range[ix_e].range.low_val    <<= (CORBA::Long) 1;
	range[ix_e++].range.high_val <<= (CORBA::Long) 1000000;
     }
     if ( ! smap.is_set((unsigned int) 10) && ((otype == RDI_ECHANNEL) ||
	  (otype == RDI_S_SeqPRX) || (otype == RDI_C_ADMIN)) ) {
	range[ix_e].name = CosN_PacingInterval;
	// range[ix_e].range.low_val    <<= (TimeBase::TimeT) 0;
	// range[ix_e++].range.high_val <<= a_qos.pacingInterval;
     }
  }
  range.length(ix_e);

  return succ;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//                          AdminQoS Properties                          //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

CosN_AdminProperties* RDI_AdminQoS::to_admin() const
{
  CORBA::Any::from_boolean f_bvl(rejectNewEvents);
  CosN_AdminProperties* aqos = new CosN_AdminProperties;
  RDI_Assert(aqos, "Memory allocation failure - AdminProperties");
  aqos->length(12);
  (*aqos)[0].name    = CORBA::string_dup("MaxQueueLength");
  (*aqos)[0].value <<= maxQueueLength;
  (*aqos)[1].name    = CORBA::string_dup("MaxConsumers");
  (*aqos)[1].value <<= maxConsumers;
  (*aqos)[2].name    = CORBA::string_dup("MaxSuppliers");
  (*aqos)[2].value <<= maxSuppliers;
  (*aqos)[3].name    = CORBA::string_dup("RejectNewEvents");
  (*aqos)[3].value <<= f_bvl;
  (*aqos)[4].name    = CORBA::string_dup("NumAdminGroups");
  (*aqos)[4].value <<= numAdminGroups;
  (*aqos)[5].name    = CORBA::string_dup("NumAdminThreads");
  (*aqos)[5].value <<= numAdminThreads;
  (*aqos)[6].name    = CORBA::string_dup("NumProxyThreads");
  (*aqos)[6].value <<= numProxyThreads;
  (*aqos)[7].name    = CORBA::string_dup("NumPushThreads");
  (*aqos)[7].value <<= numPushThreads;
  (*aqos)[8].name    = CORBA::string_dup("NumPullThreads");
  (*aqos)[8].value <<= numPullThreads;
  (*aqos)[9].name    = CORBA::string_dup("PullEventPeriod");
  (*aqos)[9].value <<= pullEventPeriod;
  (*aqos)[10].name    = CORBA::string_dup("NumOChangeThreads");
  (*aqos)[10].value <<= numOChangeThreads;
  (*aqos)[11].name    = CORBA::string_dup("NumSChangeThreads");
  (*aqos)[11].value <<= numSChangeThreads;
  return aqos;
}

void RDI_AdminQoS::from_admin(const CosN_AdminProperties& r_qos)
{
  CORBA::Any::to_boolean vbool(rejectNewEvents);  
  for (CORBA::ULong ix=0; ix < r_qos.length(); ix++) {
     if ( strcmp(r_qos[ix].name, "MaxQueueLength") == 0 )
        r_qos[ix].value >>= maxQueueLength;
     else if ( strcmp(r_qos[ix].name, "MaxConsumers") == 0 )
        r_qos[ix].value >>= maxConsumers;
     else if ( strcmp(r_qos[ix].name, "MaxSuppliers") == 0 )
        r_qos[ix].value >>= maxSuppliers;
     else if ( strcmp(r_qos[ix].name, "RejectNewEvents") == 0 )
        r_qos[ix].value >>= vbool;
     else if ( strcmp(r_qos[ix].name, "NumAdminGroups") == 0 )
        r_qos[ix].value >>= numAdminGroups;
     else if ( strcmp(r_qos[ix].name, "NumAdminThreads") == 0 ) 
        r_qos[ix].value >>= numAdminThreads;
     else if ( strcmp(r_qos[ix].name, "NumProxyThreads") == 0 ) 
        r_qos[ix].value >>= numProxyThreads;
     else if ( strcmp(r_qos[ix].name, "NumPushThreads") == 0 ) 
        r_qos[ix].value >>= numPushThreads;
     else if ( strcmp(r_qos[ix].name, "NumPullThreads") == 0 ) 
        r_qos[ix].value >>= numPullThreads;
     else if ( strcmp(r_qos[ix].name, "PullEventPeriod") == 0 ) 
        r_qos[ix].value >>= pullEventPeriod;
     else if ( strcmp(r_qos[ix].name, "NumOChangeThreads") == 0 ) 
        r_qos[ix].value >>= numOChangeThreads;
     else if ( strcmp(r_qos[ix].name, "NumSChangeThreads") == 0 ) 
        r_qos[ix].value >>= numSChangeThreads;
  }
}

CORBA::Boolean RDI_AdminQoS::validate(
                        const CosN_AdminProperties&  r_qos,
                              CosN_PropertyErrorSeq& error,
                              CORBA::Boolean                     isfac)
{
  CORBA::Boolean succ=1;
  CORBA::UShort  sval=0;
  CORBA::ULong   ix_e=0;

  error.length(r_qos.length());
  for (CORBA::ULong ix=0; ix < r_qos.length(); ix++) {
     CORBA::TypeCode_var r_tc = r_qos[ix].value.type();
     if ( strcmp(r_qos[ix].name, "MaxQueueLength") == 0 ) {
	if ( ! CORBA::_tc_long->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; 
           continue;
        }
        // We allow changes in the queue size to occur dynamically.
        // However,  depending on the provided value, these changes
        // may not be reflected immediately in the queue status....
     } else if ( (strcmp(r_qos[ix].name, "MaxConsumers") == 0) ||
                 (strcmp(r_qos[ix].name, "MaxSuppliers") == 0) ) {
	if ( ! CORBA::_tc_long->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0; 
           continue;
        }
        // We allow changes to these values to be done dynamically.
        // However,  depending on the provided value, these changes
        // may not be reflected immediately in the channel status..
     } else if ( strcmp(r_qos[ix].name, "RejectNewEvents") == 0 ) {
	if ( ! CORBA::_tc_boolean->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        // Here we assume that we can turn on and off this property
        // and, thus, we do not need to check the value requested..
     } else if ( strcmp(r_qos[ix].name, "NumAdminGroups") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( (isfac && sval == 0) || (!isfac && sval != numAdminGroups) ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= 
                        (isfac ? (CORBA::UShort) 1    : numAdminGroups);
           error[ix_e].available_range.high_val <<= 
                        (isfac ? (CORBA::UShort) 1000 : numAdminGroups);
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "NumAdminThreads") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( (isfac && sval == 0) || (!isfac && sval != numAdminThreads) ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= 
                        (isfac ? (CORBA::UShort) 1  : numAdminThreads);
           error[ix_e].available_range.high_val <<= 
                        (isfac ? (CORBA::UShort) 50 : numAdminThreads);
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "NumProxyThreads") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( !isfac && sval != numProxyThreads ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= numProxyThreads;
           error[ix_e].available_range.high_val <<= numProxyThreads;
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "NumPushThreads") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( !isfac && sval != numPushThreads ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= numPushThreads;
           error[ix_e].available_range.high_val <<= numPushThreads;
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "NumPullThreads") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( !isfac && sval != numPullThreads ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= numPullThreads;
           error[ix_e].available_range.high_val <<= numPullThreads;
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "NumOChangeThreads") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( !isfac && sval != numOChangeThreads ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= numOChangeThreads;
           error[ix_e].available_range.high_val <<= numOChangeThreads;
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "NumSChangeThreads") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
        r_qos[ix].value >>= sval;
        if ( !isfac && sval != numSChangeThreads ) {
           error[ix_e].code = CosN_BAD_VALUE;
           error[ix_e].name = r_qos[ix].name;
           error[ix_e].available_range.low_val  <<= numSChangeThreads;
           error[ix_e].available_range.high_val <<= numSChangeThreads;
           ix_e += 1; succ  = 0;
           continue;
        }
     } else if ( strcmp(r_qos[ix].name, "PullEventPeriod") == 0 ) {
	if ( ! CORBA::_tc_ushort->equivalent(r_tc) ) {
           error[ix_e].code = CosN_BAD_TYPE;
           error[ix_e].name = r_qos[ix].name;
           ix_e += 1; succ  = 0;
           continue;
        }
     } else {
        error[ix_e].code = CosN_BAD_PROPERTY;
        error[ix_e].name = r_qos[ix].name;
        ix_e += 1; succ  = 0;
        RDI_DUMP("BAD AdminQoS: " <<  r_qos[ix].name << " requested");
     }
  }
  error.length(ix_e);   // Correct the length of this list
  return succ;
}

RDI_AdminQoS& RDI_AdminQoS::operator = (const RDI_AdminQoS& qos)
{
  maxQueueLength  = qos.maxQueueLength;
  maxConsumers    = qos.maxConsumers;
  maxSuppliers    = qos.maxSuppliers;
  rejectNewEvents = qos.rejectNewEvents;
  numAdminGroups  = qos.numAdminGroups;
  numAdminThreads = qos.numAdminThreads;
  numProxyThreads = qos.numProxyThreads;
  numPushThreads  = qos.numPushThreads;
  numPullThreads  = qos.numPullThreads;
  pullEventPeriod = qos.pullEventPeriod;
  numOChangeThreads  = qos.numOChangeThreads;
  numSChangeThreads  = qos.numSChangeThreads;
  return *this;
}

ostream& operator << (ostream& out, const RDI_AdminQoS& aqos)
{
  out << "MaxQueueLength  "     << setw(6) << aqos.maxQueueLength  <<
    " | MaxConsumers    " << setw(7) << aqos.maxConsumers    <<
    " | MaxSuppliers      " << setw(3) << aqos.maxSuppliers    << endl <<
    "RejectNewEvents "     << (aqos.rejectNewEvents ? "   YES":"    NO")<<
    " | NumAdminGroups  " << setw(7) << aqos.numAdminGroups  <<
    " | NumAdminThreads   " << setw(3) << aqos.numAdminThreads << endl <<
    "NumProxyThreads "     << setw(6) << aqos.numProxyThreads <<
    " | NumPushThreads  " << setw(7) << aqos.numPushThreads  <<
    " | NumPullThreads    " << setw(3) << aqos.numPullThreads  << endl <<
    "PullEventPeriod "     << setw(6) << aqos.pullEventPeriod << 
    " | NumOChangeThreads   " << setw(3) << aqos.numOChangeThreads  <<
    " | NumSChangeThreads " << setw(3) << aqos.numSChangeThreads  << endl;
  return out;
}
