// -*- Mode: C++; -*-
//                              File      : CosEventProxy.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of CORBA Event Service Proxies
//
 
/*
$Log: CosEventProxy.cc,v $
Revision 1.20  2000/11/19 19:02:30  alcfp
removed unnecessary debug output

Revision 1.19  2000/11/16 05:32:14  alcfp
fixes for aCC

Revision 1.18  2000/11/15 21:17:29  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.17  2000/11/05 04:48:09  alcfp
changed in defaults, env variable overrride, try_pull variants

Revision 1.16  2000/10/30 04:39:28  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.15  2000/10/01 13:39:52  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.14.2.1  2000/09/30 15:39:51  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.14  2000/08/22 18:23:53  alcfp
added description to each file

Revision 1.13  2000/08/16 20:19:25  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include "RDI.h"
#include "RDITimeDefs.h"
#include "RDIOplocksMacros.h"
#include "CosEventChannelAdmin_i.h"
#include "CosNotifyChannelAdmin_i.h"

// ------------------------------------------------------------ //
// EventProxyPushConsumer_i                                     //
// ------------------------------------------------------------ //

EventProxyPushConsumer_i::EventProxyPushConsumer_i(SupplierAdmin_i* admin, 
					 	   EventChannel_i*  evchn) :
  _oplockptr(0), _channel(evchn), _myadmin(admin), 
  _nevents(0), _pxstate(RDI_NotConnected)
{
  RDI_OPLOCK_INIT;
  _supplier = CosEventComm::PushSupplier::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

void EventProxyPushConsumer_i::connect_push_supplier
(CosEventComm::PushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate  = RDI_Connected;
  _supplier = CosEventComm::PushSupplier::_duplicate(supplier);
  RDI_OPLOCK_RELEASE;
}

void EventProxyPushConsumer_i::push(const CORBA::Any& data WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  if ( _channel->new_any_event(data) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  }
  _nevents += 1;
  RDI_OPLOCK_RELEASE;
}

void EventProxyPushConsumer_i::disconnect_push_consumer(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  _disconnect_client_and_dispose(1);
}

void EventProxyPushConsumer_i::disconnect_client_and_dispose
(CORBA::Boolean remove_proxy_from_admin) {
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_client_and_dispose(remove_proxy_from_admin);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void EventProxyPushConsumer_i::_disconnect_client_and_dispose
(CORBA::Boolean remove_proxy_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("EventProxyPushConsumer_i::disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // This proxy does not need to worry about inuse count
  // because it is never bumped above zero.
  if (remove_proxy_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("EventProxyPushConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _supplier = CosEventComm::PushSupplier::_nil();
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
  _oplockptr = 0; // speeds up acquire check
}

ostream& operator << (ostream& out, const EventProxyPushConsumer_i& prx)
{
  out << & prx << " CosEvent ProxyPushConsumer";
  out << prx._pxstate;
  return out << " #Push "<< prx._nevents;
}

// ------------------------------------------------------------ //
// EventProxyPullConsumer_i                                     //
// ------------------------------------------------------------ //

/** We use unbound threads here for we assume that in the majority
  * of configuations a thread pool will be used for pulling events
  * from suppliers and pushing events to consumers.
  */

class EventPullWorker : public omni_thread {
public:
  typedef void (EventProxyPullConsumer_i::*Method)(void);
  EventPullWorker(EventProxyPullConsumer_i* p, Method m) :
    omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  EventProxyPullConsumer_i* _proxy;
  Method                    _method;
  EventPullWorker()  {;}
};

// ------------------------------------------------------------ //

EventProxyPullConsumer_i::EventProxyPullConsumer_i(SupplierAdmin_i* admin, 
                                                   EventChannel_i*  evchn) :
  _oplockptr(0), _channel(evchn), _myadmin(admin), 
  _pworker(0), _thrdone(0), _nevents(0), 
  _pxstate(RDI_NotConnected), _timeout_s(0), _timeout_n(0)
{
  RDI_OPLOCK_INIT;
  _supplier = CosEventComm::PullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
    _pworker = new EventPullWorker(this, &EventProxyPullConsumer_i::_pull);
    RDI_AssertAllocThrowNo(_pworker, "Memory allocation failed -- omni_thread");
    _pworker->start();
    _thrdone = 0;
    RDI_DUMP("Pull thread for proxy " << (void*)this << " -- " << _pworker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void EventProxyPullConsumer_i::connect_pull_supplier
(CosEventComm::PullSupplier_ptr supplier WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(supplier) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate  = RDI_Connected;
  _supplier = CosEventComm::PullSupplier::_duplicate(supplier);
  if ( _pworker ) {
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_pull_supplier ) {
    _channel->_pull_supplier->signal_pull_threads();
  }
  RDI_OPLOCK_RELEASE;
}

// is_available determines whether a pull should occur.
// in this case the requirement is that either the pull interval is zero
// or an interval timeout has occurred
CORBA::Boolean EventProxyPullConsumer_i::is_available(unsigned long* wait_s, unsigned long* wait_n) {
  RDI_OPLOCK_ACQUIRE(return 0);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    return 0;
  }

  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
  if ( (pull_interval_s == 0) && (pull_interval_n == 0) ) {
    _timeout_s = 0; _timeout_n = 0;
    RDI_OPLOCK_RELEASE;
    return 1; // pull as fast as possible
  }

  omni_thread::get_time(&time_s, &time_n);

  if ( (_timeout_s == 0) && (_timeout_n == 0) ) {
    // proxy has not established timeout yet : set it now
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  }
  if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
    RDI_OPLOCK_RELEASE;
    return 1; // interval timeout occurred for this proxy
  }
  // update wait_s/_n to reflect the future timeout point of this proxy
  if ( ( ((*wait_s) == 0) && ((*wait_n) == 0)                   ) ||
       ( ((*wait_s) > _timeout_s)                               ) ||
       ( ((*wait_s) == _timeout_s) && ((*wait_n) > _timeout_n)) ) {
    (*wait_s) = _timeout_s;
    (*wait_n) = _timeout_n;
  }
  RDI_OPLOCK_RELEASE;
  return 0;
}

// REQUIREMENT: A thread from pull pool only calls this method
// if is_available() is true and it thinks state is connected
// (For safety we verify that these conditions actually hold.)
void EventProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }

  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;
  RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);

  CORBA::Any*     event = 0;
  CORBA::Boolean  hasev = 0;
  invalid = 0;

  // sanity check: if pull interval is positive, we should only be called if a timeout occurred
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&time_s, &time_n);
    if ((_timeout_s == 0) && (_timeout_n == 0)) { // timeout not set; set it now
      omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
    } 
    if ( (time_s < _timeout_s) || ((time_s == _timeout_s) && (time_n < _timeout_n)) ) {
      RDI_DUMP("** INTERNAL ERROR: ProxyPullConsumer_i::pull_event called too soon");
      RDI_OPLOCK_DEBUMP_RELEASE;
      return;
    }
  }

  // update timeout before releasing OPLOCK -- this means we do the update
  // before doing the try_pull (seems OK)
  if (pull_interval_s || pull_interval_n) {
    omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
  } else {
    _timeout_s = 0; _timeout_n = 0;
  }

  // do not hold OPLOCK across try_pull call
  RDI_OPLOCK_RELEASE;

  CORBA::Boolean outcall_worked = 0;
  try {
    event = _supplier->try_pull(hasev);
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Consumer object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Exception while pushing event to consumer");
  }

  RDI_OPLOCK_REACQUIRE(return);

  // add new event to channel even if we have been disconnected in the meantime
  if (outcall_worked && hasev && event ) {
    _nevents += 1;
    _channel->new_any_event(*event);
  }
  // new_any_event copies event, so we delete regardless
  if ( event ) {
    delete event; event = 0;
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during try_pull -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      _pxstate = RDI_Exception;
      invalid  = 1;
    }
  }

  RDI_OPLOCK_DEBUMP_RELEASE;
}

void EventProxyPullConsumer_i::_pull()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  CORBA::Any*    event = 0;
  CORBA::Boolean hasev = 0, update_timeout = 0;
  CORBA::Boolean do_yield = 0;
  unsigned long pull_interval_s, pull_interval_n;
  unsigned long time_s, time_n;

  // invariant: oplock is held at top of loop
  while ( 1 ) {
    do_yield = 1;
    while ( 1 ) {
      // must recompute these here because they can change across a wait/timedwait
      RDI_Millisec2SecNanosec(_channel->pull_period(), pull_interval_s, pull_interval_n);
      if (pull_interval_s || pull_interval_n) {
	// if timeout not set, or if update_timeout true due to pull, recompute timeout
	if ( update_timeout || ((_timeout_s == 0) && (_timeout_n == 0)) ) {
	  omni_thread::get_time(&_timeout_s, &_timeout_n, pull_interval_s, pull_interval_n);
	}
      } else {
	_timeout_s = 0; _timeout_n = 0;
      }
      update_timeout = 0;
      // breaking from inner loop requires a pull interval timeout (if set)
      // and state must be RDI_Connected
      if ((_pxstate!=RDI_NotConnected) && (_pxstate!=RDI_Connected)) {
	break; // also break on exceptional state
      }
      if (_pxstate==RDI_Connected) {
	if ((_timeout_s == 0) && (_timeout_n == 0)) {
	  break; // pulling as fast as possible so pull now
	}
	omni_thread::get_time(&time_s, &time_n);
	if ( (time_s > _timeout_s) || ((time_s == _timeout_s) && (time_n > _timeout_n)) ) {
	  break; // pull interval has passed so pull now
	}
	do_yield = 0;
	RDI_OPLOCK_TIMEDWAIT(_timeout_s, _timeout_n); // must wait for pull interval
      } else {
	do_yield = 0;
	RDI_OPLOCK_WAIT; // must wait for _pxstate to change
      }
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("pull thread");
    }

    // do not hold OPLOCK across try_pull call
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    CORBA::Boolean outcall_worked = 0;
    try {
      event = _supplier->try_pull(hasev);
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Consumer object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this << " - Communication Failure");
    } catch (...) {
      RDI_DUMP("Pull pool calling pull_event for proxy " << (void*)this <<
	       " - Exception while pushing event to consumer");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("pull thread [**unexpected REACQUIRE failure**]"));

    update_timeout = 1; // force update of timeout at top of loop

    // add new event to channel even if we have been disconnected in the meantime
    if (outcall_worked && hasev && event) {
      _nevents += 1;
      _channel->new_any_event(*event);
    }
    // new_any_event copies event so we can delete it regardless
    if ( event ) {
      delete event; event = 0;
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during try_pull -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	_pxstate = RDI_Exception;
      }
    }
  }
}

void EventProxyPullConsumer_i::disconnect_pull_consumer(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  _disconnect_client_and_dispose(1);
}

void EventProxyPullConsumer_i::disconnect_client_and_dispose
(CORBA::Boolean remove_proxy_from_admin)
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_client_and_dispose(remove_proxy_from_admin);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void EventProxyPullConsumer_i::_disconnect_client_and_dispose
(CORBA::Boolean remove_proxy_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("EventProxyPullConsumer_i::disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding try_pull call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_proxy_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("EventProxyPullConsumer_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _supplier = CosEventComm::PullSupplier::_nil();
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
  _oplockptr = 0; // speeds up acquire check
}

ostream& operator << (ostream& out, const EventProxyPullConsumer_i& prx)
{
  out << & prx << " CosEvent ProxyPullConsumer";
  out << prx._pxstate;
  return out << "#Push "<< prx._nevents;
}

// ------------------------------------------------------------ //
// EventProxyPushSupplier_i                                     //
// ------------------------------------------------------------ //

/** We use unbound threads here for we assume that in the majority
  * of configuations a thread pool will be used for pulling events
  * from suppliers and pushing events to consumers.
  */

class EventPushWorker : public omni_thread {
public:
  typedef void (EventProxyPushSupplier_i::*Method)(void);
  EventPushWorker(EventProxyPushSupplier_i* p, Method m) :
    omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  EventProxyPushSupplier_i* _proxy;
  Method                    _method;
  EventPushWorker()  {;}
};

// ------------------------------------------------------------ //

EventProxyPushSupplier_i::EventProxyPushSupplier_i(ConsumerAdmin_i* admin, 
						   EventChannel_i*  evchn) :
  _oplockptr(0), _channel(evchn), _myadmin(admin), 
  _pworker(0),  _thrdone(0), _nevents(0), 
  _pxstate(RDI_NotConnected), _ntfqueue()
{
  RDI_OPLOCK_INIT;
  _consumer = CosEventComm::PushConsumer::_nil();
  // When the number of push threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->push_threads() == 0 ) {
    _pworker = new EventPushWorker(this, &EventProxyPushSupplier_i::_push);
    RDI_AssertAllocThrowNo(_pworker, "Memory allocation failed -- omni_thread");
    _pworker->start();
    _thrdone = 0;
    RDI_DUMP("Pull thread for proxy " << (void*)this << " -- " << _pworker->id());
  }
  WRAPPED_REGISTER_IMPL(this);
}

void EventProxyPushSupplier_i::connect_push_consumer
(CosEventComm::PushConsumer_ptr consumer WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( CORBA::is_nil(consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate  = RDI_Connected;
  _consumer = CosEventComm::PushConsumer::_duplicate(consumer);
  if ( _pworker ) {
    RDI_OPLOCK_SIGNAL;
  }
  if ( _channel->_push_consumer ) {
    _channel->_push_consumer->signal_push_threads();
  }
  RDI_OPLOCK_RELEASE;
}

void EventProxyPushSupplier_i::add_event(RDI_StructuredEvent* event)
{
  RDI_OPLOCK_ACQUIRE(return);
  if ( (_pxstate == RDI_Connected) && event ) {
    event->incr_ref_counter( RDI_LOCK_EVENT );
    _ntfqueue.insert_tail(event);
    if ( _pworker ) {
      RDI_OPLOCK_SIGNAL;
    }
    if ( _channel->_push_consumer ) {
      _channel->_push_consumer->signal_push_threads();
    }
  }
  RDI_OPLOCK_RELEASE;
}

void EventProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on method exit

  RDI_StructuredEvent* event = 0;
  unsigned int         qsize = 0;

  invalid = 0;
  if ( (_pxstate != RDI_Connected) || (_ntfqueue.length() == 0) ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    return;
  }
  event = _ntfqueue.get_head();
  qsize = _ntfqueue.length() - 1;
  _ntfqueue.remove_head();
  _nevents += 1;

  // do not hold OPLOCK across push call
  RDI_OPLOCK_RELEASE;

  CORBA::Boolean outcall_worked = 0;
  if (  RDI_LOCK_EVENT  ) event->mylock().lock();
  try {
    const CosN_StructuredEvent& cosev = event->get_cos_event();
    if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
      _consumer->push(cosev.remainder_of_body);
    } else {
      CORBA::Any anyev;
      anyev <<= cosev;
      _consumer->push(anyev);
    }
    outcall_worked = 1;
  } catch ( CORBA::INV_OBJREF& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Invalid object reference");
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Consumer object does not exist");
  } catch ( CORBA::COMM_FAILURE& e ) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Communication Failure");
  } catch (...) {
    RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Exception while pushing event to consumer");
  }

  RDI_OPLOCK_REACQUIRE(return);

  if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
  event->decr_ref_counter( RDI_LOCK_EVENT );

  if (outcall_worked) {
    _channel->incr_num_notifications(qsize);
  }

  if (_pxstate != RDI_Connected) {
    // disconnected/disposed during push -- do nothing else with my state
  } else {
    if (!outcall_worked) { // exception
      _pxstate = RDI_Exception;
      invalid  = 1;
    }
  }
  RDI_OPLOCK_DEBUMP_RELEASE;
}

void EventProxyPushSupplier_i::_push()
{
  RDI_OPLOCK_ACQUIRE_BUMP(return); // debump on thread exit

  RDI_StructuredEvent* event = 0;
  unsigned int         qsize = 0;
  CORBA::Boolean       do_yield = 0, outcall_worked = 0;

  // invariant : OPLOCK held at top of loop
  while ( 1 ) {
    do_yield = 1;
    while ( ((_pxstate == RDI_Connected) && (_ntfqueue.length() == 0)) ||
	    (_pxstate == RDI_NotConnected) ) {
      do_yield = 0;
      RDI_OPLOCK_WAIT;
    }
    if ( _pxstate != RDI_Connected ) {
      RDI_OPLOCK_DEBUMP_RELEASE;
      RDI_THREAD_EXIT("push thread");
    }
    event = _ntfqueue.get_head();
    qsize = _ntfqueue.length() - 1;
    _ntfqueue.remove_head();
    _nevents += 1;

    // do not hold OPLOCK across push call
    RDI_OPLOCK_RELEASE;

    // if we did not wait, yield here (wait or yield each time through outer loop)
    if (do_yield) {
      omni_thread::yield();
    }

    if (  RDI_LOCK_EVENT  ) event->mylock().lock();
    outcall_worked = 0;
    try {
      const CosN_StructuredEvent& cosev = 
	event->get_cos_event();
      if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
	_consumer->push(cosev.remainder_of_body);
      } else {
	CORBA::Any anyev;
	anyev <<= cosev;
	_consumer->push(anyev);
      }
      outcall_worked = 1;
    } catch ( CORBA::INV_OBJREF& e ) {
      RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Invalid object reference");
    } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
      RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Consumer object does not exist");
    } catch ( CORBA::COMM_FAILURE& e ) {
      RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Communication Failure");
    } catch (...) {
      RDI_DUMP("Push pool calling push_event for proxy " << (void*)this << " - Exception while pushing event to consumer");
    }

    RDI_OPLOCK_REACQUIRE(RDI_THREAD_EXIT("push thread [**unexpected REACQUIRE failure**]"));

    if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
    event->decr_ref_counter( RDI_LOCK_EVENT );

    if (outcall_worked) {
      _channel->incr_num_notifications(qsize);
    }

    if (_pxstate != RDI_Connected) {
      // disconnected/disposed during push -- do nothing else with my state
    } else {
      if (!outcall_worked) { // exception
	_pxstate = RDI_Exception;
      }
    }
  }
}

void EventProxyPushSupplier_i::disconnect_push_supplier(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  _disconnect_client_and_dispose(1);
}

void EventProxyPushSupplier_i::disconnect_client_and_dispose
(CORBA::Boolean remove_proxy_from_admin)
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_client_and_dispose(remove_proxy_from_admin);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void EventProxyPushSupplier_i::_disconnect_client_and_dispose
(CORBA::Boolean remove_proxy_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("EventProxyPushSupplier_i::disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _pxstate = RDI_Disconnected; // acts as guard; following only entered by 1 thread
  // Wait for inuse to drop to zero.  This is not strictly necessary, but it
  // ensures that any outstanding push call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_proxy_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("EventProxyPushSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _consumer = CosEventComm::PushConsumer::_nil();
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter( RDI_LOCK_EVENT );
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
  _oplockptr = 0; // speeds up acquire check
}

ostream& operator << (ostream& out, const EventProxyPushSupplier_i& prx)
{
  out << & prx << " CosEvent ProxyPushSupplier";
  out << prx._pxstate;
  return out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
}

// ------------------------------------------------------------ //
// EventProxyPullSupplier_i                                     //
// ------------------------------------------------------------ //

EventProxyPullSupplier_i::EventProxyPullSupplier_i(ConsumerAdmin_i* admin, 
						   EventChannel_i*  evchn) :
  _oplockptr(0), _channel(evchn), 
  _myadmin(admin), _nevents(0), 
  _pxstate(RDI_NotConnected), _ntfqueue()
{
  RDI_OPLOCK_INIT;
  _consumer = CosEventComm::PullConsumer::_nil();
  WRAPPED_REGISTER_IMPL(this);
}

void EventProxyPullSupplier_i::connect_pull_consumer
(CosEventComm::PullConsumer_ptr consumer WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  RDI_DUMP("CosEvent consumer to connect to " << (void*)this);
  if ( CORBA::is_nil(consumer) ) {
    RDI_OPLOCK_RELEASE;
    throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate  = RDI_Connected;
  _consumer = CosEventComm::PullConsumer::_duplicate(consumer);
  RDI_DUMP("CosEvent consumer connected to " << (void*)this);
  RDI_OPLOCK_RELEASE;
}

void EventProxyPullSupplier_i::add_event(RDI_StructuredEvent* event)
{
  RDI_OPLOCK_ACQUIRE(return);
  if ( (_pxstate == RDI_Connected) && event ) {
    event->incr_ref_counter( RDI_LOCK_EVENT );
    _ntfqueue.insert_tail(event);
    RDI_OPLOCK_SIGNAL;
  }
  RDI_OPLOCK_RELEASE;
}

CORBA::Any* EventProxyPullSupplier_i::pull(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE_BUMP(RDI_THROW_INV_OBJREF); // debump on method exit
  if ( _pxstate == RDI_NotConnected ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    throw CosEventComm::Disconnected();
  }
  CORBA::ULong qsize = 0;
  while ( (_pxstate == RDI_Connected) && ((qsize = _ntfqueue.length()) == 0) ) {
    RDI_OPLOCK_WAIT;
  }
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_DEBUMP_RELEASE;
    throw CosEventComm::Disconnected();
  }
  RDI_StructuredEvent* event = _ntfqueue.get_head();
  const CosN_StructuredEvent& cosev = event->get_cos_event();
  _ntfqueue.remove_head();
  _nevents += 1;
  RDI_DUMP("pull event " << event << " from queue of " << (void*)this);

  CORBA::Any* res = 0;
  if (  RDI_LOCK_EVENT  ) event->mylock().lock();
  if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
    res = new CORBA::Any(cosev.remainder_of_body);
  } else {
    res = new CORBA::Any;
    (*res) <<= cosev;
  }
  if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
  event->decr_ref_counter( RDI_LOCK_EVENT );

  _channel->incr_num_notifications(qsize);
  RDI_OPLOCK_DEBUMP_RELEASE;
  return res;
}

CORBA::Any* EventProxyPullSupplier_i::try_pull
(CORBA::Boolean& has_event WRAPPED_IMPLARG)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  if ( _pxstate != RDI_Connected ) {
    RDI_OPLOCK_RELEASE;
    throw CosEventComm::Disconnected();
  }
  CORBA::ULong qsize = _ntfqueue.length();
  if (qsize == 0 ) {
    RDI_OPLOCK_RELEASE;
    has_event = 0;
    return new CORBA::Any();
  }
  RDI_StructuredEvent* event = _ntfqueue.get_head();
  const CosN_StructuredEvent& cosev = event->get_cos_event();
  _ntfqueue.remove_head();
  _nevents += 1;
  RDI_DUMP("try_pull event " << event << " from queue of " << (void*)this);

  if (  RDI_LOCK_EVENT  ) event->mylock().lock();
  CORBA::Any* res = 0;
  has_event = 1;
  if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
    res = new CORBA::Any(cosev.remainder_of_body);
  } else {
    res = new CORBA::Any;
    (*res) <<= cosev;
  }
  if (  RDI_LOCK_EVENT  ) event->mylock().unlock();
  event->decr_ref_counter( RDI_LOCK_EVENT );

  _channel->incr_num_notifications(qsize);
  RDI_OPLOCK_RELEASE;
  return res;
}

void EventProxyPullSupplier_i::disconnect_pull_supplier(WRAPPED_IMPLARG_VOID)
{
  RDI_OPLOCK_ACQUIRE(RDI_THROW_INV_OBJREF);
  _disconnect_client_and_dispose(1);
}

void EventProxyPullSupplier_i::disconnect_client_and_dispose(CORBA::Boolean remove_proxy_from_admin)
{
  RDI_OPLOCK_ACQUIRE(return);
  _disconnect_client_and_dispose(remove_proxy_from_admin);
}

// *** lock must be held
// Effect: initiate impl dispose + release lock
void EventProxyPullSupplier_i::_disconnect_client_and_dispose(CORBA::Boolean remove_proxy_from_admin)
{
  if (_pxstate == RDI_Disconnected) {
    RDI_DUMP("EventProxyPullSupplier_i::disconnect_client_and_dispose called twice on same proxy!");
    RDI_OPLOCK_RELEASE;
    return;
  }
  _pxstate = RDI_Disconnected; // acts as guard -- only one thread does the following
  // Wait for inuse to drop to zero.  This ensures that any outstanding blocked pull
  // call has a chance to complete cleanly.
  while (_oplockptr->inuse() > 0) {
    RDI_OPLOCK_BROADCAST;
    RDI_OPLOCK_INUSEZERO_WAIT;
  }
  if (remove_proxy_from_admin) {
    // do not hold OPLOCK across upcalls involving this
    RDI_OPLOCK_BUMP_RELEASE;
    _myadmin->remove_proxy(this);
    RDI_OPLOCK_REACQUIRE_DEBUMP(RDI_METHOD_RETURN_NULL("EventProxyPullSupplier_i::_disconnect_client_and_dispose [**unexpected REACQUIRE failure**]"));
  }
  _consumer = CosEventComm::PullConsumer::_nil();
  RDI_StructuredEvent* event = 0;
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
    _ntfqueue.remove_head();
    event->decr_ref_counter( RDI_LOCK_EVENT );
  }
  RDI_OPLOCK_DISPOSE_IMPL_AND_RELEASE;
  _oplockptr = 0; // speeds up acquire check
}

ostream& operator << (ostream& out, const EventProxyPullSupplier_i& prx)
{
  out << & prx << " CosEvent ProxyPullSupplier";
  out << prx._pxstate;
  return out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
}
