// -*- Mode: C++; -*-
//                              File      : RDIUtil.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIUtil.h,v $
Revision 1.8  2000/08/22 18:23:52  alcfp
added description to each file

Revision 1.7  2000/08/16 20:19:20  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _RDI_UTIL_H_
#define _RDI_UTIL_H_

#include <sys/types.h>
#include "RDIDefs.h"

/** MISCELLANEOUS UTILITY FUNCTIONS
  *
  */

inline size_t RDI_Align(size_t size, size_t base_power_2)
{ return (size + base_power_2 - 1) & ~(base_power_2 -1); }

/** HASH AND RANK FUNCTIONS
  * Hash and Rank function definitions, together with several
  * specialized implementations for some primitive data types
  */

typedef int          (* RDI_FuncRank) (const void *, const void *);
typedef unsigned int (* RDI_FuncHash) (const void *);

inline int RDI_VoidRank(const void* L, const void* R)
{ return *((unsigned int *) L) - *((unsigned int *) R); }

inline int RDI_SIntRank(const void* L, const void* R)
{ return *((int *) L) - *((int *) R); }

inline int RDI_UIntRank(const void* L, const void* R)
{ unsigned int l = *((unsigned int *) L), r = *((unsigned int *) R); 
  return (l < r) ? -1 : ((l > r) ? 1 : 0); 
}

inline int RDI_SLongRank(const void* L, const void* R)
{ return *((long *) L) - *((long *) R); }

inline int RDI_ULongRank(const void* L, const void* R)
{ unsigned long l = *((unsigned long *) L), r = *((unsigned long *) R);
  return (l < r) ? -1 : ((l > r) ? 1 : 0); 
}

inline int RDI_StrRank(const void* L, const void* R)
{ return ::strcmp((const char*) L, (const char *) R); }

inline unsigned int RDI_VoidHash(const void* K)
{ return *((unsigned int *) K); }

inline unsigned int RDI_SIntHash(const void* K)
{ int l = *((int *) K); return (l < 0) ? -l : l; }

inline unsigned int RDI_UIntHash(const void* K)
{ return *((unsigned int *) K); }

inline unsigned int RDI_SLongHash(const void* K)
{ long l = *((long *) K); return (l < 0) ? -l : l; }

inline unsigned int RDI_ULongHash(const void* K)
{ return *((unsigned long *) K); }

inline unsigned int RDI_StrHash(const void* K)
{
  unsigned char *p = (unsigned char *) K;
  unsigned int   g, h=0;
  while ( (g = *p++) )
        h += (h << 7) + g + 987654321L;
  return h;
}

/** STOP AND DELTA WATCH
  *
  * The following classes offer simple and delta timers for time
  * measurements in a multi-threaded environment,  assuming that
  * the omni_thread module is used
  */

#define RDI_NS2US(x) ((x) / 1000)
#define RDI_NS2MS(x) ((x) / 1000000)

#define RDI_SC2MS(x) ((x) * 1000)
#define RDI_SC2US(x) ((x) * 1000000)
#define RDI_SC2NS(x) ((x) * 1000000000)

#define RDI_TIMEDIFF_MSECS(start_tval_s,start_tval_n,end_tval_s,end_tval_n)\
  ( (end_tval_n > start_tval_n) ? \
    (RDI_SC2MS(end_tval_s - start_tval_s) + \
     RDI_NS2MS(end_tval_n - start_tval_n)) : \
    (RDI_SC2MS(end_tval_s - start_tval_s - 1) + \
     RDI_NS2MS((RDI_SC2NS(1) + end_tval_n) - start_tval_n)))

#define RDI_TIMEDIFF_USECS(start_tval_s,start_tval_n,end_tval_s,end_tval_n)\
  ( (end_tval_n > start_tval_n) ? \
    (RDI_SC2US(end_tval_s - start_tval_s) + \
     RDI_NS2US(end_tval_n - start_tval_n)) : \
    (RDI_SC2US(end_tval_s - start_tval_s - 1) + \
     RDI_NS2US((RDI_SC2NS(1) + end_tval_n) - start_tval_n)))

class RDI_Watch {
public:
  RDI_Watch(void)        {;}

  void start(void)      { omni_thread::get_time(&_start_s, &_start_n); }
  void stop(void)       { omni_thread::get_time(&_end_s, &_end_n); }

  unsigned int millisecs() const 
                { return RDI_TIMEDIFF_MSECS(_start_s,_start_n,_end_s,_end_n); }
  unsigned int microsecs() const 
                { return RDI_TIMEDIFF_USECS(_start_s,_start_n,_end_s,_end_n); }
  friend ostream& operator<<(ostream& out, const RDI_Watch& w)
                { return out << "Stop watch: "<< w.millisecs() << " msecs"; }
private:
  unsigned long _start_s;
  unsigned long _start_n;
  unsigned long _end_s;
  unsigned long _end_n;
};

#define RDI_MAX_WATCH_DELTAS 128

class RDI_DeltaWatch {
public:
  RDI_DeltaWatch(void) : _deltas(-1) {;}

  void delta(void) {
    if ( (_deltas+1) < RDI_MAX_WATCH_DELTAS ) {
      _deltas++;
      omni_thread::get_time(&(_tms_s[_deltas]), &(_tms_n[_deltas]));
    }
  }
  int  deltas(void)     { return _deltas; }

  unsigned long millisecs(int interval) const {
    return ((interval < 1) || (_deltas < interval)) ? 0 :
      RDI_TIMEDIFF_MSECS(_tms_s[interval-1], _tms_n[interval-1],
                         _tms_s[interval], _tms_n[interval]);
  }
  unsigned long microsecs(int interval) const {
    return ((interval < 1) || (_deltas < interval)) ? 0 :
      RDI_TIMEDIFF_USECS(_tms_s[interval-1], _tms_n[interval-1],
                         _tms_s[interval], _tms_n[interval]);
  }

  friend ostream& operator<<(ostream& out, const RDI_DeltaWatch& w)
        { out << "Interval stop watch: "<< endl;
          if (w._deltas < 1) { return out << "\tnot enough deltas" << endl; }
          for (int interval = 1; interval <= w._deltas; interval++) {
                out << "\tInterval " << interval << " : " <<
                       w.millisecs(interval) << " millisecs" << endl;
          }
          return out << endl;
        }
private:
  int _deltas;
  unsigned long _tms_s[RDI_MAX_WATCH_DELTAS];
  unsigned long _tms_n[RDI_MAX_WATCH_DELTAS];
};

#endif
