// -*- Mode: C++; -*-
//                              File      : RDIOSet.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIOSet.h,v $
Revision 1.6  2000/08/22 18:23:51  alcfp
added description to each file

Revision 1.5  2000/08/16 20:19:18  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _RDI_ORDERED_SET_H_
#define _RDI_ORDERED_SET_H_

#include "RDIUtil.h"

template <class SetEntry> class RDI_OSet;

/** AN ORDERED SET CURSOR
  * The set cursor provides an easy way to traverse a set.  You should
  * keep in mind that the cursor is NOT safe, i.e., if the set changes,
  * the cursor may be in an inconsistent state.
  *
  * Usage scenarios:
  *
  *	RDI_OSet<SetEntry> s(...);
  *	RDI_OSetCursor<SetEntry> c(s);
  *	// Forward scanning of the set
  *	for (unsigned int indx = 0; indx < s.length(); i++, c++)
  *		const SetEntry& entry = *c;
  *	// Backward scanning of the list
  *	c.reverse();
  *	for (unsigned int indx = 0; indx < s.length(); i++, c--)
  *		const SetEntry& entry = *c;
  */

template <class SetEntry> class RDI_OSetCursor {
	friend class RDI_OSet<SetEntry>;
public:
  typedef RDI_OSetCursor<SetEntry> Cursor;
  typedef RDI_OSet<SetEntry>       OSet;

  RDI_OSetCursor() : _oset(0), _indx(0)	{;}
  RDI_OSetCursor(const OSet* oset) : _oset(oset), _indx(oset->_oset_head) {;}
  RDI_OSetCursor(const Cursor& cur) : _oset(cur._oset),  _indx(cur._indx) {;}
  ~RDI_OSetCursor() {;}

  void reset()		{ _indx = _oset->_oset_head; }
  void reverse()	{ _indx = _oset->_oset_tail; }

  Cursor operator = (const Cursor& cursor)
		{ _oset = cursor._oset; _indx = cursor._indx; return *this; }

  int operator==(const Cursor& cur)
                { return ((_oset == cur._oset) && (_indx == cur._indx)); }
  int operator!=(const Cursor& cur)
                { return ((_oset != cur._oset) || (_indx != cur._indx)); }

  Cursor& operator++()	   { _oset->next_index(_indx); return *this; }
  Cursor operator++(int)   { Cursor __tmp = *this; ++*this; return __tmp; }

  Cursor& operator--()	   { _oset->prev_index(_indx); return *this;  }
  Cursor operator--(int)   { Cursor __tmp = *this; --*this; return __tmp; }
 
  const SetEntry& operator *() const { return _oset->entry(_indx); }

private:
  const OSet*  _oset;
  unsigned int _indx;

};


/** AN ORDERED SET CONTAINING HOMOGENEOUS ELEMENTS
  * The set is implemented as an array.  The initial size of the set is
  * set to a power of 2 that is greater or equeal to 'size'.  The array
  * is sorted according to 'rank' in an ascending order.
  *
  * NOTE: the class 'SetEntry' should provide a default constructor and
  *       have the assignment operator defined.
  */

template <class SetEntry> class RDI_OSet {
	friend class RDI_OSetCursor<SetEntry>;
public:
  typedef RDI_OSetCursor<SetEntry> Cursor;
  typedef RDI_OSet<SetEntry>       OSet;

  inline RDI_OSet(RDI_FuncRank rank_func, unsigned int init_size=4);
  inline RDI_OSet(const OSet& oset);
  inline ~RDI_OSet();

  inline int insert(const SetEntry& entry);
  inline int contains(const SetEntry& entry);

  // Remove all entries having the same value as 'entry'. If only
  // some of them should be removed,  use a cursor to locate them
  // first, and then remove them using second remove member

  inline void remove(const SetEntry& entry);
  inline void remove(const Cursor& position, unsigned int nentries=1);

  // Locate a given entry, if exists, and initialize the 'cursor'
  // to point to this entry.  If there are multiple entries with
  // the same value, the 'cursor' points to the very first one
 
  inline int lookup(const SetEntry& entry, Cursor& cursor);

  // The number of entries that have the same value as 'entry'

  inline unsigned int count(const SetEntry& entry);

  unsigned int length(void)	{ return _num_items; } 

  Cursor cursor(void) const	{ return this; }

  void drain()			{ _num_items = _oset_head = _oset_tail = 0; }
private:
  unsigned int _num_items;
  unsigned int _curr_size;
  unsigned int _oset_head;
  unsigned int _oset_tail;
  RDI_FuncRank _rank_func;
  SetEntry*    _entry;

  const SetEntry& entry(unsigned int index) const { return _entry[index]; }

  // Locate a given entry, if it exists,  and set its position.  For
  // sets with multiple entries having the same value,  the position
  // of the "leftmost" one is set

  inline int lookup(const SetEntry& entry, unsigned int& position);

  // Double the size of the set -- size of the set is a power of two

  inline int resize();

  // Remove several entries starting  at a given position in the set
  // No checks are performed here; these checks are performed by the
  // caller functions

  inline void remove(unsigned int position, unsigned int nentries);

  // Given a VALID list index, compute the next/previous VALID index

  inline void next_index(unsigned int &index) const;
  inline void prev_index(unsigned int &index) const;
};

// --------------------------------------------------------- //

template <class SetEntry> inline
RDI_OSet<SetEntry>::RDI_OSet(RDI_FuncRank rank, unsigned int size) :
		_num_items(0), _oset_head(0), _oset_tail(0), _rank_func(rank)
{ 
  _curr_size = 1;
  while ( _curr_size < size )   	// Size of array is a power
        _curr_size <<= 1;       	// of 2 that is >= size
  _entry = new SetEntry[_curr_size];
}

template <class SetEntry> inline
RDI_OSet<SetEntry>::RDI_OSet(const RDI_OSet<SetEntry>& oset) :
		_num_items(oset._num_items), _curr_size(oset._curr_size),
		_oset_head(oset._oset_head), _oset_tail(oset._oset_tail), 
		_rank_func(oset._rank_func)
{ unsigned int mask = _curr_size - 1;  
  _entry = new SetEntry[_curr_size];
  for (unsigned int i = _oset_head; i < _oset_head + _num_items; i++)
	_entry[i &  mask] = oset._entry[i &  mask];
}

template <class SetEntry> inline
RDI_OSet<SetEntry>::~RDI_OSet()
{ drain(); delete [] _entry; }

template <class SetEntry> inline
int RDI_OSet<SetEntry>::insert(const SetEntry& entry)
{
  if ( (_num_items == _curr_size) && (resize() == -1) )
	return -1;
  if ( _num_items == 0 ) {
	_oset_head = _oset_tail = 0;
  } else {
	_oset_tail = (_oset_tail == (_curr_size - 1)) ? 0 : (_oset_tail + 1);
  }

  // Append the entry at the end of the set and sort the array

  _entry[_oset_tail] = entry;
  _num_items += 1;

  if ( _num_items == 1 )	// No sorting required here
	return 0;

  unsigned int up = _oset_tail;
  unsigned int lo = (up == 0) ? (_curr_size - 1) : (up - 1);

  while ( (up != _oset_head) && (_rank_func(&_entry[lo], &_entry[up]) > 0) ) {
	SetEntry e = _entry[up];
	_entry[up] = _entry[lo];
	_entry[lo] = e;
	up = lo;
	lo = (up == 0) ? (_curr_size - 1) : (up - 1);
  }

  return 0;
}

template <class SetEntry> inline
int RDI_OSet<SetEntry>::contains(const SetEntry& entry)
{ 
  unsigned int pos;
  return this->lookup(entry, pos);
}

template <class SetEntry> inline
void RDI_OSet<SetEntry>::remove(const SetEntry& entry)
{
  unsigned int pos;
  unsigned int cntr = 0;
  unsigned int mask = _curr_size - 1;

  if ( lookup(entry, pos) ) {
	unsigned int indx = pos;
  	while ( _rank_func(&_entry[indx], &entry) == 0 ) {
		cntr += 1;
		indx  = (indx + 1) & mask;
  	}
  	this->remove(pos, cntr);
  }
}

template <class SetEntry> inline
void RDI_OSet<SetEntry>::remove(const RDI_OSetCursor<SetEntry>& cursor,
				unsigned int nentries)
{
  unsigned int indx = cursor._indx;

  if ( cursor._oset != this )
	return;

  if ((_oset_head > _oset_tail) && (indx < _oset_head) && (indx > _oset_tail))
	return;
  if ((_oset_head < _oset_tail) && ((indx < _oset_head) || (indx > _oset_tail)))
	return;
  
  // Compute number of entries to the right of 'indx'. If this number
  // is less than 'nentries', no action is taken

  unsigned int nume = (indx < _oset_tail) ? (_oset_tail - indx) :
			       		    (_curr_size + _oset_tail - indx);

  if ( nentries > nume + 1 )
	return;
  this->remove(indx, nentries);
}

template <class SetEntry> inline
int RDI_OSet<SetEntry>::lookup(const SetEntry& entry,
				RDI_OSetCursor<SetEntry>& cursor)
{
  unsigned int pos = 0;
  if ( lookup(entry, pos) ) {
	cursor._oset = this;
	cursor._indx = pos;
	return 1;
  }
  return 0;
}

template <class SetEntry> inline
unsigned int RDI_OSet<SetEntry>::count(const SetEntry& entry)
{
  unsigned int cnt = 0, pos = 0, mask = _curr_size - 1;

  if ( ! lookup(entry, pos) )
	return 0;
  while ( (pos != _oset_tail) && (_rank_func(&entry, &_entry[pos]) == 0) ) {
	cnt += 1;
	pos  = (pos + 1) & mask;
  }
  if ( (pos == _oset_tail) && (_rank_func(&entry, &_entry[pos]) == 0) )
	cnt += 1;
  return cnt;
}

template <class SetEntry> inline
int RDI_OSet<SetEntry>::lookup(const SetEntry& entry, unsigned int& pos)
{
  unsigned int mask = _curr_size - 1;
  int up, lo, md;

  if ( _num_items == 0 ) 
	return 0;
  up = (_oset_tail < _oset_head) ? (_oset_tail + _curr_size) : _oset_tail;
  md = lo = _oset_head;

  while ( lo <= up ) {		// Binary search to locate 'entry'
	md = (lo + up + 1) / 2;
	if ( _rank_func(&entry, &_entry[md & mask]) < 0 ) {
		up = md - 1;
	} else if ( _rank_func(&entry, &_entry[md & mask]) > 0 ) {
		lo = md + 1;
	} else {
		break;
	}
  }

  if ( lo > up ) 
	return 0;

  pos = md;
  while ( (pos != _oset_head) && (_rank_func(&entry, &_entry[pos]) == 0) ) 
	pos = (pos == 0) ? (_curr_size - 1) : (pos - 1);

  if ( (pos != _oset_head) || (_rank_func(&entry, &_entry[pos]) != 0) )
	pos = (pos + 1) & mask;

  return 1;
}

template <class SetEntry> inline
int RDI_OSet<SetEntry>::resize()
{
  unsigned int new_size = 2 * _curr_size;
  unsigned int mask = _curr_size - 1;
  SetEntry* tmp_buff = 0;

  if ( (tmp_buff = new SetEntry[new_size]) == (SetEntry *) 0 )  
	return -1;
  for (unsigned int i=0; i < _num_items; i++)
	tmp_buff[i] = _entry[(_oset_head + i) & mask];
  _curr_size = new_size;
  _oset_head = 0;
  _oset_tail = _num_items - 1;
  delete [] _entry;
  _entry = tmp_buff;
  return 0;
}

template <class SetEntry> inline
void RDI_OSet<SetEntry>::remove(unsigned int pos, unsigned int num)
{
  unsigned int mask = _curr_size - 1;
  unsigned int l = pos, r = (pos + num - 1) & mask;
  unsigned int i, nl, nr;
  
  _num_items -= num;

  if ( _num_items == 0 ) {	// No entries left in the set
	_oset_head = 0;
	_oset_tail = 0;
	return;
  } else if ( l == _oset_head ) {
	_oset_head = (_oset_head + num) & mask;
  } else if ( r == _oset_tail ) {
	_oset_tail = (_oset_tail + _curr_size - num) & mask;
  } else {
        nl= (l > _oset_head) ? (l - _oset_head) : (_curr_size + l - _oset_head);
        nr= (r < _oset_tail) ? (_oset_tail - r) : (_curr_size + _oset_tail - r);

        if ( nr < nl ) {
                for (i=0; i < nr; i++) {
                        _entry[(l + i) & mask] = _entry[(r + i + 1) & mask];
		}
                _oset_tail = (_oset_tail + _curr_size - num) & mask;
        } else {
                for (i=0; i < nl; i++) {
                        _entry[(r + _curr_size - i) & mask] = 
					_entry[(l + _curr_size - i - 1) & mask];
		}
                _oset_head = (_oset_head + num) & mask;
        }
  }
}

template <class SetEntry> inline
void RDI_OSet<SetEntry>::next_index(unsigned int &idx) const
{ idx = (idx==_oset_tail) ? _oset_head : ((idx + 1) % _curr_size); }

template <class SetEntry> inline
void RDI_OSet<SetEntry>::prev_index(unsigned int &idx) const
{ idx = (idx==_oset_head) ? _oset_tail : (idx ? (idx - 1) : (_curr_size - 1)); }

#endif
