// -*- Mode: C++; -*-
//                              File      : RDIFileSys.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIFileSys.h,v $
Revision 1.6  2000/11/15 21:18:17  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.5  2000/08/22 18:23:50  alcfp
added description to each file

Revision 1.4  2000/08/16 20:19:16  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _RDI_FILE_SYS_H_
#define _RDI_FILE_SYS_H_

#include <stdio.h>
#include <dirent.h>
#include <string.h>
#include <iostream.h>
#include <sys/types.h>
#include <sys/stat.h>

#if defined (__OMNIORB3__)
#include <omniORB3/CORBA.h>
#elif defined (__OMNIORB2__)
#include <omniORB2/CORBA.h>
#elif defined (__ORBIX3__)
#include "CORBA.h"
#endif

#include "omnithread.h"

class EventChannel_i;

class RDI_FileSys {
public:
  // When the provided name does not correspond to a valid file
  // name or directory name, the following exception is raised
  class InvalidName { };
  // If an error occurs during a system call, record the system
  // call and the errno value in the following exception
  class SystemError {
  public:
	SystemError(const char* oper, int erno)
		{ strncpy(_oper, oper, 32); _enum = erno; }
	friend ostream& operator << (ostream& out, const SystemError& e)
		{ return out << e._oper << " error: " << e._enum; }
  protected:
 	char _oper[32];
	int  _enum;
  };
public:
  // Monitoring file and directory changes may need to be persistent
  // and, in addition, structured events may be generated when state
  // changes are detected for the files/directories being monitored.
  // If 'logfile' is not NULL, it is scanned to reconstruct any past
  // monitoring state and record future updates. If 'echan' is NULL,
  // 'cout' is used for reporting state changes

  RDI_FileSys(const char* logfile=0, EventChannel_i* echan=0);
  ~RDI_FileSys();
  
  // The following operations register a new file or directory to be
  // monitored. If an entry already exists, the reference counter of
  // the entry is incremented.  Any of the above exceptions could be
  // raised during registration

  void add_file(const char* name);
  void add_dire(const char* name);

  // Stop monitoring a given file or directory. If an entry for this
  // object exists, its reference counter is decremented.  The entry
  // is deleted when the value of this counter reaches 0

  void remove_file(const char* name);
  void remove_dire(const char* name);

  friend ostream& operator << (ostream& out, const RDI_FileSys& f);

public:
  struct node_t {
	char*        _name;
	unsigned int _refc;	// Ref counter for monitored files
	unsigned int _dirc;	// Ref counter  for monitored dirs
	unsigned int _dirf;	// Set for files of monitored dirs
  	mode_t       _mode;
	uid_t        _usid;
	off_t        _size;
	time_t       _atms;
	time_t       _mtms;
	time_t       _ctms;
	node_t*      _next;

	node_t(const char* n=0) : _name(0), _refc(0), _dirc(0), _dirf(0), 
			_mode(0), _usid(0), _size(0), _atms(0), _mtms(0),
			_ctms(0), _next(0) 
		{ if (n) { _name=new char[strlen(n)+1]; strcpy(_name,n); } }
	~node_t()  { if (_name) delete [] _name; _name=0; _next=0; }
  };

  enum { HASH_SIZE = 256 };
private:
  omni_mutex      _mylock;
  omni_condition  _isover;
  omni_thread*    _thread;
  CORBA::Boolean  _finish;
  unsigned int    _totaln;
  EventChannel_i* _evchan;
  char*           _lgfile;
  node_t*         _htable[HASH_SIZE];

  // Various utility functions that do not require synchrnonization

  node_t* lookup(const char* name);
  int     add_entry(const char* name, CORBA::Boolean dirFile=0, 
		    CORBA::Boolean isDir=0, CORBA::Boolean isNew=0);
  int     rem_entry(const char* name, 
		    CORBA::Boolean dirFile=0, CORBA::Boolean isDir=0);
  int     add_dir_entry(const char* name);
  int     rem_dir_entry(const char* name);
  void    update_dir_entry(const char* name);
  void    establish_state(const char* lfile);
  void    append_log(const char* action, const char* name);
  void    cleanup_log();

  // The main action takes place here. This method is executed by a
  // separate thread.  However, we may have to use multiple threads
  // in the future to keep up with all the entries

  void    monitor_loop();
};

#endif

