// -*- Mode: C++; -*-
//                              File      : Filter_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of Filter_i and MappingFilter_i
//
 
/*
$Log: Filter_i.cc,v $
Revision 1.50  2000/11/16 05:32:14  alcfp
fixes for aCC

Revision 1.49  2000/11/15 21:17:29  alcfp
large number of changes to switch to use of RDIOplocks for safe object disposal support.  also reduced code duplication a little, and tried hard to make all the proxy code consistent

Revision 1.48  2000/10/30 04:40:02  alcfp
extensive changes in preparation for 1.1 release.  will add notes about changes to update.log

Revision 1.47  2000/08/22 18:23:53  alcfp
added description to each file

Revision 1.46  2000/08/16 20:19:30  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>
#include <iostream.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include "RDIUtil.h"
#include "RDIList.h"
#include "RDIDebug.h"
#include "RDIOstreamUtils.h"
#include "CosNotifyChannelAdmin_i.h"
#include "CosNotifyFilter_i.h"

////////////////////////////////////////////////////////////////////

ConstraintImpl* ConstraintImpl::create(const CosNF_ConstraintExp& constraint)
{
  ConstraintImpl* cimpl = new ConstraintImpl();
  RDI_AssertAllocThrowNo(cimpl, "Memory allocation failed for ConstraintImpl");
  if ( strcasecmp(constraint.constraint_expr, "true") == 0 ) {
	cimpl->just_types = 1;
	cimpl->node = 0;
  } else {
	cimpl->node = new RDI_PCState();
	RDI_AssertAllocThrowNo(cimpl->node, "Memory allocation failed for RDI_PCState");
	cimpl->just_types = 0;
	cimpl->node->parse_string(constraint.constraint_expr);
	if ( cimpl->node->e ) {
		RDI_DUMP("Parsing error: " << cimpl->node->b);
		delete cimpl;
		return 0;
	}
	cimpl->node->r_ops->finalize();
  }
  return cimpl;
}

////////////////////////////////////////////////////////////////////

omni_mutex                 Filter_i::_classlock;
unsigned int               Filter_i::_classctr = 0;
Filter_i::RDIFilterKeyMap* Filter_i::_class_keymap =
	 new Filter_i::RDIFilterKeyMap(RDI_ULongHash, RDI_ULongRank,128,20);

Filter_i::Filter_i(const char* grammar) :
	_oplock(), _idcounter(0), _hashvalue(0),
	_constraint_grammar(CORBA_STRING_DUP(grammar)), 
	_constraints(0), _constraint_impls(0),
	_callback_serial(1), _callback_i_serial(1),
	_callbacks(RDI_ULongHash, RDI_ULongRank),
	_callbacks_i(RDI_ULongHash, RDI_ULongRank),
	_flt_dom_ev_types(RDI_EventType::hash, RDI_EventType::rank),
	_flt_all_ev_types(RDI_EventType::hash, RDI_EventType::rank)
{
#ifndef NDEBUG
  _dbgout = 1;
#endif

  _constraints      = new CosNF_ConstraintInfoSeq();
  _constraint_impls = new ConstraintImplSeq();
  RDI_AssertAllocThrowNo(_constraints, "Memory allocation failed - ConstraintInfoSeq");
  RDI_AssertAllocThrowNo(_constraint_impls, "Memory allocation failed - ConstraintImplSeq");
  _constraints->length(0);
  _constraint_impls->length(0);
  // Have to register at this point so that we can get an
  // object reference and let clients interact with the filter ...
  WRAPPED_REGISTER_IMPL(this);
  _classlock.lock();
  _fid = ++_classctr;
#if defined(__OMNIORB3__) || defined(__OMNIORB2__)
  CosNotifyFilter::Filter_var  fltr;
  RDIFilterKeyMapEntry *temp=0, *fent=new RDIFilterKeyMapEntry(this); 
  RDI_AssertAllocThrowNo(fent, "Memory allocation failed - RDIFilterKeyMapEntry");
  fltr = WRAPPED_IMPL2OREF(CosNotifyFilter::Filter, this);
  _hashvalue = fltr->_hash(ULONG_MAX);
  if ( _class_keymap->lookup(_hashvalue, temp) ) {
     fent->next = temp->next;
     temp->next = fent;
  } else {
     _class_keymap->insert(_hashvalue, fent);
  }
#endif
  _classlock.unlock();
}

Filter_i::~Filter_i()
{
  RDI_HashCursor<CosNF_CallbackID, RDINotifySubscribe_ptr> curs;

  RDI_DUMP("Destructor for filter " << (void*)this << " was called");
  // First, remove all constraints.  This will force notifications
  // to be sent and updates to take place at the TypeMap.
  remove_all_constraints();

#if defined(__OMNIORB3__) || defined(__OMNIORB2__)
  // Remove filter from the global hash table used for mapping the
  // CosN_Filter objects to Filter_i objects
  _classlock.lock();

  RDIFilterKeyMapEntry *curr=0, *prev=0;
  if ( _class_keymap->lookup(_hashvalue, curr) ) {
     while ( curr ) {
	CosNF_Filter_var f1 = WRAPPED_IMPL2OREF(CosNF_Filter, curr->iref); 
	CosNF_Filter_var f2 = WRAPPED_IMPL2OREF(CosNF_Filter, this);
	if ( f1->_is_equivalent(f2) ) {
	   if ( prev ) {
	      prev->next = curr->next;
	   } else if ( curr->next ) {
	      _class_keymap->replace(_hashvalue, curr->next);
	   } else {
	      _class_keymap->remove(_hashvalue);
	   }
	   delete curr;
	   break;
	} else {
	   prev = curr;
	   curr = curr->next;
	}
     }
  }
  _classlock.unlock();
#endif

  _oplock.lock();

  // Notify all READY callback subscribers about the destruction 
  for (curs=_callbacks_i.cursor(); curs.is_valid(); ++curs)
     curs.val()->filter_destroy_i(this);

  CORBA_STRING_FREE(_constraint_grammar);
  if ( _constraints )
     delete _constraints;
  if ( _constraint_impls ) {
     for (CORBA::ULong ix=0; ix < _constraint_impls->length(); ix++) {
        delete (*_constraint_impls)[ix];
        (*_constraint_impls)[ix] = 0;
     }
     delete _constraint_impls;
  }
  _constraints = 0;
  _constraint_impls = 0;
  _callbacks.clear();
  _callbacks_i.clear();
  _flt_dom_ev_types.clear();
  _flt_all_ev_types.clear();
  _oplock.unlock();
}

void Filter_i::destroy( WRAPPED_IMPLARG_VOID )
{
  WRAPPED_DISPOSE_IMPL(this);
} 

ostream& operator << (ostream& out, const Filter_i& f)
{
  CosNF_ConstraintInfoSeq& cons     = *f._constraints;
  ConstraintImplSeq&  conimpls = *f._constraint_impls;

  out << "Filter [" << f._fid  << "] #constraints = " << cons.length() << endl;
  for (unsigned int i = 0; i < cons.length(); i++) {
     out << "  Constraint " << setw(5) << cons[i].constraint_id << 
	    " Types " << cons[i].constraint_expression.event_types << endl;
     if ( conimpls[i]->just_types ) {
	out << "\tJUST_TYPES (cexpr: TRUE)" << endl;
     } else {
	out << "\tExpression: " << 
	       (const char*)(cons[i].constraint_expression.constraint_expr) << 
               endl;
     }
  }
  return out << endl;
}

CosNF_ConstraintInfoSeq* 
Filter_i::add_constraints(const CosNF_ConstraintExpSeq& clist  WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq add_types;
  CosN_EventTypeSeq del_types;
  CosN_EventTypeSeq star_star;
  _CORBA_ULong  size = clist.length();
  _CORBA_ULong  base = 0;
  unsigned int  ix = 0;

  CosNF_ConstraintInfoSeq* const_res = new CosNF_ConstraintInfoSeq();
  ConstraintImpl** impl_cseq = new ConstraintImpl* [ size ];

  RDI_AssertAllocThrowNo(const_res, "Memory allocation failed - ConstraintInfoSeq");
  RDI_AssertAllocThrowNo(impl_cseq, "Memory allocation failed - ConstraintImplSeq");

  const_res->length(size);
  add_types.length(0);
  del_types.length(0);
  star_star.length(1);
  star_star[0].domain_name = CORBA_STRING_DUP("*");
  star_star[0].type_name   = CORBA_STRING_DUP("*");

  _oplock.lock();

  // To guarantee atomic modification of the filter object, we first 
  // parse the provided constraints and validate their expressions

  for ( ix = 0; ix < size; ix++ ) {
     if ( ! (impl_cseq[ix] = ConstraintImpl::create(clist[ix])) ) {
	_oplock.unlock();
	delete const_res; 
	delete [] impl_cseq;
	throw (CosNF_InvalidConstraint(clist[ix]));
     }
  }

  // Being here implies that all constraints were valid,  and we can
  // now install them in the filter.  First, we need to increase the
  // sizes of the sequences we use to track the constraints

  base = _constraints->length();
  _constraints->length(base + size);
  _constraint_impls->length(base + size);

  for ( ix = 0; ix < size; ix++ ) {
     (*_constraints)[base+ix].constraint_id = _idcounter;
     (*_constraints)[base+ix].constraint_expression.event_types =
		clist[ix].event_types.length()?clist[ix].event_types:star_star;
     (*_constraints)[base+ix].constraint_expression.constraint_expr =
		CORBA_STRING_DUP(clist[ix].constraint_expr);
     (*_constraint_impls)[base+ix] = impl_cseq[ix];
     _update_ev_tables( (*_constraints)[base+ix].constraint_expression, 
			add_types, del_types);
     (*const_res)[ix].constraint_id = _idcounter++;
     (*const_res)[ix].constraint_expression.event_types = clist[ix].event_types;
     (*const_res)[ix].constraint_expression.constraint_expr = 
		CORBA_STRING_DUP(clist[ix].constraint_expr);
  }

  // Notify all internal entities about any updates in the event types
  // referenced in the constraints used by this filter -- this will in
  // turn update the TypeMap

  if ( add_types.length() || del_types.length() )
     notify_subscribers_i(add_types, del_types);

  _oplock.unlock();
  delete [] impl_cseq;

  return const_res;
}

void Filter_i::modify_constraints(const CosNF_ConstraintIDSeq&   del_list,
				  const CosNF_ConstraintInfoSeq& mod_list  WRAPPED_IMPLARG )
{
  ConstraintImpl** impl_cseq = new ConstraintImpl* [ mod_list.length() ];
  CosN_EventTypeSeq    add_types;
  CosN_EventTypeSeq    del_types;
  CosN_EventTypeSeq    star_star;
  CORBA::ULong     indx,ix;

  RDI_AssertAllocThrowNo(impl_cseq, "Memory allocation failed for ConstraintImpl[]");

  add_types.length(0);
  del_types.length(0);
  star_star.length(1);
  star_star[0].domain_name = CORBA_STRING_DUP("*");
  star_star[0].type_name   = CORBA_STRING_DUP("*");

  _oplock.lock();

  // To achieve the atomicity semantics stated in the specification, 
  // we need to make sure that all constraint IDs referenced in both
  // lists are valid and the constraint expressions are valid too

  for ( ix = 0; ix < del_list.length(); ix++ ) {
     if ( ! _exists_constraint(del_list[ix], indx) ) {
	_oplock.unlock();
	delete [] impl_cseq;
	RDI_DUMP("Invalid contraint ID " << del_list[ix]);
	throw (CosNF_ConstraintNotFound(del_list[ix]));
     }
  }
  for ( ix = 0; ix < mod_list.length(); ix++ ) {
     const CosNF_ConstraintExp& const_expr = mod_list[ix].constraint_expression;
     if ( ! _exists_constraint(mod_list[ix].constraint_id, indx) ) {
	_oplock.unlock();
	delete [] impl_cseq;
	RDI_DUMP("Invalid contraint ID " << mod_list[ix].constraint_id);
	throw (CosNF_ConstraintNotFound(mod_list[ix].constraint_id));
     }
     if ( ! (impl_cseq[ix] = ConstraintImpl::create(const_expr)) ) {
	_oplock.unlock();
	delete [] impl_cseq;
	throw (CosNF_InvalidConstraint(const_expr));
     }
  }

  // Being here implies that the provided arguments are valid. Thus,
  // we can go ahead and update the constrains. First, we remove any
  // constraints specified in 'del_list' and then we update those in
  // 'mod_list'

  for ( ix = 0; ix < del_list.length(); ix++ ) {
     RDI_DUMP("Removing contraint " << del_list[ix]);
     _remove_constraint(del_list[ix], add_types, del_types);
  }

  for ( ix = 0; ix < mod_list.length(); ix++ ) {
     const CosNF_ConstraintExp& cexpr = mod_list[ix].constraint_expression;
     const CosNF_ConstraintID&  cstid = mod_list[ix].constraint_id;

     _remove_constraint(cstid, add_types, del_types);
     indx = _constraints->length();
     _constraints->length(indx + 1);
     (*_constraints)[indx].constraint_id = cstid;
     (*_constraints)[indx].constraint_expression.event_types = 
		cexpr.event_types.length()?cexpr.event_types:star_star;
     (*_constraints)[indx].constraint_expression.constraint_expr =
		CORBA_STRING_DUP(cexpr.constraint_expr);
     _update_ev_tables((*_constraints)[indx].constraint_expression, add_types, del_types);
     _constraint_impls->length(indx + 1);
     (*_constraint_impls)[indx] = impl_cseq[ix];
  }

  // Notify all internal entities about any updates in the event types
  // referenced in the constraints used by this filter -- this will in
  // turn update the TypeMap

  notify_subscribers_i(add_types, del_types);

  _oplock.unlock();
  delete [] impl_cseq;
}

CosNF_ConstraintInfoSeq* Filter_i::get_constraints(const CosNF_ConstraintIDSeq& id_list  WRAPPED_IMPLARG )
{
  CORBA::ULong indx, spos, size=id_list.length();
  CosNF_ConstraintInfoSeq* clst = new CosNF_ConstraintInfoSeq();

  RDI_AssertAllocThrowNo(clst, "Memory allocation failed - CosNF_ConstraintInfoSeq object");
  clst->length(size);
  _oplock.lock();
  for (indx = 0; indx < size; indx++) {
     if ( ! _exists_constraint(id_list[indx], spos) ) {
	_oplock.unlock();
	delete clst;
	throw (CosNF_ConstraintNotFound(id_list[indx]));
     }
     (*clst)[indx].constraint_id = 
		id_list[indx];
     (*clst)[indx].constraint_expression.event_types = 
		(*_constraints)[spos].constraint_expression.event_types;
     (*clst)[indx].constraint_expression.constraint_expr =
		(*_constraints)[spos].constraint_expression.constraint_expr;
  }
  _oplock.unlock();
  return clst;
}

CosNF_ConstraintInfoSeq* Filter_i::get_all_constraints( WRAPPED_IMPLARG_VOID )
{
  CosNF_ConstraintInfoSeq* clst=new CosNF_ConstraintInfoSeq();
  CORBA::ULong  indx, size=0;

  RDI_AssertAllocThrowNo(clst, "Memory allocation failed - CosNF_ConstraintInfoSeq object");
  _oplock.lock();
  size = _constraints->length();
  clst->length(size);

  for ( indx = 0; indx < size; indx++ ) {
     (*clst)[indx].constraint_id = 
		(*_constraints)[indx].constraint_id;
     (*clst)[indx].constraint_expression.event_types = 
		(*_constraints)[indx].constraint_expression.event_types;
     (*clst)[indx].constraint_expression.constraint_expr =
		(*_constraints)[indx].constraint_expression.constraint_expr;
  }
  _oplock.unlock();
  return clst;
}

void Filter_i::remove_all_constraints()
{
  CosNF_ConstraintIDSeq cstridseq;
  CosN_EventTypeSeq    add_types;
  CosN_EventTypeSeq    rem_types;
  CORBA::ULong     indx;
  omni_mutex_lock  lock(_oplock);

  // First, collect all constraint IDs into a list
  cstridseq.length( _constraints->length() );
  for ( indx = 0; indx < _constraints->length(); indx++ )
     cstridseq[indx] = (*_constraints)[indx].constraint_id;

  // Next, iterate over this list and remove the constraints
  add_types.length(0);
  rem_types.length(0);
  for ( indx = 0; indx < cstridseq.length(); indx++ )
     _remove_constraint(cstridseq[indx], add_types, rem_types);

  // Finally, notify subscriber about the event type changes 
  notify_subscribers_i(add_types, rem_types);
}

// We do not perform any lookups in the following method since we 
// assume that it is called once by a given admin or proxy object

CosNF_CallbackID Filter_i::attach_callback(CosNC_NotifySubscribe_ptr callback  WRAPPED_IMPLARG )
{
  CosNF_CallbackID cbkid;
  _oplock.lock();
  cbkid = _callback_serial++;
  _callbacks.insert(cbkid, callback);
  _oplock.unlock();
  return cbkid;
}

void Filter_i::detach_callback(CosNF_CallbackID callbackID  WRAPPED_IMPLARG )
{
  _oplock.lock();
  _callbacks.remove(callbackID);
  _oplock.unlock();
}

CosNF_CallbackIDSeq* Filter_i::get_callbacks( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNF_CallbackID, CosNC_NotifySubscribe_ptr> curs;
  CosNF_CallbackIDSeq* cb_list = new CosNF_CallbackIDSeq();
  CORBA::ULong indx = 0;

  RDI_AssertAllocThrowNo(cb_list, "Memory allocation failed - CosNF_CallbackIDSeq object");
  _oplock.lock();
  cb_list->length(_callbacks.length());
  for ( curs = _callbacks.cursor(); curs.is_valid(); ++curs ) {
     (*cb_list)[indx++] = curs.key();
     indx += 1;
  }
  _oplock.unlock();
 return cb_list;
}

// We do not perform any lookups in the following method since we 
// assume that it is called once by a given admin or proxy object

CosNF_CallbackID Filter_i::attach_callback_i(RDINotifySubscribe_ptr callback) 
{
  RDI_HashCursor<RDI_EventType, void *> curs;
  CosN_EventTypeSeq* add_types = new CosN_EventTypeSeq();
  CosN_EventTypeSeq* del_types = new CosN_EventTypeSeq();
  CosNF_CallbackID    cbkid=0;
  CORBA::ULong   indx=0;

  RDI_AssertAllocThrowNo(add_types, "Memory allocation failed - EventTypeSeq object");
  RDI_AssertAllocThrowNo(del_types, "Memory allocation failed - EventTypeSeq object");

  _oplock.lock();
  cbkid = _callback_i_serial++;
  _callbacks_i.insert(cbkid, callback);
  // We need to invoke the callback with the event types present
  // in the dominating event types list for this filter ........
  add_types->length( _flt_dom_ev_types.length() );
  del_types->length( 0 );
  for (curs = _flt_dom_ev_types.cursor(); curs.is_valid(); ++curs) {
     (*add_types)[indx].domain_name = curs.key().domain_name;
     (*add_types)[indx++].type_name = curs.key().type_name;
  }
  callback->propagate_subscription_change(*add_types, *del_types, this);
  _oplock.unlock();
  delete add_types;
  delete del_types;
  return cbkid;
}

void Filter_i::detach_callback_i(CosNF_CallbackID  callbackID)
{
  omni_mutex_lock lock(_oplock);
  _callbacks_i.remove(callbackID);
}

void Filter_i::create_ev_types_from_dom_list(CosN_EventTypeSeq& lst_seq)
{
  RDI_HashCursor<RDI_EventType, void *> curs;
  CORBA::ULong cntr = 0;
  _oplock.lock();
  lst_seq.length( _flt_dom_ev_types.length() );
  for ( curs = _flt_dom_ev_types.cursor(); curs.is_valid(); ++curs ) {
     lst_seq[cntr].domain_name = CORBA_STRING_DUP(curs.key().domain_name);
     lst_seq[cntr++].type_name = CORBA_STRING_DUP(curs.key().type_name);
  }
  _oplock.unlock();
}

void Filter_i::notify_subscribers_i(const CosN_EventTypeSeq& add_seq, 
				    const CosN_EventTypeSeq& del_seq)
{
  RDI_HashCursor<CosNF_CallbackID, RDINotifySubscribe_ptr> curs1;
  RDI_HashCursor<CosNF_CallbackID, CosNC_NotifySubscribe_ptr>   curs2;
  CosN_EventTypeSeq new_add_seq;
  CosN_EventTypeSeq new_del_seq;
  CORBA::ULong ix, iz, sz;

  new_add_seq.length(0);
  new_del_seq.length(0);

  // The two lists may contain common entries. Before we notify all
  // registered entities about the changes in the subscription,  we
  // clean up the two lists first

  for (ix=0; ix < add_seq.length(); ix++) {
     for (iz=0; iz < del_seq.length(); iz++) {
	if ( (strcmp(add_seq[ix].type_name, del_seq[iz].type_name) == 0) &&
	     (strcmp(add_seq[ix].domain_name, del_seq[iz].domain_name) == 0) ) 
	   break;
     }
     if ( iz == del_seq.length() ) {
	sz = new_add_seq.length();
	new_add_seq.length(sz + 1);
	new_add_seq[sz] = add_seq[ix];
     } 
  }

  for (ix=0; ix < del_seq.length(); ix++) {
     for (iz=0; iz < add_seq.length(); iz++) {
        if ( (strcmp(del_seq[ix].type_name, add_seq[iz].type_name) == 0) &&
             (strcmp(del_seq[ix].domain_name, add_seq[iz].domain_name) == 0) ) 
           break;
     }
     if ( iz == add_seq.length() ) {
        sz = new_del_seq.length();
        new_del_seq.length(sz + 1);
        new_del_seq[sz] = del_seq[ix]; 
     }
  }

  if ( new_add_seq.length() != 0 || new_del_seq.length() != 0 ) {
     for ( curs1 = _callbacks_i.cursor(); curs1.is_valid(); ++curs1) {
	curs1.val()->propagate_subscription_change(new_add_seq, new_del_seq, this);
     }
     for (curs2 = _callbacks.cursor(); curs2.is_valid(); ++curs2) {
	curs2.val()->subscription_change(new_add_seq, new_del_seq);
     }
  }
}

////////////////////////////////////////////////////////////////////

CORBA::Boolean Filter_i::match(const CORBA::Any & event  WRAPPED_IMPLARG )
{
  RDI_DUMP("Not Implemented Yet");
  return 1;
}

CORBA::Boolean Filter_i::match_structured(const CosN_StructuredEvent & event  WRAPPED_IMPLARG )
{
  RDI_StructuredEvent* rev = new RDI_StructuredEvent(event);
  CORBA::Boolean res = 0;
  try { res = rdi_match(rev); }
  catch (...) { throw; }
  delete rev;
  return res;
}

CORBA::Boolean Filter_i::match_typed(const CosN_PropertySeq & event  WRAPPED_IMPLARG )
{
  RDI_DUMP("Not Implemented Yet");
  return 1;
}

CORBA::Boolean Filter_i::rdi_match(RDI_StructuredEvent* se,
				   EventChannel_i* channel) 
{
  unsigned int num=0;
  RDI_RVM*     rvm=0;
  CORBA::Boolean      delrvm=0;

  if ( channel ) channel->incr_num_rdi_match();

  // Lock the event so that only one thread evaluates the event
  // at a time.  This may be too restrictive in some cases and,
  // hence,  we should either use different copies of the event
  // or synchronize the extraction from Any/DynAny only .......

  se->mylock().lock();   

  _oplock.lock();

#ifndef NDEBUG
    RDI_DUMP("rdi_match called for filter " << _fid << " event " << (void*)se);
    if ( _dbgout ) {
      	RDI_DUMP(*this);
    }
    _dbgout=0;
#endif

  // In case the filter does not have any constraints attached
  // to it, we assume that the filter evaluates to TRUE ......

  if ( ! _constraint_impls || ! _constraint_impls->length() ) {
     _oplock.unlock();
     se->mylock().unlock();
     RDI_DUMP("No constraints -- evaluation is TRUE");
     return 1;
  }

  if ( channel ) {
	rvm = channel->get_rvm();
  } else {
	RDI_DUMP("XXX FILTER_I HAS NO CHANNEL, creating new RDI_RVM");
	rvm = new RDI_RVM();
	RDI_AssertAllocThrowNo(rvm, "Memory allocation failed - RDI_RVM object");
	delrvm = 1;
  }

  for ( CORBA::ULong idx = 0; idx < _constraints->length(); idx++ ) {
     CORBA::Boolean isdominated = 0;
     const CosN_EventTypeSeq& types = 
		(*_constraints)[idx].constraint_expression.event_types;
     // Need to check if the event type is dominated by the event
     // types, if any, specified in the constraint

     isdominated = 0;
     for (CORBA::ULong i = 0; i < types.length() ; i++) {
	if ( RDI_EventType::dominates(types[i], se->get_type()) ) {
	   isdominated = 1;
	   break;
	}
     }

     if ( ! isdominated )
	continue;

     if ( (*_constraint_impls)[idx]->just_types ) {
	_oplock.unlock();
	se->mylock().unlock();
	if ( delrvm && rvm ) delete rvm;
	RDI_DUMP("Dominating event type -- evaluation is TRUE");
	return 1;
     }

     while ( num <= 3 ) {
	try {
	   rvm->init((*_constraint_impls)[idx]->node->r_ops);
	   rvm->cexprs = 
		(*_constraints)[idx].constraint_expression.constraint_expr;
	   rvm->eval(se);
	   if ( channel )
	      channel->incr_num_rvm_eval();
	   break;
	} 
	catch (...) {
	   RDI_FDUMP("XXX exception in rvm->init or rvm->eval: " << num);
	   if ( num == 0 ) {
	      CosNF_ConstraintExp& cxpr= (*_constraints)[idx].constraint_expression;
	      RDI_FDUMP("XXX constraint "<<(const char*)(cxpr.constraint_expr));
	   }
	   if ( ++num == 3 ) {
	      rvm->r_code = RDI_RTRet_OK;
	      rvm->r_bool = 0;
	      break;
	  }
	}
     }

     if (rvm->r_code == RDI_RTRet_OK) {
	if ( rvm->r_bool ) {
	   _oplock.unlock();
	   se->mylock().unlock();
	   if ( delrvm && rvm ) delete rvm;
     	   RDI_DUMP("Constraint evaluation -- evaluation is TRUE");
	   return 1;
 	}
     } else {
   	// We interpret non OK value as false, not as an exception 
	// and, thus, we do nothing - we actually return true only 
	// if the r_bool is true AND the return code is OK. In the
	// future we might want to distinguish some of these codes
	// and do something more useful with this information.
    }
  }

  _oplock.unlock();
  se->mylock().unlock();
  if ( delrvm && rvm ) delete rvm;
  return 0;
}

////////////////////////////////////////////////////////////////////

CORBA::Boolean Filter_i::_exists_constraint(const CosNF_ConstraintID& cstid,
					    CORBA::ULong& position) const
{
  for (unsigned int ix = 0; ix < _constraints->length(); ix++) {
	if ( (*_constraints)[ix].constraint_id == cstid ) {
		position = ix;
		return 1;
	}
  }
  return 0;
}

// Update the tables that maintain information about the event types
// which are referenced in the constraints of the filter.  As a side
// effect,  compute information about dominating event types and any
// updates that resulted from the addition of the new constraint ...

void Filter_i::_update_ev_tables(const CosNF_ConstraintExp& cexpr,
				 CosN_EventTypeSeq& add_ev_types,
				 CosN_EventTypeSeq& rem_ev_types)
{
  RDI_HashCursor<RDI_EventType, void *> curs;
  RDI_EventType etype;
  CORBA::ULong  value=0;
  CosN_EventTypeSeq deltypes;

  // We need to keep track of the event types that are removed from
  // the dominating event type table.  We do not use 'rem_ev_types'
  // since this list may not be empty.
  deltypes.length(0);

  for (unsigned int ix=0; ix < cexpr.event_types.length(); ix++) {
	etype = cexpr.event_types[ix];
	// If this event type has been used in a previous constraint,
	// increment its reference counter and continue with next one
	if ( _flt_all_ev_types.lookup(etype, value) ) {
		_flt_all_ev_types.replace(etype, ++value);
		continue;
	}

	// New event type: insert it into the appropriate hash table,
	// and check if it is dominated by an existing event type
	value = 1;
	if ( _flt_all_ev_types.insert(etype, value) ) {
		RDI_Fatal("Failed to upadate event type hash table");
	}
    	if ( _event_is_dominated(etype) )
      		continue;

	CORBA::Boolean dstar = (strcmp(etype.domain_name, "*") == 0) ? 1 : 0;
	CORBA::Boolean tstar = (strcmp(etype.type_name, "*") == 0) ? 1 : 0;

	if ( dstar && tstar ) {		// Case 1: "*::*"
		// All existing entries are removed from the dominating list
		for (curs=_flt_dom_ev_types.cursor(); curs.is_valid(); ++curs)
			_add_ev_type(deltypes, curs.key());
	} else if ( dstar ) {		// Case 2: " *::T"
		// Entries with 'domain_name!=* && type_name==T' are removed
		for (curs=_flt_dom_ev_types.cursor(); curs.is_valid(); ++curs) {
			if ( strcmp(curs.key().domain_name, "*") &&
			     !strcmp(curs.key().type_name, etype.type_name) )
				_add_ev_type(deltypes, curs.key());
		}
	} else if ( tstar ) { 		// Case 3: "D::*"
		// Entries with 'domain_name==D && type_name!=*' are removed
		for (curs=_flt_dom_ev_types.cursor(); curs.is_valid(); ++curs) {
			if ( !strcmp(curs.key().domain_name, etype.domain_name)
			     && strcmp(curs.key().type_name, "*") )
				_add_ev_type(deltypes, curs.key());
		}
	}

	// Remove all event type entries that are present in the list
	// containing the deleted dominating event types and add this
	// event type to the dominating list and the added list
	for ( unsigned int ix=0; ix < deltypes.length(); ix++) {
		_flt_dom_ev_types.remove(deltypes[ix]);
		_add_ev_type(rem_ev_types, deltypes[ix]);
	}
	_flt_dom_ev_types.insert(etype, 0);
	_add_ev_type(add_ev_types, etype);
  }
}

// Remove an existing constraint from the filter. The removal of the
// constraint may result in the update of the dominating event types
// table.  The 'add_ev_types' and  'rem_ev_types' sequences are used
// to record changes that occur at the dominating event types table.

void Filter_i::_remove_constraint(const CosNF_ConstraintID& cstrid,
				  CosN_EventTypeSeq& add_ev_types,
				  CosN_EventTypeSeq& rem_ev_types)
{
  RDI_HashCursor<RDI_EventType, CORBA::ULong> acurs;
  CosN_EventTypeSeq deltypes;
  CosN_EventTypeSeq null_evseq;
  RDI_EventType etype;
  unsigned int  cix, idx;
  CORBA::ULong  rfctr=0;
  CORBA::ULong  numet=0;
  void*         nullv=0;
  CORBA::Boolean dstar=0;
  CORBA::Boolean tstar=0;

  // Locate the entry of the constraint into the constraint list
  for (cix = 0; cix < _constraints->length(); cix++) {
        if ( (*_constraints)[cix].constraint_id == cstrid )
                break;
  }
  RDI_Assert((cix != _constraints->length()), "Corrupted data structures");
  deltypes.length(0);
  null_evseq.length(0);

  // Remove the event types referenced in the removed constraint
  // from '_flt_all_ev_types'.  If the deletion of an event type
  // from this table triggers the removal of the event type from
  // '_flt_dom_ev_types', add the event type to 'deltypes'

  numet = (*_constraints)[cix].constraint_expression.event_types.length();
  for (idx = 0; idx < numet; idx++) {
	etype = (*_constraints)[cix].constraint_expression.event_types[idx];
	if ( ! _flt_all_ev_types.lookup(etype, rfctr) ) {
		RDI_Fatal("Corrupted data structures -- did not find entry");
	}
	if ( --rfctr > 0 ) {
		_flt_all_ev_types.replace(etype, rfctr);
		continue;
	} else {
		_flt_all_ev_types.remove(etype);
		if ( _flt_dom_ev_types.lookup(etype, nullv) ) {
			_flt_dom_ev_types.remove(etype);
			_add_ev_type(deltypes, etype);
		}
	}
  }

  // If the 'deltypes' list is not empty,  we may need to update
  // the event types in '_flt_dom_ev_types'.  Updates take place
  // only when "*::*", "*::T",  or "D::*" belongs to 'deltypes'.
  // NOTE: if "*::*" belongs to 'deltypes', no other type should
  //       exist due to the construction of '_flt_dom_ev_types'.

  for ( idx = 0; idx < deltypes.length(); idx++) {
	etype = deltypes[idx];
	dstar = (strcmp(etype.domain_name, "*") == 0) ? 1 : 0;
	tstar = (strcmp(etype.type_name, "*") == 0) ? 1 : 0;

	if ( dstar && tstar ) {
		// First we will add all "D::*" and "*::T" types
		// and then the dominating "D::T" types
		acurs = _flt_all_ev_types.cursor();
		while ( acurs.is_valid() ) {
			if ( (strcmp(acurs.key().type_name, "*") == 0) ||
			     (strcmp(acurs.key().domain_name, "*") == 0) ) {
				_flt_dom_ev_types.insert(acurs.key(), 0);
				_add_ev_type(add_ev_types, acurs.key());
			}
			acurs++;
		}
		acurs = _flt_all_ev_types.cursor();
		while ( acurs.is_valid() ) {
			if ( strcmp(acurs.key().type_name, "*") &&
			     strcmp(acurs.key().domain_name, "*") &&
			     ! _event_is_dominated(acurs.key()) ) {
				_flt_dom_ev_types.insert(acurs.key(), 0);
				_add_ev_type(add_ev_types, acurs.key());
			}
			acurs++;
		}
	} else if ( dstar ) {
		// Add all "D::type_name" event type entries
		acurs = _flt_all_ev_types.cursor();
		while ( acurs.is_valid() ) {
			if ( !strcmp(acurs.key().type_name,etype.type_name) ) {
				_flt_dom_ev_types.insert(acurs.key(), 0);
				_add_ev_type(add_ev_types, acurs.key());
			}
			acurs++;
		}
	} else if ( tstar ) {
		// Add all "domain_name::T" event type entries
		acurs = _flt_all_ev_types.cursor();
		while ( acurs.is_valid() ) {
			if (!strcmp(acurs.key().domain_name,etype.domain_name)){
				_flt_dom_ev_types.insert(acurs.key(), 0);
                                _add_ev_type(add_ev_types, acurs.key());
			}
			acurs++;
		}
	}

	_add_ev_type(rem_ev_types, etype);
  }

  // Remove the entries that corresponds to the deleted constraint 
  // from the two constraint lists -- if the deleted constraint is
  // not the last entry in these lists, we ``move'' the last entry
  // into the location of the constraint.

  idx = _constraints->length() - 1;
  if ( cix != idx ) {
	(*_constraints)[cix].constraint_id = 
		(*_constraints)[idx].constraint_id;
	(*_constraints)[cix].constraint_expression.event_types =
		(*_constraints)[idx].constraint_expression.event_types;
	(*_constraints)[cix].constraint_expression.constraint_expr =
		(*_constraints)[idx].constraint_expression.constraint_expr;
	delete (*_constraint_impls)[cix];
	(*_constraint_impls)[cix] = (*_constraint_impls)[idx];
	(*_constraint_impls)[idx] = 0;
  } else {
	delete (*_constraint_impls)[cix];
	(*_constraint_impls)[cix] = 0;
  }

  // Null-out the last entry of the constraint sequence to avoid 
  // holding references to unused objects

  (*_constraints)[idx].constraint_expression.event_types = null_evseq;
  (*_constraints)[idx].constraint_expression.constraint_expr = (const char*)"";

  _constraints->length(idx);
  _constraint_impls->length(idx);
}

// Check if the given event type is dominated by at least one entry
// in the dominating event type table '_flt_dom_ev_types'

CORBA::Boolean Filter_i::_event_is_dominated(const CosN_EventType& etype)
{
  RDI_HashCursor<RDI_EventType, void *> curs;
  for (curs = _flt_dom_ev_types.cursor(); curs.is_valid(); ++curs) {
        if ( RDI_EventType::dominates(curs.key(), etype) )
                return 1;
  }
  return 0;
}

void Filter_i::_add_ev_type(CosN_EventTypeSeq& type_seq, 
			    const RDI_EventType& type)
{
  CORBA::ULong indx = type_seq.length();
  type_seq.length(indx + 1);
  type_seq[indx] = type;
}

Filter_i* Filter_i::Filter2Filter_i(CosNF_Filter_ptr f) 
{
#if defined(__OMNIORB3__) || defined(__OMNIORB2__)
  unsigned long hashval = f->_hash(ULONG_MAX);
#else
#error "Filter2Filter_i not defined for the ORB that is in use"
#endif
  RDIFilterKeyMapEntry *curr=0;
  _classlock.lock();
  if ( _class_keymap->lookup(hashval, curr) ) {
     while ( curr ) {
	CosNF_Filter_var fltr = WRAPPED_IMPL2OREF(CosNF_Filter, curr->iref);
	if ( f->_is_equivalent( fltr ) ) {
	   _classlock.unlock();
	   return curr->iref;
	} else {
	   curr = curr->next;
	}
     } 
    _classlock.unlock();
    return 0;
  } else { 	// not found : not a local Filter_i object
    _classlock.unlock();
    return 0;
  }
}

////////////////////////////////////////////////////////////////////

MappingFilter_i::MappingFilter_i(const char* grammar, const CORBA::Any& value) :
	_oplock(), _constraint_grammar(CORBA_STRING_DUP(grammar)),
	_def_value(value), _constraint_serial(0), _constraints(0)
{ 
  _constraints = new CosNF_MappingConstraintInfoSeq();
  RDI_AssertAllocThrowNo(_constraints, "Memory allocation failed - ConstraintInfoSeq");
  _constraints->length(0);
  WRAPPED_REGISTER_IMPL(this);
}

MappingFilter_i::~MappingFilter_i()
{
  RDI_DUMP("Destructor for MappingFilter " << (void*)this << " was called");
  remove_all_mapping_constraints();
  _oplock.lock();
  CORBA_STRING_FREE(_constraint_grammar);
  if ( _constraints )
     delete _constraints;
  _constraints = 0;
  _oplock.unlock();
}

void MappingFilter_i::destroy( WRAPPED_IMPLARG_VOID )
{ 
  WRAPPED_DISPOSE_IMPL(this);
}

CORBA::TypeCode_ptr MappingFilter_i::value_type()
{ 
  return _def_value.type();
}

CORBA::Any* MappingFilter_i::default_value()
{ 
  return new CORBA::Any(_def_value);
}

CosNF_MappingConstraintInfoSeq* MappingFilter_i::add_mapping_constraints(
				const CosNF_MappingConstraintPairSeq & pair_list)
{
  CORBA::ULong size = pair_list.length();
  CORBA::ULong base = 0;
  CosNF_MappingConstraintInfoSeq* const_res = new CosNF_MappingConstraintInfoSeq();

  RDI_AssertAllocThrowNo(const_res, "Memory allocation failed - MappingConstraintInfoSeq");
  const_res->length(size);

  // NOTE: we do not validate the constraints yet since we do not 
  // implement matching in the current release 

  _oplock.lock();
  base = _constraints->length();
  _constraints->length(base + size);

  for (CORBA::ULong ix=0; ix < size; ix++) {
     (*_constraints)[base+ix].constraint_id = _constraint_serial;
     (*_constraints)[base+ix].constraint_expression.event_types =
	pair_list[ix].constraint_expression.event_types;
     (*_constraints)[base+ix].constraint_expression.constraint_expr =
	CORBA::string_dup(pair_list[ix].constraint_expression.constraint_expr);
     (*_constraints)[base+ix].value <<= pair_list[ix].result_to_set;
     (*const_res)[ix].constraint_id = _constraint_serial++;
     (*const_res)[ix].constraint_expression.event_types =
	pair_list[ix].constraint_expression.event_types;
     (*const_res)[ix].constraint_expression.constraint_expr =
	CORBA::string_dup(pair_list[ix].constraint_expression.constraint_expr);
     (*const_res)[ix].value <<= pair_list[ix].result_to_set;
  }
 
  _oplock.unlock();
  return const_res;
}

void MappingFilter_i::modify_mapping_constraints(
				const CosNF_ConstraintIDSeq & del_list,
				const CosNF_MappingConstraintInfoSeq & mod_list  WRAPPED_IMPLARG )
{ 
  CORBA::ULong ix, iz, cs;
  CosN_EventTypeSeq null_evseq;
  null_evseq.length(0);
  _oplock.lock();

  // First delete all constraints referenced in 'del_list'

  for (ix=0; ix < del_list.length(); ix++) {
     cs = _constraints->length();
     for (iz=0; iz < cs; iz++) {
	if ( (*_constraints)[iz].constraint_id == del_list[ix] ) {
	   // Move the last entry in the place of the deleted one
	   (*_constraints)[iz].constraint_id = 
		(*_constraints)[cs-1].constraint_id; 
	   (*_constraints)[iz].constraint_expression.event_types =
		(*_constraints)[cs-1].constraint_expression.event_types;
	   (*_constraints)[iz].constraint_expression.constraint_expr =
		(*_constraints)[cs-1].constraint_expression.constraint_expr;
	   (*_constraints)[iz].value <<= (*_constraints)[cs-1].value;

	   // Null-out the last entry so that reference counters are
	   // decremented appropriately at this time
	   (*_constraints)[cs-1].constraint_id = 0;
	   (*_constraints)[cs-1].constraint_expression.event_types = null_evseq;
	   (*_constraints)[cs-1].constraint_expression.constraint_expr = 
			(const char*) "";
	   (*_constraints)[cs-1].value <<= CORBA::Object::_nil();

	   _constraints->length(cs - 1);
	   break;
	}
     }
     if ( iz == cs ) {
        _oplock.unlock();
        throw CosNF_ConstraintNotFound(del_list[ix]);
     }
  }

  // Next, modify any constraints present in 'mod_list'

  for (ix=0; ix < mod_list.length(); ix++) {
     for (iz=0; iz < _constraints->length(); iz++) {
	if ( (*_constraints)[iz].constraint_id == mod_list[ix].constraint_id ) {
	   (*_constraints)[iz].constraint_expression.event_types =
		mod_list[ix].constraint_expression.event_types;
	   (*_constraints)[iz].constraint_expression.constraint_expr =
		mod_list[ix].constraint_expression.constraint_expr;
	   // NOTE: we should check the validity of the new constraint
	   // and raise an exception if the value is not appropriate..
	   (*_constraints)[iz].value <<= mod_list[ix].value;
	   // NOTE: we should check the validity of the value provided 
	   // and raise an exception if the value is not appropriate..
	   break;
	}
     }
     if ( iz == _constraints->length() ) {
	_oplock.unlock();
	throw CosNF_ConstraintNotFound(mod_list[ix].constraint_id);
     }
  }

  _oplock.unlock();
}

CosNF_MappingConstraintInfoSeq* MappingFilter_i::get_mapping_constraints(
				const CosNF_ConstraintIDSeq & id_list  WRAPPED_IMPLARG )
{ 
  CORBA::ULong size= id_list.length();
  CosNF_MappingConstraintInfoSeq* const_res = new CosNF_MappingConstraintInfoSeq();
  RDI_AssertAllocThrowNo(const_res, "Memory allocation failed - MappingConstraintInfoSeq");
  const_res->length(size);

  _oplock.lock();
  for (CORBA::ULong ix=0; ix < size; ix++) {
     CORBA::ULong iz;
     for (iz=0; iz < _constraints->length(); iz++) {
   	if ( (*_constraints)[iz].constraint_id == id_list[ix] ) {
	   (*const_res)[ix].constraint_id = (*_constraints)[iz].constraint_id;
	   (*const_res)[ix].constraint_expression.event_types =
		(*_constraints)[iz].constraint_expression.event_types;
	   (*const_res)[ix].constraint_expression.constraint_expr =
		(*_constraints)[iz].constraint_expression.constraint_expr;
	   (*const_res)[ix].value = (*_constraints)[iz].value;
	   break;
	}
     }
     if ( iz == _constraints->length() ) {
     	_oplock.unlock();
     	delete const_res;
     	throw CosNF_ConstraintNotFound( id_list[ix] );
     }
  }
  _oplock.unlock();
  return const_res;
}

CosNF_MappingConstraintInfoSeq* MappingFilter_i::get_all_mapping_constraints( WRAPPED_IMPLARG_VOID )
{ 
  CosNF_MappingConstraintInfoSeq* const_res = new CosNF_MappingConstraintInfoSeq();
  RDI_AssertAllocThrowNo(const_res, "Memory allocation failed - MappingConstraintInfoSeq");
  _oplock.lock();
  CORBA::ULong size = _constraints->length();
  const_res->length(size);
  for (CORBA::ULong ix=0; ix < size; ix++) {
     (*const_res)[ix].constraint_id = (*_constraints)[ix].constraint_id;
     (*const_res)[ix].constraint_expression.event_types =
	(*_constraints)[ix].constraint_expression.event_types;
     (*const_res)[ix].constraint_expression.constraint_expr =
	(*_constraints)[ix].constraint_expression.constraint_expr;
     (*const_res)[ix].value = (*_constraints)[ix].value;
  }
  _oplock.unlock();
  return const_res;
}

void MappingFilter_i::remove_all_mapping_constraints()
{
  _oplock.lock();
  delete _constraints;	// This should clean up all entries
  _constraints = new CosNF_MappingConstraintInfoSeq();
  _constraints->length(0);
  _oplock.unlock();
}

CORBA::Boolean MappingFilter_i::match(const CORBA::Any & event, 
			      WRAPPED_OUTARG_TYPE(CORBA::Any) result_to_set  WRAPPED_IMPLARG )
{ return 0; }

CORBA::Boolean MappingFilter_i::match_structured(
				const CosN_StructuredEvent & event,
				WRAPPED_OUTARG_TYPE(CORBA::Any) result_to_set  WRAPPED_IMPLARG )
{ return 0; }

CORBA::Boolean MappingFilter_i::match_typed(const CosN_PropertySeq & event,
			    WRAPPED_OUTARG_TYPE(CORBA::Any) result_to_set  WRAPPED_IMPLARG )
{ return 0; }

