// -*- Mode: C++; -*-
//                              File      : RDIStringDefs.h
//                               (formerly RDIDefs.h)
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2001 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIStringDefs.h,v $
Revision 1.8  2001/10/06 03:19:41  alcfp
minor fix

Revision 1.7  2001/10/02 14:07:50  alcfp
improved error messages for command line errors

Revision 1.6  2001/09/27 17:20:04  alcfp
better error messages for prop combo problem

Revision 1.5  2001/09/05 18:02:43  alcfp
more interactive commands working

Revision 1.4  2001/08/03 17:54:15  alcfp
added support for AttNotification

Revision 1.3  2001/06/26 20:01:10  alcfp
updated copyright notices, added support for omniORB4, switched default to POA

Revision 1.2  2001/06/25 20:35:18  alcfp
all mention of omniORB should be in corba_wrappers

Revision 1.1  2001/06/22 07:00:29  alcfp
moved to new logging scheme

*/
 
#ifndef __RDI_STRING_DEFS_H__
#define __RDI_STRING_DEFS_H__

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "corba_wrappers.h"

//-------------------------------------------------------------------------
// RDI_DELNULL : only invoke on reference to obj allocated using new
#define RDI_DELNULL(x)      if (x) {delete x; x=0;}
#define RDI_DELNULL2(x,y)   RDI_DELNULL(x) RDI_DELNULL(y)
#define RDI_DELNULL3(x,y,z) RDI_DELNULL(x) RDI_DELNULL(y) RDI_DELNULL(z)

//-------------------------------------------------------------------------
// STRING MANIPULATION FUNCTIONS / UTILS

#define RDI_STRLEN(s)             ::strlen(s)
#define RDI_STRCHR(s, c)          ::strchr(s, c)
#define RDI_STR_RANK(s1,s2)       ::strcmp(s1,s2)

#define RDI_STRCPY(dst, src)      ::strcpy(dst, src)
#define RDI_STRNCPY(dst, src, n)  ::strncpy(dst, src, n)

#define RDI_STRTOL(ptr, endptr_out)        ::strtol(ptr, endptr_out, 0)
#define RDI_STRTOUL(ptr, endptr_out)       ::strtoul(ptr, endptr_out, 0)
#define RDI_STRTOD(ptr, endptr_out)        ::strtod(ptr, endptr_out)

#define RDI_STR_EQ(s1,s2)  (RDI_STR_RANK(s1,s2)==0)
#define RDI_STR_NEQ(s1,s2) (RDI_STR_RANK(s1,s2)!=0)
#define RDI_STR_LT(s1,s2)  (RDI_STR_RANK(s1,s2)<0)
#define RDI_STR_LE(s1,s2)  (RDI_STR_RANK(s1,s2)<=0)
#define RDI_STR_GT(s1,s2)  (RDI_STR_RANK(s1,s2)>0)
#define RDI_STR_GE(s1,s2)  (RDI_STR_RANK(s1,s2)>=0)

#define RDI_STRN_EQ(s1,s2,n)     (::strncmp(s1,s2,n)==0)
#define RDI_STRN_NEQ(s1,s2,n)    (::strncmp(s1,s2,n)!=0)

#define RDI_IS_ALPHA_OR_UNDERSCORE(c)     ((c == '_') || isalpha(c))
#define RDI_IS_ALPHANUM_OR_UNDERSCORE(c)  ((c == '_') || isalnum(c))
#define RDI_IS_SPACE(c)                   isspace(c)

// compare, ignoring case
#if defined(_MSC_VER)
#  define RDI_STR_EQ_I(s1,s2)    (_stricmp(s1,s2)==0)
#  define RDI_STRN_EQ_I(s1,s2,n) (_strnicmp(s1,s2,n)==0) // ???
#else
#  define RDI_STR_EQ_I(s1,s2)    (strcasecmp(s1,s2)==0)
#  define RDI_STRN_EQ_I(s1,s2,n) (strncasecmp(s1,s2,n)==0)
#endif

// RDIParseCmd: constructor parses cmd, fills in members argc and argv.
// argv is a newly allocated array that is freed when the object is deleted.
// (cmd is not modified)
//
// Example:
//     do {
//       RDIParseCmd p(cmd);
//       ... use p.argc and p.argv ...
//     } while (0);
// ** p's space is freed up, including p.argv

#define RDI_PARSECMD_MAXARGS 64  // our commands are very simple; leave room anyway

class RDIParseCmd {
public:
  int argc;
  char** argv;
  RDIParseCmd(const char* cmd) {
    argc = 0;
    argv = new char*[RDI_PARSECMD_MAXARGS];
    for (unsigned int i = 0; i < RDI_PARSECMD_MAXARGS; i++) {
      argv[i] = 0;
    }
    char* argloc = (char*)cmd;
    char* cmdend = argloc + strlen(cmd);
    while (1) {
      if (argloc >= cmdend) return; // done
      while (RDI_IS_SPACE(*argloc)) {
	argloc++; // skip leading white space
	if (argloc >= cmdend) return; // done
      }
      // found something
      char* sploc = argloc + 1;
      int len = 1;
      while ((sploc < cmdend) && (! RDI_IS_SPACE(*sploc))) {
	sploc++; // skip the arg
	len++;
      }
      char* buf = new char[len+1];
      RDI_STRNCPY(buf, argloc, len);
      buf[len] = '\0';
      argloc = sploc + 1;
      argv[argc++] = buf;
      // the following test uses MAXARGS-1 to leave space for final NULL in argv
      if (argc == (RDI_PARSECMD_MAXARGS-1)) return; // done
    }
  }
  ~RDIParseCmd() {
    for (unsigned int i = 0; i < RDI_PARSECMD_MAXARGS; i++) {
      if (argv[i]) {
	delete [] argv[i];
	argv[i] = 0;
      }
    }
    delete [] argv;
    argv = 0;
  }
};

// removes arg at index, modifies argc
inline void RDI_rm_arg(int& argc, char** argv, int& index) {
  if (index >= argc) return;
  for (int i = index; i+1 < argc; i++) {
    argv[i] = argv[i+1];
  }
  argc--;
}

//-------------------------------------------------------------------------
// CORBA-specific cases

#define CORBA_STRING_ALLOC(sz) CORBA::string_alloc(sz)
#define CORBA_STRING_DUP(str)  CORBA::string_dup(str)
#define CORBA_STRING_FREE(str) CORBA::string_free(str)

#define RDI_STRDELNULL(x)      if (x) {CORBA_STRING_FREE(x); x=0;}
#define RDI_STRDELNULL2(x,y)   RDI_STRDELNULL(x) RDI_STRDELNULL(y)
#define RDI_STRDELNULL3(x,y,z) RDI_STRDELNULL(x) RDI_STRDELNULL(y) RDI_STRDELNULL(z)

inline char*
RDI_BUILD_STRING(const char* s1) { return CORBA_STRING_DUP(s1); }

inline char*
RDI_BUILD_STRING(const char* s1, 
		 const char* s2,
		 const char* s3 = "",
		 const char* s4 = "",
		 const char* s5 = "",
		 const char* s6 = "",
		 const char* s7 = "",
		 const char* s8 = "") {
  int len = RDI_STRLEN(s1) + RDI_STRLEN(s2) + RDI_STRLEN(s3) + RDI_STRLEN(s4) + 
    RDI_STRLEN(s5) + RDI_STRLEN(s6) + RDI_STRLEN(s7) + RDI_STRLEN(s8);
  char* result = CORBA_STRING_ALLOC(len);
  sprintf(result, "%s%s%s%s%s%s%s%s", s1, s2, s3, s4, s5, s6, s7, s8);
  return(result);
}

// RDIParseId: returns -1 on error.
//   A # >= zero means buf contains an identifier made up of the
//   expected prefix and a non-neg number, with no extraneous characters.

inline CORBA::Long RDIParseId(const char* prefix, char* buf) {
  if (RDI_STRN_EQ_I(prefix, buf, RDI_STRLEN(prefix))) {
    char* delim = 0;
    char* strid = buf + RDI_STRLEN(prefix);
    CORBA::Long id = (CORBA::Long)RDI_STRTOL(strid, &delim);
    if ( (delim == strid) || (*delim != '\0') ) {
      return -1;
    }
    return id;
  }
  return -1;
}


#endif  /* __RDI_STRING_DEFS_H__ */



